local helpers = require('test.functional.helpers')(after_each)

local clear = helpers.clear
local exec_lua = helpers.exec_lua
local eq = helpers.eq
local mkdir_p = helpers.mkdir_p
local rmdir = helpers.rmdir
local nvim_dir = helpers.nvim_dir
local test_build_dir = helpers.test_build_dir
local nvim_prog = helpers.nvim_prog
local is_os = helpers.is_os

local nvim_prog_basename = is_os('win') and 'nvim.exe' or 'nvim'

local test_basename_dirname_eq = {
  '~/foo/',
  '~/foo',
  '~/foo/bar.lua',
  'foo.lua',
  ' ',
  '',
  '.',
  '..',
  '../',
  '~',
  '/usr/bin',
  '/usr/bin/gcc',
  '/',
  '/usr/',
  '/usr',
  'c:/usr',
  'c:/',
  'c:',
  'c:/users/foo',
  'c:/users/foo/bar.lua',
  'c:/users/foo/bar/../',
}

local tests_windows_paths = {
  'c:\\usr',
  'c:\\',
  'c:',
  'c:\\users\\foo',
  'c:\\users\\foo\\bar.lua',
  'c:\\users\\foo\\bar\\..\\',
}

before_each(clear)

describe('vim.fs', function()
  describe('parents()', function()
    it('works', function()
      local test_dir = nvim_dir .. '/test'
      mkdir_p(test_dir)
      local dirs = exec_lua([[
        local test_dir, test_build_dir = ...
        local dirs = {}
        for dir in vim.fs.parents(test_dir .. "/foo.txt") do
          dirs[#dirs + 1] = dir
          if dir == test_build_dir then
            break
          end
        end
        return dirs
      ]], test_dir, test_build_dir)
      eq({test_dir, nvim_dir, test_build_dir}, dirs)
      rmdir(test_dir)
    end)
  end)

  describe('dirname()', function()
    it('works', function()
      eq(test_build_dir, exec_lua([[
        local nvim_dir = ...
        return vim.fs.dirname(nvim_dir)
      ]], nvim_dir))

      local function test_paths(paths)
        for _, path in ipairs(paths) do
          eq(
            exec_lua([[
              local path = ...
              return vim.fn.fnamemodify(path,':h'):gsub('\\', '/')
            ]], path),
            exec_lua([[
              local path = ...
              return vim.fs.dirname(path)
            ]], path),
            path
          )
        end
      end

      test_paths(test_basename_dirname_eq)
      if is_os('win') then
        test_paths(tests_windows_paths)
      end
    end)
  end)

  describe('basename()', function()
    it('works', function()

      eq(nvim_prog_basename, exec_lua([[
        local nvim_prog = ...
        return vim.fs.basename(nvim_prog)
      ]], nvim_prog))

      local function test_paths(paths)
        for _, path in ipairs(paths) do
          eq(
            exec_lua([[
              local path = ...
              return vim.fn.fnamemodify(path,':t'):gsub('\\', '/')
            ]], path),
            exec_lua([[
              local path = ...
              return vim.fs.basename(path)
            ]], path),
            path
          )
        end
      end

      test_paths(test_basename_dirname_eq)
      if is_os('win') then
        test_paths(tests_windows_paths)
      end
    end)
  end)

  describe('dir()', function()
    it('works', function()
      eq(true, exec_lua([[
        local dir, nvim = ...
        for name, type in vim.fs.dir(dir) do
          if name == nvim and type == 'file' then
            return true
          end
        end
        return false
      ]], nvim_dir, nvim_prog_basename))
    end)

    it('works with opts.depth and opts.skip', function()
      helpers.funcs.system 'mkdir -p testd/a/b/c'
      helpers.funcs.system('touch '..table.concat({
        'testd/a1',
        'testd/b1',
        'testd/c1',
        'testd/a/a2',
        'testd/a/b2',
        'testd/a/c2',
        'testd/a/b/a3',
        'testd/a/b/b3',
        'testd/a/b/c3',
        'testd/a/b/c/a4',
        'testd/a/b/c/b4',
        'testd/a/b/c/c4',
      }, ' '))

      local function run(dir, depth, skip)
         local r = exec_lua([[
          local dir, depth, skip = ...
          local r = {}
          local skip_f
          if skip then
            skip_f = function(n)
              if vim.tbl_contains(skip or {}, n) then
                return false
              end
            end
          end
          for name, type_ in vim.fs.dir(dir, { depth = depth, skip = skip_f }) do
            r[name] = type_
          end
          return r
        ]], dir, depth, skip)
        return r
      end

      local exp = {}

      exp['a1'] = 'file'
      exp['b1'] = 'file'
      exp['c1'] = 'file'
      exp['a'] = 'directory'

      eq(exp, run('testd', 1))

      exp['a/a2'] = 'file'
      exp['a/b2'] = 'file'
      exp['a/c2'] = 'file'
      exp['a/b'] = 'directory'

      eq(exp, run('testd', 2))

      exp['a/b/a3'] = 'file'
      exp['a/b/b3'] = 'file'
      exp['a/b/c3'] = 'file'
      exp['a/b/c'] = 'directory'

      eq(exp, run('testd', 3))
      eq(exp, run('testd', 999, {'a/b/c'}))

      exp['a/b/c/a4'] = 'file'
      exp['a/b/c/b4'] = 'file'
      exp['a/b/c/c4'] = 'file'

      eq(exp, run('testd', 999))
    end)
  end)

  describe('find()', function()
    it('works', function()
      eq({test_build_dir}, exec_lua([[
        local dir = ...
        return vim.fs.find('build', { path = dir, upward = true, type = 'directory' })
      ]], nvim_dir))
      eq({nvim_prog}, exec_lua([[
        local dir, nvim = ...
        return vim.fs.find(nvim, { path = dir, type = 'file' })
      ]], test_build_dir, nvim_prog_basename))
    end)

    it('accepts predicate as names', function()
      eq({test_build_dir}, exec_lua([[
        local dir = ...
        local opts = { path = dir, upward = true, type = 'directory' }
        return vim.fs.find(function(x) return x == 'build' end, opts)
      ]], nvim_dir))
      eq({nvim_prog}, exec_lua([[
        local dir, nvim = ...
        return vim.fs.find(function(x) return x == nvim end, { path = dir, type = 'file' })
      ]], test_build_dir, nvim_prog_basename))
      eq({}, exec_lua([[
        local dir = ...
        local opts = { path = dir, upward = true, type = 'directory' }
        return vim.fs.find(function(x) return x == 'no-match' end, opts)
      ]], nvim_dir))
    end)
  end)

  describe('normalize()', function()
    it('works with backward slashes', function()
      eq('C:/Users/jdoe', exec_lua [[ return vim.fs.normalize('C:\\Users\\jdoe') ]])
    end)
    it('works with ~', function()
      if is_os('win') then
        pending([[$HOME does not exist on Windows ¯\_(ツ)_/¯]])
      end
      eq(os.getenv('HOME') .. '/src/foo', exec_lua [[ return vim.fs.normalize('~/src/foo') ]])
    end)
    it('works with environment variables', function()
      local xdg_config_home = test_build_dir .. '/.config'
      eq(xdg_config_home .. '/nvim', exec_lua([[
        vim.env.XDG_CONFIG_HOME = ...
        return vim.fs.normalize('$XDG_CONFIG_HOME/nvim')
      ]], xdg_config_home))
    end)
  end)
end)
