/*
 * Copyright (c) 1998 Regents of The University of Michigan.
 * All Rights Reserved.  See COPYRIGHT.
 */

/**********          depmap.c          **********/

#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <netinet/in.h>

#include <pwd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

#include "nefu.h"
#include "ll.h"


    void
stdout_levels( struct test *t_node, int level )
{
    int			tmp;

    if ( t_node != NULL ) {
	for ( tmp = 0; tmp < level; tmp++ ) {
	    printf( "    " );
	}

	printf( "%s : %s", t_node->t_machine->m_name, t_node->t_full_name );

	if ( t_node->t_rlist != NULL ) {
	    printf( " : %s\n", t_node->t_rlist );
	} else {
	    printf( "\n" );
	}

	stdout_levels( t_node->t_child, level + 1 );
	stdout_levels( t_node->t_peer, level );
    }
}


    /* Prints a text representation of the tree formed by a given config
     * file.  Usefull for making sure the data structures are being built
     * and arranged properly.
     */

    void
depmap_stdout( void )
{
    printf( "Network map:\n" );

    if ( root_nodes == NULL ) {
	printf( "There are no root nodes!\n" );
    } else {
	stdout_levels( root_nodes, 0 );
    }
    fflush( stdout );
}


    /* prints the dependancy tree, all recursive style */

    void
html_machine_path_to_root( FILE *f, struct test *t )
{
    if ( t == NULL ) {
	/* print root link */
	fprintf( f, "<A HREF=\"../%s\">%s</A> >> ", HTML_DEPMAP_ROOT,
		nefu_uname );

    } else {
	/* recursion */
	html_machine_path_to_root( f, t->t_parent );

	/* print our name as a link */
	fprintf( f, "<A HREF=\"%s.html\">%s</A> >> ", t->t_machine->m_name,
		t->t_machine->m_name );
    }
}

    
    /* html out: t's ping's args, and all of t's service level tests */

    void
html_machine_services( FILE *f, struct test *t, int directory_levels )
{
    struct stab_entry	*st;
    struct rcode	*r;
    int			x;

    /* first row has rcodes */
    fprintf( f, "<TABLE BORDER><TR><TD></TD>\n" );
    for ( st = t->t_machine->m_rcodes; st != NULL; st = st->st_next ) {
	r = (struct rcode*)st->st_data;
	fprintf( f, "<TD><A HREF=\"" );
	for ( x = 0; x < directory_levels; x++ ) {
	    fprintf( f, "../" );
	}
	fprintf( f, "%s\">%s</A></TD>\n", r->r_html_name, r->r_code );
    }
    fprintf( f, "</TR><TR>\n" );

    /* next row is ping */
    fprintf( f, "<TD>%s</TD>\n", t->t_full_name );

    for ( st = t->t_machine->m_rcodes; st != NULL; st = st->st_next ) {
	if ( ll_lookup( t->t_rcodes, st->st_key ) != NULL ) {
	    fprintf( f, "<TD><CENTER>*</CENTER></TD>\n" );
	} else {
	    fprintf( f, "<TD></TD>\n" );
	}
    }

    for ( t = t->t_child; t != NULL; t = t->t_peer ) {
	if ( strcmp( t->t_name, T_DEFAULT ) == 0 ) {
	    continue;
	}

	fprintf( f, "<TR><TD>%s</TD>\n", t->t_full_name );

	for ( st = t->t_machine->m_rcodes; st != NULL; st = st->st_next ) {
	    if (( r = (struct rcode*)ll_lookup( t->t_rcodes, st->st_key ))
		    != NULL ) {
		fprintf( f, "<TD><CENTER>*</CENTER></TD>\n" );
	    } else {
		fprintf( f, "<TD></TD>\n" );
	    }
	}

	fprintf( f, "</TR>\n" );
    }

    fprintf( f, "</TR>\n</TABLE>" );
}


    /* html out: t's dependants, directory style */

    void
html_machine_dependants( FILE *f, struct test *t, int directory_levels )
{
    for ( ; t != NULL; t = t->t_peer ) {
	if ( strcmp( t->t_name, T_DEFAULT ) != 0 ) {
	    continue;
	}

	fprintf( f, "<TR>\n<TD><B>" );
	if ( t->t_router < 1 ) {
	    /* print flat name */
	    fprintf( f, "%s ", t->t_machine->m_name );
	} else {
	    /* print a link */
	    if ( t->t_parent == NULL ) {
		fprintf( f, "<A HREF=\"%s/%s.html\">%s</A> ", HTML_DEPMAP_DIR,
			t->t_machine->m_name, t->t_machine->m_name );

	    } else { 
		fprintf( f, "<A HREF=\"%s.html\">%s</A> ",
			t->t_machine->m_name, t->t_machine->m_name );
	    }
	}

	fprintf( f, "</B></TD><TD>\n" );

	html_machine_services( f, t, directory_levels );

	fprintf( f, "</TD></TR>\n" );

    }
}


    void
html_machine_index( void )
{
    char		html[ MAXPATHLEN ];
    int			fd;
    FILE		*f;
    struct stab_entry	*st;
    struct machine	*m;

    /* index page */
    sprintf( html, "%s/%s", nefu_html_dir, HTML_MACHINES );

    if (( fd = creat( html, 0666 )) < 0 ) {
	fprintf( stderr, "open: %s: ", html );
	perror( NULL );
	exit( 1 );
    }

    if (( f = fdopen( fd, "w" )) == NULL ) {
	perror( "fdopen" );
	exit( 1 );
    }

    /* print header */
    if ( html_header( ERROR_STDERR, f, 0, "machines" ) != 0 ) {
	exit( 1 );
    }

    for ( st = machine_stab; st != NULL; st = st->st_next ) {
	m = (struct machine*)st->st_data;
	fprintf( f, "<A HREF=\"%s\">%s</A><BR>\n", m->m_html_name, m->m_name );
    }

    /* print footer */
    if ( html_footer( ERROR_STDERR, f, 0 ) != 0 ) {
	exit( 1 );
    }

    /* done with machine file */
    if ( fclose( f ) != 0 ) {
	perror( "fclose" );
	exit( 1 );
    }
}


    /* a machine gets its own page if it acts as a router.
     * a router is a machine that has any other machine as its child.
     */

    void
html_machine( void *data )
{
    char		html[ MAXPATHLEN ];
    int			fd;
    FILE		*f;
    struct machine	*m;

    m = (struct machine*)data;

    /* first note our html page name */
    if ( m->m_test->t_parent != NULL ) {
	/* link to our parent's page */
	sprintf( html, "%s/%s.html", HTML_DEPMAP_DIR,
		m->m_test->t_parent->t_machine->m_name );

    } else {
	/* link to the root page */
	sprintf( html, "%s", HTML_DEPMAP_ROOT );
    }

    if (( m->m_html_name = strdup( html )) == NULL ) {
	perror( "strdup" );
	exit( 1 );
    }

    /* if we don't route, we don't need a page */
    if ( m->m_test->t_router < 1 ) {
	return;
    }

    /* here's our page */
    sprintf( html, "%s/%s/%s.html", nefu_html_dir, HTML_DEPMAP_DIR,
	    m->m_name );

    if (( fd = open( html, O_WRONLY|O_CREAT|O_EXCL, 0666 )) < 0 ) {
	fprintf( stderr, "open: %s: ", html );
	perror( NULL );
	exit( 1 );
    }

    if (( f = fdopen( fd, "w" )) == NULL ) {
	perror( "fdopen" );
	exit( 1 );
    }


    /* print header */
    if ( html_header( ERROR_STDERR, f, 1, "dependencies" ) != 0 ) {
	exit( 1 );
    }

    fprintf( f, "<TABLE><TR><TD>\n" );

    /* path to root */
    html_machine_path_to_root( f, m->m_test->t_parent );

    /* print machine's name, plaintext */
    fprintf( f, "<TH>%s</TH>\n</TD><TD>", m->m_name );

    html_machine_services( f, m->m_test, 1 );

    /* comment */
    fprintf( f, "</TD></TR></TABLE>\n<HR NOSHADE>\n<TABLE>" );

    /* print our dependants */
    html_machine_dependants( f, m->m_test->t_child, 1 );

    /* comment */
    fprintf( f, "</TABLE>\n" );

    /* print footer */
    if ( html_footer( ERROR_STDERR, f, 1 ) != 0 ) {
	exit( 1 );
    }

    /* done with machine file */
    if ( fclose( f ) != 0 ) {
	perror( "fclose" );
	exit( 1 );
    }
}


    /* create a html dependancy map */

    void
depmap_html( void )
{
    char		name[ MAXPATHLEN ];
    int			fd;
    FILE		*f;

    if ( nefu_html_dir == NULL ) {
	return;
    }

    sprintf( name, "%s/%s", nefu_html_dir, HTML_DEPMAP_DIR );
    if ( directory( name ) != 0 ) {
	exit( 1 );
    }

    /* make the root's dependancy file */
    sprintf( name, "%s/%s", nefu_html_dir, HTML_DEPMAP_ROOT );

    if (( fd = creat( name, 0666 )) < 0 ) {
	fprintf( stderr, "creat: %s: ", name );
	perror( NULL );
	exit( 1 );
    }

    if (( f = fdopen( fd, "w" )) == NULL ) {
	perror( "fdopen" );
	exit( 1 );
    }

    /* print header */
    if ( html_header( ERROR_STDERR, f, 0, "dependencies" ) != 0 ) {
	exit( 1 );
    }

    /* print machine's name, plaintext */
    fprintf( f, "<TABLE><TR><TH>%s</TH></TR></TABLE>\n", nefu_uname );
    fprintf( f, "<HR NOSHADE>\n<TABLE>" );

    /* print out our dependants */
    html_machine_dependants( f, root_nodes, 0 );

    /* comment */
    fprintf( f, "</TABLE>\n" );

    /* print footer */
    if ( html_footer( ERROR_STDERR, f, 0 ) != 0 ) {
	exit( 1 );
    }

    /* done with root file */
    if ( fclose( f ) != 0 ) {
	fprintf( stderr, "fclose: %s: ", name );
	perror( NULL );
	exit( 1 );
    }

    /* build machine db */
    ll_walk( machine_stab, html_machine );

    /* build machine db index */
    html_machine_index();
}
