#include "nds.hh"

#include <memory>
#include <cstdlib>
#include <cassert>
#include <iostream>
#include <sstream>
#include <string>

#include "test_macros.hh"
#include "nds_testing.hh"

int
main(int argc, char *argv[])
{
  using NDS::connection;

  std::vector<std::string> args = convert_args( argc, argv );

  //---------------------------------------------------------------------
  // Obtain the port of the server
  //---------------------------------------------------------------------
  connection::port_type port = 31200;
  if ( ::getenv("NDS_TEST_PORT") )
  {
    std::istringstream  ps( ::getenv("NDS_TEST_PORT") );

    ps >> port;
  }

  std::string hostname("localhost");
  if ( ::getenv("NDS_TEST_HOST") )
  {
    hostname = ::getenv( "NDS_TEST_HOST" );
  }

  connection::protocol_type proto = connection::PROTOCOL_TRY;
  if ( find_opt( "-proto-1", args ) )
  {
    proto = connection::PROTOCOL_ONE;
  }
  else if ( find_opt( "-proto-2", args ) )
  {
    proto = connection::PROTOCOL_TWO;
  }
  else if ( ::getenv("NDS_TEST_PROTO") )
  {
    std::string ps( ::getenv("NDS_TEST_PROTO") );
    if ( ps.compare("1") == 0 ) proto = connection::PROTOCOL_ONE;
    else if ( ps.compare("2") == 0 ) proto = connection::PROTOCOL_TWO;
  }

  bool fast_iterate = false;
  if ( ::getenv("NDS_FAST_ITERATE")
       || ( find_opt( "-fast", args ) ) )
  {
    fast_iterate = true;
  }

  pointer< connection > conn( new connection( hostname, port, proto ));

  connection::channel_names_type cn;
  cn.push_back("H1:ISI-BS_ST1_SENSCOR_GND_STS_X_BLRMS_30M_100M.mean,s-trend");

  // This channel has an availability with a gap
  // H-H1_T:1115756400-1116285600 H-H1_T:1116286200-1118538000
  NDS::buffer::gps_second_type start = 1116286100, end = start + 200, stride = 50;
  NDS::buffer::gps_second_type data_start = 1116286200;
  NDS::buffer::gps_second_type last_start = 0;
  int samples_per_segment = stride;

  if (proto == connection::PROTOCOL_ONE)
  {
    start = 1130797740;
    stride = 3000;
    end = start + stride*3;
    samples_per_segment = stride/60;

    cn.clear();
    cn.push_back("X1:CDS-DACTEST_COSINE_OUT_DQ.n,m-trend");
  }

  // --------------------------------------------------------------------

  conn->set_parameter("ITERATE_USE_GAP_HANDLERS", "false");
  conn->iterate(start, end, cn);

  NDS::buffers_type  bufs;
  if (fast_iterate)
    conn->next( bufs );
  else
    bufs = conn->next();

  std::cout << "Retreived a buffer " << bufs[0]->Start() << "-" << bufs[0]->Stop() << std::endl;
  std::cout << " sample count - " << bufs[0]->Samples();
  std::cout << std::endl << std::endl;

  // --------------------------------------------------------------------

  conn->set_parameter("ITERATE_USE_GAP_HANDLERS", "false");
  conn->iterate(start, end, cn);

  int count = 0;
  try
  {
    while( true )
    {
      std::cout << "iteration: " << count << std::endl;
      if (fast_iterate)
        conn->next( bufs );
      else
        bufs = conn->next();
      ++count;
    }
  }
  catch( const std::out_of_range& /* Error */)
  {
    // Reached end of buffers so just do nothing
  }

  std::cout << "Retreived a buffer " << bufs[0]->Start() << "-" << bufs[0]->Stop() << std::endl;
  std::cout << " sample count - " << bufs[0]->Samples();
  std::cout << std::endl << std::endl;

  // --------------------------------------------------------------------

  conn->set_parameter("ITERATE_USE_GAP_HANDLERS", "true");
  conn->set_parameter("GAP_HANDLER", "STATIC_HANDLER_ZERO");
  conn->iterate(start, end, stride, cn);
  try
  {
    while ( true ) {
      std::cerr << ".................." << std::endl;
      if (fast_iterate)
    conn->next( bufs );
      else
    bufs = conn->next( );
      assert( bufs.size() == 1 );
      const NDS::buffer& cur_buf = *bufs[0];
      std::cerr << "Got buffer at " << cur_buf.Start() << " with " << cur_buf.Samples() << " samples." << std::endl;
      if (last_start == 0) {
    assert(cur_buf.Start() == start);
      } else {
    assert(cur_buf.Start() == (last_start + stride));
      }
      assert(cur_buf.Stop() == end || cur_buf.Samples() == samples_per_segment);
      last_start = cur_buf.Start();
    }
  }
  catch( const std::out_of_range& /* Error */ )
  {
    // Reached end of buffers so just do nothing
  }
  catch(...) {
      bool err = true;
  }

  return 0;
}
