#include "nds.hh"
#include "nds_channel.hh"
#include "nds_buffer_internal.hh"
#include "nds_connection.hh"
#include "nds_connection_ptype.hh"
#include "nds_composer.hh"
#include "nds_db.hh"
#include "nds_helper.hh"
#include "nds_str_helper.hh"
#include "nds_foreach.hh"

#include "nds_errno.hh"

#include "debug_stream.hh"



#include "nds_os.h"

#ifdef HAVE_IO_H
#include <io.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <cassert>
#include <cctype>
#include <cmath>
#include <cstring>

#include <algorithm>
#include <iostream>
#include <sstream>
#include <new>
#include <utility>

#include "nds_channel_internal.hh"

namespace {
    template <typename T, typename U>
    void destructive_translate_to_shared_ptr(T& input, U& output) {
      U intermediate;       // use this to maintain our invariants on output, don't
      intermediate.reserve(input.size());
      for (typename T::iterator cur = input.begin(); cur !=input.end(); ++cur) {
        NDS_SHARED_PTR<typename T::value_type> tmp(new typename T::value_type());
        tmp->swap(*cur);
        intermediate.push_back(tmp);
      }
      output.swap(intermediate);
    };
}

namespace NDS
{
  //---------------------------------------------------------------------
  // Connection
  //---------------------------------------------------------------------

  connection::
  connection( )
  {
    p_ = new p_type( this, "", 0, PROTOCOL_TRY );
  }

  connection::
  connection( const host_type& host,
              port_type port,
              protocol_type protocol)
  {
    //-------------------------------------------------------------------
    // Parameter validation
    //-------------------------------------------------------------------
    if ( ( protocol <= PROTOCOL_INVALID )
         || ( protocol > PROTOCOL_TRY ) )
    {
      std::ostringstream        msg;
      msg << "The value "
          << protocol
          << " is an invalid protocol request"
        ;
      throw connection::error( msg.str( ) );
    }
    //-------------------------------------------------------------------
    // Go on with establishing the connection
    // If there is any issue with allocating the internal private data
    //   structure, let the exception pass up to the caller for
    //   them to handle as this class has no meaning without the
    //   private data structure.
    //-------------------------------------------------------------------
    p_ = new p_type( this, host, port, protocol);
    
    // connect = nds2_open(host, port, protocol);
  }


  connection::
  ~connection( )
  {
    // nds2_close( connect );
    
    delete p_;
    p_ = (p_type*)NULL;
  }

  void connection::
  close( )
  {
    p_->shutdown();
    // nds2_shutdown( connect );
  }

  availability_list_type connection::
  get_availability( const channel_names_type& channel_names )
  {
    return p_->get_availability( 0, buffer::GPS_INF, channel_names );
  }

  bool connection::
  check( buffer::gps_second_type gps_start,
   buffer::gps_second_type gps_stop,
   const channel_names_type& channel_names )
  {
    return p_->check( gps_start, gps_stop, channel_names );
  }

  buffers_type connection::
  fetch( buffer::gps_second_type gps_start,
	 buffer::gps_second_type gps_stop,
	 const channel_names_type& channel_names )
  {
    buffers_type retval;
    buffers_type_intl intermediate;
    p_->fetch( gps_start, gps_stop, channel_names, intermediate );
    destructive_translate_to_shared_ptr(intermediate, retval);
    return retval;
  }

  void connection::
  fetch( buffer::gps_second_type gps_start,
   buffer::gps_second_type gps_stop,
   const channel_names_type& channel_names, buffers_type& output )
  {
    buffers_type_intl intermediate;
    p_->fetch( gps_start, gps_stop, channel_names, intermediate );
    destructive_translate_to_shared_ptr(intermediate, output);
    NDS::dout() << "p_type::fetch returned" << std::endl;
  }

  connection::count_type connection::
  count_channels( std::string channel_glob,
		 channel::channel_type channel_type_mask,
		 channel::data_type data_type_mask,
		 channel::sample_rate_type min_sample_rate,
		 channel::sample_rate_type max_sample_rate )
  {
      return p_->count_channels(channel_glob,
                                channel_type_mask,
                                data_type_mask,
                                min_sample_rate,
                                max_sample_rate
                                );
  }

  channels_type connection::
  find_channels( std::string channel_glob,
		 channel::channel_type channel_type_mask,
		 channel::data_type data_type_mask,
		 channel::sample_rate_type min_sample_rate,
		 channel::sample_rate_type max_sample_rate )
  {
    channels_type results;
    channels_type_intl intermediate;
    p_->find_channels( intermediate,
            channel_glob,
            channel_type_mask,
            data_type_mask,
            min_sample_rate,
            max_sample_rate );
    results.reserve(intermediate.size());
    destructive_translate_to_shared_ptr(intermediate, results);
//    for (channels_type_intl::iterator cur = intermediate.begin();
//            cur != intermediate.end();
//            ++cur)
//    {
//        results.push_back(NDS_SHARED_PTR< channel > (new channel(*cur)));
//    }
    return results;
  }

  connection::host_type connection::
  get_host( ) const
  {
    return p_->host;
  }

  connection::port_type connection::
  get_port( ) const
  {
    return p_->port;
  }

  
  connection::protocol_type connection::
  get_protocol( ) const
  {
    return p_->protocol;
  }

  bool connection::
  set_parameter(const std::string &parameter, const std::string &value)
  {
    bool result = p_->parameters.set(parameter, value);
    if (result)
      p_->sync_parameters();
    return result;
  }

  std::string connection::
  get_parameter(const std::string &parameter) const
  {
    return p_->parameters.get(parameter);
  }

  connection::parameters_type connection::
  get_parameters( ) const
  {
    return p_->parameters.parameters();
  }


  const channel::hash_type& connection::
  hash( ) const
  {
    return p_->hash();
  }

  epochs_type connection::
  get_epochs()
  {
    return p_->get_epochs();
  }

  bool connection::
  set_epoch(std::string epoch)
  {
    return p_->set_epoch(epoch);
  }

  bool connection::
  set_epoch(buffer::gps_second_type gps_start,
    buffer::gps_second_type gps_stop)
  {
    return p_->set_epoch(gps_start, gps_stop);
  }

  epoch connection::
  current_epoch() const
  {
    return p_->current_epoch();
  }

  connection& connection::
  iterate( buffer::gps_second_type gps_start,
	   buffer::gps_second_type gps_stop,
	   buffer::gps_second_type stride,
	   const channel_names_type& channel_names )
  {
    // all online requests take the fast path
    if (gps_start == 0) {
      p_->iterate_fast(gps_start, gps_stop, stride, channel_names);
      return *this;
    }
    // below this line everything is an offline request
    if (get_protocol() == PROTOCOL_TWO) {
      // all nds2 offline queries w/o gaps take the fast path
      bool has_no_gaps = false;
      try {
        has_no_gaps = p_->check(gps_start, gps_stop, channel_names);
      } catch(...) {}
      if (has_no_gaps) {
        p_->iterate_fast(gps_start, gps_stop, stride, channel_names);
        return *this;
      }
    } else if (get_protocol() == PROTOCOL_ONE) {
      if (p_->parameters.iterate_uses_gap_handler()) {
        // fastish path for nds1 w/ gap handling turned on
        p_->iterate_simple_gaps(gps_start, gps_stop, stride, channel_names);
      } else {
        // return available data is a fast path
        p_->iterate_fast(gps_start, gps_stop, stride, channel_names);
      }
      return *this;
    }

    // These are the 'hard' cases, offline nds2 with gaps.
	if (p_->parameters.iterate_uses_gap_handler()) {
		p_->iterate_full( gps_start, gps_stop, stride, channel_names );
	} else {
		p_->iterate_available( gps_start, gps_stop, stride, channel_names );
	}
    return *this;
  }
  
  bool connection::
  has_next( )
  {
      return p_->has_next();
  }

  buffers_type connection::
  next( )
  {
    buffers_type output;

    next(output);
    return output;
  }

  void connection::
  next( buffers_type& output )
  {
    buffers_type_intl intermediate;
    if ( ! has_next( ) )
    {
      throw std::out_of_range( "No Next" );
    }
    p_->next( intermediate );
    destructive_translate_to_shared_ptr(intermediate, output);
  }

  bool connection::
  request_in_progress() const
  {
    return p_->request_in_progress( );
  }

  void connection::
  shutdown( )
  {
    p_->shutdown( );
  }

  //---------------------------------------------------------------------
  // 
  //---------------------------------------------------------------------
  daq_accessor::
  daq_accessor( const connection& Server )
    : server( Server )
  {
  }

  daq_t* daq_accessor::
  operator()( )
  {
    return &(server.p_->handle);
  }

  //---------------------------------------------------------------------
  // Connection::error
  //---------------------------------------------------------------------
  connection::error::
  error( const std::string& What, int ErrNo )
    : std::runtime_error( What )
  {
    if ( ErrNo != 0 )
    {
      errno = ErrNo;
    }
  }

  //---------------------------------------------------------------------
  // Connection::already_closed_error
  //---------------------------------------------------------------------
  connection::already_closed_error::
  already_closed_error( )
    : error( "Connection has already been closed",
	     NDS::ALREADY_CLOSED )
  {
  }

  //---------------------------------------------------------------------
  // Connection::minute_trend_error
  //---------------------------------------------------------------------
  connection::minute_trend_error::
  minute_trend_error( )
    : error( "minute trend ranges need to be aligned on minute boundries",
	     NDS::MINUTE_TRENDS )
  {
  }


  //---------------------------------------------------------------------
  // Connection::transfer_busy_error
  //---------------------------------------------------------------------
  connection::transfer_busy_error::
  transfer_busy_error( )
    : error( "Connection already transfering data",
	     NDS::TRANSFER_BUSY )
  {
  }

  //---------------------------------------------------------------------
  // Connection::unexpected_channels_received_error
  //---------------------------------------------------------------------
  connection::unexpected_channels_received_error::
  unexpected_channels_received_error( )
    : error( "An unexpected number of channels were received",
	     NDS::UNEXPECTED_CHANNELS_RECEIVED )
  {
  }

  //---------------------------------------------------------------------
  // Connection::daq_error
  //---------------------------------------------------------------------
  connection::daq_error::
  daq_error(int daq_code )
	: error( format( daq_code),
		 NDS::ERROR_LAST + daq_code ),
	  error_code( daq_code ) {  }

  connection::daq_error::
  daq_error(int daq_code, const std::string &additional_information )
	: error( format( daq_code, additional_information.c_str()),
		 NDS::ERROR_LAST + daq_code ),
	  error_code( daq_code ) {  }

  connection::daq_error::
  daq_error(int daq_code, const char* additional_information )
	: error( format( daq_code, additional_information),
		 NDS::ERROR_LAST + daq_code ),
	  error_code( daq_code ) { }

  std::string connection::daq_error::
  format(int DAQCode , const char *extra)
   {
    std::ostringstream	msg;

    msg << "Low level daq error occured" 
	<< " [" << DAQCode << "]"
	<< ": " << daq_strerror( DAQCode );
	if (extra) {
		msg << "\n" << extra;
	}
	if (DAQCode == DAQD_ONTAPE) {
		msg << "\nThis is not a fatal error, it is a notification that your data is not\n";
		msg << "immediately available.  If you set DATA_ON_TAPE to 1 you may reissue your request.\n";
		msg << "Please note that retrieving data from tape is slow, and may take several minutes or longer.\n";
		msg << "\nYou may request data on tape by doing either of\n";
		msg << "1. setting the 'NDS2_CLIENT_ALLOW_DATA_ON_TAPE' environment variable to '1'\n";
		msg << "or 2. calling the set_parameter(\"ALLOW_DATA_ON_TAPE\", \"1\") on your connection object.";
	}
    return msg.str( );
  }

  void buffer_initializer::
  reset_buffer(buffer *cur_buffer, const channel &channel_info) const
  {
    if (cur_buffer->Name() == channel_info.Name() &&
        cur_buffer->Type() == channel_info.Type() &&
        cur_buffer->DataType() == channel_info.DataType() &&
        cur_buffer->SampleRate() == channel_info.SampleRate() &&
        cur_buffer->Start() == gps_start && cur_buffer->Stop() == gps_stop)
    {
      NDS::dout() << "Buffer already initialized, not resetting" << std::endl;
      return;
    }
    NDS::dout() << "initializing buffer for " << channel_info.NameLong();
    buffer::size_type N = static_cast<buffer::size_type>((gps_stop - gps_start)*channel_info.SampleRate());
    NDS::dout() << " - N="<< N << " samples " << static_cast<void*>(cur_buffer) << std::endl;
    cur_buffer->reset_channel_info( channel_info, gps_start, 0 );
    cur_buffer->resize( N );
  }

  std::ostream &operator<<(std::ostream &os,const connection &conn) {
    os << "<" << conn.get_host() << ":" << conn.get_port();
    os << " (protocol version ";
    os << (conn.get_protocol() == NDS::connection::PROTOCOL_TWO ? "2)>" : "1)>");
    return os;
  }

}
