/* -*- mode: C++ ; c-basic-offset: 2; indent-tabs-mode: nil -*- */

#ifndef SWIG__COMMON__NDS_BUFFER_HH
#define SWIG__COMMON__NDS_BUFFER_HH

#include <cmath>
#include <iostream>

#include <vector>

#include "nds_export.hh"
#include "nds_channel.hh"

namespace NDS
{

  //---------------------------------------------------------------------
  /// \brief A buffer holds the data contents of a channel.
  ///
  /// This class facilitates the storage of data and accompanying
  /// meta data.
  ///
  /// \headerfile nds.hh nds.hh
  //---------------------------------------------------------------------
  class buffer
    : public channel
  {
  public:
    //-------------------------------------------------------------------
    /// \brief Type second portion of a gps time
    //-------------------------------------------------------------------
    typedef long gps_second_type;
    //-------------------------------------------------------------------
    /// \brief Type nano second portion of a gps time
    //-------------------------------------------------------------------
    typedef long gps_nanosecond_type;
    //-------------------------------------------------------------------
    /// \brief Type appropriate for length.
    //-------------------------------------------------------------------
    typedef size_t size_type;
    //-------------------------------------------------------------------
    /// \brief Storage type for the data.
    //-------------------------------------------------------------------
    typedef std::vector< unsigned char > data_type;

    static const gps_second_type GPS_INF = 1999999999;

    //-------------------------------------------------------------------
    /// \brief Default constructor.
    ///
    /// \return
    ///     New object instance
    //-------------------------------------------------------------------
    DLL_EXPORT
    buffer( );

    //-------------------------------------------------------------------
    /// \brief Constructor
    ///
    /// \param[in] ChannelInfo
    ///     Channel metadata.
    /// \param[in] Second
    ///     GPS second start time of channel data.
    /// \param[in] NanoSecond
    ///     GPS nanosecond start time of channel data.
    /// \param[in] Buffer
    ///     Channel data.
    /// \param[in] BufferSize
    ///     Number of bytes in Buffer.
    ///
    /// \return
    ///     New object instance
    //-------------------------------------------------------------------
    DLL_EXPORT
    buffer( const channel& ChannelInfo,
            gps_second_type Second,
            gps_nanosecond_type NanoSecond,
            const void* Buffer,
            size_type BufferSize );

    //-------------------------------------------------------------------
    /// \brief Retrieve the number of samples being stored
    ///
    /// \return
    ///     The number of samples being stored
    //-------------------------------------------------------------------
    DLL_EXPORT size_type Samples( ) const;

    //-------------------------------------------------------------------
    /// \brief Retrieve the GPS start second of the data
    ///
    /// \return
    ///     The GPS start second of the data
    //-------------------------------------------------------------------
    DLL_EXPORT gps_second_type Start( ) const;

    //-------------------------------------------------------------------
    /// \brief Retrieve the GPS start nano second of the data
    ///
    /// \return
    ///     The GPS start nano second of the data
    //-------------------------------------------------------------------
    DLL_EXPORT gps_nanosecond_type StartNano( ) const;

    //-------------------------------------------------------------------
    /// \brief Retrieve the GPS stop second of the data
    ///
    /// \return
    ///     The GPS stop second of the data
    //-------------------------------------------------------------------
    DLL_EXPORT gps_second_type Stop( ) const;

    //-------------------------------------------------------------------
    /// \brief Exchange the contents of this container with another.
    //-------------------------------------------------------------------
    DLL_EXPORT void swap( buffer& Source );

    //-------------------------------------------------------------------
    /// \brief Resize the container to hold N elements
    ///
    /// \param[in] N
    ///     The new number of elements to be stored.
    //-------------------------------------------------------------------
    DLL_EXPORT void resize( size_type N );

    //-------------------------------------------------------------------
    /// \brief Returns a reference to the Nth element.
    ///
    /// \param[in] N
    ///     The index of the desired element.
    ///
    /// \return
    ///     A reference to the Nth element.
    //-------------------------------------------------------------------
    DLL_EXPORT data_type::const_reference operator[]( size_type N ) const;

    // Helper functions

    //-------------------------------------------------------------------
    /// \brief Convert relative sample offsets to relative second offsets.
    ///
    /// \param[in] offset_samples
    ///     An sample number [0,...)
    ///
    /// \return
    ///     The number of seconds since the beginning of this buffer that
    ///     this sample refers located at.  This is the number of seconds
    ///     since Start().
    //-------------------------------------------------------------------
    DLL_EXPORT gps_second_type samples_to_seconds(size_type offset_samples) const;

    //-------------------------------------------------------------------
    /// \brief Convert relative byte offsets to relative sample offsets.
    ///
    /// \param[in] offset_bytes
    ///     An byte number [0,...)
    ///
    /// \return
    ///     The number of samples since the beginning of this buffer that
    //      this byte location refers to.
    //-------------------------------------------------------------------
    DLL_EXPORT size_type bytes_to_samples(size_type offset_bytes) const;

    //-------------------------------------------------------------------
    /// \brief Convert relative second offsets to relative sample offsets.
    ///
    /// \param[in] offset_seconds
    ///     A second number [0,...).  This is the number of seconds since Start()
    /// |param[in] offset_nano
    ///     A nanosecond offset, defaults to 0
    ///
    /// \return
    ///     The number of samples since the beginning of this buffer that
    //      this time location refers to.
    //-------------------------------------------------------------------
    DLL_EXPORT size_type
    seconds_to_samples(gps_second_type offset_seconds,
                       gps_nanosecond_type offset_nano = 0) const;

    //-------------------------------------------------------------------
    /// \brief Convert relative sample offsets to relative byte offsets.
    ///
    /// \param[in] offset_samples
    ///     A sample number [0,...).  This is the number of samples since 
    ///     the start of this buffer
    ///
    /// \return
    ///     The byte offset that this sample would refer to relative to
    ///     the start of this buffer.
    //-------------------------------------------------------------------
    DLL_EXPORT size_type samples_to_bytes(size_type offset_samples) const;

    //-------------------------------------------------------------------
    /// \brief Reset the channel type.  Set the sample count to 0.
    ///
    /// \param[in] ChannelInfo channel meta data to copy into this buffer
    /// |param[in] Second 
    ///     A second number [0,...) that the buffer should start at
    /// |param[in] NanoSecond
    ///     The number of nanoseconds atfer Second that this buffer starts.
    ///
    /// \return
    ///     The number of samples since the beginning of this buffer that
    //      this time location refers to.
    //-------------------------------------------------------------------
    DLL_EXPORT void
    reset_channel_info(const channel& ChannelInfo,
                       gps_second_type Second,
                       gps_nanosecond_type NanoSecond);


  private:
    friend class swig_buffer_helper;

    gps_second_type     gps_second;
    gps_nanosecond_type gps_nanosecond;
    size_type           elements;
    data_type           data;

  protected:
    bool is_minute_trend() const;
  };

  typedef std::vector< NDS_SHARED_PTR < buffer > > buffers_type;

  inline bool buffer::
  is_minute_trend() const
  {
    return Type() == CHANNEL_TYPE_MTREND;
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::gps_second_type buffer::
  samples_to_seconds(buffer::size_type offset_samples) const
  {
    if (is_minute_trend()) return static_cast<gps_second_type>(offset_samples * 60);
    return static_cast<gps_second_type>(offset_samples / SampleRate());
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::size_type buffer::
  bytes_to_samples(buffer::size_type offset_bytes) const
  {
    return offset_bytes/DataTypeSize();
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::size_type buffer::
  seconds_to_samples(buffer::gps_second_type offset_seconds, buffer::gps_nanosecond_type offset_nano) const
  {
    if (is_minute_trend()) return static_cast<size_type>(offset_seconds / 60);
    return static_cast<size_type>(offset_seconds * SampleRate());
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::size_type buffer::
  samples_to_bytes(buffer::size_type offset_samples) const
  {
    return offset_samples * DataTypeSize();
  }

  //---------------------------------------------------------------------
  /// The number of samples in the data set.
  /// The number of bytes stored is the number of samples multiplied by
  /// size of the data type.
  //---------------------------------------------------------------------
  inline buffer::size_type buffer::
  Samples( ) const
  {
    return data.size( ) / DataTypeSize( );
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::gps_second_type buffer::
  Start( ) const
  {
    return gps_second;
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::gps_nanosecond_type buffer::
  StartNano( ) const
  {
    return gps_nanosecond;
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::gps_second_type buffer::
  Stop( ) const
  {
    return gps_second + samples_to_seconds(Samples( ));
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline void buffer::
  resize( size_type N )
  {
    data.resize( N * DataTypeSize( ) );
  }

  //---------------------------------------------------------------------
  //---------------------------------------------------------------------
  inline buffer::data_type::const_reference buffer::
  operator[]( size_type N ) const
  {
    return data[ N ];
  }

  DLL_EXPORT extern std::ostream& operator<<(std::ostream& os, const buffer& obj);

  DLL_EXPORT extern std::ostream& operator<<(std::ostream& os, const buffers_type& obj);

}

#endif /* SWIG__COMMON__NDS_BUFFERL_HH */
