import binascii
import gzip
import hashlib
import httplib
import json
import os
import os.path
import re
import time

def isfloat(val):
	try:
		float(val)
	except ValueError:
		return False
	return True

class CloseButNoAbort(Exception):
	pass

class StorageBlob(object):
	blob_dir = "./blobs"
	hex_re = re.compile('^[a-fA-F0-9]+$')
	
	def __init__(self, data=None, hash=None):
		if (data == None and hash == None) or (data != None and hash != None):
			raise Exception("StorageBlobs require either data or a hash")
		self.__data = ""
		self.__sum = ""
		h = hashlib.sha1()
		if data != None:
			self.__data = data
			h.update(self.__data)
			self.__sum = h.hexdigest()
		elif hash != None:
			# sanity check on the hash structure
			if len(hash) != (h.digest_size * 2) or self.hex_re.match(hash) == None:
				raise Exception("Invalid hash passed")

			fname = os.path.join(self.blob_dir, hash)
			gzfname = fname + ".gz"

			# read the data
			# try raw data first
			if os.path.exists(fname):
				with open(fname, "rb") as f:
					self.__data = f.read()
			elif os.path.exists(gzfname):
				# python2.6 cannot use GzipFile with the 'with' statement
				gz = gzip.GzipFile(gzfname, "rb")
				self.__data = gz.read()
				gz.close()
			else:
				if os.environ.has_key('REPLAY_BLOB_REMOTE_CACHE'):
					url = os.environ['REPLAY_BLOB_REMOTE_CACHE'] + "/" + hash + ".gz"
					print "Searching for blob at %s" % url
					if url.lower().startswith('http://'):
						url = url[7:]
					elif url.lower().startswith('https://'):
						raise Exception('https not supported for blob cache, it is already checksumed')
					host, remote_path = url.split('/', 1)
					conn = httplib.HTTPConnection(host, timeout=5)
					conn.request("GET", "/" + remote_path)
					resp = conn.getresponse()
					data = resp.read()
					if resp.status == 200:
						print "Found %s in remote cache" % hash
						if not os.path.exists(self.blob_dir):
							os.mkdir(self.blob_dir)
						gz = open(gzfname, "wb")
						gz.write(data)
						gz.close()
			
						gz = gzip.GzipFile(gzfname, "rb")
						self.__data = gz.read()
						gz.close()

					data = ""

					conn.close()
				else:
					raise Exception("Data blob %s was not found please update %s or set REPLAY_BLOB_REMOTE_CACHE" % (hash, self.blob_dir))

			h.update(self.__data)
			self.__sum = hash
			if h.hexdigest() != self.__sum:
				raise Exception("Checksum incorrect for data")

	def save(self):
		if not os.path.exists(self.blob_dir):
			os.mkdir(self.blob_dir)
		fname = os.path.join(self.blob_dir, self.__sum + ".gz")
		print 'saving blob to %s' %fname
		if os.path.exists(fname):
			print 'blob exists, skipping'
			return
		tmpname = fname + ".tmp"
		print 'tmp name = %s' % tmpname
		try:
			with gzip.GzipFile(tmpname, "wb") as f:
				f.write(self.__data)
			os.rename(tmpname, fname)
			print 'Replaced %s' % fname
		finally:
			try:
				os.unlink(tmpname)
			except:
				pass

	def data(self):
		return self.__data

	def sum(self):
		return self.__sum

class Script(object):
	"""A script object waits for a command (a given byte sequence, plain text) and response with a response (a given byte sequence, may be binary)"""
	min_blob = 4*1024

	def __init__(self, cmd, response):
		self.cmd = cmd
		self.response = response

	def match(self, inp):
		received = ""
		for cur in self.cmd:
			ch = inp.read(1)
			received = received + ch
			if cur != ch:
				raise Exception("Unexpected command, wanted '%s', mismatch at '%s'" % (self.cmd, received))

	def __call__(self, inp, out):
		self.match(inp)
		print "Received: '%s'" % self.cmd
		#if cur == 'authorize':
		#	raise CloseButNoAbort()
		print "Sending '%s'" % self.response[:15]
		out.write(self.response)
		out.flush()
		
	def __must_be_binary(self, data):
		if len(data) > self.min_blob:
			return True
		for ch in data:
			o = ord(ch)
			if o == 0 or o > 127:
				return True
		return False


	def serialize(self):
		chunk = 162

		result =  { "type": "script", "command": self.cmd }
		if not self.__must_be_binary(self.response):
			if len(self.response) <= chunk:
				result["response"] = self.response
			else:
			 	data = self.response
			 	responses = []
			 	while len(data) > 0:
			 		responses.append(data[:chunk])
			 		data = data[chunk:]
			 	result["responses"] = responses
		else:
			if len(self.response) <= chunk/2:
				result["response_hex"] = binascii.hexlify(self.response)
			else:
				# large pieces of data or binary data
				blob = StorageBlob(data=self.response)
				blob.save()
				result["response_blob"] = blob.sum()
		return result

class CompoundScript(object):
	"""A CompoundScript is a container object which executes a series of script like objects"""
	def __init__(self, scripts):
		self.scripts = scripts

	def __call__(self, inp, out):
		for cur in self.scripts:
			cur(inp, out)
		print "Compound script done"

	def serialize(self):
		scripts = []
		for script in self.scripts:
			scripts.append(script.serialize())
		return { "type": "compound", "scripts": scripts }

class NamedScript(object):
	"""A Named script is a container that holds a single script object referenced by name from a external script container"""
	def __init__(self, name, container):
		self.name = name
		self.container = container

	def __call__(self, inp, out):
		return self.container[self.name](inp, out)

	def serialize(self):
		return { "type": "named", "name": self.name }

class CommandScript(object):
	"""A CommandScript takes no input and executes a programable command"""

	modules = {}

	@staticmethod
	def load_module(name):
		try:
			m = __import__(name)
			CommandScript.modules[name] = m
			print "Loaded "+name+" module"
		except ImportError:
			return

	@staticmethod
	def validate_action(action, args):
		if action == "sleep":
			if len(args) == 1 and isfloat(args[0]):
				return
		elif action == "import" and len(args) == 1:
			CommandScript.load_module(args[0])
			return
		else:
			for ext in CommandScript.modules:
				if action.startswith(ext + '.'):
					CommandScript.modules[ext].validate_action(action[len(ext)+1:], args)
					return
		raise Exception("Unsupported command action and args %s %s" % (action, str(args)))

	def __init__(self, action, args):
		CommandScript.validate_action(action, args)
		self.action = action
		self.args = args

	def __call__(self, inp, out):
		if self.action == "sleep":
			time.sleep(float(self.args[0]))
		elif self.action == "import":
			# no-op, this happens in the validate stage
			return
		else:
			for ext in CommandScript.modules:
				if self.action.startswith(ext + '.'):
					CommandScript.modules[ext].execute_action(self.action[len(ext)+1:], self.args, inp, out)
					return
		raise Exception("Unknown command action " + self.action)

	def serialize(self):
		return { "type": "command", "action": self.action, "args": self.args }

def save_scripts(scripts, f):
	data = {}
	for name in scripts:
		data[name] = scripts[name].serialize()

	json.dump(data, f, indent=4, ensure_ascii=True)

def __unserialize_script(data, container):
	if data["type"] == "script":
		if data.has_key("response"):
			return Script(data["command"], data["response"])
		elif data.has_key("responses"):
			response = ""
			for val in data["responses"]:
				response = response + val
			return Script(data["command"], response)
		elif data.has_key("response_hex"):
			return Script(data["command"], binascii.unhexlify(data["response_hex"]))
		elif data.has_key("response_blob"):
			blob = StorageBlob(hash=data["response_blob"])
			return Script(data["command"], blob.data())
		raise Exception("Invalid script serialization")
	elif data["type"] == "compound":
		scripts = []
		for script in data["scripts"]:
			scripts.append(__unserialize_script(script, container))
		return CompoundScript(scripts)
	elif data["type"] == "named":
		return NamedScript(data["name"], container)
	elif data["type"] == "command":
		return CommandScript(data["action"], data["args"])
	else:
		raise Exception("Unknown script type %s" % data["type"])

def load_scripts(f):
	scripts = {}
	data = json.load(f)
	for name in data:
		scripts[name] = __unserialize_script(data[name], scripts)
	return scripts