%
% Unit tests against mock-up NDS 1 server
%
% Copyright (C) 2014  Leo Singer <leo.singer@ligo.org>
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation; either version 2 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License along
% with this program; if not, write to the Free Software Foundation, Inc.,
% 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
%

nds2;

raw_bytepatterns = {
    cast([0xB0; 0x00], 'uint8'),
    cast([0xDE; 0xAD; 0xBE; 0xEF], 'uint8'),
    cast([0x00; 0x00; 0x00; 0x00; 0xFE; 0xED; 0xFA; 0xCE], 'uint8'),
    cast([0xFE; 0xED; 0xFA; 0xCE], 'uint8')
};

min_bytepatterns = {
    cast([0x01; 0x01], 'uint8'),
    cast([0x02; 0x02; 0x02; 0x02], 'uint8'),
    cast([0x03; 0x03; 0x03; 0x03; 0x03; 0x03; 0x03; 0x03], 'uint8'),
};

max_bytepatterns = {
    cast([0x11; 0x11], 'uint8'),
    cast([0x12; 0x12; 0x12; 0x12], 'uint8'),
    cast([0x13; 0x13; 0x13; 0x13; 0x13; 0x13; 0x13; 0x13], 'uint8'),
};

mean_bytepatterns = {
    cast([0x1A; 0x1A; 0x1A; 0x1A; 0x1A; 0x1A; 0x1A; 0x1A], 'uint8'),
    cast([0x2A; 0x2A; 0x2A; 0x2A; 0x2A; 0x2A; 0x2A; 0x2A], 'uint8'),
    cast([0x3A; 0x3A; 0x3A; 0x3A; 0x3A; 0x3A; 0x3A; 0x3A], 'uint8')
};

rms_bytepatterns = {
    cast([0x1B; 0x1B; 0x1B; 0x1B; 0x1B; 0x1B; 0x1B; 0x1B], 'uint8'),
    cast([0x2B; 0x2B; 0x2B; 0x2B; 0x2B; 0x2B; 0x2B; 0x2B], 'uint8'),
    cast([0x3B; 0x3B; 0x3B; 0x3B; 0x3B; 0x3B; 0x3B; 0x3B], 'uint8')
};

n_bytepatterns = {
    cast([0x1C; 0x1C; 0x1C; 0x1C], 'uint8'),
    cast([0x2C; 0x2C; 0x2C; 0x2C], 'uint8'),
    cast([0x3C; 0x3C; 0x3C; 0x3C], 'uint8')
};

port = str2num(getenv('NDS_TEST_PORT'))

conn = nds2.connection('localhost', port, 1)
assert (conn.get_port(), port)
assert (conn.get_protocol(), 1)

channels = conn.find_channels('*')
base_channel_count = 9
assert (length(channels), base_channel_count * (5*2 + 1))

channels = conn.find_channels('X*:{A,B,C,G,H,I}')
assert (length(channels), 6)
assert (sprintf(channels{1}), '<X1:A (16Hz, ONLINE, INT16)>')
assert (sprintf(channels{2}), '<X1:B (16Hz, ONLINE, INT32)>')
assert (sprintf(channels{3}), '<X1:C (16Hz, ONLINE, INT64)>')
assert (sprintf(channels{4}), '<X1:G (16Hz, ONLINE, UINT32)>')
assert (sprintf(channels{5}), '<X1:H (65536Hz, ONLINE, FLOAT32)>')
assert (sprintf(channels{6}), '<X1:I (16384Hz, ONLINE, UINT32)>')

channels = conn.find_channels('X1:G*')
expectedChannels = sort({"<X1:G (16Hz, ONLINE, UINT32)>", "<X1:G.max,m-trend (0.0166667Hz, MTREND, UINT32)>",\
    "<X1:G.max,s-trend (1Hz, STREND, UINT32)>", "<X1:G.mean,m-trend (0.0166667Hz, MTREND, FLOAT64)>",\
    "<X1:G.mean,s-trend (1Hz, STREND, FLOAT64)>", "<X1:G.min,m-trend (0.0166667Hz, MTREND, UINT32)>",\
    "<X1:G.min,s-trend (1Hz, STREND, UINT32)>", "<X1:G.rms,m-trend (0.0166667Hz, MTREND, FLOAT64)>",\
    "<X1:G.rms,s-trend (1Hz, STREND, FLOAT64)>", "<X1:G.n,m-trend (0.0166667Hz, MTREND, INT32)>",\
    "<X1:G.n,s-trend (1Hz, STREND, INT32)>"})
assert (length(channels), length(expectedChannels))
actualChannels = cell(length(channels), 1)
for i = 1:length(channels)
    actualChannels{i} = sprintf(channels{i})
end
actualChannels = sort(actualChannels)
for i = 1:length(channels)
    assert (expectedChannels{i}, actualChannels{i})
end

bufs = conn.fetch(1000000000, 1000000004, {'X1:A', 'X1:B', 'X1:C', 'X1:G'})
assert (length(bufs), 4)
for i = 1:length(bufs)
    buf = bufs{i};
    bytepattern = raw_bytepatterns{i};
    assert (buf.gps_seconds, 1000000000)
    assert (buf.gps_nanoseconds, 0)
    assert (length(buf.data), 4 * buf.channel.sample_rate)
    assert (typecast(buf.data, 'uint8'), repmat(bytepattern, length(buf.data), 1))
end

for tselect = 1:2
    if (tselect == 1)
        start = 1000000000
        end_time = 1000000020
        mult = 1
        samples = 20
        trend = "s-trend"
    else
        start = 1000000020
        end_time = 1000000140
        mult = 60
        samples = 2
        trend = "m-trend"
    end

    channel_list = {["X1:A.min," trend], ["X1:B.min," trend], ["X1:C.min," trend]}
    bufs = conn.fetch(start, end_time, channel_list)
    assert (length(bufs), 3)
    for i = 1:length(bufs)
        buf = bufs{i}
        cross_check = conn.find_channels([buf.channel.name "," trend]){1}
        assert (cross_check.channel.data_type, buf.channel.data_type)
        bytepattern = min_bytepatterns{i}
        assert (buf.gps_seconds, start)
        assert (buf.gps_nanoseconds, 0)
        assert (length(buf.data), samples * (buf.channel.sample_rate * mult))
        assert (typecast(buf.data, 'uint8'), repmat(bytepattern, length(buf.data) , 1))
    end

    channel_list = {["X1:A.max," trend], ["X1:B.max," trend], ["X1:C.max," trend]}
    bufs = conn.fetch(start, end_time, channel_list)
    assert (length(bufs), 3)
    for i = 1:length(bufs)
        buf = bufs{i}
        cross_check = conn.find_channels([buf.channel.name "," trend]){1}
        assert (cross_check.channel.data_type, buf.channel.data_type)
        bytepattern = max_bytepatterns{i}
        assert (buf.gps_seconds, start)
        assert (buf.gps_nanoseconds, 0)
        assert (length(buf.data), samples * (buf.channel.sample_rate * mult))
        assert (typecast(buf.data, 'uint8'), repmat(bytepattern, length(buf.data) , 1))
    end

    channel_list = {["X1:A.mean," trend], ["X1:B.mean," trend], ["X1:C.mean," trend]}
    bufs = conn.fetch(start, end_time, channel_list)
    assert (length(bufs), 3)
    for i = 1:length(bufs)
        buf = bufs{i}
        cross_check = conn.find_channels([buf.channel.name "," trend]){1}
        assert (cross_check.channel.data_type, buf.channel.data_type)
        bytepattern = mean_bytepatterns{i}
        assert (buf.gps_seconds, start)
        assert (buf.gps_nanoseconds, 0)
        assert (length(buf.data), samples * (buf.channel.sample_rate * mult))
        assert (typecast(buf.data, 'uint8'), repmat(bytepattern, length(buf.data) , 1))
    end

    channel_list = {["X1:A.rms," trend], ["X1:B.rms," trend], ["X1:C.rms," trend]}
    bufs = conn.fetch(start, end_time, channel_list)
    assert (length(bufs), 3)
    for i = 1:length(bufs)
        buf = bufs{i}
        cross_check = conn.find_channels([buf.channel.name "," trend]){1}
        assert (cross_check.channel.data_type, buf.channel.data_type)
        bytepattern = rms_bytepatterns{i}
        assert (buf.gps_seconds, start)
        assert (buf.gps_nanoseconds, 0)
        assert (length(buf.data), samples * (buf.channel.sample_rate * mult))
        assert (typecast(buf.data, 'uint8'), repmat(bytepattern, length(buf.data) , 1))
    end

    channel_list = {["X1:A.n," trend], ["X1:B.n," trend], ["X1:C.n," trend]}
    bufs = conn.fetch(start, end_time, channel_list)
    assert (length(bufs), 3)
    for i = 1:length(bufs)
        buf = bufs{i}
        cross_check = conn.find_channels([buf.channel.name "," trend]){1}
        assert (cross_check.channel.data_type, buf.channel.data_type)
        bytepattern = n_bytepatterns{i}
        assert (buf.gps_seconds, start)
        assert (buf.gps_nanoseconds, 0)
        assert (length(buf.data), samples * (buf.channel.sample_rate * mult))
        assert (typecast(buf.data, 'uint8'), repmat(bytepattern, length(buf.data) , 1))
    end

endfor

conn.iterate({'X1:A', 'X1:B', 'X1:C'})
bufs = conn.next()
assert (length(bufs), 3)
for i = 1:length(bufs)
    buf = bufs{i};
    bytepattern = raw_bytepatterns{i};
    assert (buf.gps_seconds, 1000000000)
    assert (buf.gps_nanoseconds, 0)
    assert (length(buf.data), buf.channel.sample_rate)
    assert (typecast(buf.data, 'uint8'), repmat(bytepattern, length(buf.data), 1))
end

% connection automatically closed when garbage collected,
% but test explicit close anyway
conn.close()
