/*
 * SWIG bindings for high-level NDS client library
 *
 * TO DO
 *  - For some reason, the default arguments are not being translated into
 *	overloaded methods in Java.
 *
 * Copyright (C) 2013  Leo Singer <leo.singer@ligo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

%module nds2

/* Tell swig to include the header file in the C wrapper that it is creating,
 * but don't process the declarations in it yet.
 */
%{
/*
 * FIXME: The Octave headers define these macros. This is probably a bug in
 * Octave---it needs these internally, but it should not expose them in its API.
 * This project, like all Autotools-based projects, also defines these macros,
 * which causes a preprocessor warning.
 */
#undef PACKAGE_NAME
#undef PACKAGE_STRING
#undef PACKAGE_TARNAME
#undef PACKAGE_VERSION
#include "daq_config.h"

#include "nds.h"
#include "bash_pattern.h"
#ifndef __cplusplus
typedef int bool;
#endif
#if HAVE_STDIO_H
#include <stdio.h>
#endif /* HAVE_STDIO_H */

#if ! HAVE_ASPRINTF
#include "asprintf.c"
#endif /* HAVE_ASPRINTF */
%}

/* Some built-in swig interface files that we will need. */
%include <exception.i>
%include <constraints.i>
#ifdef SWIGJAVA
%include "jniloader.i"
#endif

/* Special typedef to turn off special string handling for MATLAB.
 * MATLAB does not automatically convert java.lang.String return values to its
 * own char arrays, so for convenient calling from MATLAB we turn C strings into
 * double-scripted char arrays, which MATLAB knows to interpret as its own char
 * arrays. */
typedef char *c_str;
%{
	typedef char *c_str;
%}
%apply char * {c_str};

%typemap(javapackage) nds2;

/* Ignore all enum items: we'll map these onto data members of classes later. */
%rename("$ignore", %$isenumitem) "";

/* nds2_connection is an anonymous type. To add methods to it, we have to
 * trick SWIG a little bit. */
%ignore nds2_connection;
%rename _nds2_connection connection;

/* bash_pattern is tricky in the same way. */
%ignore bash_pattern;
%rename bash_pattern_t pattern;

/* Strip package prefix from class names */
%rename("%(strip:[nds2_])s", %$isclass) "";

#if defined(SWIGJAVA)
%include "nds2_java.i"
#elif defined(SWIGPYTHON)
%include "nds2_python.i"
#elif defined(SWIGOCTAVE)
%include "nds2_octave.i"
#endif

/* Ignore all functions: we'll map these onto methods of classes later. */
%rename("$ignore", %$isfunction, regextarget=1) "^nds2_";
%rename("$ignore", %$isfunction, regextarget=1) "^bash_";

/* Exception handling */
%{
#include <errno.h>
%}

%define NDS2_RAISE(code)
{
	#if defined(SWIGJAVA)
		SWIG_JavaException(jenv, SWIG_RuntimeError, nds2_strerror(errno));
		$cleanup
		return $null;
	#elif defined(SWIGPYTHON)
		if (errno == NDS2_EOS)
		{
			PyErr_SetString(PyExc_StopIteration, nds2_strerror(errno));
			SWIG_fail;
		} else {
			SWIG_exception(SWIG_RuntimeError, nds2_strerror(errno));
		}
	#else
		SWIG_exception(SWIG_RuntimeError, nds2_strerror(errno));
	#endif
}
%enddef

/** Exception handling for routines that must return a non-NULL result. */
%define NDS2_SWIG_EXCEPTION_NONNULL
	%exception {
		$action
		if (!result)
			NDS2_RAISE(errno)
	}
%enddef

/** Exception handling for routines that return a dynamically allocated array
 * with the number of elements returned by pointer. The return value must be
 * non-NULL unlesss the number of returned elements is zero. */
%define NDS2_SWIG_EXCEPTION_NONNULL_ARRAY
	%exception {
		$action
		if (!result && *arg2)
			NDS2_RAISE(errno)
	}
%enddef

/** Exception handling for routines that must return a non-negative result. */
%define NDS2_SWIG_EXCEPTION_NONNEG
	%exception {
		$action
		if (result < 0)
			NDS2_RAISE(errno)
	}
%enddef

/** Exception handling for routines that must not set errno. */
%define NDS2_SWIG_EXCEPTION_NOERRNO
	%exception {
		errno = 0;
		$action
		if (errno)
			NDS2_RAISE(errno)
	}
%enddef

/** Trivial exception handling for routines that do not require error checking. */
%define NDS2_SWIG_EXCEPTION_NOOP
	%exception;
%enddef

/* Turn on automatic docstrings. */
%feature("autodoc");

/* Disable all default constructors. */
%nodefaultctor;

%newobject _nds2_connection::find_channels;
%newobject _nds2_connection::fetch;
%newobject _nds2_connection::next;

/* Process declarations in the header file. */
%include "nds.h"

struct bash_pattern_t {
	%extend {

		NDS2_SWIG_EXCEPTION_NONNULL
		bash_pattern_t(c_str expression)
		{
			return bash_pattern_compile(expression);
		}

		NDS2_SWIG_EXCEPTION_NONNEG
		int matches(c_str text)
		{
			return bash_pattern_matches($self, text);
		}

		NDS2_SWIG_EXCEPTION_NOOP
		~bash_pattern_t()
		{
			bash_pattern_free($self);
		}

	}
};

/**
 * Make the next class that we declare implement the Iterator
 * interface in Java.
 */
%typemap(javaimports) SWIGTYPE "import java.util.Iterator;";
%typemap(javainterfaces) SWIGTYPE "Iterator<buffer []>";
%typemap(javacode) SWIGTYPE %{
	public void remove() {
		throw new UnsupportedOperationException();
	}
%}

/* Add some methods to nds2_connection.
 * Sadly, SWIG doesn't create overloads for keyword arguments in Java,
 * so we have to add the overloads by hand. */
struct _nds2_connection {
	%extend {

		static const int PROTOCOL_ONE = NDS2_PROTOCOL_ONE;
		static const int PROTOCOL_TWO = NDS2_PROTOCOL_TWO;
		static const int PROTOCOL_TRY = NDS2_PROTOCOL_TRY;

		NDS2_SWIG_EXCEPTION_NONNULL
		_nds2_connection(c_str host, int port, nds2_protocol protocol)
		{
			return nds2_open(host, port, protocol);
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		_nds2_connection(c_str host, int port)
		{
			return nds2_open(host, port, NDS2_PROTOCOL_TRY);
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		_nds2_connection(c_str host)
		{
			return nds2_open(host, 31200, NDS2_PROTOCOL_TRY);
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		char *get_host()
		{
			return nds2_get_host($self);
		}

		NDS2_SWIG_EXCEPTION_NONNEG
		int get_port()
		{
			return nds2_get_port($self);
		}

		NDS2_SWIG_EXCEPTION_NONNEG
		nds2_protocol get_protocol()
		{
			return nds2_get_protocol($self);
		}

		NDS2_SWIG_EXCEPTION_NONNULL_ARRAY
		nds2_channel ** find_channels(
			size_t *count_channels_ptr,
			c_str channel_glob,
			nds2_channel_type channel_type_mask,
			nds2_data_type data_type_mask,
			double min_smaple_rate,
			double max_sample_rate)
		{
			return nds2_find_channels(
				$self,
				count_channels_ptr,
				channel_glob,
				channel_type_mask,
				data_type_mask,
				min_smaple_rate,
				max_sample_rate);
		}

		NDS2_SWIG_EXCEPTION_NONNULL_ARRAY
		nds2_channel ** find_channels(
			size_t *count_channels_ptr,
			c_str channel_glob,
			nds2_channel_type channel_type_mask,
			nds2_data_type data_type_mask)
		{
			return nds2_find_channels(
				$self,
				count_channels_ptr,
				channel_glob,
				channel_type_mask,
				data_type_mask,
				0.f, 1e12f);
		}

		NDS2_SWIG_EXCEPTION_NONNULL_ARRAY
		nds2_channel ** find_channels(
			size_t *count_channels_ptr,
			c_str channel_glob,
			nds2_channel_type channel_type_mask)
		{
			return nds2_find_channels(
				$self,
				count_channels_ptr,
				channel_glob,
				channel_type_mask,
				(nds2_data_type) 0xFF,
				0.f, 1e12f);
		}

		NDS2_SWIG_EXCEPTION_NONNULL_ARRAY
		nds2_channel ** find_channels(
			size_t *count_channels_ptr,
			c_str channel_glob)
		{
			return nds2_find_channels(
				$self,
				count_channels_ptr,
				channel_glob,
				(nds2_channel_type) 0xFF,
				(nds2_data_type) 0xFF,
				0.f, 1e12f);
		}

		NDS2_SWIG_EXCEPTION_NONNULL_ARRAY
		nds2_channel ** find_channels(
			size_t *count_channels_ptr)
		{
			return nds2_find_channels(
				$self,
				count_channels_ptr,
				NULL,
				(nds2_channel_type) 0xFF,
				(nds2_data_type) 0xFF,
				0.f, 1e12f);
		}

		NDS2_SWIG_EXCEPTION_NONNULL_ARRAY
		nds2_buffer ** fetch(
			size_t *count_channels_ptr,
			long gps_start,
			long gps_stop,
			const char **channel_names,
			size_t count_channels)
		{
			nds2_buffer **ret = nds2_fetch(
				$self,
				gps_start,
				gps_stop,
				channel_names,
				count_channels);
			if (ret)
				*count_channels_ptr = count_channels;
			return ret;
		}

		/* The connection.iterate() method actually returns a pointer to the
		 * connection object itself.*/
		%typemap(out, noblock=1) nds2_connection * {
			#if defined(SWIGJAVA)
			$result = 0;
			#elif defined(SWIGPYTHON)
			Py_INCREF($self);
			$result = $self;
			#elif defined(SWIGOCTAVE)
			$result = args(0);
			#endif
		}
		#if defined(SWIGJAVA)
		%typemap(javaout) nds2_connection * {
			$jnicall;
			return this;
		}
		#endif

		NDS2_SWIG_EXCEPTION_NONNULL
		nds2_connection *iterate(
			const char **channel_names,
			size_t count_channels
		)
		{
			int result = nds2_iterate(
				$self,
				0,
				0,
				0,
				channel_names,
				count_channels
			);
			if (result)
				return NULL;
			else
				return $self;
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		nds2_connection *iterate(
			long stride,
			const char **channel_names,
			size_t count_channels
		)
		{
			int result = nds2_iterate(
				$self,
				0,
				0,
				stride,
				channel_names,
				count_channels
			);
			if (result)
				return NULL;
			else
				return $self;
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		nds2_connection *iterate(
			long gps_start,
			long gps_stop,
			const char **channel_names,
			size_t count_channels
		)
		{
			int result = nds2_iterate(
				$self,
				gps_start,
				gps_stop,
				0,
				channel_names,
				count_channels
			);
			if (result)
				return NULL;
			else
				return $self;
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		nds2_connection *iterate(
			long gps_start,
			long gps_stop,
			long stride,
			const char **channel_names,
			size_t count_channels
		)
		{
			int result = nds2_iterate(
				$self,
				gps_start,
				gps_stop,
				stride,
				channel_names,
				count_channels
			);
			if (result)
				return NULL;
			else
				return $self;
		}

		NDS2_SWIG_EXCEPTION_NONNEG
		bool has_next()
		{
			return nds2_has_next($self);
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		c_str __str__()
		{
			char *ret = NULL;
			asprintf(&ret, "<%s:%d (protocol version %d)>",
				nds2_get_host($self),
				nds2_get_port($self),
				nds2_get_protocol($self)
			);
			return ret;
		}

		NDS2_SWIG_EXCEPTION_NONNULL
		nds2_buffer ** next(
			size_t *count_channels_ptr
		)
		{
			return nds2_next(
				$self,
				count_channels_ptr
			);
		}

		NDS2_SWIG_EXCEPTION_NONNEG
		int clear_cache()
		{
			return nds2_clear_cache($self);
		}

		NDS2_SWIG_EXCEPTION_NOOP
		void close()
		{
			nds2_shutdown($self);
		}

		NDS2_SWIG_EXCEPTION_NOOP
		~_nds2_connection()
		{
			nds2_close($self);
		}
	}
};

/**
 * Stop implementing the Iterator interface in Java.
 */
%typemap(javaimports) SWIGTYPE "";
%typemap(javainterfaces) SWIGTYPE "";
%typemap(javacode) SWIGTYPE "";

/* Add string conversion routine for nds2_channel. */
%extend nds2_channel {

	static const int CHANNEL_TYPE_ONLINE = NDS2_CHANNEL_TYPE_ONLINE;
	static const int CHANNEL_TYPE_RAW = NDS2_CHANNEL_TYPE_RAW;
	static const int CHANNEL_TYPE_RDS = NDS2_CHANNEL_TYPE_RDS;
	static const int CHANNEL_TYPE_STREND = NDS2_CHANNEL_TYPE_STREND;
	static const int CHANNEL_TYPE_MTREND = NDS2_CHANNEL_TYPE_MTREND;
	static const int CHANNEL_TYPE_TEST_POINT = NDS2_CHANNEL_TYPE_TEST_POINT;
	static const int CHANNEL_TYPE_STATIC = NDS2_CHANNEL_TYPE_STATIC;

	static const int DATA_TYPE_INT16 = NDS2_DATA_TYPE_INT16;
	static const int DATA_TYPE_INT32 = NDS2_DATA_TYPE_INT32;
	static const int DATA_TYPE_INT64 = NDS2_DATA_TYPE_INT64;
	static const int DATA_TYPE_FLOAT32 = NDS2_DATA_TYPE_FLOAT32;
	static const int DATA_TYPE_FLOAT64 = NDS2_DATA_TYPE_FLOAT64;
	static const int DATA_TYPE_COMPLEX32 = NDS2_DATA_TYPE_COMPLEX32;

	NDS2_SWIG_EXCEPTION_NONNULL
	static char *data_type_to_string(nds2_data_type data_type)
	{
		return nds2_data_type_to_string(data_type);
	}
	NDS2_SWIG_EXCEPTION_NONNULL
	static char *channel_type_to_string(nds2_channel_type channel_type)
	{
		return nds2_channel_type_to_string(channel_type);
	}
	NDS2_SWIG_EXCEPTION_NONNULL
	c_str __str__()
	{
		char *ret = NULL;
		const char *channel_type_str;
		const char *data_type_str;
		switch ($self->channel_type)
		{
			case NDS2_CHANNEL_TYPE_ONLINE:
				channel_type_str = "ONLINE";
				break;
			case NDS2_CHANNEL_TYPE_RAW:
				channel_type_str = "RAW";
				break;
			case NDS2_CHANNEL_TYPE_RDS:
				channel_type_str = "RDS";
				break;
			case NDS2_CHANNEL_TYPE_STREND:
				channel_type_str = "STREND";
				break;
			case NDS2_CHANNEL_TYPE_MTREND:
				channel_type_str = "MTREND";
				break;
			case NDS2_CHANNEL_TYPE_TEST_POINT:
				channel_type_str = "TEST_POINT";
				break;
			case NDS2_CHANNEL_TYPE_STATIC:
				channel_type_str = "STATIC";
				break;
			default:
				channel_type_str = NULL;
		}
		switch ($self->data_type)
		{
			case NDS2_DATA_TYPE_INT16:
				data_type_str = "INT16";
				break;
			case NDS2_DATA_TYPE_INT32:
				data_type_str = "INT32";
				break;
			case NDS2_DATA_TYPE_INT64:
				data_type_str = "INT64";
				break;
			case NDS2_DATA_TYPE_FLOAT32:
				data_type_str = "FLOAT32";
				break;
			case NDS2_DATA_TYPE_FLOAT64:
				data_type_str = "FLOAT64";
				break;
			case NDS2_DATA_TYPE_COMPLEX32:
				data_type_str = "COMPLEX32";
				break;
			default:
				data_type_str = NULL;
		}
		asprintf(&ret, "<%s (%gHz, %s, %s)>",
			$self->name,
			(double)$self->sample_rate,
			channel_type_str,
			data_type_str
		);
		return ret;
	}

	NDS2_SWIG_EXCEPTION_NOOP
};


/* Add string conversion routine for nds2_buffer. */
%extend nds2_buffer {

	NDS2_SWIG_EXCEPTION_NONNULL
	c_str __str__()
	{
		char *ret = NULL;
		asprintf(&ret, "<%s (GPS time %ld.%09ld, %ld samples)>",
			$self->channel.name,
			$self->gps_seconds,
			$self->gps_nanoseconds,
			$self->length
		);
		return ret;
	}

	NDS2_SWIG_EXCEPTION_NOOP

};


NDS2_SWIG_EXCEPTION_NOOP
