/*
 * High-level NDS client library
 *
 * Copyright (C) 2012  Leo Singer <leo.singer@ligo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */


#ifndef SWIG
#include <stddef.h> /* For size_t */
#endif /* SWIG */


#ifndef NDS_H
#define NDS_H


#ifndef DLL_EXPORT
#if WIN32 || WIN64
#define DLL_EXPORT __declspec(dllexport)
#else
#define DLL_EXPORT
#endif /* WIN32 || WIN64 */
#endif /* DLL_EXPORT */

/**
 * Macro NDS2_CXX_ONLY to mark code that should only be used when the compiler is
 * in C++ mode. (example: default values for arguments are supported by C++ but not C).
 */
#ifdef __cplusplus
#define NDS2_CXX_ONLY(arg) arg
#else
#define NDS2_CXX_ONLY(arg)
#endif


#ifdef SWIG
#define SWIG_IMMUTABLE %immutable;
#else
#define SWIG_IMMUTABLE
#endif


#ifdef __cplusplus
extern "C" {
#elif 0
} /* so that editors will match preceding brace */
#endif


/**
 * Connection data structure; return type of nds2_open.
 */
struct _nds2_connection;
typedef struct _nds2_connection nds2_connection;


/** Error codes. */
enum nds2_errnum {
	NDS2_EOS = 256, /* end of stream */
	NDS2_CHANNEL_NAME_TOO_LONG, /* channel name is too long */
	NDS2_UNEXPECTED_CHANNELS_RECEIVED, /* server sent more channels than expected */
	NDS2_MINUTE_TRENDS, /* some minute trends were requested, but start and stop times were not divisible by 60 */
	NDS2_TRANSFER_BUSY, /* another transfer is already in progress */
    NDS2_NO_SUCH_CHANNEL, /* no such channel */
	NDS2_ELAST /* last error code */
};


/**
 * Enumeration of NDS data types.
 */
typedef enum {
	/** Signed 16-bit integer. */
	NDS2_DATA_TYPE_INT16 = (1 << 0),

	/** Signed 32-bit integer. */
	NDS2_DATA_TYPE_INT32 = (1 << 1),

	/** Signed 64-bit integer. */
	NDS2_DATA_TYPE_INT64 = (1 << 2),

	/** 32-bit float. */
	NDS2_DATA_TYPE_FLOAT32 = (1 << 3),

	/** 64-bit float. */
	NDS2_DATA_TYPE_FLOAT64 = (1 << 4),

	/** 64-bit complex consisiting of 32-bit real and imaginary parts. */
	NDS2_DATA_TYPE_COMPLEX32 = (1 << 5),
} nds2_data_type;


/**
 * Enumeration of NDS channel types.
 */
typedef enum {
	/** "Online" channel. */
	NDS2_CHANNEL_TYPE_ONLINE = (1 << 0),

	/** "Raw" channel. */
	NDS2_CHANNEL_TYPE_RAW = (1 << 1),

	/** "Reduced data set" channel. */
	NDS2_CHANNEL_TYPE_RDS = (1 << 2),

	/** "Second trend" channel. */
	NDS2_CHANNEL_TYPE_STREND = (1 << 3),

	/** "Minute trend" channel. */
	NDS2_CHANNEL_TYPE_MTREND = (1 << 4),

	/** "Test point" channel. */
	NDS2_CHANNEL_TYPE_TEST_POINT = (1 << 5),

	/** "Static" channel. */
	NDS2_CHANNEL_TYPE_STATIC = (1 << 6)
} nds2_channel_type;


/**
 * Enumeration of NDS protocol versions.
 */
typedef enum {
	/** An invalid protocol. */
	NDS2_PROTOCOL_INVALID = -1,

	/** Use version 1 protocol only. */
	NDS2_PROTOCOL_ONE = 1,

	/** Use version 2 protocol only. */
	NDS2_PROTOCOL_TWO = 2,

	/** Try version 2 protocol first; revert to version 1 on failure. */
	NDS2_PROTOCOL_TRY = 3
} nds2_protocol;


/**
 * Channel metadata structure.
 */
typedef struct {
	/** Name of channel. */
	SWIG_IMMUTABLE char name[255];

	/** Channel type. */
	SWIG_IMMUTABLE nds2_channel_type channel_type;

	/** Data type. */
	SWIG_IMMUTABLE nds2_data_type data_type;

	/** Sample rate in Hertz. */
	SWIG_IMMUTABLE double sample_rate;

	/** Calibration metadata. */
	SWIG_IMMUTABLE float signal_gain;
	SWIG_IMMUTABLE float signal_slope;
	SWIG_IMMUTABLE float signal_offset;
	SWIG_IMMUTABLE char signal_units[40];
} nds2_channel;


/**
 * Buffer data structure.  The data member is a freely sized array.
 * the size of the allocated memory is
 *     sizeof(buf) + buf.length * nds2_sizeof_data_type(buf.channel.data_type).
 */
typedef struct {
	/** Channel metadata. */
	SWIG_IMMUTABLE nds2_channel channel;

	/** Timestamp of first sample in buffer, in seconds + nanoseconds. */
	SWIG_IMMUTABLE long gps_seconds;
	SWIG_IMMUTABLE long gps_nanoseconds;

	/** Number of samples in buffer. */
	SWIG_IMMUTABLE size_t length;

	/** Freely sized array of bytes. */
	SWIG_IMMUTABLE unsigned char data[];
} nds2_buffer;


/**
 * Convert a data type or a bitwise combination of data types to a newly
 * allocated string.
 */
DLL_EXPORT
char *nds2_data_type_to_string(nds2_data_type);


/**
 * Convert a channel type or a bitwise combination of channel types to a newly
 * allocated string.
 */
DLL_EXPORT
char *nds2_channel_type_to_string(nds2_channel_type);


/**
 * Open a new NDS connection.
 */
DLL_EXPORT
nds2_connection *nds2_open(
	const char *host,
	int port NDS2_CXX_ONLY(= 31200),
	nds2_protocol protocol NDS2_CXX_ONLY(= NDS2_PROTOCOL_TRY)
);


/**
 * Close an NDS connection.
 */
DLL_EXPORT
void nds2_close(nds2_connection *connection);


/**
 * Get host name of connection.  Caller is responsible for deallocating with
 * free().
 */
DLL_EXPORT
char *nds2_get_host(nds2_connection *connection);


/**
 * Get port number of connection.
 */
DLL_EXPORT
int nds2_get_port(nds2_connection *connection);


/**
 * Get protocol version of connection.
 */
DLL_EXPORT
nds2_protocol nds2_get_protocol(nds2_connection *connection);


/**
 * Get metadata for channels matching the specified criteria.
 *
 * @param[in] connection         Connection object.
 *
 * @param[out] count_channels    Address (non-NULL) of a pointer to a size_t
 *                               in which to put the number of found channels.
 *                               Not modified on failure.
 *
 * @param[in] channel_glob       Channel name pattern, optionally containing
 *                               the wildcards * to match any string and ? to
 *                               match any character.
 *                               (suggested default: "*" or NULL to match any)
 *
 * @param[in] channel_type_mask  Bitmask of channel types of interest, for
 *                               example, (NDS2_CHANNEL_TYPE_STREND |
 *                               NDS2_CHANNEL_TYPE_MTREND).
 *                               (suggested default: 0xFF to match any)
 *
 * @param[in] data_type_mask     Bitmask of data types of interest, for example,
 *                               (NDS2_DATA_TYPE_INT16 | NDS2_DATA_TYPE_INT32).
 *                               (suggested default: 0xFF to match any)
 *
 * @param[in] min_sample_rate    Minimum sample rate.
 *                               (suggested default: 0)
 *
 * @param[in] max_sample_rate    Maximum sample rate.
 *                               (suggested default: MAXFLOAT)
 *
 * @returns                      Pointer to address of first returned channel,
 *                               or NULL on failure.  Caller is responsible for
 *                               deallocating all of the returned channels, and
 *                               the array itself, with free().
 */
DLL_EXPORT
nds2_channel **nds2_find_channels(
	nds2_connection *connection,
	size_t *count_channels,
	const char *channel_glob NDS2_CXX_ONLY(= NULL),
	nds2_channel_type channel_type_mask NDS2_CXX_ONLY(= (nds2_channel_type) 0xFF),
	nds2_data_type data_type_mask NDS2_CXX_ONLY(= (nds2_data_type) 0xFF),
	double min_sample_rate NDS2_CXX_ONLY(= 0.),
	double max_sample_rate NDS2_CXX_ONLY(= 1e12)
);


/** Determine size in bytes of the specified data type. */
DLL_EXPORT
int nds2_sizeof_data_type(nds2_data_type data_type);


/**
 * Fetch data from a given start time to a given stop time.
 *
 * @param[in] connection      Connection object.
 *
 * @param[in] gps_start       Start GPS time in seconds.
 *
 * @param[in] gps_stop        Stop GPS time in seconds.
 *
 * @param[in] channel_names   Address (non-NULL) of a pointer to name of
 *                            first desired channel.
 *
 * @param[in] count_channels  Number of requested channel names.
 *
 * @returns                   Pointer to address of first returned buffer,
 *                            or NULL on failure.  Caller is responsible for
 *                            deallocating all of the returned buffers and
 *                            the array itself with free().
 */
DLL_EXPORT
nds2_buffer **nds2_fetch(
	nds2_connection *connection,
	long gps_start,
	long gps_stop,
	const char **channel_names,
	size_t count_channels
);


/**
 * Start retrieving data progressively.  Call nds2_next() until it returns
 * -NDS2_EOS to retrieve data.
 *
 * @param[in] connection      Connection object.
 *
 * @param[in] gps_start       Start GPS time in seconds.
 *
 * @param[in] gps_stop        Stop GPS time in seconds.
 *
 * @param[in] stride          Stride in seconds, or 0 for default.
 *
 * @param[in] channel_names   Address (non-NULL) of a pointer to name of
 *                            first desired channel.
 *
 * @param[in] count_channels  Number of requested channel names.
 *
 * @returns 0 on success, or a negative number on failure.
 */
DLL_EXPORT
int nds2_iterate(
	nds2_connection *connection,
	long gps_start,
	long gps_stop,
	long stride,
	const char **channel_names,
	size_t count_channels
);


/**
 * Retrieve next buffer.  See also nds2_iterate.
 *
 * @param[in] connection       Connection object.
 *
 * @param[out] count_channels  Address (non-NULL) of a pointer to a size_t
 *                             in which to put the number of found channels.
 *                             Not modified on failure.
 *
 * @returns                    Pointer to address of first returned buffer, or
 *                             NULL on failure.  Caller is responsible for
 *                             deallocating all of the returned buffers and the
 *                             array itself with free().
 */
DLL_EXPORT
nds2_buffer **nds2_next(
	nds2_connection *connection,
	size_t *count_channels
);


/**
 * Determine if there are any more buffers available on an iterator.
 *
 * @param[in] iterator       Iterator object.
 *
 * @returns                    1, 0, or on failure, -1.
 */
DLL_EXPORT
int nds2_has_next(
	nds2_connection *connection
);


/**
 * Clear contents of cache associated with the specified connection object.
 *
 * @params[in] connection      Connection object.
 * @returns                    0 on success, or -1 on failure.
 */
DLL_EXPORT
int nds2_clear_cache(nds2_connection *connection);


/**
 * Convert the last error code to a string, prefixed with s followed by ": ".
 * See perror in <stdio.h>
 */
DLL_EXPORT
void nds2_perror(const char *s);


/**
 * Return a string version of the error code errnum.
 * Return value is a pointer to static memory and
 * /should not be freed by the caller/.
 */
DLL_EXPORT
const char *nds2_strerror(int errnum);


#if 0
{ /* so that editors will match succeeding brace */
#elif defined(__cplusplus)
} /* extern "C" */
#endif


#endif /* NDS_H */
