/* -*- mode: c; c-basic-offset: 4; -*- */
#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /*HAVE_CONFIG_H */

#include <stdarg.h>

#include "nds_os.h"
#include "nds_mex_utils.h"

#include <string.h>
#if HAVE_STRINGS_H
#include <strings.h>
#endif /* HAVE_STRINGS_H */
#include <stdio.h>

#include "nds_log.h"

/*  Copy the contents of the specfied field of an input array to the 
 *  specified field of the output array
 */
void
copy_mxarray(const mxArray* iarray, int ifield, mwSize inx, 
	     mxArray* oarray, int ofield, mwSize oinx) {
    mxArray* temp = mxDuplicateArray(mxGetFieldByNumber(iarray, inx, ifield));
    mxSetFieldByNumber(oarray, oinx, ofield, temp);
    /* mxDestroyArray(temp); */
}

/*  Get the specified integer data item from the array.
 */
int
get_mxarray_int(const mxArray* array, int field, mwSize inx) {
    int* p = (int*)mxGetData(mxGetFieldByNumber(array, inx, field));
    return *p;
}

/*  Get the specified integer data item from the array.
 */
double
get_mxarray_float(const mxArray* array, int field, mwSize inx) {
    const mxArray* mxa = mxGetFieldByNumber(array, inx, field);
    double val;
    switch (mxGetClassID(mxa)) {
    case mxSINGLE_CLASS:
	val = *(float*)mxGetData(mxa);
	break;
    case mxDOUBLE_CLASS:
	val = *(double*)mxGetData(mxa);
	break;
    case mxINT32_CLASS:
	val = *(int*)mxGetData(mxa);
	break;
    default:
	printf("get_mxarray_float: Unrecognized class: %i field: %i\n", 
	       mxGetClassID(mxa), field);
	return 9999.;
    }
    return val;
}

/*  Parse a host IP specifier of the form "<ip-addr>:<port>"
 */
void
parse_host_name(char* in, short* port) {
    char* p = strchr(in, ':');
    if (p) {
	*p++  = 0;
	*port = strtol(p, 0, 0);
    }
}

/*  Put the specified bool data into the array.
 */
int
put_mxarray_bool(mxArray* array, int field, mwSize inx, int val) {
    mxArray* temp = mxCreateLogicalScalar((mxLogical)val);
    if(temp == NULL) return -1;
    mxSetFieldByNumber(array, inx, field, temp);
    /* mxDestroyArray(temp); */
    return 0;
}

/*  Put the specified double data into the array.
 */
int
put_mxarray_double(mxArray* array, int field, mwSize inx, double val) {
    mwSize dims = 1;
    mxArray* temp = mxCreateNumericArray(1, &dims, mxDOUBLE_CLASS, mxREAL);
    if(temp == NULL) return -1;
    *(double*)mxGetData(temp) = val;
    mxSetFieldByNumber(array, inx, field, temp);
    /* mxDestroyArray(temp); */
    return 0;
}

/*  Put the specified float data into the array.
 */
int
put_mxarray_float(mxArray* array, int field, mwSize inx, float val) {
    mwSize dims = 1;
    mxArray* temp = mxCreateNumericArray(1, &dims, mxSINGLE_CLASS, mxREAL);
    if(temp == NULL) return -1;
    *(float*)mxGetData(temp) = val;
    mxSetFieldByNumber(array, inx, field, temp);
    /* mxDestroyArray(temp); */
    return 0;
}

/*  Put the specified int data into the array.
 */
int
put_mxarray_int(mxArray* array, int field, mwSize inx, int val) {
    mwSize dims = 1;
    mxArray* temp = mxCreateNumericArray(1, &dims, mxINT32_CLASS, mxREAL);
    if(temp == NULL) return -1;
    *(int*)mxGetData(temp) = val;
    mxSetFieldByNumber(array, inx, field, temp);
    /* mxDestroyArray(temp); */
    return 0;
}
	
/*  Put the specified char data into the array.
 */
int
put_mxarray_str(mxArray* array, int field, mwSize inx, const char* val) {
    mxArray* temp = mxCreateString(val);
    if(temp == NULL) return -1;
    mxSetFieldByNumber(array, inx, field, temp);
    return 0;
}

void
stride_copy( void* dest, void* src, mwSize n, mwSize stride )
{
    char* srcc = (char*)src;
    char* destc = (char*)dest;

    mwSize strideX2 = stride * 2;
    while( n > 0 )
    {
	memcpy( destc, srcc, (size_t)stride );
	destc += stride;
	srcc += strideX2;
	--n;
    }
}
