function datalist = NDS_GetMinuteTrend(namelist, start, duration,...
				       server, chanlist)
%NDS_GetMinuteTrend  Get time series data from a LIGO NDS server
%
%  This function is designed to have exactly the same name and syntax as
%  the analagous function used to talk to NDS version 1 servers. It is 
%  implemented as a call to the NDS2 version of NDS_GetData() by appending
%  the appropriate channel type (m-trend) to each requested channel name.
% 
%  Usage:
%
%    datalist = NDS_GetMinuteTrend(namelist, start, duration, server, chanlist)
%
%  where:
%
%    namelist is a list of channel names, e.g. {'H1:LSC-DARM_ERR' 'H2:LSC-DARM_ERR'}
%    start    is the GPS start time of the requested data
%    duration is the number of seconds to fetch.
%    server   is a string with the NDS2 server IP address (or name) and port ('<ip-addr>:<port>')
%    chanlist is a Matlab cell-array containing channel information, in the
%             format returned by NDS_GetChannels. 
%
%    datalist is a Matlab cell-array containing requested data
%             name           Channel name
%             group_num      Group number (Channel type in NDS2)
%             rate           Sample rate
%             tpnum          Test-point number (nds1 only)
%             bps            Bytes per sample
%             data_type      Data type
%             signal_gain    Online signal gain
%             signal_offset  Online signal offset
%             signal_slope   Online signal slope
%             signal_units   Online Signal unit string
%             start_gps_sec  Data start GPS time
%             duration_sec   Data duration in seconds
%             data           Data array
%             exists         Existence flag (non-existent == 0).
%
% See also NDS_GetChannels, NDS_GetData, NDS2_GetChannels, NDS2_GetData.
mtlist = {};
  for i=1:size(namelist) 
    mtlist(i) = strcat(deblank(namelist(i)), ',m-trend');
  end
  datalist = NDS_GetData(mtlist, start, duration, server, chanlist);
end
