/*
 * High-level NDS client library
 *
 * Copyright (C) 2012  Leo Singer <leo.singer@ligo.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#if HAVE_CONFIG_H
#include "daq_config.h"
#endif /* HAVE_CONFIG_H */

#include "nds.h"
#include <daqc.h>
#include <daqc_response.h>
#include <channel.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <math.h>
#include <sqlite3.h>

#include "nds_os.h"

#include "bash_pattern.h"
#include "asprintf.c"

/* includes for locating database file */
#ifdef _WIN32
#include <Shlobj.h>
#define snprintf _snprintf
#else
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>
#endif

struct _nds2_connection
{
	daq_t daq;
	sqlite3 *db;
	char *host;
	int port;

	/* State variables for making iterative data requests */
	long requested_end_time;
	int request_in_progress;
};


const char *_nds2_query_create_tables =
"CREATE TABLE IF NOT EXISTS servers ("
"	protocol INTEGER NOT NULL DEFAULT 3,"
"	timestamp INTEGER NOT NULL DEFAULT 0"
");"
"CREATE TABLE IF NOT EXISTS channels ("
"	name TEXT NOT NULL,"
"	channel_type INTEGER NOT NULL,"
"	data_type INTEGER NOT NULL,"
"	sample_rate DOUBLE NOT NULL,"
"	signal_gain FLOAT NOT NULL,"
"	signal_slope FLOAT NOT NULL,"
"	signal_offset FLOAT NOT NULL,"
"	signal_units TEXT NOT NULL"
");"
"CREATE INDEX IF NOT EXISTS channel_name_index ON channels (name);"
;


static const int _NDS2_TYPICAL_BYTES_PER_FRAME = (1 << 26);


typedef struct {
	char *expr;
	bash_pattern *pattern;
} _bash_pattern_auxdata;


static void _bash_pattern_free_auxdata(void *arg)
{
	_bash_pattern_auxdata *auxdata = (_bash_pattern_auxdata *) arg;
	if (auxdata)
	{
		free(auxdata->expr);
		auxdata->expr = NULL;
		bash_pattern_free(auxdata->pattern);
		auxdata->pattern = NULL;
		free(auxdata);
	}
}


static void _bash_pattern_matches_func(sqlite3_context *ctx, int nargs, sqlite3_value **args)
{
	_bash_pattern_auxdata *auxdata;
	const char *expr;
	const char *text;
	int ret = 0;

	/* If there are not exactly two arguments, then fail. */
	if (nargs != 2)
	{
		sqlite3_result_error(ctx, "exptected 2 arguments", -1);
		goto fail;
	}

	/* If the pattern expression is NULL, return false. */
	expr = (const char *)sqlite3_value_text(args[0]);
	if (!expr)
		goto done;

	/* If the text is NULL, return false. */
	text = (const char *)sqlite3_value_text(args[1]);
	if (!text)
		goto done;

	auxdata = (_bash_pattern_auxdata *) sqlite3_get_auxdata(ctx, 0);
	if (!auxdata || strcmp(expr, auxdata->expr))
	{
		auxdata = malloc(sizeof(_bash_pattern_auxdata));
		if (!auxdata)
		{
			sqlite3_result_error_nomem(ctx);
			goto fail;
		}
		auxdata->expr = strdup(expr);
		if (!auxdata->expr)
		{
			free(auxdata);
			sqlite3_result_error_nomem(ctx);
			goto fail;
		}
		auxdata->pattern = bash_pattern_compile(expr);
		if (!auxdata->pattern)
		{
			free(auxdata->expr);
			free(auxdata);
			sqlite3_result_error(ctx, "cannot compile pattern", -1);
			goto fail;
		}
		sqlite3_set_auxdata(ctx, 0, auxdata, _bash_pattern_free_auxdata);
	}

	ret = bash_pattern_matches(auxdata->pattern, text);
done:
	sqlite3_result_int(ctx, ret);
fail:
	return;
}


static int _bash_pattern_matches_register_func(sqlite3 *db)
{
	return sqlite3_create_function(db, "bash_pattern_matches", 2, SQLITE_UTF8, NULL, _bash_pattern_matches_func, NULL, NULL);
}


static int _nds2_db_start_transaction(sqlite3 *db)
{
	return sqlite3_exec(db, "BEGIN", NULL, NULL, NULL)
		|| sqlite3_exec(db, _nds2_query_create_tables, NULL, NULL, NULL);
}


static int _nds2_db_end_transaction(sqlite3 *db)
{
	return sqlite3_exec(db, "COMMIT", NULL, NULL, NULL);
}


static nds2_channel_type _nds2_chantype_to_channel_type(chantype_t c)
{
	nds2_channel_type channel_type;

	switch (c)
	{
		case cOnline:
			channel_type = NDS2_CHANNEL_TYPE_ONLINE;
			break;
		case cRaw:
			channel_type = NDS2_CHANNEL_TYPE_RAW;
			break;
		case cRDS:
			channel_type = NDS2_CHANNEL_TYPE_RDS;
			break;
		case cSTrend:
			channel_type = NDS2_CHANNEL_TYPE_STREND;
			break;
		case cMTrend:
			channel_type = NDS2_CHANNEL_TYPE_MTREND;
			break;
		case cTestPoint:
			channel_type = NDS2_CHANNEL_TYPE_TEST_POINT;
			break;
		case cStatic:
			channel_type = NDS2_CHANNEL_TYPE_STATIC;
			break;
		default:
			channel_type = 0;
	}

	return channel_type;
}


static chantype_t _nds2_chantype_to_channel_type_reverse(nds2_channel_type c)
{
	chantype_t channel_type;

	switch (c)
	{
		case NDS2_CHANNEL_TYPE_ONLINE:
			channel_type = cOnline;
			break;
		case NDS2_CHANNEL_TYPE_RAW:
			channel_type = cRaw;
			break;
		case NDS2_CHANNEL_TYPE_RDS:
			channel_type = cRDS;
			break;
		case NDS2_CHANNEL_TYPE_STREND:
			channel_type = cSTrend;
			break;
		case NDS2_CHANNEL_TYPE_MTREND:
			channel_type = cMTrend;
			break;
		case NDS2_CHANNEL_TYPE_TEST_POINT:
			channel_type = cTestPoint;
			break;
		case NDS2_CHANNEL_TYPE_STATIC:
			channel_type = cStatic;
			break;
		default:
			channel_type = cUnknown;
	}

	return channel_type;
}


static nds2_data_type _nds2_data_to_data_type(daq_data_t c)
{
	nds2_data_type data_type;

	switch (c)
	{
		case _16bit_integer:
			data_type = NDS2_DATA_TYPE_INT16;
			break;
		case _32bit_integer:
			data_type = NDS2_DATA_TYPE_INT32;
			break;
		case _64bit_integer:
			data_type = NDS2_DATA_TYPE_INT64;
			break;
		case _32bit_float:
			data_type = NDS2_DATA_TYPE_FLOAT32;
			break;
		case _64bit_double:
			data_type = NDS2_DATA_TYPE_FLOAT64;
			break;
		case _32bit_complex:
			data_type = NDS2_DATA_TYPE_COMPLEX32;
			break;
		default:
			data_type = 0;
	}

	return data_type;
}


static daq_data_t _nds2_data_to_data_type_reverse(nds2_data_type c)
{
	daq_data_t data_type;

	switch (c)
	{
		case NDS2_DATA_TYPE_INT16:
			data_type = _16bit_integer;
			break;
		case NDS2_DATA_TYPE_INT32:
			data_type = _32bit_integer;
			break;
		case NDS2_DATA_TYPE_INT64:
			data_type = _64bit_integer;
			break;
		case NDS2_DATA_TYPE_FLOAT32:
			data_type = _32bit_float;
			break;
		case NDS2_DATA_TYPE_FLOAT64:
			data_type = _64bit_double;
			break;
		case NDS2_DATA_TYPE_COMPLEX32:
			data_type = _32bit_complex;
			break;
		default:
			data_type = _undefined;
	}

	return data_type;
}


static int _nds2_chan_req_to_channel(nds2_channel *channel, chan_req_t *chanreq)
{
	int ret = -1;

	/* Check to make sure name field will fit. */
	if (strlen(chanreq->name) >= sizeof(channel->name))
	{
		errno = NDS2_CHANNEL_NAME_TOO_LONG;
		goto done;
	}

	/* Copy fields. */
	strncpy(channel->name, chanreq->name, sizeof(channel->name));
	channel->channel_type = _nds2_chantype_to_channel_type(chanreq->type);
	channel->data_type = _nds2_data_to_data_type(chanreq->data_type);
	channel->sample_rate = chanreq->rate;
	channel->signal_gain = chanreq->s.signal_gain;
	channel->signal_slope = chanreq->s.signal_slope;
	channel->signal_offset = chanreq->s.signal_offset;
	strncpy(channel->signal_units, chanreq->s.signal_units, sizeof(channel->signal_units));

	/* Set return value to indicate success. */
	ret = 0;

done:
	/* Done! */
	return ret;
}


/**
 * Determine the location in which to store the cache database.
 * On success, return a string that should be released with free().
 * On failure, return NULL.
 */
static char *_nds2_db_get_path(const char *host, int port)
{
	char *ret = NULL;

#if _WIN32
	char base_path[MAX_PATH];

	/* Locate, and create if necessary, the directory
	 * "C:\Documents and Settings\username\Local Settings\Application Data". */
	if (!SUCCEEDED(SHGetFolderPathA(NULL, CSIDL_LOCAL_APPDATA | CSIDL_FLAG_CREATE, NULL, 0, base_path)))
		goto fail;

	/* Form path to database: e.g.
	 * "C:\Documents and Settings\username\Local Settings\Application Data\nds2-client_0.10_nds.ligo-la.caltech.edu_31200.sqlite". */
	if (asprintf(&ret, "%s\\" PACKAGE_NAME "_" PACKAGE_VERSION "_%s_%d.sqlite", base_path, host, port) < 0)
		goto fail;
#else
	/* Look up ID of current user. */
	uid_t uid = getuid();

	/* Form path to database: e.g.
	 * "/var/tmp/nds2-client_0.10_501_nds.ligo-la.caltech.edu_31200.sqlite". */
	if (asprintf(&ret, "/var/tmp/" PACKAGE_NAME "_" PACKAGE_VERSION "_%lu_%s_%d.sqlite", (long unsigned)uid, host, port) < 0)
		goto fail;
#endif

fail:
	return ret;
}


/**
 * Attempt to determine protocol version by database lookup.
 */
static nds2_protocol _nds2_db_get_protocol(sqlite3 *db)
{
	nds2_protocol ret = -1;
	int result;
	sqlite3_stmt *stmt = NULL;

	result = sqlite3_prepare_v2(db,
		"SELECT protocol FROM servers", -1, &stmt, NULL);
	if (result != SQLITE_OK)
		goto fail;

	result = sqlite3_step(stmt);
	if (result != SQLITE_ROW)
		goto fail;

	ret = sqlite3_column_int(stmt, 0);

fail:
	sqlite3_finalize(stmt);
	return ret;
}


/**
 * Record protocol version in database.
 */
static int _nds2_db_put_protocol(sqlite3 *db, nds2_protocol protocol)
{
	int result;
	int old_protocol;
	sqlite3_stmt *stmt;

	result = sqlite3_exec(db, "BEGIN", NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto fail;

	result = sqlite3_exec(db, _nds2_query_create_tables, NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto rollback;

	result = sqlite3_prepare_v2(db,
		"SELECT protocol FROM servers",
		-1, &stmt, NULL);
	if (result == SQLITE_OK)
		result = sqlite3_step(stmt);
	if (result == SQLITE_ROW)
		old_protocol = sqlite3_column_int(stmt, 0);
	sqlite3_finalize(stmt);

	/* If we were able to find the old protocol in the database, and it matches
	 * the new protocol, then stop here. */
	if (result == SQLITE_ROW && protocol == old_protocol)
		goto done;

	result = sqlite3_exec(db, "DELETE FROM servers", NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto rollback;

	result = sqlite3_prepare_v2(db,
		"INSERT INTO servers (protocol) VALUES (?)",
		-1, &stmt, NULL);
	if (result != SQLITE_OK)
		goto rollback;
	result = sqlite3_bind_int(stmt, 1, protocol);
	if (result == SQLITE_OK)
		result = sqlite3_step(stmt);

	sqlite3_finalize(stmt);

	if (result != SQLITE_DONE)
		goto rollback;

done:
	sqlite3_exec(db, "COMMIT", NULL, NULL, NULL);
	return 0;
rollback:
	sqlite3_exec(db, "ROLLBACK", NULL, NULL, NULL);
fail:
	errno = EIO;
	return -1;
}


static int strjoin(char **dest, const char *next, const char *sep)
{
	char *new_dest;
	size_t new_len = strlen(next);

	if (*dest)
	{
		size_t dest_len = strlen(*dest);
		size_t sep_len = strlen(sep);
		size_t new_dest_len = dest_len + new_len + sep_len;
		new_dest = realloc(*dest, new_dest_len + 1);
		if (!new_dest) {
			free(*dest);
			*dest = NULL;
			return -1;
		}
		memcpy(new_dest + dest_len, sep, sep_len);
		memcpy(new_dest + dest_len + sep_len, next, new_len);
		new_dest[new_dest_len] = '\0';
	} else {
		new_dest = malloc(new_len + 1);
		if (!new_dest) {
			return -1;
		}
		memcpy(new_dest, next, new_len + 1);
	}

	*dest = new_dest;
	return 0;
}

char *nds2_data_type_to_string(nds2_data_type data_type)
{
	static const char *sep = " | ";
	char *ret = NULL;

	if (data_type & NDS2_DATA_TYPE_INT16)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_INT16)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_INT32)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_INT32)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_INT64)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_INT64)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_FLOAT32)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_FLOAT32)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_FLOAT64)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_FLOAT64)), sep))
			return ret;
	if (data_type & NDS2_DATA_TYPE_COMPLEX32)
		if (strjoin(&ret, data_type_name(_nds2_data_to_data_type_reverse(NDS2_DATA_TYPE_COMPLEX32)), sep))
			return ret;

	if (!ret)
		ret = strdup(data_type_name(0));

	return ret;
}


char *nds2_channel_type_to_string(nds2_channel_type channel_type)
{
	static const char *sep = " | ";
	char *ret = NULL;

	if (channel_type & NDS2_CHANNEL_TYPE_ONLINE)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_ONLINE)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_RAW)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_RAW)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_RDS)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_RDS)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_STREND)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_STREND)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_MTREND)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_MTREND)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_TEST_POINT)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_TEST_POINT)), sep))
			return ret;
	if (channel_type & NDS2_CHANNEL_TYPE_STATIC)
		if (strjoin(&ret, cvt_chantype_str(_nds2_chantype_to_channel_type_reverse(NDS2_CHANNEL_TYPE_STATIC)), sep))
			return ret;

	if (!ret)
		ret = strdup(cvt_chantype_str(0));

	return ret;
}


/* Initialize NDS client library.  FIXME: this static initialization is not
 * entirely thread safe. */
static int _nds2_startup()
{
	static int started = 0;
	int ret;
	if (started)
	{
		ret = 0;
	} else {
		int result = daq_startup();
		if (result == DAQD_OK)
		{
			ret = 0;
			started = 1;
		} else {
			ret = -1;
			errno = NDS2_ELAST + result;
		}
	}
	return ret;
}


nds2_connection *nds2_open(const char *host, int port, nds2_protocol protocol)
{
	int result;
	nds2_connection *connection = NULL;
	char *db_path = NULL;

	if (_nds2_startup())
		goto fail;

	if (protocol <= NDS2_PROTOCOL_INVALID || protocol > NDS2_PROTOCOL_TRY)
	{
		errno = EINVAL;
		goto fail;
	}

	connection = calloc((size_t)1, sizeof(struct _nds2_connection));
	if (!connection)
		goto fail;

	connection->host = strdup(host);
	if (!connection->host)
		goto fail;

	/* Populate fields. */
	connection->port = port;
	connection->request_in_progress = 0;
	connection->requested_end_time = 0;

	/* Determine name of sqlite database. */
	db_path = _nds2_db_get_path(host, port);
	if (!db_path)
		goto fail;

	/* If the protocol is TRY, first try to look up the protocol in the database,
	 * if the database exists. */
	if (protocol == NDS2_PROTOCOL_TRY)
	{
		result = sqlite3_open_v2(db_path, &connection->db, SQLITE_OPEN_READWRITE | SQLITE_OPEN_SHAREDCACHE, NULL);
		if (result == SQLITE_OK) {
			protocol = _nds2_db_get_protocol(connection->db);
		} else {
			sqlite3_close(connection->db);
			connection->db = NULL;
		}
		if (protocol <= NDS2_PROTOCOL_INVALID)
			protocol = NDS2_PROTOCOL_TRY;
	}

	/* Open the connection. */
	if (protocol == NDS2_PROTOCOL_TRY)
		result = daq_connect(&connection->daq, host, port, nds_try);
	else
		result = daq_connect(&connection->daq, host, port, protocol);

	if (result != DAQD_OK)
	{
		errno = NDS2_ELAST + result;
		goto fail;
	}

	/* Open the database for read-write, if we haven't already. */
	if (!connection->db)
	{
		result = sqlite3_open_v2(db_path, &connection->db, SQLITE_OPEN_READWRITE | SQLITE_OPEN_CREATE | SQLITE_OPEN_SHAREDCACHE, NULL);
		if (result != SQLITE_OK)
			goto fail;
	}

	/* Record the protocol version. */
	result = _nds2_db_put_protocol(connection->db, nds2_get_protocol(connection));
	if (result)
		goto fail;

	/* Register bash_pattern_matches as a custom function. */
	result = _bash_pattern_matches_register_func(connection->db);
	if (result != SQLITE_OK)
	{
		errno = EIO;
		goto fail;
	}

	goto done;
fail:
	/* nds2_close calls daq_disconnect, and daq_disconnect may set errno.
	 * Save errno before calling nds2_close. */
	result = errno;
	nds2_close(connection);
	errno = result;
	connection = NULL;
done:
	free(db_path);
	return connection;
}


void nds2_close(nds2_connection *connection)
{
	if (connection)
	{
		if (connection->db)
		{
			sqlite3_close(connection->db);
			connection->db = NULL;
		}
		daq_disconnect(&connection->daq);
		free(connection->host);
		connection->host = NULL;
		free(connection);
	} else
		errno = EFAULT;
}


char *nds2_get_host(nds2_connection *connection)
{
	char *ret;

	if (connection) {
		ret = strdup(connection->host);
	} else {
		ret = NULL;
		errno = EFAULT;
	}

	return ret;
}


int nds2_get_port(nds2_connection *connection)
{
	int ret;

	if (connection) {
		ret = connection->port;
	} else {
		ret = -EFAULT;
		errno = -ret;
	}

	return ret;
}


nds2_protocol nds2_get_protocol(nds2_connection *connection)
{
	int ret;

	if (connection) {
		ret = connection->daq.nds_versn;
		if (ret == nds_try)
			ret = NDS2_PROTOCOL_TRY;
	} else {
		errno = EFAULT;
		ret = -1;
	}

	return ret;
}


static nds2_channel **_nds2_db_find_channels(
	nds2_connection *connection,
	size_t *count_channels,
	const char *channel_glob,
	nds2_channel_type channel_type_mask,
	nds2_data_type data_type_mask,
	double min_sample_rate,
	double max_sample_rate,
	int use_glob
)
{
	nds2_channel **ret = NULL;
	nds2_channel **channels = malloc(sizeof(nds2_channel *));
	size_t count_channels_found = 0, count_channels_allocated = 1;
	int retval;
	sqlite3_stmt *stmt1 = NULL, *stmt2 = NULL;
	time_t expiry_time = time(NULL);
	int pattern_is_flat;
	const char *stmt2_sql;

	if (!connection->db)
		goto fail;

	if (!channels)
		goto fail;

	if (!channel_glob)
		channel_glob = "*";

	/* Determine if the channel name pattern is flat (has only literals and
	 * wildcards, but no alternatives. */
	{
		bash_pattern *pattern = bash_pattern_compile(channel_glob);
		if (!pattern)
			goto fail;
		pattern_is_flat = bash_pattern_is_flat(pattern);
		bash_pattern_free(pattern);
	}

	/* If the channel name pattern is flat, then compare channel names using
	 * GLOB instead of bash_pattern_matches. GLOB is faster than
	 * bash_pattern_matches because it can use SQL column indices. */
	if (!use_glob)
		stmt2_sql = "SELECT * FROM channels WHERE (name = ?) AND (channel_type & ?) AND (data_type & ?) AND (sample_rate >= ?) AND (sample_rate <= ?)";
	else if (pattern_is_flat)
		stmt2_sql = "SELECT * FROM channels WHERE (name GLOB ?) AND (channel_type & ?) AND (data_type & ?) AND (sample_rate >= ?) AND (sample_rate <= ?)";
	else
		stmt2_sql = "SELECT * FROM channels WHERE bash_pattern_matches(?, name) AND (channel_type & ?) AND (data_type & ?) AND (sample_rate >= ?) AND (sample_rate <= ?)";

	retval = _nds2_db_start_transaction(connection->db)
	|| sqlite3_prepare_v2(connection->db,
		"SELECT timestamp + 86400 >= ? FROM servers",
		-1, &stmt1, NULL)
	|| sqlite3_prepare_v2(connection->db, stmt2_sql, -1, &stmt2, NULL)
	|| sqlite3_bind_int64(stmt1, 1, expiry_time)
	|| sqlite3_bind_text(stmt2, 1, channel_glob, -1, SQLITE_STATIC)
	|| sqlite3_bind_int(stmt2, 2, (int) channel_type_mask)
	|| sqlite3_bind_int(stmt2, 3, (int) data_type_mask)
	|| sqlite3_bind_double(stmt2, 4, min_sample_rate)
	|| sqlite3_bind_double(stmt2, 5, max_sample_rate);

	if (retval != SQLITE_OK)
		goto fail;

	if (sqlite3_step(stmt1) != SQLITE_ROW)
		goto fail;

	if (!sqlite3_column_int(stmt1, 0))
		goto fail;

	while ((retval = sqlite3_step(stmt2)) == SQLITE_ROW)
	{
		nds2_channel *channel;
		const char *name;
		const char *signal_units;

		if (count_channels_found >= count_channels_allocated)
		{
			nds2_channel **new_channels;
			count_channels_allocated <<= 1;
			new_channels = realloc(channels, sizeof(nds2_channel *) * count_channels_allocated);
			if (!new_channels)
				goto fail;
			channels = new_channels;
		}

		channel = malloc(sizeof(nds2_channel));
		if (!channel)
			goto fail;

		name = (const char *) sqlite3_column_text(stmt2, 0);
		if (strlen(name) >= sizeof(channel->name))
		{
			free(channel);
			goto fail;
		}

		strncpy(channel->name, name, sizeof(channel->name));
		channel->channel_type = sqlite3_column_int(stmt2, 1);
		channel->data_type = sqlite3_column_int(stmt2, 2);
		channel->sample_rate = sqlite3_column_double(stmt2, 3);
		channel->signal_gain = (float)sqlite3_column_double(stmt2, 4);
		channel->signal_slope = (float)sqlite3_column_double(stmt2, 5);
		channel->signal_offset = (float)sqlite3_column_double(stmt2, 6);
		signal_units = (const char *)sqlite3_column_text(stmt2, 7);
		if (strlen(signal_units) >= sizeof(channel->signal_units))
		{
			free(channel);
			goto fail;
		}
		strncpy(channel->signal_units, signal_units, sizeof(channel->signal_units));
		channels[count_channels_found] = channel;
		count_channels_found ++;
	}

	if (retval != SQLITE_DONE)
		goto fail;

	if (count_channels_found > 0)
	{
		ret = realloc(channels, sizeof(nds2_channel *) * count_channels_found);
		if (!ret)
			goto fail;
	}
	channels = NULL;
	*count_channels = count_channels_found;
fail:
	if (channels)
	{
		size_t i;
		for (i = 0; i < count_channels_found; i ++)
			free(channels[i]);
		free(channels);
		channels = NULL;
	}
	sqlite3_finalize(stmt1);
	sqlite3_finalize(stmt2);
	_nds2_db_end_transaction(connection->db);
	return ret;
}


static int _nds2_db_put_channels(
	nds2_connection *connection,
	daq_channel_t *channels,
	size_t count_channels,
	time_t expiry_time
)
{
	int result;
	size_t i;
	sqlite3_stmt *stmt;
	char channel_name_buf[sizeof(((daq_channel_t *)NULL)->name) + 16];

	if (!connection->db)
		goto done;

	result = _nds2_db_start_transaction(connection->db);
	if (result != SQLITE_OK)
		goto commit;

	/* Wipe cache */
	result = sqlite3_exec(connection->db, "DELETE FROM servers; DELETE FROM channels;", NULL, NULL, NULL);
	if (result != SQLITE_OK)
		goto rollback;

	/* Write server metadata (protocol and timestamp) */
	result = sqlite3_prepare_v2(connection->db,
		"INSERT INTO servers (protocol, timestamp) VALUES (?, ?)", -1, &stmt, NULL)
	|| sqlite3_bind_int(stmt, 1, nds2_get_protocol(connection))
	|| sqlite3_bind_int64(stmt, 2, expiry_time);
	if (result == SQLITE_OK)
		result = sqlite3_step(stmt);
	sqlite3_finalize(stmt);
	if (result != SQLITE_DONE)
		goto rollback;

	/* Write channel list */
	result = sqlite3_prepare_v2(connection->db,
		"INSERT INTO channels VALUES (?, ?, ?, ?, ?, ?, ?, ?)", -1, &stmt, NULL);
	if (result != SQLITE_OK)
	{
		sqlite3_finalize(stmt);
		goto rollback;
	}
	switch (nds2_get_protocol(connection))
	{
		case NDS2_PROTOCOL_ONE:
			for (i = 0; i < count_channels; i ++)
			{
				static const char *reductions[] = {"min", "max", "mean", "rms"};
				static const char *trend_types[] = {"s-trend", "m-trend"};
				nds2_channel_type channel_type = (channels[i].type == cUnknown ? NDS2_CHANNEL_TYPE_ONLINE : _nds2_chantype_to_channel_type(channels[i].type));
				nds2_data_type data_type = _nds2_data_to_data_type(channels[i].data_type);
				int j, k;

				result = sqlite3_bind_text(stmt, 1, channels[i].name, -1, SQLITE_STATIC)
				|| sqlite3_bind_int(stmt, 2, (int) channel_type)
				|| sqlite3_bind_int(stmt, 3, (int) data_type)
				|| sqlite3_bind_double(stmt, 4, channels[i].rate)
				|| sqlite3_bind_double(stmt, 5, channels[i].s.signal_gain)
				|| sqlite3_bind_double(stmt, 6, channels[i].s.signal_slope)
				|| sqlite3_bind_double(stmt, 7, channels[i].s.signal_offset)
				|| sqlite3_bind_text(stmt, 8, channels[i].s.signal_units, -1, SQLITE_STATIC);
				if (result == SQLITE_OK)
					result = sqlite3_step(stmt);
				if (result != SQLITE_DONE)
				{
					sqlite3_finalize(stmt);
					goto rollback;
				}
				result = sqlite3_reset(stmt);
				if (result != SQLITE_OK)
				{
					sqlite3_finalize(stmt);
					goto rollback;
				}

				/* Fill in records for (implied!) trend channels.
				 * See trend_dtype in trench.c. */
				for (j = 0; j < sizeof(reductions) / sizeof(*reductions); j ++)
				{
					const char *reduction = reductions[j];
					nds2_data_type new_data_type;
					if (j < 2)
					{
						if (data_type <= NDS2_DATA_TYPE_INT64)
							new_data_type = NDS2_DATA_TYPE_INT32;
						else
							new_data_type = NDS2_DATA_TYPE_FLOAT32;
					} else {
						new_data_type = NDS2_DATA_TYPE_FLOAT64;
					}

					for (k = 0; k < sizeof(trend_types) / sizeof(*trend_types); k ++)
					{
						const char *trend_type = trend_types[k];
						nds2_channel_type new_channel_type;
						double rate;

						if (k == 0)
						{
							new_channel_type = NDS2_CHANNEL_TYPE_STREND;
							rate = 1;
						} else {
							new_channel_type = NDS2_CHANNEL_TYPE_MTREND;
							rate = 1. / 60;
						}

						if (snprintf(channel_name_buf, sizeof(channel_name_buf), "%s.%s,%s", channels[i].name, reduction, trend_type) < 0)
						{
							sqlite3_finalize(stmt);
							goto rollback;
						}

						result = sqlite3_bind_text(stmt, 1, channel_name_buf, -1, SQLITE_STATIC)
						|| sqlite3_bind_int(stmt, 2, (int) new_channel_type)
						|| sqlite3_bind_int(stmt, 3, (int) new_data_type)
						|| sqlite3_bind_double(stmt, 4, rate)
						|| sqlite3_bind_double(stmt, 5, channels[i].s.signal_gain)
						|| sqlite3_bind_double(stmt, 6, channels[i].s.signal_slope)
						|| sqlite3_bind_double(stmt, 7, channels[i].s.signal_offset)
						|| sqlite3_bind_text(stmt, 8, channels[i].s.signal_units, -1, SQLITE_STATIC);
						if (result == SQLITE_OK)
							result = sqlite3_step(stmt);
						if (result != SQLITE_DONE)
						{
							sqlite3_finalize(stmt);
							goto rollback;
						}
						result = sqlite3_reset(stmt);
						if (result != SQLITE_OK)
						{
							sqlite3_finalize(stmt);
							goto rollback;
						}
					}
				}
			}
			break;
		case NDS2_PROTOCOL_TWO:
			for (i = 0; i < count_channels; i ++)
			{
				nds2_channel_type channel_type = _nds2_chantype_to_channel_type(channels[i].type);
				nds2_data_type data_type = _nds2_data_to_data_type(channels[i].data_type);
				char *channel_name;

				switch (channel_type)
				{
					case NDS2_CHANNEL_TYPE_RDS:
						if (snprintf(channel_name_buf, sizeof(channel_name_buf), "%s,rds", channels[i].name) < 0)
						{
							sqlite3_finalize(stmt);
							goto rollback;
						}
						channel_name = channel_name_buf;
						break;
					case NDS2_CHANNEL_TYPE_STREND:
						if (snprintf(channel_name_buf, sizeof(channel_name_buf), "%s,s-trend", channels[i].name) < 0)
						{
							sqlite3_finalize(stmt);
							goto rollback;
						}
						channel_name = channel_name_buf;
						break;
					case NDS2_CHANNEL_TYPE_MTREND:
						if (snprintf(channel_name_buf, sizeof(channel_name_buf), "%s,m-trend", channels[i].name) < 0)
						{
							sqlite3_finalize(stmt);
							goto rollback;
						}
						channel_name = channel_name_buf;
						break;
					default:
						channel_name = channels[i].name;
				}
				result = sqlite3_bind_text(stmt, 1, channel_name, -1, SQLITE_STATIC)
				|| sqlite3_bind_int(stmt, 2, (int) channel_type)
				|| sqlite3_bind_int(stmt, 3, (int) data_type)
				|| sqlite3_bind_double(stmt, 4, channels[i].rate)
				|| sqlite3_bind_double(stmt, 5, channels[i].s.signal_gain)
				|| sqlite3_bind_double(stmt, 6, channels[i].s.signal_slope)
				|| sqlite3_bind_double(stmt, 7, channels[i].s.signal_offset)
				|| sqlite3_bind_text(stmt, 8, channels[i].s.signal_units, -1, SQLITE_STATIC);
				if (result == SQLITE_OK)
					result = sqlite3_step(stmt);
				if (result != SQLITE_DONE)
				{
					sqlite3_finalize(stmt);
					goto rollback;
				}
				result = sqlite3_reset(stmt);
				if (result != SQLITE_OK)
				{
					sqlite3_finalize(stmt);
					goto rollback;
				}
			}
			break;
		default:
			/* Should not be reached */
			abort();
	}
	sqlite3_finalize(stmt);

commit:
	result = 0;
	_nds2_db_end_transaction(connection->db);
done:
	return result;
rollback:
	result = -1;
	sqlite3_exec(connection->db, "ROLLBACK", NULL, NULL, NULL);
	return result;
}


static nds2_channel **_nds2_find_channels(
	nds2_connection *connection,
	size_t *count_channels,
	const char *channel_glob,
	nds2_channel_type channel_type_mask,
	nds2_data_type data_type_mask,
	double min_sample_rate,
	double max_sample_rate,
	int use_glob
)
{
	time_t expiry_time;
	nds2_channel **ret;
	int nchannels, retval;
	daq_channel_t *daq_channels = NULL;

	if (!connection || !count_channels)
	{
		errno = EFAULT;
		goto fail;
	}

	/* Attempt query against database. */
	{
		size_t count_channels_found = 1;
		ret = _nds2_db_find_channels(connection, &count_channels_found, channel_glob, channel_type_mask, data_type_mask, min_sample_rate, max_sample_rate, use_glob);
		if (ret || !count_channels_found)
		{
			*count_channels = count_channels_found;
			return ret;
		}
	}

	/* Fail if there is another transfer already in progress. */
	if (connection->request_in_progress)
	{
		errno = NDS2_TRANSFER_BUSY;
		goto fail;
	}

	/* Error reading from cache database or cache database out of date, so
	 * retrieve channel list from server. */
	retval = daq_recv_channels(&connection->daq, NULL, 0, &nchannels);

	if (retval)
	{
		errno = NDS2_ELAST + retval;
		goto fail;
	}

	daq_channels = malloc(sizeof(daq_channel_t) * nchannels);

	if (!daq_channels)
		goto fail;

	expiry_time = time(NULL);
	retval = daq_recv_channels(&connection->daq, daq_channels,
		nchannels, &nchannels);

	if (retval)
	{
		free(daq_channels);
		errno = NDS2_ELAST + retval;
		goto fail;
	}

	if (_nds2_db_put_channels(connection, daq_channels, (size_t) nchannels, expiry_time))
	{
		free(daq_channels);
		errno = EIO;
		goto fail;
	}
	free(daq_channels);

	/* Attempt query against database again. */
	{
		size_t count_channels_found = 1;
		ret = _nds2_db_find_channels(connection, &count_channels_found, channel_glob, channel_type_mask, data_type_mask, min_sample_rate, max_sample_rate, use_glob);
		if (ret || !count_channels_found)
		{
			*count_channels = count_channels_found;
			return ret;
		}
	}

	/* Database query failed. */
	errno = EIO;

fail:
	return NULL;
}


nds2_channel **nds2_find_channels(
	nds2_connection *connection,
	size_t *count_channels,
	const char *channel_glob,
	nds2_channel_type channel_type_mask,
	nds2_data_type data_type_mask,
	double min_sample_rate,
	double max_sample_rate
)
{
	return _nds2_find_channels(connection, count_channels, channel_glob, channel_type_mask, data_type_mask, min_sample_rate, max_sample_rate, 1);
}


int nds2_sizeof_data_type(nds2_data_type data_type)
{
	int ret;

	switch (data_type) {
		case NDS2_DATA_TYPE_INT16:
			ret = 2;
			break;
		case NDS2_DATA_TYPE_INT32:
		case NDS2_DATA_TYPE_FLOAT32:
			ret = 4;
			break;
		case NDS2_DATA_TYPE_INT64:
		case NDS2_DATA_TYPE_FLOAT64:
		case NDS2_DATA_TYPE_COMPLEX32:
			ret = 8;
			break;
		default:
			ret = -EINVAL;
	}

	if (ret < 0)
		errno = -ret;
	return ret;
}


nds2_buffer **nds2_fetch(
	nds2_connection *connection,
	long gps_start,
	long gps_stop,
	const char **channel_names,
	size_t count_channels
) {
	nds2_buffer **ret = NULL;
	unsigned char **ptrs = malloc(sizeof(char *) * count_channels);
	unsigned char **ends = malloc(sizeof(char *) * count_channels);
	int result;
	size_t count_received_channels;
	size_t i;

	if (!connection)
	{
		errno = EFAULT;
		goto fail;
	}

	/* Check that all temporary arrays were allocated. */
	if (!ptrs || !ends)
		goto fail;

	/* Try to request data for indicated times and channels. */
	result = nds2_iterate(connection, gps_start, gps_stop, 0, channel_names,
		count_channels);
	if (result)
		goto fail;

	/* Try to retrieve the first set of buffers. */
	ret = nds2_next(connection, &count_received_channels);
	if (!ret)
		goto fail;

	/* Check that the correct number of buffers arrived. */
	if (count_channels != count_received_channels)
	{
		errno = NDS2_UNEXPECTED_CHANNELS_RECEIVED;
		goto fail;
	}

	/* Reallocate the received buffers to fit the entire request. */
	for (i = 0; i < count_channels; i ++)
	{
		nds2_buffer *new_buf;
		size_t unit_size = nds2_sizeof_data_type(ret[i]->channel.data_type);
		size_t needed_samples = (size_t)( (gps_stop - gps_start) * ret[i]->channel.sample_rate );
		size_t needed_bytes = needed_samples * unit_size;
		new_buf = realloc(ret[i], sizeof(nds2_buffer) + needed_bytes);
		if (!new_buf)
			goto fail;
		ret[i] = new_buf;
		ptrs[i] = ret[i]->data + ret[i]->length * unit_size;
		ends[i] = ret[i]->data + needed_bytes;
		ret[i]->length = needed_samples;
	}

	/* Retrieve more buffers as they arrive. */
	while (1) {
		chan_req_t *chan_reqs;
		char *block_data;

		/* Check to see if we need to retrieve any more data. */
		for (i = 0; i < count_channels; i ++)
			if (ptrs[i] < ends[i])
				break;
		if (i == count_channels)
			break;

		result = daq_recv_next(&connection->daq);
		if (result != DAQD_OK)
		{
			errno = NDS2_ELAST + result;
			goto fail;
		}

		/* Check that the correct number of buffers arrived. */
		if (count_channels != connection->daq.num_chan_request)
		{
			errno = NDS2_UNEXPECTED_CHANNELS_RECEIVED;
			goto fail;
		}

		/* Copy data buffer by buffer. */
		chan_reqs = connection->daq.chan_req_list;
		block_data = daq_get_block_data(&connection->daq);
		for (i = 0; i < count_channels; i ++)
		{
			/* Determine number of bytes received for this channel. */
			int nbytes = chan_reqs[i].status;

			/* Check that no error condition occurred on this channel. */
			if (nbytes < 0)
			{
				errno = NDS2_ELAST - nbytes;
				goto fail;
			}

			/* Check that there is room to copy the new data. */
			if (ptrs[i] + nbytes > ends[i])
			{
				errno = NDS2_UNEXPECTED_DATA_RECEIVED;
				goto fail;
			}

			/* Copy the new data. */
			memcpy(ptrs[i], block_data + chan_reqs[i].offset, (size_t) nbytes);
			ptrs[i] += nbytes;
		}
	}

	/* NDS1 transfers end with a 'termination block', an empty block that
	 * is indistinguisable from a 'data not found' condition. If this is
	 * an NDS1 connection, we must digest the termination block. */
	if (connection->request_in_progress && connection->daq.nds_versn == nds_v1)
	{
		result = daq_recv_next(&connection->daq);
		if (result != DAQD_NOT_FOUND)
		{
			errno = NDS2_ELAST + result;
			goto fail;
		}
	}

	connection->request_in_progress = 0;

	/* No errors; jump to end! */
	goto done;
fail:
	/* Uh-oh, something wrong happened. */
	if (ret)
	{
		for (i = 0; i < count_channels; i ++)
			free(ret[i]);
		free(ret);
		ret = NULL;
	}
done:
	/* Done! */
	free(ptrs);
	free(ends);
	return ret;
}


int nds2_iterate(
	nds2_connection *connection,
	long gps_start,
	long gps_stop,
	long stride,
	const char **channel_names,
	size_t count_channels
)
{
	int ret = -1;
	int result;
	size_t i, j;
	int have_minute_trends = 0;
	nds2_channel_type channel_type_mask = 0xFF;
	double bps = 0;

	if (!connection)
	{
		errno = EFAULT;
		goto fail;
	}

	/* Fail if there is another transfer already in progress. */
	if (connection->request_in_progress)
	{
		errno = NDS2_TRANSFER_BUSY;
		goto fail;
	}

	daq_clear_channel_list(&connection->daq);

	 /* NDS2 connections refuse to serve channels 'online'
	  * (i.e., streaming from the current GPS time) unless the
	  * channel type is 'online'. */
	if (nds2_get_protocol(connection) == NDS2_PROTOCOL_TWO)
	{
		if (gps_start)
		{
			/* Exclude online channels. */
			channel_type_mask ^= NDS2_CHANNEL_TYPE_ONLINE;
		} else {
			/* Exclude raw (i.e., not online) channels. */
			channel_type_mask ^= NDS2_CHANNEL_TYPE_RAW;
		}
	}

	for (i = 0; i < count_channels; i ++)
	{
		daq_channel_t channel;
		size_t count_channels_found = 1;
		nds2_channel **channels = _nds2_find_channels(connection, &count_channels_found, channel_names[i], channel_type_mask, 0xFF, 0., 1e9, 0);

		/* channels is NULL but count_channels_found is 1:
		 * an error occurred in _nds2_find_channels, so fail. */
		if (!channels && count_channels_found)
			goto fail;

		/* count_channels_found is 0: no error, but no channels were found,
		 * so fail. */
		if (!count_channels_found)
		{
			free(channels);
			errno = NDS2_NO_SUCH_CHANNEL;
			goto fail;
		}

		if (channels[0]->channel_type == NDS2_CHANNEL_TYPE_MTREND)
			have_minute_trends = 1;

		memset(&channel, 0, sizeof(channel));
		daq_init_channel(&channel, channels[0]->name, _nds2_chantype_to_channel_type_reverse(channels[0]->channel_type), channels[0]->sample_rate, _nds2_data_to_data_type_reverse(channels[0]->data_type));

		bps += channels[0]->sample_rate * nds2_sizeof_data_type(channels[0]->data_type);

		for (j = 0; j < count_channels_found; j ++)
			free(channels[j]);
		free(channels);

		result = daq_request_channel_from_chanlist(&connection->daq, &channel);
		if (result != DAQD_OK)
		{
			errno = NDS2_ELAST + result;
			goto fail;
		}
	}

	/* NDS2 protocol does not support indefinite GPS stop time */
	if (nds2_get_protocol(connection) == NDS2_PROTOCOL_TWO && !gps_stop)
	{
		gps_stop = 1893024016; /* A safely huge value of January 1, 2040. */
	}

	if (stride == 0)
	{
		if (gps_start) /* offline request */
		{
			if (have_minute_trends) /* request includes minute trends */
			{
				/* start with smallest stride that is a multiple of 60 seconds and
				 * whose frame size in bytes is greater than or equal to
				 * _NDS2_TYPICAL_BYTES_PER_FRAME. */
				stride = 60 * ceil(_NDS2_TYPICAL_BYTES_PER_FRAME / (60 * bps));
			} else { /* request does not include minute trends */
				/* start with smallest stride that is a multiple of 1 second and
				 * whose frame size in bytes is greater than or equal to
				 * _NDS2_TYPICAL_BYTES_PER_FRAME. */
				stride = ceil(_NDS2_TYPICAL_BYTES_PER_FRAME / bps);
			}

			if (stride > gps_stop - gps_start) /* stride is longer than request duration */
			{
				stride = gps_stop - gps_start; /* set stride to request duration */
			}
		} else { /* online request */
			if (have_minute_trends) /* request includes minute trends */
				stride = 60; /* use shortest possible stride, 60 seconds */
			else
				stride = 1; /* use shortest possible stride, 1 second */
		}
	}

	/* Raise an error if any of the requested channels are minute-trends and
	 * the start or stop times are indivisible by 60. */
	if (have_minute_trends && (gps_start % 60 || gps_stop % 60 || stride % 60))
	{
		errno = NDS2_MINUTE_TRENDS;
		goto fail;
	}

	result = daq_request_data(&connection->daq, (time_t) gps_start, (time_t) gps_stop, stride);
	if (result != DAQD_OK)
	{
		errno = NDS2_ELAST + result;
		goto fail;
	}

	connection->requested_end_time = gps_stop;
	connection->request_in_progress = 1;

	ret = 0;
fail:
	return ret;
}


nds2_buffer **nds2_next(
	nds2_connection *connection,
	size_t *count_channels
)
{
	nds2_buffer **ret = NULL;
	size_t count_received_channels;
	chan_req_t *chan_reqs;
	size_t i;
	int result;

	if (!connection)
	{
		errno = EFAULT;
		goto done;
	}

	if (!connection->request_in_progress)
	{
		errno = NDS2_EOS;
		goto done;
	}

	result = daq_recv_next(&connection->daq);
	if (result != DAQD_OK)
	{
		errno = NDS2_ELAST + result;
		goto done;
	}

	count_received_channels = connection->daq.num_chan_request;

	ret = malloc(sizeof(nds2_buffer *) * count_received_channels);
	if (!ret)
		goto done;

	chan_reqs = connection->daq.chan_req_list;

	for (i = 0; i < count_received_channels; i ++)
	{
		int nbytes = chan_reqs[i].status;

		if (nbytes < 0)
		{
			errno = NDS2_ELAST - nbytes;
			break;
		}

		/* Allocate buffer, making room for data payload. */
		ret[i] = malloc(sizeof(nds2_buffer) + nbytes);
		if (!ret[i])
			break;

		/* Copy channel information. */
		result = _nds2_chan_req_to_channel(&ret[i]->channel, &chan_reqs[i]);
		if (result)
		{
			free(ret[i]);
			break;
		}

		/* Set buffer's metadata. */
		ret[i]->gps_seconds = daq_get_block_gps(&connection->daq);
		ret[i]->gps_nanoseconds = daq_get_block_gpsn(&connection->daq);
		ret[i]->length = nbytes / nds2_sizeof_data_type(ret[i]->channel.data_type);

		/* Copy buffer payload. */
		memcpy(ret[i]->data, daq_get_block_data(&connection->daq) + chan_reqs[i].offset, (size_t) nbytes);
	}

	if (i != count_received_channels)
	{
		size_t j;
		for (j = 0; j < i; j ++)
			free(ret[j]);
		free(ret);
		ret = NULL;
		goto done;
	}

	if (connection->requested_end_time != 0)
	{
		int has_next = 0;
		for (i = 0; i < count_received_channels; i ++)
		{
			if (ret[i]->gps_seconds < connection->requested_end_time
				&& (connection->requested_end_time - ret[i]->gps_seconds)
				* ret[i]->channel.sample_rate > ret[i]->length)
				has_next = 1;
		}
		if (!has_next)
		{
			connection->request_in_progress = 0;

			/* NDS1 transfers end with a 'termination block', an empty block that
			 * is indistinguisable from a 'data not found' condition. If this is
			 * an NDS1 connection, we must digest the termination block. */
			if (connection->daq.nds_versn == nds_v1)
			{
				result = daq_recv_next(&connection->daq);
				if (result != DAQD_NOT_FOUND)
				{
					errno = NDS2_ELAST + result;
					for (i = 0; i < count_received_channels; i ++)
						free(ret[i]);
					free(ret);
					ret = NULL;
					goto done;
				}
			}
		}
	}

	*count_channels = count_received_channels;
done:
	return ret;
}


int nds2_has_next(nds2_connection *connection)
{
	int ret;

	if (!connection)
	{
		errno = EFAULT;
		ret = -1;
	} else {
		return connection->request_in_progress;
	}

	return ret;
}


int nds2_clear_cache(nds2_connection *connection)
{
	int ret = -1;
	int result;

	if (!connection)
	{
		errno = EFAULT;
		goto fail;
	}

	result = sqlite3_exec(connection->db,
		"DROP INDEX IF EXISTS channel_name_index; DROP TABLE IF EXISTS servers; DROP TABLE IF EXISTS channels;", NULL, NULL, NULL);
	if (result != SQLITE_OK)
	{
		errno = EIO;
		goto fail;
	}

	ret = 0;
fail:
	return ret;
}


void nds2_perror(const char *s)
{
	fprintf(stderr, "%s: %s\n", s, nds2_strerror(errno));
}


const char *nds2_strerror(int errnum)
{
	const char *errstr;

	switch (errnum) {
		case NDS2_EOS:
			errstr = "Reached end of stream.";
			break;
		case NDS2_CHANNEL_NAME_TOO_LONG:
			errstr = "Channel name is too long.";
			break;
		case NDS2_UNEXPECTED_CHANNELS_RECEIVED:
			errstr = "Server sent more channels than were expected.";
			break;
		case NDS2_UNEXPECTED_DATA_RECEIVED:
			errstr = "Server sent more data than were expected.";
			break;
		case NDS2_MINUTE_TRENDS:
			errstr = "One or more requested channels are minute-trends, but requested start and stop times are not multiples of 60 seconds.";
			break;
		case NDS2_TRANSFER_BUSY:
			errstr = "Another transfer is already in progress. Complete the transfer or retry on a new connection object.";
			break;
		case NDS2_NO_SUCH_CHANNEL:
			errstr = "No such channel.";
			break;
		default:
			if (errnum >= NDS2_ELAST)
				errstr = daq_strerror(errnum - NDS2_ELAST);
			else
				errstr = strerror(errnum);
	}

	return errstr;
}
