/*
 * Decompiled with CFR 0.152.
 */
package org.mvndaemon.mvnd.common;

import java.io.IOException;
import java.lang.management.ManagementFactory;
import java.nio.BufferOverflowException;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.MappedByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.channels.FileLock;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.mvndaemon.mvnd.common.BufferCaster;
import org.mvndaemon.mvnd.common.BufferHelper;
import org.mvndaemon.mvnd.common.DaemonException;
import org.mvndaemon.mvnd.common.DaemonExpirationStatus;
import org.mvndaemon.mvnd.common.DaemonInfo;
import org.mvndaemon.mvnd.common.DaemonState;
import org.mvndaemon.mvnd.common.DaemonStopEvent;
import org.mvndaemon.mvnd.common.Os;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DaemonRegistry
implements AutoCloseable {
    private static final Logger LOGGER = LoggerFactory.getLogger(DaemonRegistry.class);
    private static final int MAX_LENGTH = 32768;
    private static final long LOCK_TIMEOUT_MS = 20000L;
    private final Path registryFile;
    private static final Map<Path, Object> locks = new ConcurrentHashMap<Path, Object>();
    private final Object lck;
    private final FileChannel channel;
    private MappedByteBuffer buffer;
    private long size;
    private final Map<String, DaemonInfo> infosMap = new HashMap<String, DaemonInfo>();
    private final List<DaemonStopEvent> stopEvents = new ArrayList<DaemonStopEvent>();
    private static final int PROCESS_ID = DaemonRegistry.getProcessId0();

    public DaemonRegistry(Path registryFile) {
        Path absPath = registryFile.toAbsolutePath().normalize();
        this.lck = locks.computeIfAbsent(absPath, p -> new Object());
        this.registryFile = absPath;
        try {
            if (!Files.isRegularFile(absPath, new LinkOption[0]) && !Files.isDirectory(absPath.getParent(), new LinkOption[0])) {
                Files.createDirectories(absPath.getParent(), new FileAttribute[0]);
            }
            this.channel = FileChannel.open(absPath, StandardOpenOption.CREATE, StandardOpenOption.READ, StandardOpenOption.WRITE);
            this.size = this.nextPowerOf2(this.channel.size(), 32768L);
            this.buffer = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.size);
        }
        catch (IOException e) {
            throw new DaemonException(e);
        }
    }

    private long nextPowerOf2(long a, long min) {
        long b;
        for (b = min; b < a; b <<= 1) {
        }
        return b;
    }

    @Override
    public void close() {
        try {
            this.channel.close();
        }
        catch (IOException e) {
            throw new DaemonException("Error closing registry", e);
        }
    }

    public Path getRegistryFile() {
        return this.registryFile;
    }

    public DaemonInfo get(String daemonId) {
        this.read();
        return this.infosMap.get(daemonId);
    }

    public List<DaemonInfo> getAll() {
        this.read();
        return new ArrayList<DaemonInfo>(this.infosMap.values());
    }

    public List<DaemonInfo> getIdle() {
        this.read();
        return this.infosMap.values().stream().filter(di -> di.getState() == DaemonState.Idle).collect(Collectors.toList());
    }

    public List<DaemonInfo> getNotIdle() {
        return this.infosMap.values().stream().filter(di -> di.getState() != DaemonState.Idle).collect(Collectors.toList());
    }

    public List<DaemonInfo> getCanceled() {
        this.read();
        return this.infosMap.values().stream().filter(di -> di.getState() == DaemonState.Canceled).collect(Collectors.toList());
    }

    public void remove(String daemonId) {
        this.update(() -> this.infosMap.remove(daemonId));
    }

    public void markState(String daemonId, DaemonState state) {
        LOGGER.debug("Marking busy by id: {}", (Object)daemonId);
        this.update(() -> this.infosMap.computeIfPresent(daemonId, (id, di) -> di.withState(state)));
    }

    public void storeStopEvent(DaemonStopEvent stopEvent) {
        LOGGER.debug("Storing daemon stop event with timestamp {}", (Object)stopEvent.getTimestamp());
        this.update(() -> this.stopEvents.add(stopEvent));
    }

    public List<DaemonStopEvent> getStopEvents() {
        this.read();
        return this.doGetDaemonStopEvents();
    }

    protected List<DaemonStopEvent> doGetDaemonStopEvents() {
        return new ArrayList<DaemonStopEvent>(this.stopEvents);
    }

    public void removeStopEvents(Collection<DaemonStopEvent> events) {
        LOGGER.debug("Removing {} daemon stop events from registry", (Object)events.size());
        this.update(() -> this.stopEvents.removeAll(events));
    }

    public void store(DaemonInfo info) {
        LOGGER.debug("Storing daemon {}", (Object)info);
        this.update(() -> this.infosMap.put(info.getId(), info));
    }

    public static int getProcessId() {
        return PROCESS_ID;
    }

    private void read() {
        this.doUpdate(null);
    }

    private void update(Runnable updater) {
        this.doUpdate(updater);
    }

    private void doUpdate(Runnable updater) {
        if (!Files.isReadable(this.registryFile)) {
            throw new DaemonException("Registry became unaccessible");
        }
        Object object = this.lck;
        synchronized (object) {
            long deadline = System.currentTimeMillis() + 20000L;
            while (System.currentTimeMillis() < deadline) {
                try (FileLock l = this.tryLock();){
                    long ns;
                    String daemonId;
                    int i;
                    BufferCaster.cast(this.buffer).position(0);
                    this.infosMap.clear();
                    int nb = this.buffer.getInt();
                    for (i = 0; i < nb; ++i) {
                        daemonId = this.readString();
                        String javaHome = this.readString();
                        String mavenHome = this.readString();
                        int pid = this.buffer.getInt();
                        String address = this.readString();
                        byte[] token = new byte[16];
                        this.buffer.get(token);
                        String locale = this.readString();
                        ArrayList<String> opts = new ArrayList<String>();
                        int nbOpts = this.buffer.getInt();
                        for (int j = 0; j < nbOpts; ++j) {
                            opts.add(this.readString());
                        }
                        DaemonState state = DaemonState.values()[this.buffer.get()];
                        long lastIdle = this.buffer.getLong();
                        long lastBusy = this.buffer.getLong();
                        DaemonInfo di = new DaemonInfo(daemonId, javaHome, mavenHome, pid, address, token, locale, opts, state, lastIdle, lastBusy);
                        this.infosMap.putIfAbsent(di.getId(), di);
                    }
                    this.stopEvents.clear();
                    nb = this.buffer.getInt();
                    for (i = 0; i < nb; ++i) {
                        daemonId = this.readString();
                        long date = this.buffer.getLong();
                        byte ord = this.buffer.get();
                        DaemonExpirationStatus des = ord >= 0 ? DaemonExpirationStatus.values()[ord] : null;
                        String reason = this.readString();
                        DaemonStopEvent se = new DaemonStopEvent(daemonId, date, des, reason);
                        this.stopEvents.add(se);
                    }
                    if (updater != null) {
                        updater.run();
                        BufferCaster.cast(this.buffer).position(0);
                        this.buffer.putInt(this.infosMap.size());
                        for (DaemonInfo di : this.infosMap.values()) {
                            this.writeString(di.getId());
                            this.writeString(di.getJavaHome());
                            this.writeString(di.getMvndHome());
                            this.buffer.putInt(di.getPid());
                            this.writeString(di.getAddress());
                            this.buffer.put(di.getToken());
                            this.writeString(di.getLocale());
                            this.buffer.putInt(di.getOptions().size());
                            for (String opt : di.getOptions()) {
                                this.writeString(opt);
                            }
                            this.buffer.put((byte)di.getState().ordinal());
                            this.buffer.putLong(di.getLastIdle());
                            this.buffer.putLong(di.getLastBusy());
                        }
                        this.buffer.putInt(this.stopEvents.size());
                        for (DaemonStopEvent dse : this.stopEvents) {
                            this.writeString(dse.getDaemonId());
                            this.buffer.putLong(dse.getTimestamp());
                            this.buffer.put((byte)(dse.getStatus() == null ? -1 : dse.getStatus().ordinal()));
                            this.writeString(dse.getReason());
                        }
                    }
                    if (this.buffer.remaining() >= this.buffer.position() * 2 && (ns = this.nextPowerOf2(this.buffer.position(), 32768L)) != this.size) {
                        this.size = ns;
                        LOGGER.info("Resizing registry to {} kb due to buffer underflow", (Object)(this.size / 1024L));
                        l.release();
                        BufferHelper.closeDirectByteBuffer(this.buffer, arg_0 -> ((Logger)LOGGER).debug(arg_0));
                        this.channel.truncate(this.size);
                        try {
                            this.buffer = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.size);
                        }
                        catch (IOException ex) {
                            throw new DaemonException("Could not resize registry " + this.registryFile, ex);
                        }
                    }
                    return;
                }
                catch (BufferOverflowException e) {
                    this.size <<= 1;
                    LOGGER.info("Resizing registry to {} kb due to buffer overflow", (Object)(this.size / 1024L));
                    try {
                        this.buffer = this.channel.map(FileChannel.MapMode.READ_WRITE, 0L, this.size);
                    }
                    catch (IOException ex) {
                        ex.addSuppressed(e);
                        throw new DaemonException("Could not resize registry " + this.registryFile, ex);
                    }
                }
                catch (IOException e) {
                    throw new DaemonException("Exception while " + (updater != null ? "updating " : "reading ") + this.registryFile, e);
                }
                catch (ArrayIndexOutOfBoundsException | IllegalStateException | BufferUnderflowException e) {
                    String absPath = this.registryFile.toAbsolutePath().normalize().toString();
                    LOGGER.warn("Invalid daemon registry info, trying to recover from this issue. If you keep getting this warning, try deleting the `registry.bin` file at [" + absPath + "]", (Throwable)e);
                    this.reset();
                    return;
                }
            }
            throw new RuntimeException("Could not lock " + this.registryFile + " within " + 20000L + " ms");
        }
    }

    private FileLock tryLock() {
        try {
            return this.channel.tryLock(0L, this.size, false);
        }
        catch (IOException e) {
            throw new DaemonException("Could not lock " + this.registryFile, e);
        }
    }

    private void reset() {
        this.infosMap.clear();
        this.stopEvents.clear();
        BufferCaster.cast(this.buffer).clear();
        this.buffer.putInt(0);
        this.buffer.putInt(0);
    }

    private static int getProcessId0() {
        String pid;
        block8: {
            if (Os.current() == Os.LINUX) {
                try {
                    Path self = Paths.get("/proc/self", new String[0]);
                    if (!Files.exists(self, new LinkOption[0])) break block8;
                    pid = self.toRealPath(new LinkOption[0]).getFileName().toString();
                    if (pid.equals("self")) {
                        LOGGER.debug("/proc/self symlink could not be followed");
                        break block8;
                    }
                    LOGGER.debug("loading own PID from /proc/self link: {}", (Object)pid);
                    try {
                        return Integer.parseInt(pid);
                    }
                    catch (NumberFormatException x) {
                        LOGGER.warn("Unable to determine PID from malformed /proc/self link `" + pid + "`");
                    }
                }
                catch (IOException ignored) {
                    LOGGER.debug("could not load /proc/self", (Throwable)ignored);
                }
            }
        }
        String vmname = ManagementFactory.getRuntimeMXBean().getName();
        pid = vmname.split("@", 0)[0];
        LOGGER.debug("loading own PID from VM name: {}", (Object)pid);
        try {
            return Integer.parseInt(pid);
        }
        catch (NumberFormatException x) {
            int rpid = new Random().nextInt(65536);
            LOGGER.warn("Unable to determine PID from malformed VM name `" + vmname + "`, picked a random number=" + rpid);
            return rpid;
        }
    }

    protected String readString() {
        short sz = this.buffer.getShort();
        if (sz == -1) {
            return null;
        }
        if (sz < -1 || sz > 1024) {
            throw new IllegalStateException("Bad string size: " + sz);
        }
        byte[] buf = new byte[sz];
        this.buffer.get(buf);
        return new String(buf, StandardCharsets.UTF_8);
    }

    protected void writeString(String str) {
        if (str == null) {
            this.buffer.putShort((short)-1);
            return;
        }
        byte[] buf = str.getBytes(StandardCharsets.UTF_8);
        if (buf.length > 1024) {
            LOGGER.warn("Attempting to write string longer than 1024 bytes: '{}'. Please raise an issue.", (Object)str);
            str = str.substring(0, 1033);
            while (buf.length > 1024) {
                str = str.substring(0, str.length() - 12) + "\u2026";
                buf = str.getBytes(StandardCharsets.UTF_8);
            }
        }
        this.buffer.putShort((short)buf.length);
        this.buffer.put(buf);
    }

    protected ByteBuffer buffer() {
        return this.buffer;
    }

    public String toString() {
        return String.format("DaemonRegistry[file=%s]", this.registryFile);
    }
}

