
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>

#include "conditionals.h"
#include "common.h"
#include "passfile.h"
#include "cipher.h"
#include "base64/base64.h"

/* Limits local to this file -- the rest of murk can deal with
   arbitarily long keys and ivs */
#define KEY_MAXLEN      256
#define IV_MAXLEN       256
#define KEYLIST_ENTRIES 32

struct keystore {
   char keyword[KEYWORD_MAXLEN +1];
   char cipher[CIPHER_TYPE_MAXLEN + 1];
   char key[KEY_MAXLEN];                  /* Base64 encoded, null terminated */
   char iv[IV_MAXLEN];                    /* Base64 encoded, null terminated */
   struct keystore *next;
};


/******** Helper functions **********/

/*
 * Finds the key for the given keyword and cipher
 * returns 0 for found
 *        -1 for not found 
 *            (entry->next pointed at next free slot)
 */
static int
find_key(struct keystore *keylist, const char *keyword, const char *cipher, 
         struct keystore **entry)
{
   struct keystore *this_entry;

   this_entry = keylist;

   /* Loop through the keys */
   while (this_entry) {

      /* This is inefficient but only done once for each entry */
      if (0 == strcmp(this_entry->keyword, keyword) &&
          0 == strcmp(this_entry->cipher, cipher) ) {
         *entry = this_entry;
         OK;
         }
     
      if (0 == this_entry->next) break;

      this_entry = this_entry->next;
      }

   /* If this point is reached the key wasn't found and this_entry->next is
      pointed at the next available slot in case insertion is required */
   *entry = this_entry;
   return -1;
}



/*
 * Reads the keyfile into the given data structure
 */
static int
read_keyfile(FILE *f, struct keystore **list)
{
   int res;
   int i;
   struct keystore *entry,
                   *prev;

   *list = 0;
   prev  = 0;
   i     = 0;

   /* Ignore empty files */
   if (NULL == f) OK;

   /* Loop through the keyfile */
   while (1) {
      entry       = XMALLOC(struct keystore, 1);
      entry->next = 0;

      /* Get the raw entry */
      res = fscanf(f, "%32s %32s %256s %256s",
                    entry->cipher, entry->keyword, entry->key, entry->iv);

      if (EOF == res) {
         XFREE(entry);
         OK;
         }
      else if (4 != res) {
         ERROR_1("Cannot parse line %d in keyfile", i + 1);
         }

      /* Make note of the head of the list */
      if (0 == *list) *list = entry;

      /* Prepare for the next iteration */
      if (prev) prev->next = entry;
      prev = entry;
      ++i;
      }
}


/*
 * Writes the datastructure to the given keyfile
 */
static int
write_keyfile(FILE *f, struct keystore *ks)
{
   struct keystore *entry;

   entry = ks;

   /* Loop through the keyfile */
   while (entry) {

      /* Write the preamble */
      fprintf(f, "%s %s %s %s\n", 
               entry->cipher, entry->keyword, entry->key, entry->iv);

      entry = entry->next;
      }

   OK;
}

/*
 * Opens the keyfile with the given mode
 */
static int
open_keyfile(char *name, const char *mode, FILE **f)
{
   char *actual;
   char *home_dir;
   size_t len;
   struct stat file_stat;

   /* Use the default if no keyfile has been specified */
   if (0 == name) {
      home_dir = getenv("HOME");

      if (0 == home_dir) {
         ERROR("Cannot get $HOME environment variable");
         }
    
      len = strlen(home_dir);
      if (len > FILENAME_MAXLEN) {
         ERROR("$HOME environment variable suspiciously long");
         }

      actual = XMALLOC(char, len + 10);
      sprintf(actual, "%s/.murk", home_dir);
      }
   else {
      actual = name;
      }

   /* Stat the target and warn if its group/world readable */
   if ( 0 == strcmp(mode, "r") &&
       -1 != stat(actual, &file_stat)) {
      if (file_stat.st_mode & (S_IROTH | S_IWOTH | S_IRGRP | S_IWGRP )) {
         WARNING("Keyfile is readable by other users");
         }
      }

   /* Open and lock the file */
   *f = fopen(actual, mode);
   if (NULL != *f) flockfile(*f);

   /* Free the name */
   if (0 == name) XFREE(actual);

   if (NULL == *f && ENOENT != errno) {
      PERROR("Cannot open keyfile");
      }

   OK;
}


/*
 * Cleans out the list of the users keys from memory
 */
void
burn_keylist(struct keystore *keylist)
{
   struct keystore *entry,
                   *next;

   entry = keylist;

   while (entry) {
      next = entry->next;
      XBURN(entry, sizeof(struct keystore));
      entry = next;
      }
}


/******** Entry points **********/

/*
 * Writes the key/iv to the given config for the given config
 */
int
restore_key_iv (struct config *conf)
{
   FILE *keyfile;
   struct keystore *entry, *keylist;
   struct cipher *cipher;
   char key_from_file[KEY_MAXLEN];
   char iv_from_file[IV_MAXLEN];
   size_t keylen, ivlen;

   int ret = 0;

   ASSERT( open_keyfile(conf->keyfile, "r", &keyfile));
   cipher  = conf->cipher;

   do {
      LASSERT( read_keyfile(keyfile, &keylist));

      if (0 != find_key(keylist, conf->keyword, cipher->name, &entry)) {
         LERROR_1("No known key for keyword %s", conf->keyword);
         }

      /* Convert the key and iv from base64 */
      keylen = base64_decode(entry->key, key_from_file);
      ivlen  = base64_decode(entry->iv, iv_from_file);

      if (-1 == keylen) {
         LERROR("Error decoding key in keyfile");
         }

      if (-1 == ivlen) {
         LERROR("Error decoding iv in keyfile");
         }

      if (cipher->keylen != keylen) {
         LERROR("Key found in keyfile but has incompatible length");
         }
      
      if (cipher->ivlen != ivlen) {
         LERROR("IV found in keyfile but has incompatible length");
         }
      
      memcpy(cipher->key, key_from_file, cipher->keylen);
      memcpy(cipher->iv,  iv_from_file,  cipher->ivlen);

      } while (0);

   burn_keylist(keylist);
   funlockfile(keyfile);
   fclose(keyfile);
   return ret;
}



/*
 * Writes the given key, cipher, keyword entry to file 
 */
int
save_key_iv (struct config *conf)
{

   FILE *keyfile;
   struct keystore *entry, 
                   *keylist,
                   *ks;
   char *key_for_file, 
        *iv_for_file;
   struct cipher *cipher;
   size_t keylen, ivlen;
   mode_t old_mask;

   int ret = 0;
   
   cipher  = conf->cipher;

   /* Convert the key and iv to base64 */
   keylen = base64_encode(cipher->key, cipher->keylen,  &key_for_file);
   ivlen  = base64_encode(cipher->iv, cipher->ivlen,  &iv_for_file);

   if (keylen > KEY_MAXLEN) {
      ERROR("Key too long to write to keyfile");
      }

   if (ivlen > IV_MAXLEN) {
      ERROR("IV too long to write to keyfile");
      }

   ASSERT( open_keyfile(conf->keyfile, "r", &keyfile));

   /* See if an existing entry needs to be edited */
   do {
      LASSERT( read_keyfile(keyfile, &keylist));

      if (0 != find_key(keylist, conf->keyword, cipher->name, &entry)) {
         /* This is an add to the keylist */
         ks = XMALLOC(struct keystore, 1);
         strcpy(ks->keyword, conf->keyword);
         strcpy(ks->cipher, cipher->name);
         strcpy(ks->key, key_for_file);
         strcpy(ks->iv, iv_for_file);
	 ks->next = 0;
   
         /* Add the entry on to the end of the list (if one exists) */
         if (entry) {
            entry->next = ks;
            }
         else {
            keylist = ks;
            }
         }
      else {
         /* This is an edit of the existing entry*/
         strcpy(entry->key, key_for_file);
         strcpy(entry->iv,  iv_for_file);
         }

      /* These were malloc'ed within the base64 function */
      XBURN(key_for_file, keylen);
      XBURN(iv_for_file, ivlen);

      /* The read has finished */
      if (NULL != keyfile) {
         funlockfile(keyfile);
         fclose(keyfile);
         }

      /* Set the umask for privacy */
      old_mask = umask(S_IROTH | S_IWOTH | S_IRGRP | S_IWGRP);

      /* Start the write */
      LASSERT( open_keyfile(conf->keyfile, "w", &keyfile));
      LASSERT( write_keyfile(keyfile, keylist));
 
      /* Reset the umask */
      umask(old_mask);

      } while(0);

   burn_keylist(keylist);
   funlockfile(keyfile);
   fclose(keyfile);

   return ret;
}
