
/* $Id: murk.c,v 1.16 2005/03/29 19:25:40 alien-science Exp $ */

#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>

#include "getopt/getopt.h"
#include "common.h"
#include "prog_config.h"
#include "cipher.h"
#include "checksum.h"
#include "file.h"

int g_verbose;

static struct option _longopts[] = {
      { "blocksize",    required_argument,   NULL, 'b' },
      { "output",       required_argument,   NULL, 'o' },
      { "no-compress",  no_argument,         NULL, '0' },
      { "cipher",       required_argument,   NULL, 'c' },
      { "digest",       required_argument,   NULL, 'i' },
      { "decrypt",      no_argument,         NULL, 'd' },
      { "help",         no_argument,         NULL, 'h' },
      { "verbose",      no_argument,         NULL, 'v' },
      { "keyfile",      required_argument,   NULL, 'k' },
      { "addkey",       no_argument,         NULL, 'a' },
      { "noprompt",     no_argument,         NULL, 'n' },
      { NULL,           0,                   NULL, 0}
      };


void
usage()
{
   printf("Usage : murk [options] keyword [file]                            \n"
          " Where options are..                                             \n"
          " -h   | --help                                                   \n"
          " -d   | --decrypt        Decrypt                                 \n"
          " -v   | --verbose        Be verbose                              \n"
          " -o f | --output f       Output to file                          \n"
          " -0   | --no-compress    Turn off compression                    \n"
          " -c x | --cipher x       Use alternative cipher                  \n"
          " -i x | --digest x       Use alternative digest                  \n"
          " -n   | --noprompt       Get the key from a file                 \n"
          " -k f | --keyfile f      Use this file for the keys              \n"
          " -b n | --blocksize n    Maximum blocksize (KB) to use           \n"
          " -a   | --addkey         Prompt for a password, generate a key   \n"
          "                         and then add this key to the keyfile    \n"
          " Tips:                                                           \n"
          " * The keyword is used to generate the salt for generating the   \n"
          "   key and the checksum used to find blocks.                     \n"
          " * If no filename is given, murk encrypts stdin to stdout.       \n"
          " * Using - for output file encrypts to stdout.                   \n"
          " * Default keyfile is $HOME/.murk                                \n"
          " * OpenSSL provides the ciphers and usually makes available:     \n"
          "   aes-128, aes-192, aes-256, des, idea, rc2-64, rc2, bf, cast,  \n"
          "   rc5, des-ede, des-ede3 and null (useful for debugging)        \n");

   exit(1);
}

int 
main (int argc, char *argv[])
{
   extern char *optarg;
   extern int optind;
   extern int optopt;
   extern int opterr;

   char ch;
   char cipher_name[CIPHER_TYPE_MAXLEN];
   char digest_name[CIPHER_TYPE_MAXLEN];
   struct config conf;
   int rc;
   size_t len;
   char *infile  = 0;
   char *outfile = 0;
   struct rlimit no_core = { 0, 0 };

   /* This may be running unattended so don't dump core */
   setrlimit(RLIMIT_CORE, &no_core);

   /* Set defaults */
   g_verbose            = 0;
   conf.is_compressing  = 1;
   conf.is_encrypting   = 1; 
   conf.max_block_size  = 20 * 1024;
   conf.cipher          = 0;
   conf.noprompt        = 0;
   conf.addkey          = 0;
   conf.keyword         = 0;
   conf.keyfile         = 0;

   strcpy(cipher_name, "bf");
   strcpy(digest_name, "md5");   /* not being usd for signatures */

   /* Overide with user options */
   while ((ch = getopt_long(argc, argv, "b:o:0c:dhvk:an", 
                            _longopts, NULL)) != -1) {
      
      switch (ch) {
         case 'b':
            sscanf(optarg,"%u", &(conf.max_block_size));
            /* Limit on max blocksize currently set by checksum algorithm */
            if (conf.max_block_size > BLOCKSIZE_MAXKB) {
               fatal("Cannot handle blocksizes more than 16MB");
               }
            conf.max_block_size *= 1024;
            break;
         case 'o':
            len = strlen(optarg);
            if (len > FILENAME_MAXLEN - 10) {
               fatal("Output filename suspiciously long");
               }
            outfile = XMALLOC(char, len + 10);
            strcpy(outfile, optarg);
            break;
         case '0':
            conf.is_compressing = 0;
            break;
         case 'c':
            strncpy(cipher_name, optarg, CIPHER_TYPE_MAXLEN - 1);
            cipher_name[CIPHER_TYPE_MAXLEN] = 0;
            break;
         case 'i':
            strncpy(digest_name, optarg, CIPHER_TYPE_MAXLEN - 1);
            digest_name[CIPHER_TYPE_MAXLEN] = 0;
            break;
         case 'd':
            conf.is_encrypting = 0; 
            break;
         case 'v':
            g_verbose = 1; 
            break;
         case 'n':
            conf.noprompt = 1;
            break;
         case 'a':
            conf.addkey = 1;
            break;
         case 'k':
            len = strlen(optarg);
            if (len > FILENAME_MAXLEN) {
               fatal("Keyfile name suspiciously long");
               }
            conf.keyfile = XMALLOC(char, len + 1);
            strcpy(conf.keyfile, optarg);
            break;
         case 'h':
         default:
            usage();
         }
      }

   argc -= optind;
   argv += optind;

   /* Check the parameters -- keyword is compulsory */
   if (argc < 1) {
      usage();
      }

   /* Consider the keyword */
   len = strlen(argv[0]);
   if (len > KEYWORD_MAXLEN) {
      fatal("Keyword longer than expected");
      }

   conf.keyword = XMALLOC(char, len + 1);
   strcpy(conf.keyword, argv[0]);

   /* Sanity check the addkey/noprompt flags */
   if (conf.addkey && conf.noprompt) {
      printf("Cannot run unattended while adding a key\n");
      usage();
      }
      
   /* Get the source */
   if (argc > 1) {
      infile  = argv[1];
      if (0 == strcmp("-", infile)) {
         infile = 0;
         }
      }
   else if (0 == infile && 0 == outfile &&
            (isatty(STDIN_FILENO) || isatty(STDOUT_FILENO))) { 
      /* Check for nothing to encrypt and no admin to do */
      if (0 == conf.addkey) {
         usage();
         }
      else {
         /* There is nothing to encrypt but the passfile needs to be updated
            with a prompted for password, this is done as part of configuring
            a cipher */
         exit (-1 == cipher_keyword_conf(&conf) ||
               -1 == conf_cipher(&conf, cipher_name, digest_name)); 
         }
      }

   /* Get the destination */
   if (0 == outfile) {
      outfile = infile;
      }

   /* Init CRC table for faster crc calculation */
   init_crc_table();

   /* Initialise config based on keyword */
   if (-1 == cipher_keyword_conf(&conf)) {
      fatal ("Keyword setup failed");
      }

   /* Do the business */
   if (conf.is_encrypting ) {

      /* Get a mask that can be used to checksum within the blocksize */
      conf.entropy_bits = get_entropy_bits(conf.max_block_size);

      /* When encrypting, the cipher config is built from the command
         line and defaults */
      rc = conf_cipher(&conf, cipher_name, digest_name);
      if (-1 != rc) {
         rc = file_encrypt(infile, outfile, &conf);
         }
      }
   else {
      /* When decrypting the cipher config is read from the file header */
      rc = file_decrypt(infile, outfile, &conf);
      }

   /* Cleanup */
   cipher_delete(conf.cipher);
   XFREE(conf.keyword);
   XFREE(conf.keyfile);
   XFREE(conf.cipher);

   if (-1 == rc) {
      fatal("I'm a failure");
      }

   return 0;
}

