using System.Collections.Generic;
using System.Runtime.CompilerServices;
using System.Runtime.InteropServices;
using System.Text;

namespace System.Reflection
{
	public class CustomAttributeData
	{
		class LazyCAttrData {
			internal Assembly assembly;
			internal IntPtr data;
			internal uint data_length;
		}

		ConstructorInfo ctorInfo;
		IList<CustomAttributeTypedArgument> ctorArgs;
		IList<CustomAttributeNamedArgument> namedArgs;
		LazyCAttrData lazyData;

		protected CustomAttributeData ()
		{
		}

		// custom-attrs.c:create_custom_attr_data ()
		internal CustomAttributeData (ConstructorInfo ctorInfo, Assembly assembly, IntPtr data, uint data_length)
		{
			this.ctorInfo = ctorInfo;
			this.lazyData = new LazyCAttrData ();
			this.lazyData.assembly = assembly;
			this.lazyData.data = data;
			this.lazyData.data_length = data_length;
		}

		internal CustomAttributeData (ConstructorInfo ctorInfo)
			: this (ctorInfo, Array.Empty<CustomAttributeTypedArgument> (), Array.Empty<CustomAttributeNamedArgument> ())
		{
		}

		internal CustomAttributeData (ConstructorInfo ctorInfo, IList<CustomAttributeTypedArgument> ctorArgs, IList<CustomAttributeNamedArgument> namedArgs)
		{
			this.ctorInfo = ctorInfo;
			this.ctorArgs = ctorArgs;
			this.namedArgs = namedArgs;
		}

		[MethodImplAttribute (MethodImplOptions.InternalCall)]
		static extern void ResolveArgumentsInternal (ConstructorInfo ctor, Assembly assembly, IntPtr data, uint data_length, out object[] ctorArgs, out object[] namedArgs); 

		void ResolveArguments ()
		{
			object[] ctor_args, named_args;
			if (lazyData == null)
				return;

			ResolveArgumentsInternal (ctorInfo, lazyData.assembly, lazyData.data, lazyData.data_length, out ctor_args, out named_args);

			this.ctorArgs = Array.AsReadOnly<CustomAttributeTypedArgument>
				(ctor_args != null ? UnboxValues<CustomAttributeTypedArgument> (ctor_args) : Array.Empty<CustomAttributeTypedArgument>());
			this.namedArgs = Array.AsReadOnly<CustomAttributeNamedArgument> 
				(named_args != null ? UnboxValues<CustomAttributeNamedArgument> (named_args) : Array.Empty<CustomAttributeNamedArgument>());
			
			lazyData = null;
		}
		
		public
		virtual
		ConstructorInfo Constructor {
			get {
				return ctorInfo;
			}
		}

		public
		virtual
		IList<CustomAttributeTypedArgument> ConstructorArguments {
			get {
				ResolveArguments ();
				return ctorArgs;
			}
		}

		public
		virtual
		IList<CustomAttributeNamedArgument> NamedArguments {
			get {
				ResolveArguments ();
				return namedArgs;
			}
		}

		public static IList<CustomAttributeData> GetCustomAttributes (Assembly target) {
			return MonoCustomAttrs.GetCustomAttributesData (target);
		}

		public static IList<CustomAttributeData> GetCustomAttributes (MemberInfo target) {
			return MonoCustomAttrs.GetCustomAttributesData (target);
		}

		internal static IList<CustomAttributeData> GetCustomAttributesInternal (RuntimeType target) {
			return MonoCustomAttrs.GetCustomAttributesData (target);
		}

		public static IList<CustomAttributeData> GetCustomAttributes (Module target) {
			return MonoCustomAttrs.GetCustomAttributesData (target);
		}

		public static IList<CustomAttributeData> GetCustomAttributes (ParameterInfo target) {
			return MonoCustomAttrs.GetCustomAttributesData (target);
		}

		virtual public Type AttributeType {
			get { return ctorInfo.DeclaringType; }
		}

		public override string ToString ()
		{
			ResolveArguments ();

			StringBuilder sb = new StringBuilder ();

			sb.Append ("[" + ctorInfo.DeclaringType.FullName + "(");
			for (int i = 0; i < ctorArgs.Count; i++) {
				sb.Append (ctorArgs [i].ToString ());
				if (i + 1 < ctorArgs.Count)
					sb.Append (", ");
			}

			if (namedArgs.Count > 0)
				sb.Append (", ");
			
			for (int j = 0; j < namedArgs.Count; j++) {
				sb.Append (namedArgs [j].ToString ());
				if (j + 1 < namedArgs.Count)
					sb.Append (", ");
			}
			sb.AppendFormat (")]");

			return sb.ToString ();
		}

		static T [] UnboxValues<T> (object [] values)
		{
			T [] retval = new T [values.Length];
			for (int i = 0; i < values.Length; i++)
				retval [i] = (T) values [i];

			return retval;
		}

        public override int GetHashCode () => base.GetHashCode ();		

		public override bool Equals (object? obj)
		{
			return obj == (object)this;
		}
	}

}

