// Copyright (c) Microsoft Open Technologies, Inc. All rights reserved. See License.txt in the project root for license information.

#pragma warning disable 1591

using System.Linq.Expressions;
using System.Reactive.Linq;

namespace System.Reactive.Joins
{
    /// <summary>
    /// Abstract base class for join patterns represented by an expression tree.
    /// </summary>
    public abstract class QueryablePattern
    {
        /// <summary>
        /// Creates a new join pattern object using the specified expression tree represention.
        /// </summary>
        /// <param name="expression">Expression tree representing the join pattern.</param>
        protected QueryablePattern(Expression expression)
        {
            Expression = expression;
        }

        /// <summary>
        /// Gets the expression tree representing the join pattern.
        /// </summary>
        public Expression Expression { get; private set; }
    }

    /* The following code is generated by a tool checked in to $/.../Source/Tools/CodeGenerators. */

    #region Joins auto-generated code (8/4/2012 1:00:32 AM)

    /// <summary>
    /// Represents a join pattern over two observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all three observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource3">The type of the elements in the third observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the two previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3> And<TSource3>(IObservable<TSource3> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource3));
            return new QueryablePattern<TSource1, TSource2, TSource3>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over three observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all four observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource4">The type of the elements in the fourth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the three previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4> And<TSource4>(IObservable<TSource4> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource4));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over four observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

#if !NO_LARGEARITY
        /// <summary>
        /// Creates a pattern that matches when all five observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource5">The type of the elements in the fifth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the four previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5> And<TSource5>(IObservable<TSource5> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource5));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }
#endif

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

#if !NO_LARGEARITY
    /// <summary>
    /// Represents a join pattern over five observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all six observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource6">The type of the elements in the sixth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the five previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6> And<TSource6>(IObservable<TSource6> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource6));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over six observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all seven observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource7">The type of the elements in the seventh observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the six previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7> And<TSource7>(IObservable<TSource7> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource7));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over seven observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all eight observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource8">The type of the elements in the eighth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the seven previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8> And<TSource8>(IObservable<TSource8> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource8));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over eight observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all nine observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource9">The type of the elements in the ninth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the eight previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9> And<TSource9>(IObservable<TSource9> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource9));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over nine observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all ten observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource10">The type of the elements in the tenth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the nine previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10> And<TSource10>(IObservable<TSource10> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource10));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over ten observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    /// <typeparam name="TSource10">The type of the elements in the tenth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all eleven observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource11">The type of the elements in the eleventh observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the ten previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11> And<TSource11>(IObservable<TSource11> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource11));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over eleven observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    /// <typeparam name="TSource10">The type of the elements in the tenth source sequence.</typeparam>
    /// <typeparam name="TSource11">The type of the elements in the eleventh source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all twelve observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource12">The type of the elements in the twelfth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the eleven previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12> And<TSource12>(IObservable<TSource12> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource12));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over twelve observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    /// <typeparam name="TSource10">The type of the elements in the tenth source sequence.</typeparam>
    /// <typeparam name="TSource11">The type of the elements in the eleventh source sequence.</typeparam>
    /// <typeparam name="TSource12">The type of the elements in the twelfth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all thirteen observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource13">The type of the elements in the thirteenth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the twelve previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13> And<TSource13>(IObservable<TSource13> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource13));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over thirteen observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    /// <typeparam name="TSource10">The type of the elements in the tenth source sequence.</typeparam>
    /// <typeparam name="TSource11">The type of the elements in the eleventh source sequence.</typeparam>
    /// <typeparam name="TSource12">The type of the elements in the twelfth source sequence.</typeparam>
    /// <typeparam name="TSource13">The type of the elements in the thirteenth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all fourteen observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource14">The type of the elements in the fourteenth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the thirteen previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14> And<TSource14>(IObservable<TSource14> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource14));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over fourteen observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    /// <typeparam name="TSource10">The type of the elements in the tenth source sequence.</typeparam>
    /// <typeparam name="TSource11">The type of the elements in the eleventh source sequence.</typeparam>
    /// <typeparam name="TSource12">The type of the elements in the twelfth source sequence.</typeparam>
    /// <typeparam name="TSource13">The type of the elements in the thirteenth source sequence.</typeparam>
    /// <typeparam name="TSource14">The type of the elements in the fourteenth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all fifteen observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource15">The type of the elements in the fifteenth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the fourteen previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15> And<TSource15>(IObservable<TSource15> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource15));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over fifteen observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    /// <typeparam name="TSource10">The type of the elements in the tenth source sequence.</typeparam>
    /// <typeparam name="TSource11">The type of the elements in the eleventh source sequence.</typeparam>
    /// <typeparam name="TSource12">The type of the elements in the twelfth source sequence.</typeparam>
    /// <typeparam name="TSource13">The type of the elements in the thirteenth source sequence.</typeparam>
    /// <typeparam name="TSource14">The type of the elements in the fourteenth source sequence.</typeparam>
    /// <typeparam name="TSource15">The type of the elements in the fifteenth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Creates a pattern that matches when all sixteen observable sequences have an available element.
        /// </summary>
        /// <typeparam name="TSource16">The type of the elements in the sixteenth observable sequence.</typeparam>
        /// <param name="other">Observable sequence to match with the fifteen previous sequences.</param>
        /// <returns>Pattern object that matches when all observable sequences have an available element.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="other"/> is null.</exception>
        public QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15, TSource16> And<TSource16>(IObservable<TSource16> other)
        {
            if (other == null)
                throw new ArgumentNullException("other");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15>);
            var m = t.GetMethod("And").MakeGenericMethod(typeof(TSource16));
            return new QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15, TSource16>(
                Expression.Call(
                    Expression,
                    m,
                    Qbservable.GetSourceExpression(other)
                )
            );

        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

    /// <summary>
    /// Represents a join pattern over sixteen observable sequences.
    /// </summary>
    /// <typeparam name="TSource1">The type of the elements in the first source sequence.</typeparam>
    /// <typeparam name="TSource2">The type of the elements in the second source sequence.</typeparam>
    /// <typeparam name="TSource3">The type of the elements in the third source sequence.</typeparam>
    /// <typeparam name="TSource4">The type of the elements in the fourth source sequence.</typeparam>
    /// <typeparam name="TSource5">The type of the elements in the fifth source sequence.</typeparam>
    /// <typeparam name="TSource6">The type of the elements in the sixth source sequence.</typeparam>
    /// <typeparam name="TSource7">The type of the elements in the seventh source sequence.</typeparam>
    /// <typeparam name="TSource8">The type of the elements in the eighth source sequence.</typeparam>
    /// <typeparam name="TSource9">The type of the elements in the ninth source sequence.</typeparam>
    /// <typeparam name="TSource10">The type of the elements in the tenth source sequence.</typeparam>
    /// <typeparam name="TSource11">The type of the elements in the eleventh source sequence.</typeparam>
    /// <typeparam name="TSource12">The type of the elements in the twelfth source sequence.</typeparam>
    /// <typeparam name="TSource13">The type of the elements in the thirteenth source sequence.</typeparam>
    /// <typeparam name="TSource14">The type of the elements in the fourteenth source sequence.</typeparam>
    /// <typeparam name="TSource15">The type of the elements in the fifteenth source sequence.</typeparam>
    /// <typeparam name="TSource16">The type of the elements in the sixteenth source sequence.</typeparam>
    public class QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15, TSource16> : QueryablePattern
    {
        internal QueryablePattern(Expression expression)
            : base(expression)
        {
        }

        /// <summary>
        /// Matches when all observable sequences have an available element and projects the elements by invoking the selector function.
        /// </summary>
        /// <typeparam name="TResult">The type of the elements in the result sequence, returned by the selector function.</typeparam>
        /// <param name="selector">Selector that will be invoked for elements in the source sequences.</param>
        /// <returns>Plan that produces the projected results, to be fed (with other plans) to the When operator.</returns>
        /// <exception cref="ArgumentNullException"><paramref name="selector"/> is null.</exception>
        public QueryablePlan<TResult> Then<TResult>(Expression<Func<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15, TSource16, TResult>> selector)
        {
            if (selector == null)
                throw new ArgumentNullException("selector");

            var t = typeof(QueryablePattern<TSource1, TSource2, TSource3, TSource4, TSource5, TSource6, TSource7, TSource8, TSource9, TSource10, TSource11, TSource12, TSource13, TSource14, TSource15, TSource16>);
            var m = t.GetMethod("Then").MakeGenericMethod(typeof(TResult));
            return new QueryablePlan<TResult>(
                Expression.Call(
                    Expression,
                    m,
                    selector
                )
            );

        }

    }

#endif

    #endregion
}

#pragma warning restore 1591