/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#define MONGO_LOGV2_DEFAULT_COMPONENT ::mongo::logv2::LogComponent::kSharding

#include "mongo/db/auth/authorization_session.h"
#include "mongo/db/commands.h"
#include "mongo/db/commands/feature_compatibility_version.h"
#include "mongo/db/s/refine_collection_shard_key_coordinator.h"
#include "mongo/db/s/sharding_state.h"
#include "mongo/s/refine_collection_shard_key_coordinator_feature_flags_gen.h"
#include "mongo/s/request_types/sharded_ddl_commands_gen.h"

namespace mongo {
namespace {

class ShardsvrRefineCollectionShardKeyCommand final
    : public TypedCommand<ShardsvrRefineCollectionShardKeyCommand> {
public:
    AllowedOnSecondary secondaryAllowed(ServiceContext*) const override {
        return Command::AllowedOnSecondary::kNever;
    }

    bool skipApiVersionCheck() const override {
        // Internal command (server to server).
        return true;
    }

    std::string help() const override {
        return "Internal command, which is exported by the primary sharding server. Do not call "
               "directly. Refines Collection shard key.";
    }

    using Request = ShardsvrRefineCollectionShardKey;

    class Invocation final : public InvocationBase {
    public:
        using InvocationBase::InvocationBase;

        void typedRun(OperationContext* opCtx) {

            uassertStatusOK(ShardingState::get(opCtx)->canAcceptShardedCommands());
            opCtx->setAlwaysInterruptAtStepDownOrUp();

            const auto coordinatorCompletionFuture = [&]() -> SharedSemiFuture<void> {
                FixedFCVRegion fixedFcvRegion(opCtx);
                const auto coordinatorType =
                    feature_flags::gFeatureFlagRecoverableRefineCollectionShardKeyCoordinator
                        .isEnabled(serverGlobalParams.featureCompatibility)
                    ? DDLCoordinatorTypeEnum::kRefineCollectionShardKey
                    : DDLCoordinatorTypeEnum::kRefineCollectionShardKeyNoResilient;

                auto coordinatorDoc = RefineCollectionShardKeyCoordinatorDocument();
                coordinatorDoc.setShardingDDLCoordinatorMetadata({{ns(), coordinatorType}});
                coordinatorDoc.setRefineCollectionShardKeyRequest(
                    request().getRefineCollectionShardKeyRequest());

                auto service = ShardingDDLCoordinatorService::getService(opCtx);
                auto refineCoordinator = checked_pointer_cast<RefineCollectionShardKeyCoordinator>(
                    service->getOrCreateInstance(opCtx, coordinatorDoc.toBSON()));
                return refineCoordinator->getCompletionFuture();
            }();

            coordinatorCompletionFuture.get(opCtx);
        }

        bool supportsWriteConcern() const override {
            return true;
        }

        void doCheckAuthorization(OperationContext*) const override {}

        /**
         * The ns() for when Request's IDL specifies "namespace: concatenate_with_db".
         */
        NamespaceString ns() const override {
            return request().getNamespace();
        }
    };

} shardsvrRefineCollectionShardKeyCommand;

}  // namespace
}  // namespace mongo
