
/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#include "mongo/platform/basic.h"

#include "mongo/executor/connection_pool_test_fixture.h"

#include "mongo/stdx/memory.h"

namespace mongo {
namespace executor {
namespace connection_pool_test_details {

TimerImpl::TimerImpl(PoolImpl* global) : _global(global) {}

TimerImpl::~TimerImpl() {
    cancelTimeout();
}

void TimerImpl::setTimeout(Milliseconds timeout, TimeoutCallback cb) {
    _timers.erase(this);

    _cb = std::move(cb);
    _expiration = _global->now() + timeout;

    _timers.emplace(this);
}

void TimerImpl::cancelTimeout() {
    _timers.erase(this);
    _cb = TimeoutCallback{};
}

void TimerImpl::clear() {
    while (!_timers.empty()) {
        auto* timer = *_timers.begin();
        timer->cancelTimeout();
    }
}

void TimerImpl::fireIfNecessary() {
    auto now = PoolImpl().now();

    auto timers = _timers;

    for (auto&& x : timers) {
        if (_timers.count(x) && (x->_expiration <= now)) {
            x->_cb();
        }
    }
}

Date_t TimerImpl::now() {
    return _global->now();
}

std::set<TimerImpl*> TimerImpl::_timers;

ConnectionImpl::ConnectionImpl(const HostAndPort& hostAndPort, size_t generation, PoolImpl* global)
    : ConnectionInterface(generation),
      _hostAndPort(hostAndPort),
      _timer(global),
      _global(global),
      _id(_idCounter++) {}

Date_t ConnectionImpl::now() {
    return _timer.now();
}

size_t ConnectionImpl::id() const {
    return _id;
}

const HostAndPort& ConnectionImpl::getHostAndPort() const {
    return _hostAndPort;
}

bool ConnectionImpl::isHealthy() {
    return true;
}

void ConnectionImpl::clear() {
    _setupQueue.clear();
    _refreshQueue.clear();
    _pushSetupQueue.clear();
    _pushRefreshQueue.clear();
}

void ConnectionImpl::pushSetup(PushSetupCallback status) {
    _pushSetupQueue.push_back(status);

    if (_setupQueue.size()) {
        auto connPtr = _setupQueue.front();
        auto callback = _pushSetupQueue.front();
        _setupQueue.pop_front();
        _pushSetupQueue.pop_front();

        auto cb = connPtr->_setupCallback;
        connPtr->indicateUsed();
        cb(connPtr, callback());
    }
}

void ConnectionImpl::pushSetup(Status status) {
    pushSetup([status]() { return status; });
}

size_t ConnectionImpl::setupQueueDepth() {
    return _setupQueue.size();
}

void ConnectionImpl::pushRefresh(PushRefreshCallback status) {
    _pushRefreshQueue.push_back(status);

    if (_refreshQueue.size()) {
        auto connPtr = _refreshQueue.front();
        auto callback = _pushRefreshQueue.front();

        _refreshQueue.pop_front();
        _pushRefreshQueue.pop_front();

        auto cb = connPtr->_refreshCallback;
        connPtr->indicateUsed();
        cb(connPtr, callback());
    }
}

void ConnectionImpl::pushRefresh(Status status) {
    pushRefresh([status]() { return status; });
}

size_t ConnectionImpl::refreshQueueDepth() {
    return _refreshQueue.size();
}

void ConnectionImpl::setTimeout(Milliseconds timeout, TimeoutCallback cb) {
    _timer.setTimeout(timeout, cb);
}

void ConnectionImpl::cancelTimeout() {
    _timer.cancelTimeout();
}

void ConnectionImpl::setup(Milliseconds timeout, SetupCallback cb) {
    _setupCallback = std::move(cb);

    _timer.setTimeout(timeout, [this] {
        _setupCallback(this, Status(ErrorCodes::NetworkInterfaceExceededTimeLimit, "timeout"));
    });

    _setupQueue.push_back(this);

    if (_pushSetupQueue.size()) {
        auto connPtr = _setupQueue.front();
        auto callback = _pushSetupQueue.front();
        _setupQueue.pop_front();
        _pushSetupQueue.pop_front();

        auto refreshCb = connPtr->_setupCallback;
        connPtr->indicateUsed();
        refreshCb(connPtr, callback());
    }
}

void ConnectionImpl::refresh(Milliseconds timeout, RefreshCallback cb) {
    _refreshCallback = std::move(cb);

    _timer.setTimeout(timeout, [this] {
        _refreshCallback(this, Status(ErrorCodes::NetworkInterfaceExceededTimeLimit, "timeout"));
    });

    _refreshQueue.push_back(this);

    if (_pushRefreshQueue.size()) {
        auto connPtr = _refreshQueue.front();
        auto callback = _pushRefreshQueue.front();

        _refreshQueue.pop_front();
        _pushRefreshQueue.pop_front();

        auto refreshCb = connPtr->_refreshCallback;
        connPtr->indicateUsed();
        refreshCb(connPtr, callback());
    }
}

std::deque<ConnectionImpl::PushSetupCallback> ConnectionImpl::_pushSetupQueue;
std::deque<ConnectionImpl::PushRefreshCallback> ConnectionImpl::_pushRefreshQueue;
std::deque<ConnectionImpl*> ConnectionImpl::_setupQueue;
std::deque<ConnectionImpl*> ConnectionImpl::_refreshQueue;
size_t ConnectionImpl::_idCounter = 1;

std::shared_ptr<ConnectionPool::ConnectionInterface> PoolImpl::makeConnection(
    const HostAndPort& hostAndPort, transport::ConnectSSLMode sslMode, size_t generation) {
    return std::make_shared<ConnectionImpl>(hostAndPort, generation, this);
}

std::shared_ptr<ConnectionPool::TimerInterface> PoolImpl::makeTimer() {
    return stdx::make_unique<TimerImpl>(this);
}

Date_t PoolImpl::now() {
    return _now.get_value_or(Date_t::now());
}

void PoolImpl::setNow(Date_t now) {
    _now = now;
    TimerImpl::fireIfNecessary();
}

boost::optional<Date_t> PoolImpl::_now;

}  // namespace connection_pool_test_details
}  // namespace executor
}  // namespace mongo
