
/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include "mongo/base/disallow_copying.h"
#include "mongo/bson/timestamp.h"
#include "mongo/db/operation_context.h"
#include "mongo/db/storage/recovery_unit.h"

namespace mongo {

/**
 * TimestampBlock is an raii type that sets a commit timestamp on the RecoveryUnit at construction
 * and clears it at destruction. RecoveryUnit::setTimestamp() should not be called while a
 * TimestampBlock is in scope.
 */
class TimestampBlock {
    MONGO_DISALLOW_COPYING(TimestampBlock);

public:
    TimestampBlock(OperationContext* opCtx, Timestamp ts) : _opCtx(opCtx), _ts(ts) {
        uassert(ErrorCodes::IllegalOperation,
                "Cannot timestamp a write operation in read-only mode",
                !storageGlobalParams.readOnly);
        if (!_ts.isNull()) {
            _opCtx->recoveryUnit()->setCommitTimestamp(_ts);
        }
    }

    ~TimestampBlock() {
        if (!_ts.isNull()) {
            _opCtx->recoveryUnit()->clearCommitTimestamp();
        }
    }

private:
    OperationContext* const _opCtx;
    Timestamp _ts;
};

}  // namespace mongo
