
/**
 *    Copyright (C) 2018-present MongoDB, Inc.
 *
 *    This program is free software: you can redistribute it and/or modify
 *    it under the terms of the Server Side Public License, version 1,
 *    as published by MongoDB, Inc.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    Server Side Public License for more details.
 *
 *    You should have received a copy of the Server Side Public License
 *    along with this program. If not, see
 *    <http://www.mongodb.com/licensing/server-side-public-license>.
 *
 *    As a special exception, the copyright holders give permission to link the
 *    code of portions of this program with the OpenSSL library under certain
 *    conditions as described in each individual source file and distribute
 *    linked combinations including the program with the OpenSSL library. You
 *    must comply with the Server Side Public License in all respects for
 *    all of the code used other than as permitted herein. If you modify file(s)
 *    with this exception, you may extend this exception to your version of the
 *    file(s), but you are not obligated to do so. If you do not wish to do so,
 *    delete this exception statement from your version. If you delete this
 *    exception statement from all source files in the program, then also delete
 *    it in the license file.
 */

#pragma once

#include <string>

#include "mongo/db/index/multikey_paths.h"
#include "mongo/db/index_names.h"
#include "mongo/db/jsobj.h"
#include "mongo/util/container_size_helper.h"
#include "mongo/util/mongoutils/str.h"

namespace mongo {

class CollatorInterface;
class MatchExpression;

/**
 * This name sucks, but every name involving 'index' is used somewhere.
 */
struct IndexEntry {
    /**
     * Use this constructor if you're making an IndexEntry from the catalog.
     */
    IndexEntry(const BSONObj& kp,
               const std::string& accessMethod,
               bool mk,
               const MultikeyPaths& mkp,
               bool sp,
               bool unq,
               const std::string& n,
               const MatchExpression* fe,
               const BSONObj& io,
               const CollatorInterface* ci)
        : keyPattern(kp),
          multikey(mk),
          multikeyPaths(mkp),
          sparse(sp),
          unique(unq),
          name(n),
          filterExpr(fe),
          infoObj(io),
          collator(ci) {
        type = IndexNames::nameToType(accessMethod);
    }

    /**
     * For testing purposes only.
     */
    IndexEntry(const BSONObj& kp,
               bool mk,
               bool sp,
               bool unq,
               const std::string& n,
               const MatchExpression* fe,
               const BSONObj& io)
        : keyPattern(kp),
          multikey(mk),
          sparse(sp),
          unique(unq),
          name(n),
          filterExpr(fe),
          infoObj(io) {
        type = IndexNames::nameToType(IndexNames::findPluginName(keyPattern));
    }

    /**
     * For testing purposes only.
     */
    IndexEntry(const BSONObj& kp, const std::string& indexName = "test_foo")
        : keyPattern(kp),
          multikey(false),
          sparse(false),
          unique(false),
          name(indexName),
          filterExpr(nullptr),
          infoObj(BSONObj()) {
        type = IndexNames::nameToType(IndexNames::findPluginName(keyPattern));
    }

    bool operator==(const IndexEntry& rhs) const {
        // Indexes are logically equal when names are equal.
        return this->name == rhs.name;
    }

    std::string toString() const;

    uint64_t estimateObjectSizeInBytes() const {

        return  // For each element in 'multikeyPaths' add the 'length of the vector * size of the
            // vector element'.
            container_size_helper::estimateObjectSizeInBytes(
                multikeyPaths,
                [](const auto& keyPath) {
                    // Calculate the size of each std::set in 'multiKeyPaths'.
                    return container_size_helper::estimateObjectSizeInBytes(keyPath);
                },
                true) +
            // Add the runtime BSONObj size of 'keyPattern' and capacity of 'name'.
            keyPattern.objsize() + name.capacity() +
            // The BSON size of the 'infoObj' is purposefully excluded since its ownership is shared
            // with the index catalog.
            // Add size of the object.
            sizeof(*this);
    }

    BSONObj keyPattern;

    bool multikey;

    // If non-empty, 'multikeyPaths' is a vector with size equal to the number of elements in the
    // index key pattern. Each element in the vector is an ordered set of positions (starting at 0)
    // into the corresponding indexed field that represent what prefixes of the indexed field cause
    // the index to be multikey.
    MultikeyPaths multikeyPaths;

    bool sparse;

    bool unique;

    std::string name;

    const MatchExpression* filterExpr;

    // Geo indices have extra parameters.  We need those available to plan correctly.
    BSONObj infoObj;

    // What type of index is this?  (What access method can we use on the index described
    // by the keyPattern?)
    IndexType type;

    // Null if this index orders strings according to the simple binary compare. If non-null,
    // represents the collator used to generate index keys for indexed strings.
    const CollatorInterface* collator = nullptr;
};

}  // namespace mongo
