/**
 *
 * mod_antispam for Apache-2.0/2.1
 *
 * Copyright 2005 by Hideo NAKAMITSU. All rights reserved
 *
 * nomo@bluecoara.net
 * http://bluecoara.net/
 *
 */

#include "apr_strings.h"
#include "ap_config.h"
#include "httpd.h"
#include "http_config.h"
#include "http_main.h"
#include "http_core.h"
#include "http_log.h"
#include "http_protocol.h"
#include "http_request.h"

/* if you need debug, comment out this */
//#define DEBUG           1
#define MODULE            "mod_antispam"
#define HEAD_LEN          1024
#define BUF_LEN           1024
/* actions */
#define ACT_TEST          0
#define ACT_REPLACE       1
#define ACT_REJECT        2
#define ACT_REPLACEREJECT 3
/* targets */
#define TARGET_FQDN       0
#define TARGET_FULL       1

typedef struct {
  int  antispam_enable;
  int  antispam_action;
  int  antispam_target;
  apr_size_t antispam_sizelimit;
  apr_time_t antispam_timeout;
  int  antispam_retry;
  const char *antispam_blacklist;
  const char *antispam_whitelist;
  const char *antispam_autoblacklist;
  const char *antispam_autowhitelist;
} antispam_config_rec;

static void *create_antispam_dir_config(apr_pool_t *p, char *d) {
    antispam_config_rec *conf = apr_palloc(p, sizeof(*conf));
    conf->antispam_enable = 0;            /* default is off         */
    conf->antispam_action = ACT_TEST;     /* default is ACT_TEST    */
    conf->antispam_target = TARGET_FULL;  /* default is TARGET_FULL */
    conf->antispam_sizelimit = 100000;    /* default is 100KB       */
    conf->antispam_timeout = 5;           /* default is 5 seconds   */
    conf->antispam_retry = 3;             /* default is 3 times     */
    conf->antispam_blacklist = NULL;      /* no default             */
    conf->antispam_whitelist = NULL;      /* no default             */
    conf->antispam_autoblacklist = NULL;  /* no default             */
    conf->antispam_autowhitelist = NULL;  /* no default             */
    return conf;
}

static const char *set_action_slot(cmd_parms *cmd,void *dir_config,
				    const char *arg) {
  antispam_config_rec *conf;
  conf=(antispam_config_rec *) dir_config;

  if ( strcasecmp("test",(char *) apr_pstrdup(cmd->pool,arg)) == 0) {
    conf->antispam_action = ACT_TEST;
    return NULL;
  } else if ( strcasecmp("replace",(char *) apr_pstrdup(cmd->pool,arg)) == 0) {
    conf->antispam_action = ACT_REPLACE;
    return NULL;
  } else if ( strcasecmp("reject",(char *) apr_pstrdup(cmd->pool,arg)) == 0) {
    conf->antispam_action = ACT_REJECT;
    return NULL;
  } else if ( strcasecmp("replacereject",(char *) apr_pstrdup(cmd->pool,arg)) == 0) {
    conf->antispam_action = ACT_REPLACEREJECT;
    return NULL;
  }
  return "Available option is Test/Replace/Reject/ReplaceReject";
}

static const char *set_target_slot(cmd_parms *cmd,void *dir_config,
				   const char *arg) {
  antispam_config_rec *conf;
  conf=(antispam_config_rec *) dir_config;

  if ( strcasecmp("fqdn",(char *) apr_pstrdup(cmd->pool,arg)) == 0) {
    conf->antispam_target = TARGET_FQDN;
    return NULL;
  } else if ( strcasecmp("full",(char *) apr_pstrdup(cmd->pool,arg)) == 0) {
    conf->antispam_target = TARGET_FULL;
    return NULL;
  }
  return "Available option is FQDN/FULL";
}

static const command_rec antispam_cmds[] = {
  AP_INIT_FLAG ("AntispamEnable", ap_set_flag_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_enable),
		 RSRC_CONF | ACCESS_CONF,
                 "Enable or not this module"),
  AP_INIT_TAKE1 ("AntispamAction", set_action_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_action),
		RSRC_CONF | ACCESS_CONF,
		"Action after receiving spam"),
  AP_INIT_TAKE1 ("AntispamTarget", set_target_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_target),
		 RSRC_CONF | ACCESS_CONF,
		 "Target"),
  AP_INIT_TAKE1 ("AntispamSizelimit", ap_set_int_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_sizelimit),
                 RSRC_CONF | ACCESS_CONF,
                 "Download size limit"),
  AP_INIT_TAKE1 ("AntispamTimeout", ap_set_int_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_timeout),
                 RSRC_CONF | ACCESS_CONF,
                 "Connection timeout"),
  AP_INIT_TAKE1 ("AntispamRetry", ap_set_int_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_retry),
                 RSRC_CONF | ACCESS_CONF,
                 "Connection retry"),
  AP_INIT_TAKE1 ("AntispamBlackList", ap_set_file_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_blacklist),
		 RSRC_CONF | ACCESS_CONF,
                 "Blacklist file which you can edit by hands"),
  AP_INIT_TAKE1 ("AntispamWhiteList", ap_set_file_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_whitelist),
		 RSRC_CONF | ACCESS_CONF,
                 "Whitelist file which you can edit by hands"),
  AP_INIT_TAKE1 ("AntispamAutoBlackList", ap_set_file_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_autoblacklist),
		 RSRC_CONF | ACCESS_CONF,
                 "Automatically created blacklist"),
  AP_INIT_TAKE1 ("AntispamAutoWhiteList", ap_set_file_slot, (void *)APR_OFFSETOF(antispam_config_rec, antispam_autowhitelist),
		 RSRC_CONF | ACCESS_CONF,
                 "Automatically created whitelist"),
  {NULL}
};

module AP_MODULE_DECLARE_DATA antispam_module;

/* msg */
char msg[16];

/* main */
static int antispam_check_access(request_rec *r) {
  apr_uri_t url;
  apr_port_t port;
  int st, i, flag = 0;
  char *host, *path, *query, *fullpath;
  char ref[HEAD_LEN], mysite[HEAD_LEN];

  antispam_config_rec *conf = ap_get_module_config(r->per_dir_config, &antispam_module);

  /* make log header */
  if (conf->antispam_action == ACT_TEST)
    strcpy(msg, "(test)");
  else if (conf->antispam_action == ACT_REPLACE)
    strcpy(msg, "(replace)");
  else if (conf->antispam_action == ACT_REJECT)
    strcpy(msg, "(reject)");
  else if (conf->antispam_action == ACT_REPLACEREJECT)
    strcpy(msg, "(replacereject)");

  /**
   * if client has no referer or
   * client is this module or
   * this module is disabled, access allowed without checking white/black lists and http connection
   */
  if (apr_table_get(r->headers_in, "User-Agent") == MODULE ||
      apr_table_get(r->headers_in, "Referer") == NULL ||
      conf->antispam_enable == 0 )
    return DECLINED;

  /* if referer from clients is longer than HEAD_LEN */
  if (strlen(apr_table_get(r->headers_in, "Referer")) >= HEAD_LEN) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s longer than %d ref=%s", MODULE, msg, HEAD_LEN, apr_table_get(r->headers_in, "Referer"));
    return antispam_ret(r);
  }

  /* if HOST header from clients is longer than HEAD_LEN, access denied except ACT_TEST/ACT_REPLACE */
  if (strlen(apr_table_get(r->headers_in, "Host")) >= HEAD_LEN) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s longer than %d host=%s", MODULE, msg, HEAD_LEN, apr_table_get(r->headers_in, "Host"));
    if (conf->antispam_action == ACT_TEST || conf->antispam_action == ACT_REPLACE)
      return DECLINED;
    return HTTP_FORBIDDEN;
  }

  /**
   * analyze referer
   *
   * EXAMPLE:
   * ref      : http://www.example.com/dir/file.cgi?str=str
   * host     : www.example.com
   * port     : 80
   * path     : /dir/file.cgi
   * query    : str=str
   * fullpath : /dir/file.cgi?str=str
   */
  strncpy(ref, apr_table_get(r->headers_in, "Referer"), HEAD_LEN);
  if (apr_uri_parse(r->pool, ref, &url) == APR_SUCCESS) {
    host = url.hostname;
    path = url.path ? url.path : apr_pstrdup(r->pool, "/");
    query = url.query;
    port = url.port ?  url.port : 80;
    fullpath = apr_palloc(r->pool, HEAD_LEN);
    if (query)
      fullpath = apr_pstrcat(r->pool, path, "?", query, NULL);
    else
      fullpath = path;
  }

#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s ref=%s, host=%s, port=%d, path=%s, query=%s, fullpath=%s", MODULE, msg, ref, host, port, path, query, fullpath);
#endif

  if (!url.scheme || strlen(url.scheme) > 4 || strncmp(url.scheme, "http", 4) != 0) {
    /**
     * supported scheme is only http.
     * https, ftp, telnet, or another methods are not supported.
     */
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s %s is not supported ref=%s", MODULE, msg, url.scheme, ref);
#endif
    flag = 1;
  }
  else if (!host) {
    /**
     * referer is invalid, module can't connect there.
     * set flag to 1 and check white/black lists and update them if needed.
     */
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s invalid referer ref=%s", MODULE, msg, ref);
#endif
    flag = 1;
  }

  /**
   * list check start.
   *
   * 1. antispam_whitelist
   * 2. antispam_blacklist
   * 3. antispam_autowhitelist
   * 4. antispam_autoblacklist
   */

  if (host && conf->antispam_target == TARGET_FQDN) {
    /* if target is only FQDN */
    strcpy(ref, host);
  }

  /* check white list */
  if (antispam_checkregexlist(r, conf->antispam_whitelist, ref) == 0)
    return DECLINED;

  /* check black list */
  if (antispam_checkregexlist(r, conf->antispam_blacklist, ref) == 0)
    return antispam_ret(r);

  /* check auto white list */
  if (antispam_checklist(r, conf->antispam_autowhitelist, ref) == 0)
    return DECLINED;

  /* check auto black list */
  if (antispam_checklist(r, conf->antispam_autoblacklist, ref) == 0)
    return antispam_ret(r);

  if (flag) {
    /* referer is invalid and can't connect there. then don't use htsearch() */
    st = 2;
  }
  else {
    /**
     * mysite is this host's hostname or searvername. have to add IP address in future because some clients don't support HTTP/1.1
     */
    strcpy(mysite, apr_table_get(r->headers_in, "Host") ? apr_table_get(r->headers_in, "Host") : ap_get_server_name(r));  
    for (i=0; i<conf->antispam_retry; i++) {
      st = antispam_htsearch(r, host, port, fullpath, mysite, conf->antispam_sizelimit, conf->antispam_timeout);
      if (st == 0 || st == 2)
	break;
    }
  }
  
  /* string found in the contents */
  if (st == 0) {
    antispam_addlist(r, ref, conf->antispam_autowhitelist);
    return DECLINED;
  }
  else if (st == 2) {
    /* string not found, add to autoblacklist */
    antispam_addlist(r, ref, conf->antispam_autoblacklist);
    return antispam_ret(r);
  }
  
  /* socket error or something. add this to blacklist */
  ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s failed to connect to %s err=%d", MODULE, msg, ref, st);
  antispam_addlist(r, ref, conf->antispam_autoblacklist);
  return antispam_ret(r);

  /* main end */
}

/**
 * 0: string found
 * 2: string not found
 * 4: apr_file_open() failed
 * 8: apr_file_lock() failed
 * 16: apr_file_unlock() failed
 * 32: apr_file_close() failed
 */
int antispam_checklist (request_rec * r, char *file, char *ref)
{
  char line[HEAD_LEN];
  apr_file_t *fd = NULL;

  if (apr_file_open(&fd, file, APR_READ, APR_OS_DEFAULT, r->pool) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_open() failed %s", MODULE, msg, file);
    return 4;
  }

  if (apr_file_lock(fd, APR_FLOCK_SHARED) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_lock() failed %s", MODULE, msg, file);
    apr_file_close(fd);
    return 8;
  }

  while (apr_file_gets(line, sizeof(line), fd) == APR_SUCCESS) {
    line[strlen(line) - 1] = '\0';
    if (strcmp(line, ref) == 0) {
#ifdef DEBUG
      ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s found in %s ref=%s", MODULE, msg, file, ref);
#endif
      return 0;
    }
  }

  if (apr_file_unlock(fd) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_unlock() failed %s", MODULE, msg, file);
    apr_file_close(fd);
    return 16;
  }

  if (apr_file_close(fd) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_close() failed %s", MODULE, msg, file);
    return 32;
  }

#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s not found in %s ref=%s", MODULE, msg, file, ref);
#endif

  return 2;
}

/*
 * 0: add success
 * 2: apr_file_printf() failed
 * 2: apr_file_open() failed
 * 4: apr_file_lock() failed
 * 8: apr_file_unlock() failed
 * 16: apr_file_close() failed
 */
int antispam_addlist (request_rec * r, char *ref, char *file)
{
  char line[HEAD_LEN];
  apr_file_t *fd = NULL;

  if (apr_file_open(&fd, file, APR_READ | APR_WRITE, APR_OS_DEFAULT, r->pool) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_open() failed %s", MODULE, msg, file);
    return 2;
  }

  if (apr_file_lock(fd, APR_FLOCK_EXCLUSIVE) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_lock() failed %s", MODULE, msg, file);
    apr_file_close(fd);
    return 4;
  }

  while (apr_file_gets(line, sizeof(line), fd) == APR_SUCCESS) {
    line[strlen(line) - 1] = '\0';
    if (strcmp(line, ref) == 0)
      return 0;
  }
  apr_file_printf(fd, "%s\n", ref);

  if (apr_file_unlock(fd) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_unlock() failed %s", MODULE, msg, file);
    apr_file_close(fd);
    return 8;
  }

  if (apr_file_close(fd) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_close() failed %s", MODULE, msg, file);
    return 16;
  }

  ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s added to %s ref=%s", MODULE, msg, file, ref);
  return 0;
}

/**
 * 0: string found
 * 2: string not found
 * 4: apr_file_open() failed
 * 8: apr_file_lock() failed
 * 16: apr_file_unlock() failed
 * 32: apr_file_close() failed
 */
int antispam_checkregexlist (request_rec * r, char *file, char *ref)
{
  char line[HEAD_LEN];
#if (AP_SERVER_MAJORVERSION_NUMBER == 2 && AP_SERVER_MINORVERSION_NUMBER == 1)
  /* for Apache-2.1 */
  ap_regex_t *regexp;
#elif (defined(AP_SERVER_MAJORVERSION) && defined(AP_SERVER_MINORVERSION))
  /* for Apache-2.0 */
  regex_t *regexp;
#endif
  apr_file_t *fd = NULL;

  if (apr_file_open(&fd, file, APR_READ, APR_OS_DEFAULT, r->pool) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_open() failed %s", MODULE, msg, file);
    return 4;
  }

  if (apr_file_lock(fd, APR_FLOCK_SHARED) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_lock() failed %s", MODULE, msg, file);
    apr_file_close(fd);
    return 8;
  }

  while (apr_file_gets(line, sizeof(line), fd) == APR_SUCCESS) {
    line[strlen(line) - 1] = '\0';

    /* ignore empty lines or comment out */
    if (line[0] == '#' || !line[0])
      continue;

#if (AP_SERVER_MAJORVERSION_NUMBER == 2 && AP_SERVER_MINORVERSION_NUMBER == 1)
    /* for Apache-2.1 */
    regexp = ap_pregcomp(r->pool, line, AP_REG_EXTENDED);
#elif (defined(AP_SERVER_MAJORVERSION) && defined(AP_SERVER_MINORVERSION))
    /* for Apache-2.0 */
    regexp = ap_pregcomp(r->pool, line, REG_EXTENDED);
#endif

    if (regexp == NULL) {
      /* bad regex */
      ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s %s in %s bad regex", MODULE, msg, line, file);
    }
    else if (ap_regexec(regexp, ref, 0, NULL, 0) == 0 ) {
      /* ref matched */
#ifdef DEBUG
      ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s %s in %s matched %s", MODULE, msg, line, file, ref);
#endif
      ap_pregfree(r->pool, regexp);
      apr_file_close(fd);
      return 0;
    }
  }

  /* not found */
#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s not found in %s ref=%s", MODULE, msg, file, ref);
#endif

  if (apr_file_unlock(fd) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_unlock() failed %s", MODULE, msg, file);
    apr_file_close(fd);
    return 16;
  }

  if (apr_file_close(fd) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s apr_file_close() failed %s", MODULE, msg, file);
    return 32;
  }
  return 2;
}

/**
 * 0: mysite found in http://host:port/path
 * 2: mysite not found
 * 4: apr_socket_timeout_set() failed
 * 8: apr_sockaddr_info_get() failed
 * 16: apr_socket_connect() failed
 * 32: apr_socket_send() failed
 */
int antispam_htsearch(request_rec *r, char *host, int port, char *path, char *mysite, int sizelimit, apr_time_t timeout) {

  apr_status_t val;
  apr_socket_t *socket;
  apr_sockaddr_t *addr;
  apr_size_t bytes_send, buf_len = BUF_LEN, total = 0;
  char buf[buf_len], *data_send, *contents;

  /* create socket */
#if (AP_SERVER_MAJORVERSION_NUMBER == 2 && AP_SERVER_MINORVERSION_NUMBER == 1)
  /* for Apache-2.1 */
  val = apr_socket_create(&socket, APR_INET, SOCK_STREAM, APR_PROTO_TCP, r->pool);
#elif (defined(AP_SERVER_MAJORVERSION) && defined(AP_SERVER_MINORVERSION))
  /* for Apache-2.0 */
  val = apr_socket_create(&socket, APR_INET, SOCK_STREAM, r->pool);
#endif

  /* set timeout */
  if (val = apr_socket_timeout_set(socket, timeout * 1000000) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_socket_timeout_set() failed, download canceled host=%s port=%d timeout=%d err=%d", MODULE, msg, host, port, timeout, val);
    return 4;
  }
#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_socket_timeout_set() success, host=%s port=%d timeout=%d", MODULE, msg, host, port, timeout);
#endif

  /* get sockaddr info */
  if (val = apr_sockaddr_info_get(&addr, host, APR_INET, port, 0, r->pool) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_sockaddr_info_get() failed, download canceled host=%s port=%d timeout=%d err=%d", MODULE, msg, host, port, timeout, val);
    return 8;
  }
#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_sockaddr_info_get() success, host=%s port=%d timeout=%d", MODULE, msg, host, port, timeout);
#endif

  /* connection */
  if (val = apr_socket_connect(socket, addr) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_socket_connect() failed, download canceled host=%s port=%d timeout=%d err=%d", MODULE, msg, host, port, timeout, val);
    apr_socket_close(socket);
    return 16;
  }
#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_socket_connect() success, host=%s port=%d timeout=%d", MODULE, msg, host, port, timeout);
#endif

  /* make get request string with Connection: close */
  data_send = apr_palloc(r->pool, strlen(host) + strlen(path) + strlen(MODULE) + 1024);
  data_send = apr_pstrcat(r->pool, "GET ", path, " HTTP/1.1", CRLF,
                       "Host: ", host, CRLF,
                       "Connection: close", CRLF,
                       "User-Agent: ", MODULE, CRLF, CRLF, NULL);
  bytes_send = strlen(data_send);

  /* send request */
  if (val = apr_socket_send(socket, data_send, &bytes_send) != APR_SUCCESS) {
    ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_socket_send() failed, download canceled host=%s port=%d timeout=%d err=%d", MODULE, msg, host, port, timeout, val);
    apr_socket_close(socket);
    return 32;
  }
#ifdef DEBUG
  ap_log_rerror(APLOG_MARK, APLOG_ERR, val, r, "%s:%s apr_socket_send() success, host=%s port=%d timeout=%d", MODULE, msg, host, port, timeout);
#endif

  contents = apr_palloc(r->pool, sizelimit);
  /* receive contents */
  while (val = apr_socket_recv(socket, buf, &buf_len) == APR_SUCCESS) {
    total += buf_len;
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s downloaded %d bytes from %s buf=%d", MODULE, msg, total, apr_table_get(r->headers_in, "Referer"), buf_len);
#endif
    if (total > sizelimit) {
      /* if contents size is too large, break here */
      ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s contents larger than sizelimit, sizelimit=%d host=%s port=%d", MODULE, msg, sizelimit, host, port);
      total -= buf_len;
      break;
    }
    else {
      contents = apr_pstrcat(r->pool, contents, buf, NULL);
    }
  }

  ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s downloaded %d bytes from %s", MODULE, msg, total, apr_table_get(r->headers_in, "Referer"));

  if (strstr(contents, mysite) ) {
    /* string found in the target */
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s %s has %s dlsize=%d", MODULE, msg, apr_table_get(r->headers_in, "Referer"), mysite, total);
#endif
    apr_socket_close(socket);
    return 0;
  }
  apr_socket_close(socket);

  /* string not found in the target */
  return 2;
}

/**
 * actions definition (Test/Replace/Reject/ReplaceReject)
 */
int antispam_ret(request_rec *r) {
  antispam_config_rec *conf = ap_get_module_config(r->per_dir_config, &antispam_module);
  if (conf->antispam_action == ACT_REPLACE) {
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s replaced to empty, access allowed ref=%s", MODULE, msg, apr_table_get(r->headers_in, "Referer"));
#endif
    apr_table_set(r->headers_in, "Referer", "");
    return DECLINED;
  }
  else if (conf->antispam_action == ACT_REJECT) {
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s access denied ref=%s", MODULE, msg, apr_table_get(r->headers_in, "Referer"));
#endif
    return HTTP_FORBIDDEN;
  }
  else if (conf->antispam_action == ACT_REPLACEREJECT) {
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s replaced to empty, access denied ref=%s", MODULE, msg, apr_table_get(r->headers_in, "Referer"));
#endif
    apr_table_set(r->headers_in, "Referer", "");
    return HTTP_FORBIDDEN;
  }
  else {
#ifdef DEBUG
    ap_log_rerror(APLOG_MARK, APLOG_ERR, 0, r, "%s:%s access allowed ref=%s", MODULE, msg, apr_table_get(r->headers_in, "Referer"));
#endif
    return DECLINED;
  }
}

static void register_hooks(apr_pool_t *p) {
  ap_hook_access_checker(antispam_check_access, NULL, NULL, APR_HOOK_MIDDLE);
}

module AP_MODULE_DECLARE_DATA antispam_module = {
  STANDARD20_MODULE_STUFF,
  create_antispam_dir_config,
  NULL,
  NULL,
  NULL,
  antispam_cmds,
  register_hooks
};
