/*
	LTOVRUDP.h

	Copyright (C) 2012 Michael Fort, Paul C. Pratt, Rob Mitchelmore

	You can redistribute this file and/or modify it under the terms
	of version 2 of the GNU General Public License as published by
	the Free Software Foundation.  You should have received a copy
	of the license along with this file; see the file COPYING.

	This file is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	license for more details.
*/

/*
	LocalTalk OVeR User Datagram Protocol
*/


#define UDP_dolog (dbglog_HAVE && 0)

/*
	Transmit buffer for localtalk data and its metadata
*/
static unsigned char tx_buffer[4 + LT_TxBfMxSz] =
	"pppp";

LOCALVAR ui5b LT_MyStamp = 0;
	/*
		randomly chosen value included in sent packets.
		if received packets have different value, then know it
			is from somebody else. (but if is same value, don't
			know for sure it isn't from somebody else.)
	*/


/*
	Receive buffer for LocalTalk data and its metadata
*/
static unsigned int rx_buffer_allocation = 1800;

LOCALVAR int sock_fd = 0;
LOCALVAR blnr udp_ok = falseblnr;

LOCALFUNC int start_udp(void)
{
	int one = 1;
	struct sockaddr_in addr;

	if ((sock_fd = socket(AF_INET, SOCK_DGRAM, IPPROTO_UDP)) < 0) {
		/* todo: some error reporting here probably a good plan */
#if UDP_dolog
		dbglog_writeCStr("socket: err ");
		dbglog_writeNum(errno);
		dbglog_writeCStr(" (");
		dbglog_writeCStr(strerror(errno));
		dbglog_writeCStr(")");
		dbglog_writeReturn();
#endif
		return -1;
	}
	setsockopt(sock_fd, SOL_SOCKET, SO_REUSEADDR,
		(const void *)&one, sizeof(one));
#if use_SO_REUSEPORT
	setsockopt(sock_fd, SOL_SOCKET, SO_REUSEPORT,
		(const void *)&one, sizeof(one));
	/*
		https://stackoverflow.com/questions/14388706/
		how-do-so-reuseaddr-and-so-reuseport-differ
			claims that SO_REUSEPORT is the same as SO_REUSEADDR for
			multicast addresses.
	*/
#endif

	/* bind it to any address it fancies */
	memset((char*)&addr, 0, sizeof(addr));
	addr.sin_family = AF_INET;
	addr.sin_addr.s_addr = INADDR_ANY;
	addr.sin_port = htons(1954);

	/* bind it */
	errno = 0;
	if (bind(sock_fd, (struct sockaddr*)&addr, sizeof(addr)) < 0) {
#if UDP_dolog
		dbglog_writeCStr("bind: err ");
		dbglog_writeNum(errno);
		dbglog_writeCStr(" (");
		dbglog_writeCStr(strerror(errno));
		dbglog_writeCStr(")");
		dbglog_writeReturn();
#endif
		return -1;
	}

	/* whack it on a multicast group */
	struct ip_mreq mreq;
	mreq.imr_multiaddr.s_addr = inet_addr("239.192.76.84");
	mreq.imr_interface.s_addr = INADDR_ANY;

	if (setsockopt(sock_fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
		(const void *)&mreq, sizeof(mreq)) < 0)
	{
#if UDP_dolog
		dbglog_writeCStr("IP_ADD_MEMBERSHIP: err ");
		dbglog_writeNum(errno);
		dbglog_writeCStr(" (");
		dbglog_writeCStr(strerror(errno));
		dbglog_writeCStr(")");
		dbglog_writeReturn();
#endif
	}

	/* non-blocking I/O is good for the soul */
#if 1
	fcntl(sock_fd, F_SETFL, O_NONBLOCK);
#else
	{
		int iResult;
		u_long iMode = 1;

		iResult = ioctlsocket(sock_fd, FIONBIO, &iMode);
		if (iResult != NO_ERROR) {
			/*
				printf("ioctlsocket failed with error: %ld\n", iResult);
			*/
		}
	}
#endif

	udp_ok = trueblnr;
	return sock_fd;
}

LOCALVAR unsigned char *MyRxBuffer = NULL;
LOCALVAR struct sockaddr_in MyRxAddress;

/*
	External function needed at startup to initialize the LocalTalk
	functionality.
*/
LOCALFUNC int InitLocalTalk(void)
{
	/* Set up UDP socket */
	start_udp();

	LT_TxBuffer = (ui3p)&tx_buffer[4];

	MyRxBuffer = malloc(rx_buffer_allocation);
	if (NULL == MyRxBuffer) {
		return falseblnr;
	}

	int fd = open("/dev/urandom", O_RDONLY);
	if (-1 == fd) {
#if UDP_dolog
		dbglog_writeCStr("open /dev/urandom fails");
		dbglog_writeNum(errno);
		dbglog_writeCStr(" (");
		dbglog_writeCStr(strerror(errno));
		dbglog_writeCStr(")");
		dbglog_writeReturn();
#endif
	} else {
		ui3b x;

		if (read(fd, &LT_MyStamp, sizeof(LT_MyStamp)) < 0) {
#if UDP_dolog
			dbglog_writeCStr("open /dev/urandom fails");
			dbglog_writeNum(errno);
			dbglog_writeCStr(" (");
			dbglog_writeCStr(strerror(errno));
			dbglog_writeCStr(")");
			dbglog_writeReturn();
#endif
		}

		/* LT_MyStamp = arc4random(); */

#if UDP_dolog
		dbglog_writelnNum("LT_MyStamp ", LT_MyStamp);
#endif

		do {
			read(fd, &x, sizeof(x));
			/* x = (ui3b)arc4random(); */
			LT_NodeHint += 1 + x;
			/*
				avoid getting in infinite loop if arc4random is broken,
				unlike if just used
					LT_NodeHint = arc4random();
			*/
		} while ((0 == LT_NodeHint) || (0xFF == LT_NodeHint));

#if UDP_dolog
		dbglog_writelnNum("LT_NodeHint ", LT_NodeHint);
#endif

		close(fd);
	}

	/* Initialized properly */
	return trueblnr;
}

LOCALPROC embedMyPID(void)
{
	/*
		embeds my process ID in network byte order in the start of the
		Tx buffer we assume a pid is at most 32 bits.  As far as I know
		there's no actual implementation of POSIX with 64-bit PIDs so we
		should be ok.
	*/
	int i;

#if LT_MayHaveEcho
	ui5r v = LT_MyStamp;
#else
	ui5r v = (ui5r)getpid();
#endif

	for (i = 0; i < 4; i++) {
		tx_buffer[i] = (v >> (3 - i)*8) & 0xff;
	}
}

GLOBALOSGLUPROC LT_TransmitPacket(void)
{
	size_t bytes;
	/* Write the packet to UDP */
#if UDP_dolog
	dbglog_writeln("writing to udp");
#endif
	embedMyPID();
	if (udp_ok) {
		struct sockaddr_in dest;
		memset((char*)&dest, 0, sizeof(dest));
		dest.sin_family = AF_INET;
		dest.sin_addr.s_addr = inet_addr("239.192.76.84");
		dest.sin_port = htons(1954);

		bytes = sendto(sock_fd,
			(const void *)tx_buffer, LT_TxBuffSz + 4, 0,
			(struct sockaddr*)&dest, sizeof(dest));
#if UDP_dolog
		dbglog_writeCStr("sent ");
		dbglog_writeNum(bytes);
		dbglog_writeCStr(" bytes");
		dbglog_writeReturn();
#endif
		(void) bytes; /* avoid warning about unused */
	}
}

/*
	pidInPacketIsMine returns 1 if the process ID embedded in the packet
	is the same as the process ID of the current process
*/
LOCALFUNC int pidInPacketIsMine(void)
{
	/* is the PID in the packet my own PID? */
	int i;
	ui5r v;

#if LT_MayHaveEcho
	v = LT_MyStamp;
#else
	v = (ui5r)getpid();
#endif

	for (i = 0; i < 4; i++) {
		if (MyRxBuffer[i] != ((v >> (3 - i)*8) & 0xff)) {
			return 0;
		}
	}

	return 1;
}

/*
	ipInPacketIsMine returns 1 if the source IP for the just-received
	UDP packet is an IP address that is attached to an interface on this
	machine.
*/
#if ! LT_MayHaveEcho
LOCALFUNC int ipInPacketIsMine(void)
{
	if (MyRxAddress.sin_family != AF_INET) {
#if UDP_dolog
		dbglog_writeln(
			"baffling error: got a non-inet packet on an inet socket");
#endif
		return 1;
			/*
				because we should drop this garbled packet on the floor
			*/
	}
	in_addr_t raddr = MyRxAddress.sin_addr.s_addr;

	/*
		Now we need to iterate through all the interfaces on the machine
	*/
	struct ifaddrs *iflist, *ifptr;
	struct sockaddr_in *addr;

	int foundAddress = 0;

	getifaddrs(&iflist);
	for (ifptr = iflist; ifptr; ifptr = ifptr->ifa_next) {
		/* if there is no address in this slot, skip it and move on */
		if (! ifptr->ifa_addr) {
			continue;
		}

		/* if it's not an af_inet then we skip it */
		if (ifptr->ifa_addr->sa_family != AF_INET) {
			continue;
		}

		addr = (struct sockaddr_in*)ifptr->ifa_addr;

		if (addr->sin_addr.s_addr == raddr) {
			foundAddress = 1;
		}
	}
	freeifaddrs(iflist);

	return foundAddress;
}
#endif

/*
	packetIsOneISent returns 1 if this looks like a packet that this
	process sent and 0 if it looks like a packet that a different
	process sent.  This provides loopback protection so that we do not
	try to consume packets that we sent ourselves.  We do this by
	checking the process ID embedded in the packet and the IP address
	the packet was sent from.  It would be neater to just look at the
	LocalTalk node ID embedded in the LLAP packet, but this doesn't
	actually work, because during address acquisition it is entirely
	legitimate (and, in the case of collision, *required*) for another
	node to send a packet from what we think is our own node ID.
*/
#if ! LT_MayHaveEcho
LOCALFUNC int packetIsOneISent(void)
{
	/*
		do the PID comparison first because it's faster and most of the
		time will disambiguate for us
	*/
	if (pidInPacketIsMine()) {
		return ipInPacketIsMine();
	}
	return 0;
}
#endif

LOCALFUNC int GetNextPacket(void)
{
	unsigned char* device_buffer = MyRxBuffer;
	socklen_t addrlen = sizeof(MyRxAddress);

	errno = 0;
	int bytes = recvfrom(sock_fd, (void *)device_buffer,
		rx_buffer_allocation, 0,
		(struct sockaddr*)&MyRxAddress, &addrlen);
	if (bytes < 0) {
		if (errno != EAGAIN) {
#if UDP_dolog
			dbglog_writeCStr("ret");
			dbglog_writeNum(bytes);
			dbglog_writeCStr(", bufsize ");
			dbglog_writeNum(rx_buffer_allocation);
			dbglog_writeCStr(", errno = ");
			dbglog_writeCStr(strerror(errno));
			dbglog_writeReturn();
#endif
		}
	} else {
#if UDP_dolog
		dbglog_writeCStr("got ");
		dbglog_writeNum(bytes);
		dbglog_writeCStr(", bufsize ");
		dbglog_writeNum(rx_buffer_allocation);
		dbglog_writeReturn();
#endif
	}
	return bytes;
}

GLOBALOSGLUPROC LT_ReceivePacket(void)
{
	int bytes;
#if ! LT_MayHaveEcho
label_retry:
#endif

	bytes = GetNextPacket();
	if (bytes > 0) {
#if LT_MayHaveEcho
		CertainlyNotMyPacket = ! pidInPacketIsMine();
#endif

#if ! LT_MayHaveEcho
		if (packetIsOneISent()) {
			goto label_retry;
		}
#endif

		{
#if UDP_dolog
			dbglog_writeCStr("passing ");
			dbglog_writeNum(bytes - 4);
			dbglog_writeCStr(" bytes to receiver");
			dbglog_writeReturn();
#endif
			LT_RxBuffer = MyRxBuffer + 4;
			LT_RxBuffSz = bytes - 4;
		}
	}
}
