/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  viewcard.c -- Dialogs for displaying card info.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "cardbase.h"
#include "cardart.h"
#include "viewcard.h"

struct viewcard
{
    GtkWidget *dialog;
    GtkWidget *image;
    GtkWidget *name_label;
    GtkWidget *cost_label;
    GtkWidget *cardtype_label;
    GtkWidget *text_label;
    GtkWidget *flavor_label;
    GtkWidget *pow_label;
    struct cardinfo *info;
};

static void do_viewcard_destroy_window (GtkWidget *, struct viewcard *);
static void do_card_image_dialog (GtkWidget *, struct viewcard *);
extern GtkWidget *new_wide_button (const char *text, int pad);

/**
 * Create a top-level window to display card info.
 */
GtkWidget *viewcard_new_window (void)
{
    GtkWidget *dialog, *image, *name_label, *cost_label, *cardtype_label;
    GtkWidget *text_label, *flavor_label, *pow_label;
    GtkWidget *layout, *namebox, *frame;
    GtkWidget *botbox, *close_button, *comp_button;
    struct viewcard *args;

    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (dialog), "View Card");
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 10);
    image = gtk_image_new ();
    name_label = gtk_label_new (NULL);
    cost_label = gtk_label_new (NULL);
    cardtype_label = gtk_label_new (NULL);
    text_label = gtk_label_new (NULL);
    flavor_label = gtk_label_new (NULL);
    pow_label = gtk_label_new (NULL);

    gtk_label_set_selectable (GTK_LABEL (name_label), TRUE);
    gtk_label_set_selectable (GTK_LABEL (cost_label), TRUE);
    gtk_label_set_selectable (GTK_LABEL (cardtype_label), TRUE);
    gtk_label_set_selectable (GTK_LABEL (text_label), TRUE);
    gtk_label_set_selectable (GTK_LABEL (flavor_label), TRUE);
    gtk_label_set_selectable (GTK_LABEL (pow_label), TRUE);

    gtk_misc_set_alignment (GTK_MISC (name_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (cost_label), 1, 0);
    gtk_misc_set_alignment (GTK_MISC (cardtype_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (text_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (flavor_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (pow_label), 1, 0);

    gtk_label_set_line_wrap (GTK_LABEL (text_label), TRUE);
    gtk_label_set_line_wrap (GTK_LABEL (flavor_label), TRUE);

    close_button = new_wide_button ("Close", 6);
    comp_button = new_wide_button ("Card Image", 6);

    layout = gtk_vbox_new (FALSE, 5);
    gtk_box_set_spacing (GTK_BOX (layout), 5);
    namebox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (namebox), name_label, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (namebox), cost_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (layout), namebox, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (layout), image, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (layout), cardtype_label, FALSE, FALSE, 0);
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), text_label);
    gtk_misc_set_padding (GTK_MISC (text_label), 5, 5);
    gtk_box_pack_start (GTK_BOX (layout), frame, TRUE, TRUE, 0);
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), flavor_label);
    gtk_misc_set_padding (GTK_MISC (flavor_label), 5, 5);
    gtk_box_pack_start (GTK_BOX (layout), frame, TRUE, TRUE, 0);
    botbox = gtk_hbox_new (FALSE, 10);
    gtk_box_pack_start (GTK_BOX (botbox), close_button, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (botbox), comp_button, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (botbox), pow_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (layout), botbox, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (dialog), layout);

    args = malloc (sizeof (struct viewcard));
    args->dialog = dialog;
    args->image = image;
    args->name_label = name_label;
    args->cost_label = cost_label;
    args->cardtype_label = cardtype_label;
    args->text_label = text_label;
    args->flavor_label = flavor_label;
    args->pow_label = pow_label;
    args->info = NULL;
    gtk_object_set_user_data (GTK_OBJECT (dialog), args);
    gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy), dialog);
    gtk_signal_connect (GTK_OBJECT (comp_button), "clicked",
			GTK_SIGNAL_FUNC (do_card_image_dialog), args);
    gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
			GTK_SIGNAL_FUNC (do_viewcard_destroy_window), args);
    return (dialog);
}

/**
 * Display the given card in the given window.
 */
void viewcard_display_cardinfo (GtkWidget *dialog, struct cardinfo *info)
{
    struct viewcard *args;
    GdkPixbuf *pixbuf;

    args = gtk_object_get_user_data (GTK_OBJECT (dialog));
    if (info->art == NULL) {
	info->art = cardart_load_cardart (info->name, info->expansion);
    }
    pixbuf = cardart_get_unscaled_art (info);
    gtk_image_set_from_pixbuf (GTK_IMAGE (args->image), pixbuf);
    gtk_label_set_text (GTK_LABEL (args->name_label), info->name);
    gtk_label_set_text (GTK_LABEL (args->cost_label), info->cost);
    gtk_label_set_text (GTK_LABEL (args->cardtype_label), info->cardtype);
    gtk_label_set_text (GTK_LABEL (args->text_label), info->text);
    gtk_label_set_text (GTK_LABEL (args->flavor_label), info->flavor);
    gtk_label_set_text (GTK_LABEL (args->pow_label), info->pow_tgh);
    args->info = info;
    gtk_widget_show_all (dialog);
    gtk_window_present (GTK_WINDOW (dialog));
}

static void do_viewcard_destroy_window (GtkWidget *w, struct viewcard *args)
{
    free (args);
}

/**
 * Create a new window with the full card image.
 */
static void do_card_image_dialog (GtkWidget *w, struct viewcard *args)
{
    struct cardinfo *info;
    GdkPixbuf *pixbuf;
    GtkWindow *parent;
    GtkWidget *dialog, *image;

    info = args->info;
    if (info->art == NULL) {
	info->art = cardart_load_cardart3 (info->name, info->expansion, TRUE);
	pixbuf = cardart_get_unscaled_art (info);
	if (pixbuf != NULL)
	    gtk_image_set_from_pixbuf (GTK_IMAGE (args->image), pixbuf);
	return;
    }
    pixbuf = cardart_get_card_image (info);
    if (pixbuf == NULL)
	return;
    parent = gtk_window_get_transient_for (GTK_WINDOW (args->dialog));
    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (dialog), info->name);
    gtk_window_set_transient_for (GTK_WINDOW (dialog), parent);
    image = gtk_image_new_from_pixbuf (pixbuf);
    gtk_container_add (GTK_CONTAINER (dialog), image);
    gtk_widget_show_all (dialog);
}
