/* -*- Mode: C ; tab-width: 4 -*-
 *
 *  Copyright (c) 2005 Federico Poloni <f.poloni@sns.it>
 *  added to Mindless Automaton:(c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  sealed.c -- sealed deck generator main file
 *
 */

#include <gtk/gtk.h>
#if (GTK_MAJOR_VERSION > 2) || (GTK_MAJOR_VERSION == 2 && GTK_MINOR_VERSION > 2)
#include <assert.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "game.h"
#include "deck.h"
#include "dialogs.h"
#include "cardbase.h"
#include "expansions.h"
#include "sealed.h"
#include "random.h"
#include "newgame.h"

/**
 * returns an array (which must be freed after use) of random integer
 * in the range 1..n, no two equal
 * the algorithm is quite a trivial one
 */
static guint *choose_subset(guint n,guint k)
{
	guint i,r,j,*subset;
	assert(n>=k);
	subset=(guint*) malloc(k*sizeof(guint));
	i=0; //no. of chosen integer
	while(i<k) {
		r=random_number(n);
		for(j=0;j<i;j++)
			if(subset[j]==r) break; //collision->breaks the FOR cycle
		if(j<i) continue; //continues the WHILE cycle --that is, tries a new random number
		subset[i++]=r;
	}
	return subset;
}

struct infoptr{
	GtkWidget *combo_box;
	GtkListStore *store,*selected;
	GtkWidget *selected_box;
};

struct bothptr{
	struct infoptr *S;
	struct infoptr *B;
	struct game *game;
	GtkWidget *win;
};

//callbacks

static gboolean do_add_set(GtkWidget *w,struct infoptr *info)
{
	GtkTreeIter iter;
	gchar *v;
	if(gtk_combo_box_get_active_iter(GTK_COMBO_BOX(info->combo_box),&iter))
	{
		gtk_tree_model_get(GTK_TREE_MODEL(info->store),&iter,0,&v,-1);
		gtk_list_store_append(info->selected,&iter);
		gtk_list_store_set(info->selected,&iter,0,v,-1);
	}
	return TRUE;
}

static gboolean do_remove_set(GtkWidget *w,struct infoptr *info)
{
	GtkTreePath *path=NULL;
	GtkTreeViewColumn *focus_column;
	GtkTreeIter iter;

	gtk_tree_view_get_cursor(GTK_TREE_VIEW(info->selected_box),&path,&focus_column);
	if(path!=NULL)
	{
		gtk_tree_model_get_iter(GTK_TREE_MODEL(info->selected),&iter,path);
		gtk_tree_path_prev(path);
		gtk_tree_view_row_activated(GTK_TREE_VIEW(info->selected_box),path,gtk_tree_view_get_column(GTK_TREE_VIEW(info->selected_box),0));
		gtk_list_store_remove(info->selected,&iter);
		gtk_tree_path_free(path);
	}
	return TRUE;
}

gboolean foreach_do_generateS(GtkTreeModel *model,GtkTreePath *path,
							  GtkTreeIter *iter,struct game *game)
{
	char *thisname;
	struct expansion *thisset;
	struct deck *deck=game->player[game->local_player]->deck;
	guint i;
	guint *chosen;

	gtk_tree_model_get(model,iter,0,&thisname,-1);
	thisset=expansions_get_from_name(game->expansions,thisname);

	//the same card must not be generated twice in the same pack

	chosen=choose_subset(thisset->num_rares,thisset->starterR);
	for(i=0;i<thisset->starterR;i++) {
		deck_set_sb_card(deck,deck->sb_size,1,thisset->rares[chosen[i]]);
	}
	free(chosen);
	chosen=choose_subset(thisset->num_uncommons,thisset->starterU);
	for(i=0;i<thisset->starterU;i++) {
		deck_set_sb_card(deck,deck->sb_size,1,thisset->uncommons[chosen[i]]);
	}
	free(chosen);
	chosen=choose_subset(thisset->num_commons,thisset->starterC);
	for(i=0;i<thisset->starterC;i++) {
		deck_set_sb_card(deck,deck->sb_size,1,thisset->commons[chosen[i]]);
	}
	free(chosen);
	return FALSE; //if TRUE, the foreach stops
}

gboolean foreach_do_generateB(GtkTreeModel *model,GtkTreePath *path,
							  GtkTreeIter *iter,struct game *game)
{
	char *thisname;
	struct expansion *thisset;
	struct deck *deck=game->player[game->local_player]->deck;
	guint i;
	guint *chosen;

	gtk_tree_model_get(model,iter,0,&thisname,-1);
	thisset=expansions_get_from_name(game->expansions,thisname);

	//the same card must not be generated twice in the same pack

	chosen=choose_subset(thisset->num_rares,thisset->boosterR);
	for(i=0;i<thisset->boosterR;i++) {
		deck_set_sb_card(deck,deck->sb_size,1,thisset->rares[chosen[i]]);
	}
	free(chosen);
	chosen=choose_subset(thisset->num_uncommons,thisset->boosterU);
	for(i=0;i<thisset->boosterU;i++) {
		deck_set_sb_card(deck,deck->sb_size,1,thisset->uncommons[chosen[i]]);
	}
	free(chosen);
	chosen=choose_subset(thisset->num_commons,thisset->boosterC);
	for(i=0;i<thisset->boosterC;i++) {
		deck_set_sb_card(deck,deck->sb_size,1,thisset->commons[chosen[i]]);
	}
	free(chosen);
	return FALSE; //if TRUE, the foreach stops
}

static gboolean do_generate(GtkWidget *w,struct bothptr *both)
{
	//actually generates a sealed deck and puts it into the sideboard
	struct deck *deck;
	struct game *game=both->game;
	char *land_names[]={"plains","island","forest","swamp","mountain"};
	int i,landgid;
	struct player *me;

	deck=deck_new();
	deck->name = strdup("Sealed Deck");
	me = game->player[game->local_player];
    if (me->deck != NULL)
		deck_destroy (me->deck);
    me->deck = deck;
	gtk_tree_model_foreach(GTK_TREE_MODEL(both->S->selected),(GtkTreeModelForeachFunc)foreach_do_generateS,game);
	gtk_tree_model_foreach(GTK_TREE_MODEL(both->B->selected),(GtkTreeModelForeachFunc)foreach_do_generateB,game);

	//add some basic lands
	for(i=0;i<5;i++)
		{
			landgid=cardbase_find_globalid(game->cardbase,land_names[i]);
			deck_set_sb_card(deck,deck->sb_size,5,landgid);
		}
	//cleaning up
	gtk_widget_destroy(both->win);
	free(both->S->store);free(both->S->selected);
	free(both->B->store);free(both->B->selected);
	free(both->S);free(both->B);free(both);

	//bring the user to the sideboard dialog
	newgame_update_deck_label(game);
	do_sideboard_dialog(game, TRUE);
	return TRUE;
}

void generate_sealed(struct game *game)
{
	struct infoptr *Sinfo,*Binfo;
	struct bothptr *bothinfo;
	GtkWidget *window, *Scombo_box, *Sframe, *Sadd_button,*Sremove_button, *Sbutton_box,*Sselected_box, *Sscroll, *layout;
	GtkWidget *Bcombo_box, *Bframe, *Badd_button,*Bremove_button, *Bbutton_box,*Bselected_box, *Bscroll;
	GtkListStore *Sstore,*Sselected;
	GtkListStore *Bstore,*Bselected;
	GtkTreeIter iter;
	guint i;
	GtkCellRenderer *randy,*randy2,*Brandy,*Brandy2;
	GtkWidget *genbutton;

	if (game->expansions == NULL) {
		game_load_expansions (game);
		if (game->expansions == NULL)
			return;
	}

	window=gtk_window_new(GTK_WINDOW_TOPLEVEL);
	if (game->new_game_dialog != NULL) {
		gtk_window_set_transient_for (GTK_WINDOW (window),
									  GTK_WINDOW (game->new_game_dialog));
		gtk_window_set_destroy_with_parent (GTK_WINDOW (window), TRUE);
	}
	gtk_window_set_title(GTK_WINDOW(window),"Generate sealed deck");
	gtk_container_set_border_width (GTK_CONTAINER (window), 10);

	layout=gtk_table_new(5,4,FALSE);

	//Starters

	Sinfo=(struct infoptr *)malloc(sizeof(struct infoptr));
	Sstore = gtk_list_store_new (1, G_TYPE_STRING);
	Sinfo->store=Sstore;
	for(i=0;i<game->expansions->num_expansions;i++) {
		if(game->expansions->db[i]->starterC>0) {
			gtk_list_store_prepend(Sstore,&iter);
			gtk_list_store_set(Sstore,&iter,0,game->expansions->db[i]->name,-1);
		}
	}
	//Starter:
	//starter combo box
 	Scombo_box=gtk_combo_box_new_with_model(GTK_TREE_MODEL(Sstore));
	gtk_combo_box_set_active(GTK_COMBO_BOX(Scombo_box),0);
	randy=gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(Scombo_box),randy,FALSE);
	gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(Scombo_box),randy,"text",0);
	Sinfo->combo_box=Scombo_box;
	Sframe=gtk_frame_new("Starter sets");
	gtk_container_add(GTK_CONTAINER(Sframe),Scombo_box);
	gtk_table_attach(GTK_TABLE(layout),Sframe,0,1,0,1,GTK_SHRINK,GTK_SHRINK,5,5);
	//starter selection box
	Sselected=gtk_list_store_new(1,G_TYPE_STRING);
	Sinfo->selected=Sselected;
	Sselected_box=gtk_tree_view_new_with_model(GTK_TREE_MODEL(Sselected));
	randy2=gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(Sselected_box),0,NULL,randy2,"text",0,NULL);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(Sselected_box),FALSE);
	Sinfo->selected_box=Sselected_box;
	Sscroll=gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(Sscroll),Sselected_box);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(Sscroll),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_widget_set_size_request(Sscroll,200,50);
	gtk_table_attach_defaults(GTK_TABLE(layout),Sscroll,2,3,0,1);
	//starter button box
	Sbutton_box=gtk_vbox_new(FALSE,5);
	Sadd_button=gtk_button_new_with_mnemonic(">>");
	gtk_box_pack_start(GTK_BOX(Sbutton_box),Sadd_button,FALSE,FALSE,0);
	gtk_signal_connect(GTK_OBJECT(Sadd_button), "clicked",GTK_SIGNAL_FUNC(do_add_set),Sinfo);
	Sremove_button=gtk_button_new_with_mnemonic("<<");
	gtk_box_pack_start(GTK_BOX(Sbutton_box),Sremove_button,FALSE,FALSE,0);
	gtk_signal_connect(GTK_OBJECT(Sremove_button), "clicked",GTK_SIGNAL_FUNC(do_remove_set),Sinfo);
	gtk_table_attach(GTK_TABLE(layout),Sbutton_box,1,2,0,1,GTK_SHRINK,GTK_SHRINK,5,5);

	//separator
	gtk_table_attach(GTK_TABLE(layout),gtk_hseparator_new(),0,3,1,2,GTK_FILL,GTK_SHRINK,5,5);
	//Booster:
	//booster combo box
	Binfo=(struct infoptr *)malloc(sizeof(struct infoptr));
	Bstore = gtk_list_store_new (1, G_TYPE_STRING);
	Binfo->store=Bstore;
	for(i=0;i<game->expansions->num_expansions;i++) {
		if(game->expansions->db[i]->boosterC>0) {
			gtk_list_store_prepend(Bstore,&iter);
			gtk_list_store_set(Bstore,&iter,0,game->expansions->db[i]->name,-1);
		}
	}
 	Bcombo_box=gtk_combo_box_new_with_model(GTK_TREE_MODEL(Bstore));
	gtk_combo_box_set_active(GTK_COMBO_BOX(Bcombo_box),0);
	Brandy=gtk_cell_renderer_text_new();
	gtk_cell_layout_pack_start(GTK_CELL_LAYOUT(Bcombo_box),Brandy,FALSE);
	gtk_cell_layout_add_attribute(GTK_CELL_LAYOUT(Bcombo_box),Brandy,"text",0);
	Binfo->combo_box=Bcombo_box;
	Bframe=gtk_frame_new("Booster sets");
	gtk_container_add(GTK_CONTAINER(Bframe),Bcombo_box);
	gtk_table_attach(GTK_TABLE(layout),Bframe,0,1,2,3,GTK_SHRINK,GTK_SHRINK,5,5);
	//booster selection box
	Bselected=gtk_list_store_new(1,G_TYPE_STRING);
	Binfo->selected=Bselected;
	Bselected_box=gtk_tree_view_new_with_model(GTK_TREE_MODEL(Bselected));
	Brandy2=gtk_cell_renderer_text_new();
	gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(Bselected_box),0,NULL,Brandy2,"text",0,NULL);
	gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(Bselected_box),FALSE);
	Binfo->selected_box=Bselected_box;
	Bscroll=gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(Bscroll),Bselected_box);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(Bscroll),GTK_POLICY_NEVER,GTK_POLICY_AUTOMATIC);
	gtk_widget_set_size_request(Bscroll,200,50);
	gtk_table_attach_defaults(GTK_TABLE(layout),Bscroll,2,3,2,3);
	//booster button box
	Bbutton_box=gtk_vbox_new(FALSE,5);
	Badd_button=gtk_button_new_with_mnemonic(">>");
	gtk_box_pack_start(GTK_BOX(Bbutton_box),Badd_button,FALSE,FALSE,0);
	gtk_signal_connect(GTK_OBJECT(Badd_button), "clicked",GTK_SIGNAL_FUNC(do_add_set),Binfo);
	Bremove_button=gtk_button_new_with_mnemonic("<<");
	gtk_box_pack_start(GTK_BOX(Bbutton_box),Bremove_button,FALSE,FALSE,0);
	gtk_signal_connect(GTK_OBJECT(Bremove_button), "clicked",GTK_SIGNAL_FUNC(do_remove_set),Binfo);
	gtk_table_attach(GTK_TABLE(layout),Bbutton_box,1,2,2,3,GTK_SHRINK,GTK_SHRINK,0,5);


	//generate button
	bothinfo=(struct bothptr *)malloc(sizeof(struct bothptr));
	bothinfo->S=Sinfo;bothinfo->B=Binfo;bothinfo->game=game;bothinfo->win=window;
	genbutton=gtk_button_new_with_mnemonic("_Generate");
	gtk_signal_connect(GTK_OBJECT(genbutton), "clicked",GTK_SIGNAL_FUNC(do_generate),bothinfo);
	gtk_table_attach(GTK_TABLE(layout),genbutton,0,3,3,4,GTK_FILL,GTK_SHRINK,5,5);
	gtk_container_add(GTK_CONTAINER(window),layout);
	gtk_widget_show_all(window);
}

#else
#include "game.h"
#include "dialogs.h"

void generate_sealed(struct game *game)
{
	do_message_dialog ("Sealed Decks are not supported by this version of gtk.");
}
#endif
