/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  playarea.c -- A play area (tabletop) is a zone where cards owned by
 *		  each player may reside.
 *
 *  Cards in the play area have (x, y) coordinates.  The cards can be
 *  moved around, tapped, etc.
 *  The play area maintains a graphical widget for displaying the cards.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "game.h"
#include "playarea.h"
#include "cardbase.h"
#include "deck.h"
#include "opponent.h"
#include "dialogs.h"
#include "zone.h"
#include "prefs.h"
#include "cardart.h"

#define CARD_WIDTH	48
#define CARD_HEIGHT	48
#define BORDER_WIDTH	3
#define GRID_WIDTH	16
#define GRID_HEIGHT	20

#define W_OFFSET	3

const char tiny_font_name[] =
	   "-schumacher-*-medium-r-normal-*-*-80-*-*-*-*-*";

struct menu_item {
    char *label;
    char accel;
    void (*callback)(struct game *game, GtkMenuItem *mitem);
};

static gint expose_event (GtkWidget *widget, GdkEventExpose *event);
static int card_overlaps (struct playarea *zone, struct table_card *tcard,
			  GdkRectangle *area);
static void playarea_get_cardart_size (struct playarea *, int *, int *);
static int paint_card_picture (struct playarea *, struct cardinfo *, int, int);
static gint table_button_press_cb (GtkWidget *, GdkEventButton *);
static gint table_motion_notify_cb (GtkWidget *, GdkEventMotion *);
static void get_card_from_table (GtkWidget *, GdkDragContext *,
				 GtkSelectionData *, guint, guint, gpointer);
static gboolean drag_motion_on_table (GtkWidget *, GdkDragContext *,
				      gint, gint, guint, gpointer);
static int snap_to_grid (int, int, int, int, struct point, struct point *);
static void drag_leave_table (GtkWidget *w, GdkDragContext *context,
			      guint time, gpointer data);
static void draw_dotted_rectangle (struct playarea *zone);
static void put_card_on_table (GtkWidget *, GdkDragContext *, gint, gint,
			       GtkSelectionData *, guint, guint);
static struct table_card *find_card (struct playarea *, int, int);
static struct table_card *raise_card (struct playarea *zone, int cid);
static void fix_popup_menu (GtkWidget *popup_menu, struct table_card *tcard);
static void paint_spiffy_card_background (struct playarea *,
					  struct table_card *,
					  struct cardinfo *);
static void paint_powtuf_background (struct playarea *, struct table_card *,
				     char *);
static void tap_card_callback (struct game *game, GtkMenuItem *mitem);
static void attack_card_callback (struct game *game, GtkMenuItem *mitem);
static void serra_attack_card_callback (struct game *game, GtkMenuItem *mitem);
static void no_untap_card_callback (struct game *game, GtkMenuItem *mitem);
static void phase_card_callback (struct game *game, GtkMenuItem *mitem);
static void flip_card_callback (struct game *game, GtkMenuItem *mitem);
static void change_color_callback (struct game *game, GtkMenuItem *mitem);
static void set_counters_callback (struct game *game, GtkMenuItem *mitem);
static int counters_dialog_callback (struct game *, gpointer, const char *);
static void add_counter_callback (struct game *game, GtkMenuItem *mitem);
static void remove_counter_callback (struct game *game, GtkMenuItem *mitem);
static void donate_card_callback (struct game *game, GtkMenuItem *mitem);
static void view_card_callback (struct game *game, GtkMenuItem *mitem);
static void set_local_card_flag (struct game *, struct table_card *,
				 int, int, int);

#define rcard_width(zone) ((int)(zone->card_width * zone->ratio))
#define rcard_height(zone) ((int)(zone->card_height * zone->ratio))
#define transform(zone, pos) ((int)(pos * zone->ratio) + W_OFFSET)
#define rev_transform(zone, pos) ((int)((pos - W_OFFSET) / zone->ratio))

static struct menu_item popup_menu_items[] = {
    { "Tap",			 'T', tap_card_callback },
    { "Attack",			 'A', attack_card_callback },
    { "Attack without Tapping",  'W', serra_attack_card_callback },
    { "Doesn't Untap as Normal",  0 , no_untap_card_callback },
    { NULL,			  0 , NULL },
    { "Phase Out",		 'P', phase_card_callback },
    { "Flip Card Over",		 'F', flip_card_callback },
    { "Color Lace",		  0 , change_color_callback },
    { NULL,			  0 , NULL },
    { "Set Counters",		 'C', set_counters_callback },
    { "Add Counter",		 'D', add_counter_callback },
    { "Remove Counter",		 'M', remove_counter_callback },
    { "Switch Controller",	  0 , donate_card_callback },
    { NULL,			  0 , NULL },
#if 0
    { "Bury",			 'B', bury_card_callback },
    { "Return to Hand",		 'H', bounce_card_callback },
    { "Return to Library",	 'L', ebb_card_callback },
    { "Remove from Game",	 'R', remove_card_callback },
    { NULL,			  0 , NULL },
#endif
    { "View Card",		  0 , view_card_callback }
};

static struct menu_item opponent_menu_items[] = {
    { "Take Control of Card",	  0 , donate_card_callback },
    { NULL,			  0 , NULL },
    { "View Card",		  0 , view_card_callback }
};

/* Colors for drawing spiffy card templates. */
static GdkColor dark_color[8];
static GdkColor lite_color[8];
static GdkColor land_color[5];
static GdkColor yellow_color;

static char *card_color_name[][2] = {
    { "#e6dab4",	"#fffaf6" },
    { "royalblue",	"#cde6f6" },
    { "grey30",		"wheat" }, 
    { "#c55441",	"#e68983" },
    { "forestgreen",	"#d5b294" },
    { "#b8860b",	"#deea5a" },
    { "#736152",	"#e6e6de" },
    { "#8c8c8c",	"#ffb900" }
};

static char *land_color_name[] = {
    "#ffe6a4", "#cde6f6", "#a3a3a3", "#ffcec5", "#9aff9a"
};

static int got_colors = FALSE;

/*
 *  playarea_new () -- Create structures to store the list of cards in
 *	the zone, and a graphical component to display the area.
 */
struct playarea *playarea_new (struct game *game)
{
    GdkColormap *cmap;
    struct playarea *playarea;
    GtkWidget *darea, *color_menu, *mitem;
    struct menu_item *mip;
    int num_items, i;

    if (!got_colors) {
	cmap = gdk_colormap_get_system ();
	for (i = 0; i < 8; i++) {
	    gdk_color_parse (card_color_name[i][0], &dark_color[i]);
	    gdk_colormap_alloc_color (cmap, &dark_color[i], FALSE, TRUE);
	    gdk_color_parse (card_color_name[i][1], &lite_color[i]);
	    gdk_colormap_alloc_color (cmap, &lite_color[i], FALSE, TRUE);
	}
	for (i = 0; i < 5; i++) {
	    gdk_color_parse (land_color_name[i], &land_color[i]);
	    gdk_colormap_alloc_color (cmap, &land_color[i], FALSE, TRUE);
	}
	gdk_color_parse ("yellow", &yellow_color);
	gdk_colormap_alloc_color (cmap, &yellow_color, FALSE, TRUE);
	got_colors = TRUE;
    }

    playarea = calloc (1, sizeof (struct playarea));
    playarea->game = game;
    playarea->card_width =
	get_int_preference (game->prefs, "card_width", CARD_WIDTH);
    playarea->card_height =
	get_int_preference (game->prefs, "card_height", CARD_HEIGHT);
    playarea_refresh_prefs (playarea);

    darea = gtk_drawing_area_new ();
    playarea->darea = darea;
    playarea->font = gdk_font_load (tiny_font_name);
    gtk_object_set_user_data (GTK_OBJECT (darea), playarea);

    gtk_signal_connect (GTK_OBJECT (darea), "expose_event",
			(GtkSignalFunc) expose_event, NULL);

    /* gtk_drag_source_set (...); */
    gtk_widget_set_events (darea, gtk_widget_get_events (darea) |
			   GDK_BUTTON_PRESS_MASK | GDK_BUTTON1_MOTION_MASK);
    gtk_signal_connect (GTK_OBJECT (darea), "button_press_event",
			GTK_SIGNAL_FUNC (table_button_press_cb), game);
    gtk_signal_connect (GTK_OBJECT (darea), "motion_notify_event",
			GTK_SIGNAL_FUNC (table_motion_notify_cb), game);
    gtk_signal_connect (GTK_OBJECT (darea), "drag_data_get",
			GTK_SIGNAL_FUNC (get_card_from_table), playarea);
    gtk_signal_connect (GTK_OBJECT (darea), "drag-motion",
			GTK_SIGNAL_FUNC (drag_motion_on_table), playarea);
    gtk_signal_connect (GTK_OBJECT (darea), "drag-leave",
			GTK_SIGNAL_FUNC (drag_leave_table), playarea);
    gtk_drag_dest_set (darea, GTK_DEST_DEFAULT_ALL, target_table,
		       num_targets, GDK_ACTION_MOVE);
    gtk_signal_connect (GTK_OBJECT (darea), "drag_data_received",
			GTK_SIGNAL_FUNC (put_card_on_table), NULL);

    color_menu = gtk_menu_new ();
    for (i = 0; color_names[i] != NULL; i++) {
	mitem = gtk_menu_item_new_with_label (color_names[i]);
	gtk_signal_connect_object (GTK_OBJECT (mitem), "activate",
				   GTK_SIGNAL_FUNC (change_color_callback),
				   (gpointer) game);
	gtk_object_set_user_data (GTK_OBJECT (mitem), GINT_TO_POINTER(i+1));
	gtk_menu_append (GTK_MENU (color_menu), mitem);
	gtk_widget_show (mitem);
    }

    playarea->popup_menu = gtk_menu_new ();
    num_items = sizeof (popup_menu_items) / sizeof (struct menu_item);
    for (i = 0; i < num_items; i++) {
	mip = &popup_menu_items[i];
	if (mip->label != NULL) {
	    mitem = gtk_menu_item_new_with_label (mip->label);
	    /* if (mip->accel != 0) ...; */
	} else {
	    mitem = gtk_menu_item_new ();
	}
	gtk_menu_append (GTK_MENU (playarea->popup_menu), mitem);
	if (mip->callback == change_color_callback)
	    gtk_menu_item_set_submenu (GTK_MENU_ITEM (mitem), color_menu);
	else if (mip->callback != NULL)
	    gtk_signal_connect_object (GTK_OBJECT (mitem), "activate",
				       GTK_SIGNAL_FUNC (mip->callback),
				       (gpointer) game);
	gtk_widget_show (mitem);
    }

    playarea->opponent_menu = gtk_menu_new ();
    num_items = sizeof (opponent_menu_items) / sizeof (struct menu_item);
    for (i = 0; i < num_items; i++) {
	mip = &opponent_menu_items[i];
	if (mip->label != NULL) {
	    mitem = gtk_menu_item_new_with_label (mip->label);
	    /* if (mip->accel != 0) ...; */
	} else {
	    mitem = gtk_menu_item_new ();
	}
	gtk_menu_append (GTK_MENU (playarea->opponent_menu), mitem);
	if (mip->callback != NULL)
	    gtk_signal_connect_object (GTK_OBJECT (mitem), "activate",
				       GTK_SIGNAL_FUNC (mip->callback),
				       (gpointer) game);
	gtk_widget_show (mitem);
    }

    return (playarea);
}

void playarea_refresh_prefs (struct playarea *zone)
{
    struct prefs *prefs = zone->game->prefs;
    char *background;

    zone->pic_type = get_int_preference (prefs, card_pic_type_key,
					 PLAYAREA_ART_ONLY);
    zone->ratio = get_float_preference (prefs, zoom_ratio_key);
    if (zone->ratio <= 0)
	zone->ratio = 1;
    background = get_preference (prefs, background_key);
    if ((background == NULL) || (zone->background == NULL) ||
	(strcmp (background, zone->background) != 0)) {
	if (zone->background != NULL) {
	    free (zone->background);
	    zone->background = NULL;
	}
	if (zone->background_pix != NULL) {
	    gdk_pixbuf_unref (zone->background_pix);
	    zone->background_pix = NULL;
	}
	if (background != NULL) {
	    zone->background_pix = gdk_pixbuf_new_from_file (background, NULL);
	    if (zone->background_pix != NULL)
		zone->background = strdup (background);
	}
    }
    zone->background_type = get_int_preference (prefs, background_type_key, 0);
}

void playarea_repaint (struct playarea *zone)
{
    gtk_widget_draw (zone->darea, NULL);
}

/*
 *  expose_event () -- This is called (by the main loop) when the
 *	play area needs to be redrawn.
 */
static gint
expose_event (GtkWidget *widget, GdkEventExpose *event)
{
    int x, y, width, height, idx;
    struct playarea *zone;
    struct game *game;
    struct table_card *tcard;
    int pw, ph, zw, zh, x_sp, y_sp, x_ep, x_ea, y_ep, y_ea;
    int x_into_pic, y_into_pic, x_on_darea, y_on_darea, count;

    zone = gtk_object_get_user_data (GTK_OBJECT (widget));
    game = zone->game;

    /*
     *  First, we fill in the drawing area.
     */
    x = event->area.x;
    y = event->area.y;
    width = event->area.width;
    height = event->area.height;
    gdk_draw_rectangle (widget->window, widget->style->black_gc, TRUE,
			x, y, width, height);
    if (zone->background_pix != NULL) {
	x_ea = x + width;
	y_ea = y + height;
	pw = gdk_pixbuf_get_width (zone->background_pix);
	ph = gdk_pixbuf_get_height (zone->background_pix);
	zw = zone->darea->allocation.width;
	zh = zone->darea->allocation.height;
	if (zone->background_type == PLAYAREA_TILE) {
	    count = 0;
	    for (y_sp = 0; y_sp < y_ea; y_sp += ph) {
		y_ep = y_sp + ph;
		if (y_ep < y)
		    continue;
		y_into_pic = (y < y_sp ? 0 : y - y_sp);
		y_on_darea = y_sp + y_into_pic;
		height = MIN (y_ep, y_ea) - y_on_darea;
		for (x_sp = 0; x_sp < x_ea; x_sp += pw) {
		    x_ep = x_sp + pw;
		    if (x_ep < x)
			continue;
		    x_into_pic = (x < x_sp ? 0 : x - x_sp);
		    x_on_darea = x_sp + x_into_pic;
		    width = MIN (x_ep, x_ea) - x_on_darea;
		    gdk_pixbuf_render_to_drawable (zone->background_pix, zone->darea->window, zone->darea->style->white_gc, x_into_pic, y_into_pic, x_on_darea, y_on_darea, width, height, GDK_RGB_DITHER_NORMAL, 0, 0);
		    count++;
		}
	    }
	    // printf ("count=%d\n", count);
	}
	else if (zone->background_type == PLAYAREA_CENTER) {
	    y_sp = (zh - ph) / 2;
	    if (y_sp < 0)
		y_sp = 0;
	    y_ep = y_sp + ph;
	    if ((y_sp < y_ea) && (y <= y_ep)) {
		y_into_pic = (y < y_sp ? 0 : y - y_sp);
		y_on_darea = y_sp + y_into_pic;
		height = MIN (y_ep, y_ea) - y_on_darea;
		x_sp = (zw - pw) / 2;
		if (x_sp < 0)
		    x_sp = 0;
		x_ep = x_sp + pw;
		if ((x_sp < x_ea) && (x <= x_ep)) {
		    x_into_pic = (x < x_sp ? 0 : x - x_sp);
		    x_on_darea = x_sp + x_into_pic;
		    width = MIN (x_ep, x_ea) - x_on_darea;
		    gdk_pixbuf_render_to_drawable (zone->background_pix, zone->darea->window, zone->darea->style->white_gc, x_into_pic, y_into_pic, x_on_darea, y_on_darea, width, height, GDK_RGB_DITHER_NORMAL, 0, 0);
		}
	    }
	}
    }
    /*
     *  Then, we draw each card.
     */
    for (idx = 0; idx < zone->size; idx++) {
	tcard = zone->cardlist[idx];
	if (card_overlaps (zone, tcard, &event->area))
	    playarea_paint_card (zone, tcard);
    }
    /*
     *  Finally, the dotted box.
     */
    if (zone->box_drawn) {
	gdk_gc_set_clip_rectangle (zone->drag_gc, &event->area);
	draw_dotted_rectangle (zone);
	gdk_gc_set_clip_rectangle (zone->drag_gc, NULL);
    }

    return (FALSE);
}

static int
card_overlaps (struct playarea *zone, struct table_card *tcard,
	       GdkRectangle *area)
{
    int x1, x2, x3, x4, y1, y2, y3, y4, horiz, vert;

    x1 = transform (zone, tcard->xpos);
    x2 = x1 + rcard_width (zone);
    x3 = area->x;
    x4 = x3 + area->width;
    y1 = transform (zone, tcard->ypos);
    y2 = y1 + rcard_height (zone);
    y3 = area->y;
    y4 = y3 + area->height;
    horiz = (((x1 <= x3) && (x2 >= x3)) || ((x3 <= x1) && (x4 >= x1)));
    vert  = (((y1 <= y3) && (y2 >= y3)) || ((y3 <= y1) && (y4 >= y1)));
    return (horiz && vert);
}

void playarea_paint_card (struct playarea *zone, struct table_card *tcard)
{
    struct cardinfo *info;
    char *name, buf[15];
    int xpos, ypos, height, start, length, i, w, nonzero, p_xpos;
    int have_picture, label_width;
    GdkGC *color_gc;

    info = game_get_cardinfo (zone->game, tcard->cid);
    paint_spiffy_card_background (zone, tcard, info);
    if ((tcard->flags & CARD_FLIPPED_FLAG) != 0)
	info = NULL;

    /*  Draw the card picture and/or name.  */
    name = (info == NULL ? NULL : info->name);
    height = zone->font->ascent + zone->font->descent + 2;
    xpos = transform (zone, tcard->xpos);
    ypos = transform (zone, tcard->ypos);
    have_picture = FALSE;
    if (zone->pic_type != PLAYAREA_NAME_ONLY)
	have_picture = paint_card_picture (zone, info, xpos, ypos);
    if (((zone->pic_type != PLAYAREA_ART_ONLY) || !have_picture) &&
	(name != NULL)) {
	xpos = xpos + 2 + BORDER_WIDTH;
	ypos = ypos + 3 + BORDER_WIDTH + zone->font->ascent;
	nonzero = TRUE;
	for (start = length = i = 0; nonzero; i++) {
	    nonzero = (name[start+i] != 0);
	    if (nonzero && (name[start+i] != ' '))
		continue;
	    if (!nonzero && (length == 0))
		break;
	    w = gdk_text_width (zone->font, name+start, i);
	    if (w+2*(2+BORDER_WIDTH) <= rcard_width (zone)) {
		length = i;
		if (nonzero)
		    continue;
		break;
	    }
	    if (length == 0)
		length = i;
	    gdk_draw_text (zone->darea->window,  zone->font,
			   zone->darea->style->white_gc,
			   xpos, ypos, name+start, length);
	    ypos += height;
	    start += length+1;
	    length = i = 0;
	}
	gdk_draw_string (zone->darea->window,  zone->font,
			 zone->darea->style->white_gc,
			 xpos, ypos, name+start);
	xpos = transform (zone, tcard->xpos);
    }
    /*  Draw the card power/toughness.  */
    name = (info == NULL ? NULL : info->pow_tgh);
    if (name != NULL) {
	paint_powtuf_background (zone, tcard, name);
	w = gdk_string_width (zone->font, name);
	p_xpos = xpos + rcard_width (zone) - w - BORDER_WIDTH - 2;
	ypos = transform (zone, tcard->ypos);
	ypos += rcard_height (zone) - zone->font->descent-BORDER_WIDTH-2;
	gdk_draw_string (zone->darea->window, zone->font,
			 zone->darea->style->black_gc, p_xpos, ypos, name);
    }
    /*  Draw the card flags.  */
    xpos = xpos + 1 + BORDER_WIDTH;
    ypos = transform (zone, tcard->ypos);
    ypos = ypos + 6 + height;
    height += 2;
    label_width = rcard_width (zone) - 1 - 2*BORDER_WIDTH;
    if ((tcard->flags & CARD_TAPPED_FLAG) != 0) {
	if ((tcard->flags & CARD_NO_UNTAP_FLAG) != 0) {
	    if (zone->color_gc == NULL)
		zone->color_gc = gdk_gc_new (zone->darea->window);
	    gdk_gc_set_foreground (zone->color_gc, &yellow_color);
	    color_gc = zone->color_gc;
	} else {
	    color_gc = zone->darea->style->white_gc;
	}
	gdk_draw_rectangle (zone->darea->window, color_gc, TRUE,
			    xpos, ypos, label_width, height);
	name = "Tapped";
	w = gdk_string_width (zone->font, name);
	gdk_draw_string (zone->darea->window,  zone->font,
			 zone->darea->style->black_gc,
			 xpos + (label_width-w)/2,
			 ypos+zone->font->ascent+2, name);
	ypos += height;
    }
    if ((tcard->flags & CARD_ATTACK_FLAG) != 0) {
	gdk_draw_rectangle (zone->darea->window, zone->darea->style->white_gc,
			    TRUE, xpos, ypos, label_width, height);
	name = "Attack";
	w = gdk_string_width (zone->font, name);
	gdk_draw_string (zone->darea->window,  zone->font,
			 zone->darea->style->black_gc,
			 xpos + (label_width-w)/2,
			 ypos+zone->font->ascent+2, name);
	ypos += height;
    }
    if (tcard->counters != 0) {
	gdk_draw_rectangle (zone->darea->window, zone->darea->style->white_gc,
			    TRUE, xpos, ypos, label_width, height);
	sprintf (buf, "%d ctr", tcard->counters);
	name = buf;
	w = gdk_string_width (zone->font, name);
	gdk_draw_string (zone->darea->window,  zone->font,
			 zone->darea->style->black_gc,
			 xpos + (label_width-w)/2,
			 ypos+zone->font->ascent+2, name);
	ypos += height;
    }
}

static void playarea_get_cardart_size (struct playarea *zone,
				       int *widthp, int *heightp)
{
    int width = rcard_width (zone) - 1 - 2*BORDER_WIDTH;
    *widthp = width;
    *heightp = (int) ((double) width * 0.725);
    /* Ensure enough space for three lines of text. */
    if (*heightp < 33) *heightp = 33;
}

static int paint_card_picture (struct playarea *zone, struct cardinfo *info,
			       int xpos, int ypos)
{
    int pic_width, pic_height;
    GdkPixbuf *small_art;

    if (info == NULL)
	return (FALSE);
    playarea_get_cardart_size (zone, &pic_width, &pic_height);
    if (info->art == NULL)
	info->art = cardart_load_cardart (info->name, info->expansion);
    small_art = cardart_get_small_art (info, pic_width, pic_height);
    if (small_art == NULL)
	return (FALSE);
    gdk_pixbuf_render_to_drawable (small_art, zone->darea->window,
				   zone->darea->style->white_gc, 0, 0,
				   xpos+1+BORDER_WIDTH, ypos+1+BORDER_WIDTH,
				   pic_width, pic_height,
				   GDK_RGB_DITHER_NORMAL, 0, 0);
    return (TRUE);
}

/*
 *  table_button_press_cb () -- This is called (by the main loop) when
 *	a mouse button is pressed in the play area.
 *	If the mouse is over a card, this function may either:
 *	(1) note that mouse motion should start dragging the card.
 *	(2) tap the card.
 *	(3) pop up a menu for the card.
 */
static gint table_button_press_cb (GtkWidget *widget, GdkEventButton *event)
{
    struct playarea *zone;
    struct game *game;
    struct table_card *tcard;
    GtkWidget *menu;

    zone = gtk_object_get_user_data (GTK_OBJECT (widget));
    game = zone->game;

    zone->drag_cid = 0;
    tcard = find_card (zone, rev_transform (zone, event->x),
		       rev_transform (zone, event->y));
    if (tcard == NULL)
	return (TRUE);
    if (event->button == 3) {
	zone->drag_cid = tcard->cid;
	if (tcard->controller == game->local_player) {
	    fix_popup_menu (zone->popup_menu, tcard);
	    menu = zone->popup_menu;
	} else {
	    menu = zone->opponent_menu;
	}
	gtk_menu_popup (GTK_MENU (menu), NULL, NULL, NULL, NULL,
			event->button, event->time);
	return (TRUE);
    }
    if (tcard->controller != game->local_player)
	return (TRUE);
    if ((event->button == 2) ||
	((event->button == 1) && (event->type == GDK_2BUTTON_PRESS))) {
	set_local_card_flag (game, tcard, CARD_TAPPED_FLAG, FALSE, TRUE);
	return (TRUE);
    }
    if (event->button == 1) {
	zone->drag_cid = tcard->cid;
	zone->box.x = tcard->xpos;
	zone->box.y = tcard->ypos;
	zone->click_location.x = event->x;
	zone->click_location.y = event->y;
    }
    return (TRUE);
}

/*
 *  table_motion_notify_cb () -- This is called (by the main loop) when
 *	a mouse button is held and the mouse is moved.
 *	If the mouse was clicked over a card, this function puts
 *	gdk into drag mode.
 */
static gint table_motion_notify_cb (GtkWidget *widget, GdkEventMotion *event)
{
    struct playarea *zone;
    int delta_x, delta_y;
    GtkTargetList *tlist;

    zone = gtk_object_get_user_data (GTK_OBJECT (widget));
    if ((zone->drag_cid > 0) && (event->state & GDK_BUTTON1_MASK) != 0) {
	delta_x = zone->click_location.x - event->x;
	delta_y = zone->click_location.y - event->y;
	if ((delta_x > 3) || (delta_x < -3) ||
	    (delta_y > 3) || (delta_y < -3)) {
	    tlist = gtk_target_list_new (target_table, num_targets);
	    gtk_drag_begin (widget, tlist, GDK_ACTION_MOVE, 1,
			    (GdkEvent *) event);
	    return TRUE;
	}
    }
    return FALSE;
}

/*
 *  get_card_from_table () -- When the user drags a card from the table
 *	and releases the mouse button, the system calls this function
 *	to ask what data should be sent to the destination widget.
 */
static void get_card_from_table (GtkWidget *w, GdkDragContext *context,
				 GtkSelectionData *selection, guint info,
				 guint time, gpointer data)
{
    /*
     *  Send zero bytes of data to the destination widget.
     *  It will use zone->drag_cid to find out what card was dragged.
     */
    gtk_selection_data_set (selection, selection->target, 8, NULL, 0);
}

/*
 *  drag_motion_on_table () -- While a card is being dragged along the table,
 *	the system calls this function periodically.
 */
static gboolean drag_motion_on_table (GtkWidget *w, GdkDragContext *context,
				      gint xpos, gint ypos, guint time,
				      gpointer data)
{
    struct playarea *zone;
    struct game *game;
    struct point box;
    GdkGC *gc;

    zone = gtk_object_get_user_data (GTK_OBJECT (w));
    game = zone->game;

    if (zone->drag_gc == NULL) {
	gc = gdk_gc_new (w->window);
	gdk_gc_copy (gc, w->style->white_gc);
	gdk_gc_set_function (gc, GDK_XOR);
	gdk_gc_set_line_attributes (gc, 1, GDK_LINE_ON_OFF_DASH,
				    GDK_CAP_NOT_LAST, GDK_JOIN_MITER);
	zone->drag_gc = gc;
    }

    xpos = rev_transform (zone, xpos);
    ypos = rev_transform (zone, ypos);
    if (snap_to_grid (xpos, ypos, zone->card_width, zone->card_height,
		      zone->box, &box) &&
	((box.x != zone->box.x) || (box.y != zone->box.y))) {
	if (zone->box_drawn)
	    draw_dotted_rectangle (zone);
	zone->box = box;
	draw_dotted_rectangle (zone);
	zone->box_drawn = TRUE;
    }
    return TRUE;
}

static int snap_to_grid (int x, int y, int card_width, int card_height,
			 struct point origin, struct point *boxp)
{
    *boxp = origin;

    if ((x < 0) || (y < 0))
	return (FALSE);

    if (x < origin.x) {
	boxp->x = ((int) (x / GRID_WIDTH)) * GRID_WIDTH;
    } else {
	x -= (card_width+1);
	if (x > origin.x)
	    boxp->x = (1 + (int) (x / GRID_WIDTH)) * GRID_WIDTH;
    }

    if (y < origin.y) {
	boxp->y = ((int) (y / GRID_HEIGHT)) * GRID_HEIGHT;
    } else {
	y -= (card_height+1);
	if (y > origin.y)
	    boxp->y = (1 + (int) (y / GRID_HEIGHT)) * GRID_HEIGHT;
    }

    return (TRUE);
}

static void drag_leave_table (GtkWidget *w, GdkDragContext *context,
			      guint time, gpointer data)
{
    struct playarea *zone;

    zone = gtk_object_get_user_data (GTK_OBJECT (w));
    if (zone->box_drawn) {
	draw_dotted_rectangle (zone);
	zone->box_drawn = FALSE;
    }
}

static void draw_dotted_rectangle (struct playarea *zone)
{
    gdk_draw_rectangle (zone->darea->window, zone->drag_gc, FALSE,
			transform (zone, zone->box.x),
			transform (zone, zone->box.y),
			rcard_width (zone), rcard_height (zone));
}

/*
 *  put_card_on_table () -- When a card is dragged and dropped on the
 *	table, the system calls this function.
 */
static void put_card_on_table (GtkWidget *w, GdkDragContext *context,
			       gint x, gint y, GtkSelectionData *data,
			       guint info, guint time)
{
    struct playarea *zone;
    struct game *game;
    GtkWidget *wsrc;
    int shift_flag, pid, srczone, idx;
    struct table_card *tcard;
    struct point box;

    zone = gtk_object_get_user_data (GTK_OBJECT (w));
    game = zone->game;
    shift_flag = is_shift_down ();

    if (zone->box_drawn) {
	draw_dotted_rectangle (zone);
	zone->box_drawn = FALSE;
    }

    snap_to_grid (rev_transform (zone, x), rev_transform (zone, y),
		  zone->card_width, zone->card_height, zone->box, &box);
    x = box.x;
    y = box.y;

    wsrc = gtk_drag_get_source_widget (context);
    if (wsrc == w) {
	playarea_move_card (zone, zone->drag_cid, x, y);
	return;
    }

    if ((srczone = widget_to_zone (game, wsrc, &pid)) < 0)
	return;
    idx = zone_dragged_card_idx (game->player[pid]->zone[srczone]);
    if (idx < 0)
	return;
    tcard = calloc (1, sizeof (struct table_card));
    tcard->cid = 0;
    tcard->xpos = x;
    tcard->ypos = y;
    if (shift_flag)
	tcard->flags = CARD_FLIPPED_FLAG;
    tcard->controller = pid;
    game_play_card (game, tcard, pid, srczone, idx, game->local_player);
    send_card_location (game, tcard);
}

/*
 *  playarea_add_card () -- When a card is dragged from any zone to
 *	the play area, the game manager calls this function.
 */
void playarea_add_card (struct playarea *zone, struct table_card *tcard)
{
    while (zone->size >= zone->alloced) {
	zone->alloced += 60;
	zone->cardlist = realloc (zone->cardlist, zone->alloced *
				  sizeof (struct table_card *));
    }
    zone->cardlist[zone->size] = tcard;
    zone->size++;
    playarea_paint_card (zone, tcard);
}

/*
 *  find_card () -- Return the topmost card at the given coordinates.
 */
static struct table_card *find_card (struct playarea *zone, int xpos, int ypos)
{
    int idx;
    struct table_card *tcard;

    for (idx = zone->size-1; idx >= 0; idx--) {
	tcard = zone->cardlist[idx];
	if ((tcard->xpos <= xpos) && (tcard->ypos <= ypos) &&
	    (tcard->xpos + zone->card_width > xpos) &&
	    (tcard->ypos + zone->card_height > ypos))
	    return (tcard);
    }
    return (NULL);
}

/*
 *  raise_card () -- Move a card to the end of the cardlist so that
 *	it will be drawn on top of all other cards.
 */
static struct table_card *raise_card (struct playarea *zone, int cid)
{
    int idx;
    struct table_card *tcard;

    for (idx = 0; idx < zone->size; idx++) {
	tcard = zone->cardlist[idx];
	if (tcard->cid == cid) {
	    for (; idx < zone->size-1;  idx++)
		zone->cardlist[idx] = zone->cardlist[idx+1];
	    zone->cardlist[zone->size-1] = tcard;
	    return (tcard);
	}
    }
    return (NULL);
}

/*
 *  playarea_move_card () -- Change the location of a card on the table.
 */
int playarea_move_card (struct playarea *zone, int cid, int xpos, int ypos)
{
    struct table_card *tcard;
    GdkRectangle rect;

    tcard = raise_card (zone, cid);
    if (tcard == NULL)
	return (FALSE);
    rect.x = transform (zone, tcard->xpos);
    rect.width = rcard_width (zone) + 1;
    rect.y = transform (zone, tcard->ypos);
    rect.height = rcard_height (zone) + 1;
    tcard->xpos = xpos;
    tcard->ypos = ypos;
    playarea_paint_card (zone, tcard);
    gtk_widget_draw (zone->darea, &rect);
    send_card_location (zone->game, tcard);
    return (TRUE);
}

int playarea_remove_card (struct playarea *zone, int cid)
{
    struct table_card *tcard;
    int flags;
    GdkRectangle rect;

    tcard = raise_card (zone, cid);
    if (tcard == NULL)
	return (-1);
    flags = tcard->flags;
    rect.x = transform (zone, tcard->xpos);
    rect.width = rcard_width (zone) + 1;
    rect.y = transform (zone, tcard->ypos);
    rect.height = rcard_height (zone) + 1;
    zone->size--;
    free (tcard);
    gtk_widget_draw (zone->darea, &rect);
    return (flags);
}

int playarea_dragged_card (struct playarea *zone)
{
    return (zone->drag_cid);
}

struct table_card *playarea_get_table_card (struct playarea *zone, int cid)
{
    int i;
    for (i = 0; i < zone->size; i++)
	if (zone->cardlist[i]->cid == cid)
	    return (zone->cardlist[i]);
    return (NULL);
}

void playarea_reset (struct playarea *zone)
{
    struct table_card *tcard;

    while (zone->size > 0) {
	tcard = zone->cardlist[zone->size-1];
	free (tcard);
	zone->size--;
    }
    gtk_widget_queue_draw (zone->darea);
}

void playarea_all_cards_stop_attacking (struct playarea *zone)
{
    int flag, i;
    struct table_card *tcard;

    flag = CARD_ATTACK_FLAG;
    for (i = 0; i < zone->size; i++) {
	tcard = zone->cardlist[i];
	if ((tcard->flags & flag) != 0) {
	    tcard->flags &= ~flag;
	    playarea_paint_card (zone, tcard);
	}
    }
}

static void fix_popup_menu (GtkWidget *popup_menu, struct table_card *tcard)
{
    GList *items;
    GtkWidget *item, *label;
    char *text;

    items = gtk_container_children (GTK_CONTAINER (popup_menu));

    text = (((tcard->flags & CARD_TAPPED_FLAG) == 0) ?
	    "Tap" : "Untap");
    item = (GtkWidget *) items->data;
    label = (GTK_BIN (item))->child;
    gtk_label_set_text (GTK_LABEL (label), text);

    items = items->next;
    text = (((tcard->flags & CARD_ATTACK_FLAG) == 0) ?
	    "Attack" : "Unattack");
    item = (GtkWidget *) items->data;
    label = (GTK_BIN (item))->child;
    gtk_label_set_text (GTK_LABEL (label), text);

    items = items->next;
    items = items->next;
    text = (((tcard->flags & CARD_NO_UNTAP_FLAG) == 0) ?
	    "Doesn't Untap as Normal" : "Untaps as Normal");
    item = (GtkWidget *) items->data;
    label = (GTK_BIN (item))->child;
    gtk_label_set_text (GTK_LABEL (label), text);

    items = items->next;
    items = items->next;
    text = (((tcard->flags & CARD_PHASED_FLAG) == 0) ?
	    "Phase Out" : "Phase In");
    item = (GtkWidget *) items->data;
    label = (GTK_BIN (item))->child;
    gtk_label_set_text (GTK_LABEL (label), text);
}

int is_shift_down (void)
{
    int x, y;
    GdkModifierType mask;

    gdk_window_get_pointer (NULL, &x, &y, &mask);
    return ((mask & GDK_SHIFT_MASK) != 0);
}

/* ----------------------------------------------------------------- */
/*
 *  This function should draw customizable images and colors and themes
 *  and cool stuff like that.
 */
static void
paint_spiffy_card_background (struct playarea *zone, struct table_card *tcard,
			      struct cardinfo *info)
{
    int xpos, ypos, pic_width, pic_height, land_num, pow_height;
    int card_width, card_height;
    char *name;
    GdkColor *boxcolor;

    if (zone->color_gc == NULL)
	zone->color_gc = gdk_gc_new (zone->darea->window);

    xpos = transform (zone, tcard->xpos);
    ypos = transform (zone, tcard->ypos);
    card_width = rcard_width (zone);
    card_height = rcard_height (zone);
    gdk_draw_rectangle (zone->darea->window,
			zone->darea->style->black_gc, TRUE, xpos, ypos,
			card_width, card_height);
    gdk_draw_rectangle (zone->darea->window,
			zone->darea->style->white_gc, FALSE, xpos, ypos,
			card_width, card_height);

    if ((tcard->color < 1) || (tcard->color > 8) ||
 	((tcard->flags & CARD_FLIPPED_FLAG) != 0)) {
	return;
    }

    land_num = 0;
    if ((tcard->color == 8) && (info != NULL) &&
	((name = info->name) != NULL)) {
	if (strcasecmp (name, "plains") == 0)
	    land_num = 1;
	else if (strcasecmp (name, "island") == 0)
	    land_num = 2;
	else if (strcasecmp (name, "swamp") == 0)
	    land_num = 3;
	else if (strcasecmp (name, "mountain") == 0)
	    land_num = 4;
	else if (strcasecmp (name, "forest") == 0)
	    land_num = 5;
    }
    boxcolor = (land_num > 0 ? &land_color[land_num-1] :
		&lite_color[tcard->color-1]);

    playarea_get_cardart_size (zone, &pic_width, &pic_height);
    gdk_gc_set_foreground (zone->color_gc, &dark_color[tcard->color-1]);
    gdk_draw_rectangle (zone->darea->window,
			zone->color_gc, TRUE, xpos+1, ypos+1,
			card_width-1, card_height-1);

    gdk_draw_rectangle (zone->darea->window,
			zone->darea->style->black_gc, TRUE,
			xpos+1+BORDER_WIDTH, ypos+1+BORDER_WIDTH,
			pic_width, pic_height);

    gdk_gc_set_foreground (zone->color_gc,  boxcolor);
    pow_height = card_height - 1 - 3*BORDER_WIDTH - pic_height;
    gdk_draw_rectangle (zone->darea->window,
			zone->color_gc,	TRUE, xpos+1+BORDER_WIDTH,
			ypos+1+2*BORDER_WIDTH+pic_height,
			pic_width, pow_height);
}

static void
paint_powtuf_background (struct playarea *zone, struct table_card *tcard,
			 char *powtuf)
{
    int height, pic_width, pic_height, pow_height, card_width, card_height;
    int width, xpos, ypos;
    GdkColor *boxcolor;

    if (*powtuf == 0)
	return;
    height = zone->font->ascent + zone->font->descent + 4;
    playarea_get_cardart_size (zone, &pic_width, &pic_height);
    card_width = rcard_width (zone);
    card_height = rcard_height (zone);
    pow_height = card_height - 1 - 3*BORDER_WIDTH - pic_height;
    if (pow_height >= height)
	return;

    width = gdk_string_width (zone->font, powtuf) + 4;
    xpos = transform (zone, tcard->xpos) + card_width-width-BORDER_WIDTH;
    ypos = transform (zone, tcard->ypos) + card_height-height-BORDER_WIDTH;
    boxcolor = &lite_color[tcard->color-1];
    gdk_gc_set_foreground (zone->color_gc,  boxcolor);
    gdk_draw_rectangle (zone->darea->window,
			zone->color_gc,	TRUE, xpos, ypos, width, height);
}

/* ----------------------------------------------------------------- */

static void tap_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    set_local_card_flag (game, tcard, CARD_TAPPED_FLAG, FALSE, TRUE);
}

static void attack_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;
    int state;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    if (tcard == NULL)
	return;
    state = ((tcard->flags & CARD_ATTACK_FLAG) == 0);
    set_local_card_flag (game, tcard, CARD_TAPPED_FLAG, state, FALSE);
    set_local_card_flag (game, tcard, CARD_ATTACK_FLAG, state, FALSE);
}

static void serra_attack_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    set_local_card_flag (game, tcard, CARD_ATTACK_FLAG, FALSE, TRUE);
}

static void no_untap_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    set_local_card_flag (game, tcard, CARD_NO_UNTAP_FLAG, FALSE, TRUE);
}

static void phase_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    set_local_card_flag (game, tcard, CARD_PHASED_FLAG, FALSE, TRUE);
}

static void flip_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    set_local_card_flag (game, tcard, CARD_FLIPPED_FLAG, FALSE, TRUE);
}

static void change_color_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;
    int color;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    color = GPOINTER_TO_INT(gtk_object_get_user_data (GTK_OBJECT (mitem)));
    if (tcard->color != color) {
	set_table_card_color (game, tcard, color, game->local_player);
	send_sleight_card (game, tcard);
    }
}

static void set_counters_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;
    char *title, *prompt;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    title = "Set Counters";
    prompt = "Enter the number of counters.";
    do_input_dialog (game, title, prompt, NULL,
		     counters_dialog_callback, tcard);
}

static int
counters_dialog_callback (struct game *game, gpointer data, const char *text)
{
    struct table_card *tcard = data;

    if ((text != NULL) && (*text != 0)) {
	set_card_counters (game, tcard, atoi (text));
	send_card_counters (game, tcard);
    }
    return (TRUE);
}

static void add_counter_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    set_card_counters (game, tcard, tcard->counters+1);
    send_card_counters (game, tcard);
}

static void remove_counter_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    set_card_counters (game, tcard, tcard->counters-1);
    send_card_counters (game, tcard);
}

static void donate_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;
    int take, pid;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    if (tcard == NULL)
	return;
    take = (tcard->controller != game->local_player);
    if (take) {
	pid = game->local_player;
    } else {
	pid = pid_of_single_opponent (game);
	if (pid < 0)
	    return;
    }
    set_card_controller (game, tcard, pid, take);
    send_card_controller (game, tcard, take);
}

static void view_card_callback (struct game *game, GtkMenuItem *mitem)
{
    struct playarea *zone;
    struct table_card *tcard;

    zone = game->playarea;
    tcard = playarea_get_table_card (zone, zone->drag_cid);
    if (tcard == NULL)
	return;
    if ((tcard->flags & CARD_FLIPPED_FLAG) != 0) {
	display_message (game, "%s is peeking at a facedown card.",
			 game->player[game->local_player]->name);
	send_peek_at_facedown (game);
    }
    do_view_card_dialog (game, tcard->cid);
}

static void
set_local_card_flag (struct game *game, struct table_card *tcard,
		     int flag, int state, int toggle)
{
    if (tcard == NULL)
	return;
    if (toggle)
	state = ((tcard->flags & flag) == 0);
    set_table_card_flag (game, tcard, flag, state, game->local_player);
    send_card_flag (game, tcard->cid, flag, state);
}
