/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  newgame.c -- The New Game dialog box.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include "deck.h"
#include "zone.h"
#include "cardbase.h"
#include "game.h"
#include "opponent.h"
#include "prefs.h"
#include "dialogs.h"
#include "sealed.h"

struct newgame_args {
    struct game *game;
    GtkWidget *deck_label;
    GtkWidget *solitaire_button;
    GtkWidget *call_button;
    GtkWidget *ipaddr_entry;
    GtkWidget *wait_button;
    GtkWidget *life_entry;
    GtkWidget *hand_entry;
    GtkWidget *message_entry;
    GtkWidget *port_entry;
    GtkWidget *status_label;

    SOCKET wait_socket;
    int wait_input_tag;
};

static void do_load_deck (GtkWidget *w, struct newgame_args *args);
static void finish_load_deck (GtkWidget *, const char *,
			      struct newgame_args *);
static void do_generate_sealed (GtkWidget *w, struct newgame_args *args);
static void do_sideboard_cb (GtkWidget *w, struct newgame_args *args);
static void newgame_dialog_ok (GtkWidget *w, struct newgame_args *args);
static void got_connection_cb (struct newgame_args *, SOCKET,
			       GdkInputCondition);
static void do_cancel_dialog (GtkWidget *w, struct newgame_args *args);
static int stop_waiting_for_call (struct newgame_args *args);

#define gtk_entry_get_text(x) \
	gtk_editable_get_chars(GTK_EDITABLE(x), 0, -1)
#define is_button_active(x) \
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(x))
#define toggle_button_set_active(x, val) \
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(x), val)

const char *game_type_key = "game_type";
const char *opponent_key = "opponent";
const char *port_number_key = "port_number";

void new_game_dialog (struct game *game)
{
    struct newgame_args *args;
    GtkWidget *dialog, *vbox, *hbox, *label, *button;
    GtkWidget *frame, *table, *entry;
    GSList *group;
    struct player *me;
    char *name, *text, *val, *ipaddr, buf[6];
    int i, port_num;

    if (game->new_game_dialog != NULL) {
	gtk_window_present (GTK_WINDOW (game->new_game_dialog));
	return;
    }
    args = calloc (1, sizeof (struct newgame_args));
    args->game = game;
    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (dialog), "New Game");
    game_make_transient (game, dialog);
    gtk_object_set_user_data (GTK_OBJECT (dialog), args);
    vbox = gtk_vbox_new (FALSE, 6);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 6);
    gtk_container_add (GTK_CONTAINER (dialog), vbox);

    hbox = gtk_hbox_new (FALSE, 0);
    label = gtk_label_new ("Deck:");
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 6);
    me = game->player[game->local_player];
    name = ((me == NULL) || (me->deck == NULL) || (me->deck->name == NULL) ?
	    "" : me->deck->name);
    label = gtk_label_new (name);
    args->deck_label = label;
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    hbox = gtk_hbox_new (TRUE, 12);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
    button = gtk_button_new_with_label ("Load...");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_load_deck), args);
    gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 3);
    button = new_wide_button ("Sealed Deck...", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_generate_sealed), args);
    gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 3);
    button = gtk_button_new_with_label ("Sideboard...");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_sideboard_cb), args);
    gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 3);
    gtk_box_pack_start (GTK_BOX (vbox), hbox, FALSE, FALSE, 0);

    if (game->opponent == NULL) {
	frame = gtk_frame_new ("Game Type");
	table = gtk_table_new (2, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), 6);
	text = "Play a Solitaire Game";
	button = gtk_radio_button_new_with_label (NULL, text);
	args->solitaire_button = button;
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	align_and_attach (table, button, 0, 2, 0, 1);
	text = "Call Opponent at:";
	button = gtk_radio_button_new_with_label (group, text);
	args->call_button = button;
	group = gtk_radio_button_group (GTK_RADIO_BUTTON (button));
	align_and_attach (table, button, 0, 1, 1, 2);
	entry = gtk_entry_new ();
	args->ipaddr_entry = entry;
	gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, 1, 2);
	text = "Wait for Call";
	button = gtk_radio_button_new_with_label (group, text);
	args->wait_button = button;
	align_and_attach (table, button, 0, 2, 2, 3);
	gtk_table_set_row_spacings (GTK_TABLE (table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (table), 6);
	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
    }

    frame = gtk_frame_new ("Game Options");
    table = gtk_table_new (2, 3, FALSE);
    gtk_container_set_border_width (GTK_CONTAINER (table), 6);
    label = gtk_label_new ("Starting Life:");
    align_and_attach (table, label, 0, 1, 0, 1);
    entry = gtk_entry_new ();
    args->life_entry = entry;
    gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, 0, 1);
    label = gtk_label_new ("Starting Cards in Hand:");
    align_and_attach (table, label, 0, 1, 1, 2);
    entry = gtk_entry_new ();
    args->hand_entry = entry;
    gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, 1, 2);
    label = gtk_label_new ("Starting Message:");
    align_and_attach (table, label, 0, 1, 2, 3);
    entry = gtk_entry_new ();
    args->message_entry = entry;
    gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, 2, 3);
    gtk_table_set_row_spacings (GTK_TABLE (table), 3);
    gtk_table_set_col_spacings (GTK_TABLE (table), 6);
    gtk_container_add (GTK_CONTAINER (frame), table);
    gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

    ipaddr = get_ip_address_text ();
    if ((game->opponent == NULL) && (ipaddr != NULL)) {
	frame = gtk_frame_new ("Network Information");
	table = gtk_table_new (2, 3, FALSE);
	gtk_container_set_border_width (GTK_CONTAINER (table), 6);
	label = gtk_label_new ("IP Address:");
	align_and_attach (table, label, 0, 1, 0, 1);
	label = gtk_label_new (ipaddr);
	align_and_attach (table, label, 1, 2, 0, 1);
	label = gtk_label_new ("Port Number:");
	align_and_attach (table, label, 0, 1, 1, 2);
	entry = gtk_entry_new ();
	args->port_entry = entry;
	gtk_table_attach_defaults (GTK_TABLE (table), entry, 1, 2, 1, 2);
	label = gtk_label_new ("Status:");
	align_and_attach (table, label, 0, 1, 2, 3);
	label = gtk_label_new ("");
	args->status_label = label;
	align_and_attach (table, label, 1, 2, 2, 3);
	gtk_table_set_row_spacings (GTK_TABLE (table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (table), 6);
	gtk_container_add (GTK_CONTAINER (frame), table);
	gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);
    }

    hbox = gtk_hbox_new (TRUE, 30);
    gtk_container_set_border_width (GTK_CONTAINER (hbox), 6);
    button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (newgame_dialog_ok), args);
    gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 20);
    button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_cancel_dialog), args);
    gtk_box_pack_start (GTK_BOX (hbox), button, TRUE, TRUE, 20);
    gtk_box_pack_end (GTK_BOX (vbox), hbox, FALSE, TRUE, 0);

    if (game->opponent == NULL) {
	val = get_preference (game->prefs, game_type_key);
	if ((val != NULL) && (strcasecmp (val, "call") == 0)) {
	    toggle_button_set_active (args->call_button, TRUE);
	    val = get_preference (game->prefs, opponent_key);
	    if (val != NULL)
		gtk_entry_set_text (GTK_ENTRY (args->ipaddr_entry), val);
	}
	else if ((val != NULL) && (strcasecmp (val, "wait") == 0)) {
	    toggle_button_set_active (args->wait_button, TRUE);
	}
	port_num = get_int_preference (game->prefs, port_number_key,
				       default_port);
	sprintf (buf, "%d", port_num);
	gtk_entry_set_text (GTK_ENTRY (args->port_entry), buf);
    }

    i = get_int_preference (game->prefs, starting_life_key, 20);
    sprintf (buf, "%d", i);
    gtk_entry_set_text (GTK_ENTRY (args->life_entry), buf);

    i = get_int_preference (game->prefs, starting_hand_size_key, 0);
    sprintf (buf, "%d", i);
    gtk_entry_set_text (GTK_ENTRY (args->hand_entry), buf);

    val = get_preference (game->prefs, starting_message_key);
    if (val == NULL)
	val = "Hello.";
    gtk_entry_set_text (GTK_ENTRY (args->message_entry), val);

    game->new_game_dialog = dialog;
    gtk_widget_show_all (dialog);
}

#if (GTK_MAJOR_VERSION>2) || (GTK_MAJOR_VERSION==2&&GTK_MINOR_VERSION>2)
static void
load_deck_response (GtkDialog *dialog, gint arg, struct newgame_args *args)
{
    const char *filename;
    switch (arg) {
    case GTK_RESPONSE_ACCEPT:
	filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
	finish_load_deck (GTK_WIDGET (dialog), filename, args);
	break;
    case GTK_RESPONSE_CANCEL:
	gtk_widget_destroy (GTK_WIDGET (dialog));
	break;
    }
}

static void
do_load_deck (GtkWidget *w, struct newgame_args *args)
{
    GtkWidget *dialog;
    GtkFileFilter *filter;

    dialog = gtk_file_chooser_dialog_new ("Load Deck",
			GTK_WINDOW (args->game->new_game_dialog),
			GTK_FILE_CHOOSER_ACTION_OPEN,
			GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
			GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
			NULL);
    gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
    filter = gtk_file_filter_new ();
    gtk_file_filter_add_pattern (filter, "*.dec");
    gtk_file_chooser_set_filter (GTK_FILE_CHOOSER (dialog), filter);
    gtk_signal_connect (GTK_OBJECT (dialog), "response",
			GTK_SIGNAL_FUNC (load_deck_response), args);
    gtk_widget_show_all (dialog);
}
#else
static void
load_deck_ok (GtkWidget *w, GtkWidget *dialog)
{
    const char *filename;
    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog));
    if (filename != NULL)
	finish_load_deck (dialog, filename,
			  gtk_object_get_user_data (GTK_OBJECT (dialog)));
}

static void
do_load_deck (GtkWidget *w, struct newgame_args *args)
{
    GtkWidget *dialog;
    GtkFileSelection *fsp;

    dialog = gtk_file_selection_new ("Load Deck");
    gtk_window_set_transient_for (GTK_WINDOW (dialog),
				  GTK_WINDOW (args->game->new_game_dialog));
    gtk_window_set_destroy_with_parent (GTK_WINDOW (dialog), TRUE);
    gtk_object_set_user_data (GTK_OBJECT (dialog), args);
    fsp = GTK_FILE_SELECTION (dialog);
    gtk_signal_connect (GTK_OBJECT (fsp->ok_button), "clicked",
			GTK_SIGNAL_FUNC (load_deck_ok), dialog);
    gtk_signal_connect_object (GTK_OBJECT (fsp->cancel_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    gtk_widget_show (dialog);
}
#endif

static void finish_load_deck (GtkWidget *dialog, const char *filename,
			      struct newgame_args *args)
{
    struct game *game;
    struct deck *deck;
    struct player *me;

    game = args->game;
    deck = deck_read_file (filename, game->cardbase);
    if (deck == NULL) {
	/* show an error */
	return;
    }
    me = game->player[game->local_player];
    if (me->deck != NULL)
	deck_destroy (me->deck);
    me->deck = deck;
    gtk_widget_destroy (dialog);

    if (deck->name != NULL)
	gtk_label_set_text (GTK_LABEL (args->deck_label), deck->name);

    display_message (game, "%s has loaded a new deck for the next game.",
		     me->name);
    send_loaded_new_deck (game);
}

static void do_generate_sealed (GtkWidget *w, struct newgame_args *args)
{
    generate_sealed (args->game);
}

static void do_sideboard_cb (GtkWidget *w, struct newgame_args *args)
{
    do_sideboard_dialog (args->game, FALSE);
}

static void newgame_dialog_ok (GtkWidget *w, struct newgame_args *args)
{
    struct game *game = args->game;
    struct player *me;
    int i, port_num, condition;
    char *cp, *hostname;

    me = game->player[game->local_player];
    if (me->deck == NULL) {
	do_message_dialog ("Load a deck first.");
	return;
    }
    /* Deck may be an (unsaved) sealed deck. */
    if (me->deck->filename != NULL)
	set_preference (game->prefs, deck_name_key, me->deck->filename);
    if (game->opponent != NULL) {
	if ((game->opponent->flags & local_wants_new) != 0)
	    return;
	game->opponent->flags |= local_wants_new;
	if (game_should_restart (game)) {
	    start_new_game (game);
	    return;
	}
	send_request_new_game (game, TRUE);
	display_message (game, "%s is requesting a new game.",
			 game->player[game->local_player]->name);
	return;
    }
    /*
     *  The local player becomes the only player, and he has no cards in
     *  any zone, including his library.
     */
    for (i = 0; i < game->num_players; i++)
	if (i != game->local_player)
	    player_destroy (game->player[i]);
    game->num_players = 1;
    game->local_player = 0;
    game->player = realloc (game->player, sizeof (struct player *) * 2);
    game->player[0] = me;
    game_empty_all_zones (game);

    if (is_button_active (GTK_TOGGLE_BUTTON (args->solitaire_button))) {
	stop_waiting_for_call (args);
	set_preference (game->prefs, game_type_key, "play");
	start_new_game (game);
	close_new_game_dialog (game);
	return;
    }

    cp = gtk_entry_get_text (args->port_entry);
    port_num = strtol (cp, NULL, 0);
    free (cp);

    if (is_button_active (GTK_TOGGLE_BUTTON (args->wait_button))) {
	if (args->wait_input_tag > 0)
	    return;
	set_preference (game->prefs, game_type_key, "wait");
	cp = "Waiting for Call...";
	gtk_label_set_text (GTK_LABEL (args->status_label), cp);

	args->wait_socket = create_socket_and_listen (port_num);
	condition = GDK_INPUT_READ | GDK_INPUT_EXCEPTION;
	args->wait_input_tag =
	    gdk_input_add (args->wait_socket, condition, (GdkInputFunction)
			   got_connection_cb, args);
	return;
    }

    if (is_button_active (GTK_TOGGLE_BUTTON (args->call_button))) {
	stop_waiting_for_call (args);
	hostname = gtk_entry_get_text (args->ipaddr_entry);
	if (*hostname == 0) {
	    free (hostname);
	    return;
	}
	set_preference (game->prefs, game_type_key, "call");
	set_preference (game->prefs, opponent_key, hostname);
	cp = "Connecting...";
	gtk_label_set_text (GTK_LABEL (args->status_label), cp);

	game->opponent = call_new_opponent (hostname, port_num);
	free (hostname);
	if (game->opponent == NULL) {
	    gtk_label_set_text (GTK_LABEL (args->status_label), "");
	    return;
	}
	gtk_label_set_text (GTK_LABEL (args->status_label), "Connected");
	game_become_client (game);
	return;
    }
}

static void got_connection_cb (struct newgame_args *args, SOCKET socket,
			       GdkInputCondition condition)
{
    struct game *game = args->game;

    game->opponent = accept_new_opponent (socket);
    if (game->opponent == NULL)
	return;
    if (args->wait_input_tag > 0) {
	gdk_input_remove (args->wait_input_tag);
	args->wait_input_tag = 0;
    }
    if (args->wait_socket > 0) {
	close_socket (args->wait_socket);
	args->wait_socket = 0;
    }
    gtk_label_set_text (GTK_LABEL (args->status_label), "Connected");
    game_become_server (game);
}

void close_new_game_dialog (struct game *game)
{
    struct newgame_args *args;

    if (game->new_game_dialog == NULL)
	return;
    args = gtk_object_get_user_data (GTK_OBJECT (game->new_game_dialog));
    gtk_widget_destroy (game->new_game_dialog);
    game->new_game_dialog = NULL;
    free (args);
}

/*
 *  Called when:
 *  1. You begin a solitaire game.
 *  2. Connection established by calling.
 *  3. Connection established by waiting.
 *  4. Opponent sends setup-begin while you're waiting for it.
 *  5. You hit ok while opponent is waiting for it.
 */
void start_new_game (struct game *game)
{
    struct newgame_args *args;
    struct player *me;
    int starting_life, starting_cards, pid, offset;
    char *cp, *starting_message;
    struct deck *deck;

    if (game->new_game_dialog == NULL)
	return;
    args = gtk_object_get_user_data (GTK_OBJECT (game->new_game_dialog));
    cp = gtk_entry_get_text (args->life_entry);
    starting_life = atoi (cp);
    free (cp);
    set_int_preference (game->prefs, starting_life_key, starting_life);
    cp = gtk_entry_get_text (args->hand_entry);
    starting_cards = atoi (cp);
    free (cp);
    set_int_preference (game->prefs, starting_hand_size_key, starting_cards);
    starting_message = gtk_entry_get_text (args->message_entry);

    if (game->opponent != NULL)
	game->opponent->flags &= ~(local_wants_new|opponent_wants_new);
    
#if GTK_MAJOR_VERSION >= 2
    {
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(game->message_box));
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	gtk_text_buffer_delete(buffer, &start, &end);
    }
#else
    gtk_editable_delete_text (GTK_EDITABLE (game->message_box), 0, -1);
#endif
    game_empty_all_zones (game);

    /*
     *  Forget all remote players's decks, in case they loaded new decks
     *  or sideboarded.
     */
    for (pid = 0; pid < game->num_players; pid++) {
	if ((pid != game->local_player) &&
	    ((deck = game->player[pid]->deck) != NULL)) {
	    deck->size = deck->sb_size = deck->tk_size = 0;
	}
    }

    set_game_turn (game, 1);
    set_game_phase (game, UNTAP_PHASE);

    me = game->player[game->local_player];
    display_message (game, "%s is setting up cards...", me->name);
    deck_remove_tokens (me->deck);
    send_player_setup (game);
    offset = player_deck_offset (game->local_player);
    zone_add_deck (me->zone[LIBRARY_ZONE], offset, me->deck->size);
    offset = player_sideboard_offset (game, game->local_player);
    zone_add_deck (me->zone[SIDEBOARD_ZONE], offset, me->deck->sb_size);
    display_message (game, "%s has a %d card sideboard.", me->name,
		     me->deck->sb_size);
    display_message (game, "%s has a %d card deck.", me->name,
		     me->deck->size);

    set_player_life (game, game->local_player, starting_life);
    send_player_life (game);
    local_player_shuffle (game);
    game_draw_cards (game, game->local_player, starting_cards);
    send_draw_cards (game, starting_cards);
    send_setup_end (game);
    display_message (game, "-- %s's Security Code: %d", me->name,
		     me->deck->security_code);
    display_message (game, "-- %s says: ''%s''", me->name, starting_message);
    send_starting_message (game, starting_message);
    free (starting_message);
}

static void do_cancel_dialog (GtkWidget *w, struct newgame_args *args)
{
    struct game *game = args->game;

    if (stop_waiting_for_call (args))
	return;
    if ((game->opponent != NULL) &&
	((game->opponent->flags & local_wants_new) != 0)) {
	game->opponent->flags &= ~local_wants_new;
	send_request_new_game (game, FALSE);
	display_message (game, "%s cancelled new game.",
			 game->player[game->local_player]->name);
    }
    close_new_game_dialog (game);
}

static int stop_waiting_for_call (struct newgame_args *args)
{
    if (args->wait_input_tag == 0)
	return (FALSE);
    gdk_input_remove (args->wait_input_tag);
    args->wait_input_tag = 0;
    if (args->wait_socket > 0) {
	close_socket (args->wait_socket);
	args->wait_socket = 0;
    }
    gtk_label_set_text (GTK_LABEL (args->status_label), "");
    return (TRUE);
}

/*
 * Display the name of the deck in the New Game dialog.
 */
void newgame_update_deck_label (struct game *game)
{
    struct newgame_args *args;
    char *name;
    struct player *me;

    if (game->new_game_dialog == NULL)
	return;
    args = gtk_object_get_user_data (GTK_OBJECT (game->new_game_dialog));
    name = "";
    me = game->player[game->local_player];
    if ((me->deck != NULL) && (me->deck->name != NULL))
	name = me->deck->name;
    gtk_label_set_text (GTK_LABEL (args->deck_label), name);
}
