/*
 *  Copyright (c) 2003  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  kplayer.c -- A player is a set of zones and an on-screen description
 *		 of those zones.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include "game.h"
#include "zone.h"
#include "deck.h"
#include "dialogs.h"

struct player *player_new (struct game *game, char *name)
{
    struct player *me;
    int znum;

    if (name == NULL)
	name = "Player";
    me = calloc (1, sizeof (struct player));
    me->name = strdup (name);
    for (znum = 0; znum < NUM_ZONES; znum++)
	me->zone[znum] = zone_new (game);
    me->zone[HAND_ZONE]->flags = ZONE_DISPLAY_HAND | ZONE_PRIVATE |
	ZONE_SHOW_ORDER;
    me->zone[LIBRARY_ZONE]->flags = ZONE_PRIVATE | ZONE_SHUFFLE_CLOSE;
    me->zone[SIDEBOARD_ZONE]->flags = ZONE_PRIVATE;
    return (me);
}

void player_destroy (struct player *me)
{
    int znum;

    if (me->name != NULL)
	free (me->name);
    if (me->deck != NULL)
	deck_destroy (me->deck);
    for (znum = 0; znum < NUM_ZONES; znum++) {
	if (me->zone[znum] != NULL)
	    zone_destroy (me->zone[znum]);
    }
    if (me->turn_frame != NULL)
	gtk_widget_destroy (me->turn_frame);
    free (me);
}

void player_set_name (struct player *me, const char *name)
{
    if (me->name != NULL) {
	free (me->name);
    }
    me->name = strdup (name);
    if (me->name_label != NULL) {
	gtk_label_set_text (GTK_LABEL (me->name_label), me->name);
    }
}

int player_get_life (struct player *me)
{
    return (me->life);
}

void player_set_life (struct player *me, int life)
{
    char buf[12];

    me->life = life;
    if (me->life_label != NULL) {
	sprintf (buf, "%d", me->life);
	gtk_label_set_text (GTK_LABEL (me->life_label), buf);
    }
}

struct zone *player_get_zone (struct player *me, int znum)
{
    return (me->zone[znum]);
}

int player_sideboard_offset (struct game *game, int pid)
{
    struct deck *deck;
    deck = game->player[pid]->deck;
    return (player_deck_offset (pid) + deck->size);
}

int player_token_offset (struct game *game, int pid)
{
    struct deck *deck;
    deck = game->player[pid]->deck;
    return (player_deck_offset (pid) + deck->size + deck->sb_size);
}

/* ----------------------------------------------------------------- */

void player_make_labels (struct player *me, GtkWidget *player_vbox)
{
    GtkWidget *frame, *vbox, *table, *label;

    frame = gtk_frame_new (NULL);
    me->turn_frame = frame;
    vbox = gtk_vbox_new (FALSE, 0);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 4);
    label = gtk_label_new (me->name);
    me->name_label = label;
    gtk_container_add (GTK_CONTAINER (vbox), label);
    table = gtk_table_new (2, 4, TRUE);

    label = gtk_label_new ("Life:");
    align_and_attach (table, label, 0, 1, 0, 1);
    label = gtk_label_new ("");
    align_and_attach (table, label, 1, 2, 0, 1);
    me->life_label = label;

    label = gtk_label_new ("Hand:");
    align_and_attach (table, label, 2, 3, 0, 1);
    label = gtk_label_new ("");
    align_and_attach (table, label, 3, 4, 0, 1);
    zone_set_label (me->zone[HAND_ZONE], label);

    label = gtk_label_new ("Grave:");
    align_and_attach (table, label, 0, 1, 1, 2);
    label = gtk_label_new ("");
    align_and_attach (table, label, 1, 2, 1, 2);
    zone_set_label (me->zone[GRAVEYARD_ZONE], label);

    label = gtk_label_new ("Lib:");
    align_and_attach (table, label, 2, 3, 1, 2);
    label = gtk_label_new ("");
    align_and_attach (table, label, 3, 4, 1, 2);
    zone_set_label (me->zone[LIBRARY_ZONE], label);

    gtk_container_add (GTK_CONTAINER (vbox), table);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    gtk_container_add (GTK_CONTAINER (player_vbox), frame);
    gtk_widget_show_all (frame);
}

void player_set_turn (struct player *me, int flag)
{
    GtkShadowType type;

    if (me->turn_frame != NULL) {
	type = (flag ? GTK_SHADOW_ETCHED_OUT : GTK_SHADOW_ETCHED_IN);
	gtk_frame_set_shadow_type (GTK_FRAME (me->turn_frame), type);
    }
}

