/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  game.c -- The Mindless Automaton game manager.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <gtk/gtk.h>
#include "game.h"
#include "cardbase.h"
#include "zone.h"
#include "deck.h"
#include "playarea.h"
#include "opponent.h"
#include "prefs.h"
#include "cardart.h"
#include "expansions.h"
#include "dialogs.h"

char *color_names[] = {
    "White", "Blue", "Black", "Red", "Green", "Gold", "Artifact", "Land",
    NULL
};
GtkTargetEntry target_table[] = {
    { "MAGIC-ZONE", GTK_TARGET_SAME_APP, 0 },
};
int num_targets = sizeof(target_table) / sizeof(target_table[0]);

static char *diff_to_string (int i);

const char *oracle_name_key = "oracle_name";
const char *cardinfo_name_key = "cardinfo_name";
const char *player_name_key = "player_name";
const char *deck_name_key = "deck_name";
const char *starting_life_key = "starting_life";
const char *starting_hand_size_key = "starting_hand_size";
const char *starting_message_key = "starting_message";
const char *card_pic_type_key = "card_pic_type";
const char *zoom_ratio_key = "zoom_ratio";
const char *background_key = "background";
const char *background_type_key = "background_type";

/*
 *  game_new () -- Return a new structure that can be used to
 *	store the game state.
 */
struct game *game_new (void)
{
    struct game *game;
    char *name;

    game = calloc (1, sizeof (struct game));
    game->prefs = prefs_open ();
    name = get_preference (game->prefs, oracle_name_key);
    if (name != NULL) {
         game->cardbase = cardbase_new_from_ort (name);
    } else {
        name = getenv ("CARDINFO_DAT");
        if (name == NULL)
	    name = get_preference (game->prefs, cardinfo_name_key);
#ifdef CARDINFO_FILE
	if (name == NULL)
	    name = CARDINFO_FILE;
#endif
	if (name != NULL)
	    game->cardbase = cardbase_new_from_dat (name);
    }
    cardart_initialize (game->prefs);
    game->num_players = 1;
    game->player = malloc (sizeof (struct player *));
    name = get_preference (game->prefs, player_name_key);
    game->player[0] = player_new (game, name);
    return (game);
}

void game_empty_all_zones (struct game *game)
{
    int pid, znum;
    struct player *me;

    for (pid = 0; pid < game->num_players; pid++) {
	me = game->player[pid];
	for (znum = 0; znum < NUM_ZONES; znum++)
	    zone_reset (me->zone[znum]);
    }
    playarea_reset (game->playarea);
}

/*
 *  game_get_cardinfo () -- If the given card exists in some player's deck,
 *	return the information about that card.
 */
struct cardinfo *game_get_cardinfo (struct game *game, int cid)
{
    int pid, globalid, idx;
    struct cardinfo *info;

    info = NULL;
    for (pid = 0; pid < game->num_players; pid++) {
	idx = cid - player_deck_offset (pid);
	if ((game->player[pid]->deck != NULL) &&
	    ((globalid = deck_lookup (game->player[pid]->deck, idx)) >= 0)) {
	    info = cardbase_get_card (game->cardbase, globalid);
	    break;
	}
    }
    return (info);
}

char *game_get_card_name (struct game *game, int cid)
{
    struct cardinfo *info;

    if ((info = game_get_cardinfo (game, cid)) == NULL)
	return (NULL);
    return (info->name);
}

int owner_of_card (struct game *game, int cid)
{
    int pid, idx;

    for (pid = 0; pid < game->num_players; pid++) {
	idx = cid - player_deck_offset (pid);
	if (deck_lookup (game->player[pid]->deck, idx) >= 0)
	    return (pid);
    }
    return (-1);
}

int controller_of_card (struct game *game, int cid)
{
    struct table_card *tcard;

    tcard = playarea_get_table_card (game->playarea, cid);
    if (tcard == NULL)
	return (0);
    return (tcard->controller);
}

int find_card_zone (struct game *game, int cid,
		    int *pidp, int *znump, int *idxp)
{
    int p, z, c;
    struct zone *zone;

    for (p = 0; p < game->num_players; p++) {
	for (z = 0; z < NUM_ZONES; z++) {
	    zone = game->player[p]->zone[z];
	    for (c = 0; c < zone->size; c++)
		if (zone->cardlist[c] == cid) {
		    *pidp = p;
		    *znump = z;
		    *idxp = c;
		    return (TRUE);
		}
	}
    }
    return (FALSE);
}

int widget_to_zone (struct game *game, GtkWidget *w, int *pidp)
{
    int pid, znum;
    struct player *me;

    for (pid = 0; pid < game->num_players; pid++) {
	me = game->player[pid];
	for (znum = 0; znum < NUM_ZONES; znum++)
	    if ((w == me->zone[znum]->list) || (w == me->zone[znum]->box)) {
		*pidp = pid;
		return (znum);
	    }
    }
    return (-1);
}

char *name_of_zone (int znum)
{
    static char *zone_names[] = { "hand", "graveyard", "library",
				  "removed from game pile", "sideboard" };
    return (zone_names[znum]);
}

/*
 *  game_draw_cards() -- Move cards from a player's library to his hand.
 */
void game_draw_cards (struct game *game, int pid, int qty)
{
    struct player *me;
    struct card_move m;
    int max, i;

    me = game->player[pid];
    if (qty == 0) {
	display_message (game, "%s draws no cards.", me->name);
	return;
    }
    max = me->zone[LIBRARY_ZONE]->size;
    if (qty > max)
	qty = max;
    m.pid = pid;
    m.srcznum = LIBRARY_ZONE;
    m.idx = -1;
    m.dstznum = HAND_ZONE;
    m.bottom = FALSE;
    m.cid = 0;
    for (i = 0; i < qty; i++)
	game_move_card (game, &m);
    if (qty == 1)
	display_message (game, "%s draws a card.", me->name);
    else
	display_message (game, "%s draws %d cards.", me->name, qty);
}

/*
 *  game_play_card () -- Move a card from a private zone to the play area.
 */
void game_play_card (struct game *game, struct table_card *tcard,
		     int pid, int znum, int idx, int who)
{
    struct player *me;
    struct cardinfo *info;
    char *pname, *cname, *owner, *zname;

    me = game->player[pid];
    tcard->cid = zone_remove_card (me->zone[znum], idx);
    if (tcard->cid <= 0) {
	printf ("zone %d does not have %d cards\n", znum, idx);
	return;
    }
    info = game_get_cardinfo (game, tcard->cid);
    if (info != NULL)
	tcard->color = color_bits_to_number (info->color);
    playarea_add_card (game->playarea, tcard);

    /* Display a message about what happened. */
    pname = game->player[who]->name;
    if (((tcard->flags & CARD_FLIPPED_FLAG) != 0) &&
	((me->zone[znum]->flags & ZONE_PRIVATE) != 0))
	cname = "a facedown card";
    else
	cname = info->name;
    if ((pid == who) && (znum == HAND_ZONE)) {
	display_message (game, "%s plays %s.", pname, cname);
	return;
    }
    owner = me->name;
    zname = name_of_zone (znum);
    display_message (game, "%s moves %s from %s's %s to tabletop.",
		     pname, cname, owner, zname);
}

/*
 *  game_unplay_card () -- Move a card from the play area to a private zone.
 */
void game_unplay_card (struct game *game, int cid, int pid, int znum,
		       int who, int bottom)
{
    int flags;
    char *pname, *cname, *owner, *zname, *bstring;

    flags = playarea_remove_card (game->playarea, cid);
    if (flags < 0)
	return;
    zone_add_card (game->player[pid]->zone[znum], cid, bottom);

    /* Display a message about what happened. */
    pname = game->player[who]->name;
    if (((flags & CARD_FLIPPED_FLAG) != 0) &&
	((game->player[pid]->zone[znum]->flags & ZONE_PRIVATE) != 0))
	cname = "a facedown card";
    else
	cname = game_get_card_name (game, cid);
    owner = game->player[pid]->name;
    switch (znum) {
    case HAND_ZONE:
	display_message (game, "%s returns %s to %s's hand.", pname,
			 cname, owner);
	break;
    case GRAVEYARD_ZONE:
	display_message (game, "%s buries %s.", pname, cname);
	break;
    case REMOVED_ZONE:
	display_message (game, "%s removes %s from the game.", pname, cname);
	break;
    default:
	zname = name_of_zone (znum);
	bstring = (bottom ? "the BOTTOM of " : "");
	display_message (game, "%s moves %s from tabletop to %s%s's %s.",
			 pname, cname, bstring, owner, zname);
    }
}

/*
 *  game_move_card () -- Move a card from one zone to another.
 */
void game_move_card (struct game *game, struct card_move *mp)
{
    struct player *me;

    me = game->player[mp->pid];
    mp->cid = zone_remove_card (me->zone[mp->srcznum], mp->idx);
    if (mp->cid > 0) {
	zone_add_card (me->zone[mp->dstznum], mp->cid, mp->bottom);
    }
}

void game_move_card_message (struct game *game, struct card_move *mp)
{
    struct player *me;
    int no_peeking;
    char text[80], *cname, *bstring, *szname, *dzname;

    me = game->player[mp->pid];
    no_peeking = ((me->zone[mp->srcznum]->flags & ZONE_PRIVATE) &&
		  (me->zone[mp->dstznum]->flags & ZONE_PRIVATE));
    if (no_peeking) {
	if ((me->zone[mp->srcznum]->flags & ZONE_PRIVATE) &&
	    (mp->idx >= 0)) {
	    sprintf (text, "card #%d", mp->idx+1);
	    cname = text;
	} else {
	    cname = "a card";
	}
    } else {
	cname = game_get_card_name (game, mp->cid);
    }
    bstring = (mp->bottom ? "the BOTTOM of " : "");
    szname = name_of_zone (mp->srcznum);
    dzname = name_of_zone (mp->dstznum);
    display_message (game, "%s moves %s from %s's %s to %s%s.",
		     me->name, cname, me->name, szname, bstring, dzname);
}

void game_rearrange_message (struct game *game, int pid, int owner, int znum)
{
    display_message (game, "%s rearranges cards in %s's %s.",
		     game->player[pid]->name, game->player[owner]->name,
		     name_of_zone (znum));
}

void game_inform_peeking (struct game *game, int peeker, int owner,
			  int znum, int quantity, int sflag)
{
    char *pname, *oname, *zname, *state, *period;

    pname = game->player[peeker]->name;
    oname = game->player[owner]->name;
    zname = name_of_zone (znum);
    if (sflag) {
	state = "is";
	period = "...";
    } else {
	state = "stops";
	period = ".";
    }
    if (quantity == 0) {
	display_message (game, "%s %s looking through %s's %s%s",
			 pname, state, oname, zname, period);
    } else {
	display_message (game, "%s %s looking through the top %d cards "
			 "of %s's %s%s", pname, state, quantity,
			 oname, zname, period);
    }
}

/*
 *  game_create_card () -- The card becomes the idx'th card in the
 *	player's list of tokens, regardless of the current number of
 *	tokens in his deck.
 */
int game_create_card (struct game *game, int pid, char *name, int idx,
		      int znum, char *powtuf, int color)
{
    int globalid, cid;
    struct player *me;

    /* Add this card to the cardbase, the deck, and the player's hand. */
    globalid = cardbase_add_card (game->cardbase, name, powtuf,
				  1 << (color-1));
    me = game->player[pid];
    cid = player_token_offset (game, pid) + idx;
    deck_set_token (me->deck, idx, 1, globalid);
    display_message (game, "%s creates new card: %s.", me->name, name);
    zone_add_card (me->zone[znum], cid, 0);
    /* display_message ("moves card..."); */
    return cid;
}

void game_make_transient (struct game *game, GtkWidget *dialog)
{
    if (get_int_preference (game->prefs, "dialog_windows", 0) == 0) {
	gtk_window_set_transient_for (GTK_WINDOW (dialog),
				      GTK_WINDOW (game->main_window));
    }
}

/* ----------------------------------------------------------------- */

void display_message (struct game *game, char *fmt, ...)
{
    gchar *xfmt, *message;
    va_list ap;

    xfmt = alloca (strlen (fmt) + 16);
    strcpy (xfmt, "\n");
    strcat_current_time (xfmt);
    strcat (xfmt, fmt);
    va_start (ap, fmt);
    message = g_strdup_vprintf (xfmt, ap);
    va_end (ap);

#if GTK_MAJOR_VERSION >= 2
    {
	GtkTextBuffer *buffer;
	GtkTextIter start, end;
	GtkTextMark *mark;
	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(game->message_box));
	gtk_text_buffer_get_bounds(buffer, &start, &end);
	gtk_text_buffer_insert (buffer, &end, message, -1);
	mark = gtk_text_buffer_create_mark(buffer, NULL, &end, FALSE);
	gtk_text_view_scroll_mark_onscreen(GTK_TEXT_VIEW(game->message_box), mark);
    }
#else
    gtk_text_insert (GTK_TEXT (game->message_box), NULL, NULL, NULL,
		     message, strlen (message));
    gtk_widget_grab_focus (game->entry_box);
#endif
    g_free (message);
}

void strcat_current_time (char *message)
{
    time_t now;
    struct tm *data;
    int hour;

    time (&now);
    data = localtime (&now);
    hour = data->tm_hour;
    if (hour > 12)
	hour -= 12;
    if (hour == 0)
	hour = 12;
    sprintf (message+strlen(message), "%d:%02d - ", hour, data->tm_min);
}

/* ----------------------------------------------------------------- */
/*
 *  Simple commands that can be issued by the local or remote player.
 */

void game_flip_coin (struct game *game, int pid, int res)
{
    display_message (game, "%s flipped a coin: %s", game->player[pid]->name,
		     (res == 0 ? "TAILS" : "HEADS"));
}

void game_roll_die (struct game *game, int pid, int res, int size)
{
    display_message (game, "%s rolled a %d, using a %d sided die.",
		     game->player[pid]->name, res, size);
}

void set_card_controller (struct game *game, struct table_card *tcard,
			  int pid, int take)
{
    char *pname, *cname, *middle;

    tcard->controller = pid;
    pname = game->player[pid]->name;
    cname = game_get_card_name (game, tcard->cid);
    middle = (take ? "has TAKEN control of" : "is now the controller of");
    display_message (game, "%s %s %s.", pname, middle, cname);
}

void set_card_counters (struct game *game, struct table_card *tcard,
			int counters)
{
    char *name;
    int diff;

    diff = counters - tcard->counters;
    if (diff == 0)
	return;
    if ((tcard->flags & CARD_FLIPPED_FLAG) != 0)
        name = "Facedown Card";
    else
	name = game_get_card_name (game, tcard->cid);
    tcard->counters = counters;
    display_message (game, "%s now has %d counters. (%s)", name, counters,
		     diff_to_string (diff));
    playarea_paint_card (game->playarea, tcard);
}

void set_table_card_flag (struct game *game, struct table_card *tcard,
			  int flag, int state, int pid)
{
    char *name, *pname;
    int old_state;

    old_state = ((tcard->flags & flag) != 0);
    if (state == old_state)
	return;
    if (state)
	tcard->flags |= flag;
    else
	tcard->flags &= ~flag;
    if (((tcard->flags & CARD_FLIPPED_FLAG) != 0) && flag != CARD_FLIPPED_FLAG)
        name = "Facedown Card";
    else
	name = game_get_card_name (game, tcard->cid);
    switch (flag) {
    case CARD_TAPPED_FLAG:
	if (state)
	    display_message (game, "%s is tapped.", name);
	else
	    display_message (game, "%s is untapped.", name);
	break;
    case CARD_ATTACK_FLAG:
	if (state)
	    display_message (game, "%s is attacking.", name);
	else
	    display_message (game, "%s stops attacking.", name);
	break;
    case CARD_NO_UNTAP_FLAG:
	pname = game->player[pid]->name;
	if (state)
	    display_message (game, "%s sets %s to NOT Untap as Normal.",
			     pname, name);
	else
	    display_message (game, "%s sets %s to Untap as Normal.",
			     pname, name);
	break;
    case CARD_PHASED_FLAG:
	if (state)
	    display_message (game, "%s has Phased Out.", name);
	else
	    display_message (game, "%s has Phased In.", name);
	break;
    case CARD_FLIPPED_FLAG:
	pname = game->player[pid]->name;
	display_message (game, "%s flipped %s over.", pname, name);
	break;
    }
    playarea_paint_card (game->playarea, tcard);
}

void set_table_card_color (struct game *game, struct table_card *tcard,
			   int color, int pid)
{
    if (tcard->color == color)
	return;
    tcard->color = color;
    playarea_paint_card (game->playarea, tcard);
}

void set_game_phase (struct game *game, int phase)
{
    GtkToggleButton *button;
    char *pname = NULL;

    button = GTK_TOGGLE_BUTTON (game->phase_button[game->phase_number]);
    gtk_toggle_button_set_active (button, FALSE);
    game->phase_number = phase;
    button = GTK_TOGGLE_BUTTON (game->phase_button[game->phase_number]);
    gtk_toggle_button_set_active (button, TRUE);

    switch (game->phase_number) {
    case UNTAP_PHASE:
	pname = "Untap";
	break;
    case UPKEEP_PHASE:
	pname = "Upkeep";
	break;
    case DRAW_PHASE:
	pname = "Draw";
	break;
    case MAIN_PHASE:
	pname = "Main";
	break;
    case COMBAT_PHASE:
	pname = "Combat";
	break;
    case CLEANUP_PHASE:
	pname = "Cleanup";
	break;
    }
    display_message (game, "It is now the %s Phase.", pname);
}

void set_game_turn (struct game *game, int turn)
{
    int delta;

    player_set_turn (game->player[game->current_player], FALSE);

    /* Figure out the current player. */
    delta = turn - game->turn_number;
    game->current_player += delta;
    while (game->current_player < 0)
	game->current_player += game->num_players;
    while (game->current_player >= game->num_players)
	game->current_player -= game->num_players;

    game->turn_number = turn;
    display_message (game, "");
    display_message (game, "Turn %d: %s", turn,
		     game->player[game->current_player]->name);
    player_set_turn (game->player[game->current_player], TRUE);
    playarea_all_cards_stop_attacking (game->playarea);
}

void set_player_life (struct game *game, int pid, int life)
{
    struct player *me;
    int offset;

    me = game->player[pid];
    offset = life - me->life;
    if (offset == 0)
	return;
    player_set_life (me, life);
    display_message (game, "%s's life is now %d. (%s)", me->name,
		     me->life, diff_to_string (offset));
}

void set_player_name (struct game *game, int pid, const char *name)
{
    struct player *me;

    me = game->player[pid];
    if ((me->name != NULL) && (me->deck != NULL)) {
	display_message (game, "%s has changed name to '%s'.",
			 me->name, name);
    }
    player_set_name (me, name);
}

void set_card_location (struct game *game, int pid, int cid,
			int dstlid, int x, int y, int face, int bottom)
{
    int srcpid, srcznum, idx, dstpid, dstznum;
    struct table_card *tcard;
    struct zone *zone;
    struct card_move m;

    tcard = playarea_get_table_card (game->playarea, cid);
    if (tcard != NULL) {
	if (tcard->controller != pid)
	    return;
	lid_to_tux (dstlid, &dstpid, &dstznum);
	if (dstlid == 0) {
	    /* should use width and height */
	    set_table_card_flag (game, tcard, CARD_FLIPPED_FLAG, face, pid);
	    playarea_move_card (game->playarea, cid, x, y);
	    return;
	}
	/*
	 *  A card's controller may move it to any of its owner's zones.
	 */
	dstpid = owner_of_card (game, cid);
	game_unplay_card (game, cid, dstpid, dstznum, pid, bottom);
	return;
    }
    if (!find_card_zone (game, cid, &srcpid, &srcznum, &idx))
	return;
    if (dstlid == 0) {
	tcard = calloc (1, sizeof (struct table_card));
	tcard->cid = cid;
	tcard->xpos = x;
	tcard->ypos = y;
	if (face)
	    tcard->flags = CARD_FLIPPED_FLAG;
	tcard->controller = srcpid;
	game_play_card (game, tcard, srcpid, srcznum, idx, pid);
	return;
    }
    lid_to_tux (dstlid, &dstpid, &dstznum);
    if ((srcpid != pid) || (dstpid != pid))
	return;
    if (srcznum == dstznum) {
	zone = game->player[srcpid]->zone[srcznum];
	cid = zone_move_card (zone, idx, x);
	if (cid > 0)
	    game_rearrange_message (game, pid, srcpid, srcznum);
	return;
    }
    m.pid = pid;
    m.srcznum = srcznum;
    m.idx = idx;
    m.dstznum = dstznum;
    m.bottom = bottom;
    m.cid = 0;
    game_move_card (game, &m);
    if (m.cid > 0)
	game_move_card_message (game, &m);
}

static char *diff_to_string (int i)
{
    static char buf[10];

    if (i > 0) {
	buf[0] = '+';
	sprintf (buf+1, "%d", i);
    } else {
	sprintf (buf, "%d", i);
    }
    return (buf);
}

void game_show_random (struct game *game, int pid, int cid)
{
    char *pname, *cname;

    pname = game->player[pid]->name;
    cname = game_get_card_name (game, cid);
    display_message (game, "%s randomly shows '%s'", pname, cname);
}

int color_name_to_number (char *ctext)
{
    int i;

    for (i = 0; color_names[i] != NULL; i++)
	if (strcasecmp (ctext, color_names[i]) == 0)
	    return (i+1);
    return (0);
}

int color_bits_to_number (int cbits)
{
    if ((cbits & 32) != 0)
	return 6;
    if (cbits == 1)
	return 1;
    if (cbits == 2)
	return 2;
    if (cbits == 4)
	return 3;
    if (cbits == 8)
	return 4;
    if (cbits == 16)
	return 5;
    if (cbits == 64)
	return 7;
    if (cbits == 128)
	return 8;
    return 0;
}

int pid_of_single_opponent (struct game *game)
{
    int i;

    for (i = 0; i < game->num_players; i++)
	if (i != game->local_player)
	    return (i);
    return (-1);
}

void game_save_deck (struct game *game, int pid)
{
    char *directory, *filename;
    int dlen, nlen, fd;
    struct player *me;

    directory = get_preference (game->prefs, "opponents_directory");
    if (directory == NULL)
	return;
    dlen = strlen (directory);
    me = game->player[pid];
    nlen = strlen (me->name);
    filename = alloca (dlen + nlen + 7);
    strcpy (filename, directory);
    if (filename[dlen-1] != '/')
	filename[dlen++] = '/';
    strcpy (filename+dlen, me->name);
    strcpy (filename+dlen+nlen, ".dec");

    fd = open (filename, O_WRONLY | O_CREAT | O_EXCL, 0666);
    if (fd >= 0) {
	close (fd);
	display_message (game, "--- Writing %s.", filename);
	deck_write_file (filename, me->deck, game->cardbase);
    }
}

/**
 * Load the expansions data.
 */
void game_load_expansions (struct game *game)
{
    struct cardbase *cardbase;
    int count, i;
    char *filename, *cp;
    struct expansions *expansions;

    /*
     * If the cardbase was loaded from an Oracle file,
     * then it doesn't have expansion data.
     */
    cardbase = game->cardbase;
    count = 0;
    for (i = 0; i < cardbase->num_cards; i++) {
	if ((cardbase->cards[i].expansion != NULL) &&
	    (cardbase->cards[i].expansion[0] != 0))
	    count++;
    }
    if (count <= cardbase->num_cards / 10) {
	cp = "Cannot create sealed deck because card database\n"
	    "does not contain expansion data.\n"
	    "Try an Apprentice CardInfo.dat";
	do_message_dialog (cp);
	return;
    }

    filename = malloc (strlen (cardbase->filename) + 96);
    strcpy (filename, cardbase->filename);
    cp = strrchr (filename, '/');
    cp = (cp == NULL ? filename : cp + 1);
    strcpy (cp, "Expan.dat");
    expansions = expansions_read_expandat (filename, cardbase);
    if (expansions == NULL) {
	strcat (filename, ": ");
	strcat (filename, strerror (errno));
	do_message_dialog (filename);
    }
    else {
	strcpy (cp, "Distro.dat");
	if (!expansions_read_distrodat (expansions, filename)) {
	    strcat (filename, ": ");
	    strcat (filename, strerror (errno));
	    do_message_dialog (filename);
	    expansions_destroy (expansions);
	    expansions = NULL;
	}
    }
    free (filename);
    game->expansions = expansions;
}
