/*
 *  Copyright (c) 2005 Federico Poloni <f.poloni@sns.it>
 *  added to Mindless Automaton:(c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  expansions.c -- parses Expan.dat and Distro.dat to create a
 *                  MTG expansion set database
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "cardbase.h"
#include "expansions.h"
#include "fgetline.h"

static int get_commonality (char *expansion, char *name);
static void expansion_destroy (struct expansion *expansion);

struct expansions *
expansions_read_expandat(const char *expandat, struct cardbase *cardb)
{
    FILE *fp;
    struct expansions *expansions;
    char buf[128], *cp;
    struct expansion *exp;
    int alloc_expansions, alloc_commons, alloc_uncommons, alloc_rares;
    int globalid, commonality;

    fp = fopen (expandat, "r");
    if (fp == NULL)
	return (NULL);
    expansions = malloc (sizeof (struct expansions));
    expansions->num_expansions = 0;
    alloc_expansions = 16;
    expansions->db = malloc (alloc_expansions * sizeof (struct expansion *));
    while (fgetline (buf, sizeof (buf), fp) != NULL) {
	cp = buf;
	while (isspace (*cp))
	    cp++;
	if (!isalnum(*cp))
	    continue;
	/* format:  UT-Urza's toilet */
	exp = calloc (1, sizeof (struct expansion));
	strncpy (exp->shortname, cp, 2);
	exp->shortname[2] = 0;
	exp->name = strdup (cp);

	alloc_commons = 60;
	exp->commons = malloc (alloc_commons * sizeof (int));
	alloc_uncommons = 50;
	exp->uncommons = malloc (alloc_uncommons * sizeof (int));
	alloc_rares = 50;
	exp->rares = malloc (alloc_rares * sizeof (int));

	for (globalid = 0; globalid < cardb->num_cards; globalid++) {
	    commonality = get_commonality (cardb->cards[globalid].expansion,
					   exp->shortname);
	    switch(commonality) {
	    case 'C':
		if (exp->num_commons >= alloc_commons) {
		    alloc_commons *= 2;
		    exp->commons = realloc (exp->commons,
					    alloc_commons * sizeof (int));
		}
		exp->commons[exp->num_commons] = globalid;
		exp->num_commons++;
		break;
	    case 'U':
		if (exp->num_uncommons >= alloc_uncommons) {
		    alloc_uncommons *= 2;
		    exp->uncommons = realloc (exp->uncommons,
					      alloc_uncommons * sizeof (int));
		}
		exp->uncommons[exp->num_uncommons] = globalid;
		exp->num_uncommons++;
		break;
	    case 'R':
		if (exp->num_rares >= alloc_rares) {
		    alloc_rares *= 2;
		    exp->rares = realloc (exp->rares,
					  alloc_rares * sizeof (int));
		}
		exp->rares[exp->num_rares] = globalid;
		exp->num_rares++;
		break;
	    }
	}
	if (expansions->num_expansions >= alloc_expansions) {
	    alloc_expansions *= 2;
	    expansions->db = realloc (expansions->db, alloc_expansions *
				      sizeof (struct expansion *));
	}
	expansions->db[expansions->num_expansions] = exp;
	expansions->num_expansions++;
    }
    fclose (fp);
    return expansions;
}

int expansions_read_distrodat (struct expansions *expansions,
			       const char *distrodat)
{
    FILE *fp;
    char buf[128], *cp, *next, *name;
    struct expansion *exp;

    fp = fopen (distrodat, "r");
    if (fp == NULL)
	return (0);
    exp = NULL;
    while (fgetline (buf, sizeof (buf), fp) != NULL) {
	cp = buf;
	while (isspace (*cp))
	    cp++;
	if (*cp == 0)
	    continue;
	/*
	 * format:
	 * [UT]
	 *    Starter=R3,U10,C32
	 *    Booster=R1,U3,C11
	 */
	if (*cp == '[') {
	    name = cp + 1;
	    cp = strchr (name, ']');
	    if (cp != NULL)
		*cp = 0;
	    exp = expansions_get_from_short (expansions, name);
	    continue;
	}
	if (exp == NULL)
	    continue;
	if (strncmp (cp, "Starter=", 8) == 0) {
	    cp += 9; /* skips "Starter=R" */
	    exp->starterR = strtol (cp, &next, 10);
	    cp = next + 2; /* skips ",U" */
	    exp->starterU = strtol (cp, &next, 10);
	    cp = next + 2; /* skips ",C" */
	    exp->starterC = strtol (cp, &next, 10);
	}
	else if (strncmp (cp, "Booster=", 8) == 0) {
	    cp += 9; /* skips "Booster=R" */
	    exp->boosterR = strtol (cp, &next, 10);
	    cp = next + 2; /* skips ",U" */
	    exp->boosterU = strtol (cp, &next, 10);
	    cp = next + 2; /* skips ",C" */
	    exp->boosterC = strtol (cp, &next, 10);
	}
    }
    return (1);
}

static int get_commonality (char *expansion, char *name)
{
    char *cp;
    int found, len;

    cp = expansion;
    found = 0;
    while (*cp != 0) {
	len = 0;
	while (isalnum (cp[len]))
	    len++;
	if ((len == strlen (name)) && (strncmp (cp, name, len) == 0))
	    found = 1;
	cp += len;
	if (*cp == '-') {
	    cp++;
	    if (found)
		return (*cp);
	    /* skip over commonality */
	    while (isalnum (*cp))
		cp++;
	}
	if (*cp == ' ')
	    found = 0;
	/* skip over commas, spaces, and garbage */
	while ((*cp != 0) && !isalnum (*cp))
	    cp++;
    }
    return 0;
}

struct expansion *expansions_get_from_name (struct expansions *db, char *name)
{
    int i;
    for (i = 0; i < db->num_expansions; i++)
	if (strcmp (name, db->db[i]->name) == 0)
	    return db->db[i];
    return NULL;
}

struct expansion *expansions_get_from_short (struct expansions *db, char *name)
{
    int i;
    for (i = 0; i < db->num_expansions; i++)
	if (strcmp (name, db->db[i]->shortname) == 0)
	    return db->db[i];
    return NULL;
}

void expansions_destroy (struct expansions *db)
{
    int i;
    for (i = 0; i < db->num_expansions; i++)
	expansion_destroy (db->db[i]);
    if (db->db != NULL)
	free (db->db);
}

static void expansion_destroy (struct expansion *expansion)
{
    if (expansion->name != NULL)
	free (expansion->name);
    if (expansion->commons != NULL)
	free (expansion->commons);
    if (expansion->uncommons != NULL)
	free (expansion->uncommons);
    if (expansion->rares != NULL)
	free (expansion->rares);
    free (expansion);
}
