/*
 *  Copyright (c) Jordan Christensen <jchristensen@blastradius.com>
 *		  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  encrypt.c -- Translate the apprentice-protocol card name into the
 *		 actual card name.
 *
 */

#include <string.h>
#include <stdlib.h>
#include <assert.h>
#include "encrypt.h"

typedef unsigned char uchar;

static char *rbuffer = NULL;
static int rbuffer_size = 0;

static int base64Decode (char *, char *, int);
static char keyFromName (char *);
static int encode_three_bytes (char *, uchar *);
static uchar nth_radical (uchar, uchar);
static void allocate_space (int);

static int base64Decode(char *out, char *in, int len)
{
  int i, j;
  char ch;

  for (i = j = 0; i < len; i++) {
    ch = in[i]-58;
    switch (i%4) {
    case 0:
      out[j] = ch << 2;
      break;
    case 1:
      out[j++] |= ch >> 4;
      out[j] = (ch & 0x0f) << 4;
      break;
    case 2:
      out[j++] |= ch >> 2;
      out[j] = (ch & 0x03) << 6;
      break;
    case 3:
      out[j++] |= ch;
      break;
    }
  }
  out[j] = 0;
  return j;
}

static char keyFromName(char *name)
{
  char key = 0;
  int len, i;

  len = strlen(name);
  for (i = 0; i < len; i++) {
    key ^= name[i];
  }
  key ^= len;
  key ^= len/2;
  return key;
}

char *decrypt_name (char *pname, char *card_crypt)
{
    char key;
    int len, i;

    allocate_space (strlen (card_crypt) + 1);
    key = keyFromName (pname);
    len = base64Decode (rbuffer, card_crypt, strlen (card_crypt));
    assert (rbuffer[0] == rbuffer[1]);
    assert (rbuffer[0] == rbuffer[2]);
    assert (rbuffer[0] + 3 <= len);
    len = rbuffer[0];
    for (i = 0; i < len; i++)
	rbuffer[i] = rbuffer[i+3] ^ key;
    rbuffer[len] = 0;
    return (rbuffer);
}

static int encode_three_bytes (char *obuf, uchar *ibuf)
{
    obuf[0] = 58 + (ibuf[0] >> 2);
    obuf[1] = 58 + (((ibuf[0] & 0x3) << 4) | (ibuf[1] >> 4));
    obuf[2] = 58 + (((ibuf[1] & 0xf) << 2) | (ibuf[2] >> 6));
    obuf[3] = 58 + (ibuf[2] & 0x3f);
    return 4;
}

static uchar nth_radical (uchar value, uchar n)
{
    uchar i = 0;

    while (1) {
	if ((i ^ n) == value)
	    return i;
	if (i == 255)
	    break;
	i++;
    }
    return 0;
}

char *encrypt_name (char *pname, char *card_plain)
{
    char key;
    int len, written, done;
    uchar ibuf[3];
    char c;

    allocate_space (strlen (card_plain) * 4 / 3 + 9);
    key = keyFromName (pname);
    len = strlen (card_plain);
    ibuf[0] = ibuf[1] = ibuf[2] = len;
    written = encode_three_bytes (rbuffer, ibuf);
    done = 0;
    while (done < len) {
	ibuf[0] = nth_radical (card_plain[done], key);
	done++;
	c = (done < len ? card_plain[done] : ' ');
	ibuf[1] = nth_radical (c, key);
	done++;
	c = (done < len ? card_plain[done] : ' ');
	ibuf[2] = nth_radical (c, key);
	done++;
	written += encode_three_bytes (rbuffer + written, ibuf);
    }
    rbuffer[written] = 0;
    return (rbuffer);
}

static void allocate_space (int space)
{
    if (rbuffer == NULL) {
	rbuffer_size = 128;
	rbuffer = malloc (rbuffer_size);
    }
    if (rbuffer_size < space) {
	rbuffer_size = space;
	rbuffer = realloc (rbuffer, space);
    }
}
