/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  prefs.c -- Read and write the ~/.mindlessrc preferences file.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "prefs.h"
#include "fgetline.h"

struct keyvalue {
    char *key;
    char *value;
};

struct prefs {
    char *filename;
    int alloced;
    int size;
    struct keyvalue **values;
    int modified;
};

static struct keyvalue *parse_line (char *line);

struct prefs *prefs_open (void)
{
    struct prefs *prefs;
    char *home, *filename, buf[256];
    int len;
    FILE *fp;
    struct keyvalue *keyval;

    home = getenv ("HOME");
    if (home == NULL) {
	fputs ("HOME environment variable is not set.\n", stderr);
	return (NULL);
    }
    prefs = calloc (1, sizeof (struct prefs));
    len = strlen (home);
    filename = malloc (len + 12);
    strcpy (filename, home);
    if (filename[len-1] != '/')
	filename[len++] = '/';
    strcpy (filename+len, ".mindlessrc");
    prefs->filename = filename;
    fp = fopen (filename, "r");
    if (fp == NULL)
	return (prefs);
    while (fgetline (buf, sizeof (buf), fp) != NULL) {
	keyval = parse_line (buf);
	if (keyval != NULL) {
	    if (prefs->size >= prefs->alloced) {
		prefs->alloced += 5;
		prefs->values = realloc (prefs->values, prefs->alloced *
					 sizeof (struct keyvalue *));
	    }
	    prefs->values[prefs->size] = keyval;
	    prefs->size++;
	}
    }
    fclose (fp);
    return (prefs);
}

static struct keyvalue *parse_line (char *line)
{
    char *key, *value;
    int klen, vlen;
    struct keyvalue *keyval;

    while (isspace (*line))
	line++;
    if ((*line == 0) || (*line == '#'))
	return NULL;
    key = line;
    while (isalnum (*line) || (*line == '_'))
	line++;
    klen = (int)(line - key);
    while (isspace (*line))
	line++;
    if (*line != '=')
	return NULL;
    line++;
    while (isspace (*line))
	line++;
    value = line;
    vlen = strlen (value);
    while ((vlen > 0) && isspace (value[vlen-1]))
	vlen--;
    keyval = malloc (sizeof (struct keyvalue));
    keyval->key = malloc (klen+1);
    strncpy (keyval->key, key, klen);
    keyval->key[klen] = 0;
    keyval->value = malloc (vlen+1);
    strncpy (keyval->value, value, vlen);
    keyval->value[vlen] = 0;
    return (keyval);
}

char *get_preference (struct prefs *prefs, const char *key)
{
    int i;

    if (prefs == NULL)
	return (NULL);
    for (i = 0; i < prefs->size; i++)
	if (strcasecmp (key, prefs->values[i]->key) == 0)
	    return (prefs->values[i]->value);
    return (NULL);
}

void set_preference (struct prefs *prefs, const char *key, const char *value)
{
    int i;
    struct keyvalue *keyval;

    if (prefs == NULL)
	return;
    for (i = 0; i < prefs->size; i++)
	if (strcasecmp (key, prefs->values[i]->key) == 0) {
	    if (strcmp (value, prefs->values[i]->value) == 0)
		return;
	    free (prefs->values[i]->value);
	    prefs->values[i]->value = strdup (value);
	    prefs->modified++;
	    return;
	}
    if (prefs->size >= prefs->alloced) {
	prefs->alloced += 5;
	prefs->values = realloc (prefs->values, prefs->alloced *
				 sizeof (struct keyvalue *));
    }
    keyval = malloc (sizeof (struct keyvalue));
    keyval->key = strdup (key);
    keyval->value = strdup (value);
    prefs->values[prefs->size] = keyval;
    prefs->size++;
    prefs->modified++;
}

int get_int_preference (struct prefs *prefs, const char *key, int def)
{
    char *val;

    val = get_preference (prefs, key);
    return ((val == NULL) || (!isdigit (*val)) ? def : atoi (val));
}

void set_int_preference (struct prefs *prefs, const char *key, int value)
{
    char buf[16];

    sprintf (buf, "%d", value);
    set_preference (prefs, key, buf);
}

float get_float_preference (struct prefs *prefs, const char *key)
{
    char *val;

    val = get_preference (prefs, key);
    return (val == NULL ? 0 : strtod (val, NULL));
}

void set_float_preference (struct prefs *prefs, const char *key, float value)
{
    char buf[16];

    sprintf (buf, "%g", value);
    set_preference (prefs, key, buf);
}

void clear_preference (struct prefs *prefs, const char *key)
{
    int i;

    if (prefs == NULL)
	return;
    for (i = 0; i < prefs->size; i++)
	if (strcasecmp (key, prefs->values[i]->key) == 0) {
	    free (prefs->values[i]->key);
	    free (prefs->values[i]->value);
	    free (prefs->values[i]);
	    prefs->values[i] = prefs->values[prefs->size-1];
	    prefs->size--;
	    prefs->modified++;
	    return;
	}
}

int save_prefs (struct prefs *prefs)
{
    FILE *fp;
    int i;

    if ((prefs == NULL) || (prefs->filename == NULL))
	return (-1);
    if (prefs->modified == 0)
	return (0);
    fp = fopen (prefs->filename, "w");
    if (fp == NULL)
	return (-1);
    for (i = 0; i < prefs->size; i++)
	fprintf (fp, "%s=%s\n", prefs->values[i]->key,
		 prefs->values[i]->value);
    prefs->modified = 0;
    return (fclose (fp));
}
