/*
 *  Copyright (c) 1999  Salvatore Valente <svalente@mit.edu>
 *
 *  This program is free software.  You can modify and distribute it under
 *  the terms of the GNU General Public License.  There is no warranty.
 *  See the file "COPYING" for more information.
 *
 *  dialogs.c -- Dialogs for setting life total, card counters, etc.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "game.h"
#include "dialogs.h"
#include "opponent.h"
#include "zone.h"
#include "deck.h"
#include "cardbase.h"
#include "playarea.h"
#include "cardart.h"
#include "prefs.h"

#if GTK_MAJOR_VERSION >= 2
#define KEY_EVENT "key_release_event"
#else
#define KEY_EVENT "key_press_event"
#endif

/* ----------------------------------------------------------------- */

void do_message_dialog (const char *message)
{
    GtkWidget *dialog, *label, *ok_button;
    GtkDialog *dp;

    dialog = gtk_dialog_new ();
    /* gtk_window_set_title (GTK_WINDOW (dialog), ""); */
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 15);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);

    label = gtk_label_new (message);
    gtk_container_add (GTK_CONTAINER (dp->vbox), label);
    ok_button = gtk_button_new_with_label ("OK");
    gtk_signal_connect_object (GTK_OBJECT (ok_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    gtk_container_add (GTK_CONTAINER (dp->action_area), ok_button);
    gtk_widget_show_all (dialog);
}

/* ----------------------------------------------------------------- */

struct input_args {
    GtkWidget *dialog;
    GtkWidget *entry;
    struct game *game;
    input_callback_t callback;
    gpointer data;
};

static int input_box_key_pressed (GtkWidget *, GdkEventKey *,
				  struct input_args *);
static void do_input_ok (GtkWidget *w, struct input_args *args);
static void do_input_cancel (GtkWidget *w, struct input_args *args);

void do_input_dialog (struct game *game, const char *title,
		      const char *prompt, const char *def,
		      input_callback_t callback, gpointer data)
{
    GtkWidget *dialog, *label, *entry, *ok_button, *cancel_button;
    GtkDialog *dp;
    struct input_args *args;

    dialog = gtk_dialog_new ();
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dp->vbox), 10);
    gtk_window_set_title (GTK_WINDOW (dialog), title);
    label = gtk_label_new (prompt);
    gtk_box_pack_start (GTK_BOX (dp->vbox), label, FALSE, FALSE, 0);
    entry = gtk_entry_new ();
    if (def != NULL)
	gtk_entry_set_text (GTK_ENTRY (entry), def);
    gtk_box_pack_start (GTK_BOX (dp->vbox), entry, FALSE, FALSE, 5);

    args = malloc (sizeof (struct input_args));
    args->dialog = dialog;
    args->entry = entry;
    args->game = game;
    args->callback = callback;
    args->data = data;

    gtk_signal_connect (GTK_OBJECT (entry), KEY_EVENT,
			GTK_SIGNAL_FUNC (input_box_key_pressed), args);
    ok_button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			GTK_SIGNAL_FUNC (do_input_ok), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), ok_button,
			TRUE, TRUE, 0);
    cancel_button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
			GTK_SIGNAL_FUNC (do_input_cancel), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), cancel_button,
			TRUE, TRUE, 0);
    gtk_widget_show_all (dialog);
    gtk_widget_grab_focus (entry);
}

static int input_box_key_pressed (GtkWidget *w, GdkEventKey *event,
				  struct input_args *args)
{
    if ((event->keyval == GDK_KP_Enter) || (event->keyval == GDK_Return))
	do_input_ok (w, args);
    return (TRUE);
}

static void do_input_ok (GtkWidget *w, struct input_args *args)
{
    const gchar *text;
    int good;

    text = gtk_entry_get_text (GTK_ENTRY (args->entry));
    good = (*args->callback)(args->game, args->data, text);
    if (!good)
	return;
    gtk_widget_destroy (args->dialog);
    free (args);
}

static void do_input_cancel (GtkWidget *w, struct input_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static void do_verify_ok (GtkWidget *w, struct input_args *args);
static void do_verify_cancel (GtkWidget *w, struct input_args *args);

void do_verify_dialog (struct game *game, const char *title,
		       const char *prompt, input_callback_t callback)
{
    GtkWidget *dialog, *label, *ok_button, *cancel_button;
    GtkDialog *dp;
    struct input_args *args;

    dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (dialog), title);
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 15);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 15);

    label = gtk_label_new (prompt);
    gtk_box_pack_start (GTK_BOX (dp->vbox), label, FALSE, FALSE, 0);

    args = malloc (sizeof (struct input_args));
    args->dialog = dialog;
    args->entry = NULL;
    args->game = game;
    args->callback = callback;
    args->data = NULL;

    ok_button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (ok_button), "clicked",
			GTK_SIGNAL_FUNC (do_verify_ok), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), ok_button,
			TRUE, TRUE, 0);
    cancel_button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (cancel_button), "clicked",
			GTK_SIGNAL_FUNC (do_verify_cancel), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), cancel_button,
			TRUE, TRUE, 0);
    gtk_widget_show_all (dialog);
}

static void do_verify_ok (GtkWidget *w, struct input_args *args)
{
    gtk_widget_destroy (args->dialog);
    (*args->callback)(args->game, NULL, NULL);
    free (args);
}

static void do_verify_cancel (GtkWidget *w, struct input_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static int life_dialog_callback (struct game *, gpointer, const char *);

void do_life_dialog (struct game *game, int what_todo)
{
    char *title, prompt[75];

    sprintf(prompt, "You have %d life.\n",
                    game->player[game->local_player]->life);
    title = NULL;
    switch (what_todo) {
    case SET_LIFE:
	title = "Set Life";
	strcat (prompt, "Enter your new life total.");
	break;
    case GAIN_LIFE:
	title = "Add Life";
	strcat (prompt, "You GAINED how many life points?");
	break;
    case LOSE_LIFE:
	title = "Subtract Life";
	strcat (prompt, "You LOST how many life points?");
	break;
    }
    do_input_dialog (game, title, prompt, NULL, life_dialog_callback,
		     (gpointer) what_todo);
}

static int
life_dialog_callback (struct game *game, gpointer data, const char *text)
{
    int what_todo, old_life, new_life;

    what_todo = (int) data;
    old_life = game->player[game->local_player]->life;
    if (*text == '+') {
	what_todo = GAIN_LIFE;
	text++;
    } else if (*text == '-') {
	what_todo = LOSE_LIFE;
	text++;
    }
    new_life = atoi (text);
    switch (what_todo) {
    case GAIN_LIFE:
	new_life = old_life + new_life;
	break;
    case LOSE_LIFE:
	new_life = old_life - new_life;
	break;
    }
    if (old_life != new_life) {
	set_player_life (game, game->local_player, new_life);
	send_player_life (game);
    }
    return (TRUE);
}

/* ----------------------------------------------------------------- */

static void view_button_callback (GtkWidget *dialog, gpointer data);
static void shuffle_button_callback (GtkWidget *dialog, gpointer data);

void do_library_dialog (struct game *game)
{
    GtkWidget *dialog, *label, *button;
    GtkDialog *dp;
    GtkContainer *area;

    dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (dialog), "Library Options");
    dp = GTK_DIALOG (dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 15);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 15);
    gtk_object_set_user_data (GTK_OBJECT (dialog), game);

    label = gtk_label_new ("What would you like to do?");
    gtk_container_add (GTK_CONTAINER (dp->vbox), label);
    area = GTK_CONTAINER (dp->action_area);
    button = gtk_button_new_with_label ("View...");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			       GTK_SIGNAL_FUNC (view_button_callback),
			       GTK_OBJECT (dialog));
    gtk_container_add (area, button);
    button = gtk_button_new_with_label ("Shuffle");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			       GTK_SIGNAL_FUNC (shuffle_button_callback),
			       GTK_OBJECT (dialog));
    gtk_container_add (area, button);
    button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect_object (GTK_OBJECT (button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    gtk_container_add (area, button);
    gtk_widget_show_all (dialog);
}

static void view_button_callback (GtkWidget *dialog, gpointer data)
{
    struct game *game;

    game = gtk_object_get_user_data (GTK_OBJECT (dialog));
    gtk_widget_destroy (dialog);
    do_zone_dialog (game, game->local_player, LIBRARY_ZONE, 0);
}

static void shuffle_button_callback (GtkWidget *dialog, gpointer data)
{
    struct game *game;

    game = gtk_object_get_user_data (GTK_OBJECT (dialog));
    gtk_widget_destroy (dialog);
    local_player_shuffle (game);
}

/* ----------------------------------------------------------------- */

struct zone_dialog_args {
    GtkWidget *dialog;
    GtkWidget *shuffle_box;
    GtkWidget *up_button;
    struct game *game;
    int pid;
    int znum;
};

static gint zone_dialog_delete (GtkWidget *w, GdkEvent *event,
				struct zone_dialog_args *args);
static void zone_dialog_ok (GtkWidget *, struct zone_dialog_args *);
static void close_zone_dialog (struct zone_dialog_args *, int);
static void do_view_card (GtkWidget *, struct zone_dialog_args *);
static void do_zone_move_card (GtkWidget *, struct zone_dialog_args *);
static void do_move_all_to_bottom (GtkWidget *, struct zone_dialog_args *);

void do_zone_dialog (struct game *game, int pid, int znum, int quantity)
{
    struct player *me;
    struct zone *zone;
    char title[256];
    GtkWidget *dialog, *table, *scrollbox, *clist, *button, *arrow;
    struct zone_dialog_args *args;
    const char *zone_names[] = { "Hand", "Graveyard", "Library",
				 "Removed Cards", "Sideboard" };

    me = game->player[pid];
    zone = me->zone[znum];
    if (zone == NULL)
	return;
    if (zone->dialog != NULL) {
	gdk_window_raise (zone->dialog->window);
	return;
    }
    if (pid == game->local_player) {
	strcpy (title, "Viewing My ");
	strcat (title, zone_names[znum]);
    } else {
	strcpy (title, "Viewing ");
	strcat (title, me->name);
	strcat (title, "'s ");
	strcat (title, zone_names[znum]);
    }
    if (quantity > 0)
	sprintf (title+strlen (title), " (%d)", quantity);
    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);

    args = malloc (sizeof (struct zone_dialog_args));
    args->dialog = dialog;
    args->shuffle_box = NULL;
    args->game = game;
    args->pid = pid;
    args->znum = znum;

    gtk_signal_connect (GTK_OBJECT (dialog), "delete_event",
			GTK_SIGNAL_FUNC (zone_dialog_delete), args);
    gtk_window_set_title (GTK_WINDOW (dialog), title);
    table = gtk_table_new (3, 4, FALSE);
    gtk_container_set_border_width (GTK_CONTAINER (table), 10);
    gtk_container_add (GTK_CONTAINER (dialog), table);

    scrollbox = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrollbox),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize (GTK_WIDGET (scrollbox), 210, 250);
    clist = zone_create_clist (zone, quantity);
    gtk_container_add (GTK_CONTAINER (scrollbox), clist);
    gtk_table_attach (GTK_TABLE (table), scrollbox, 0, 2, 0, 3,
		      GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);
    if ((pid == game->local_player) && (quantity == 0) &&
	(zone->flags & ZONE_SHUFFLE_CLOSE) != 0) {
	button = gtk_check_button_new_with_label ("Shuffle When Done");
	args->shuffle_box = button;
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (button), TRUE);
	align_and_attach (table, button, 0, 1, 3, 4);
    }
    if ((pid == game->local_player) && (quantity > 0)) {
	button = new_wide_button ("Move All to Bottom", 6);
	gtk_signal_connect (GTK_OBJECT (button), "clicked",
			    GTK_SIGNAL_FUNC (do_move_all_to_bottom), args);
	align_and_attach (table, button, 0, 1, 3, 4);
    }
    button = new_wide_button ("View", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_view_card), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 0, 1,
		      0, 0, 0, 0);
    arrow = gtk_arrow_new (GTK_ARROW_UP, GTK_SHADOW_OUT);
    button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (button), arrow);
    args->up_button = button;
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_zone_move_card), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 1, 2,
		      0, 0, 0, 0);
    arrow = gtk_arrow_new (GTK_ARROW_DOWN, GTK_SHADOW_OUT);
    button = gtk_button_new ();
    gtk_container_add (GTK_CONTAINER (button), arrow);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (do_zone_move_card), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 2, 3,
		      0, 0, 0, 0);
    button = new_wide_button ("Ok", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (zone_dialog_ok), args);
    gtk_table_attach (GTK_TABLE (table), button, 2, 3, 3, 4,
		      0, 0, 0, 0);

    gtk_table_set_row_spacings (GTK_TABLE (table), 2);
    gtk_table_set_col_spacings (GTK_TABLE (table), 5);
    zone->dialog = dialog;
    gtk_widget_show_all (dialog);

    if ((zone->flags & ZONE_PRIVATE) != 0) {
	game_inform_peeking (game, game->local_player, pid, znum, quantity, 1);
	send_inform_peeking (game, pid, znum, quantity, 1);
    }
}

void align_and_attach (GtkWidget *table, GtkWidget *child,
		       int left, int right, int top, int bottom)
{
    GtkWidget *w;

    w = gtk_alignment_new (0, 0, 0, 0);
    gtk_container_add (GTK_CONTAINER (w), child);
    gtk_table_attach (GTK_TABLE (table), w, left, right, top, bottom,
		      GTK_EXPAND | GTK_FILL, 0, 0, 0);
}

GtkWidget *new_wide_button (const char *text, int pad)
{
    GtkWidget *button, *label;

    button = gtk_button_new ();
    label = gtk_label_new (text);
    gtk_misc_set_alignment (GTK_MISC (label), 0.5, 0.5);
    gtk_misc_set_padding (GTK_MISC (label), pad, 1);
    gtk_container_add (GTK_CONTAINER (button), label);
    return (button);
}

static gint zone_dialog_delete (GtkWidget *w, GdkEvent *event,
				struct zone_dialog_args *args)
{
    close_zone_dialog (args, FALSE);
    return (FALSE);
}

static void zone_dialog_ok (GtkWidget *w, struct zone_dialog_args *args)
{
    close_zone_dialog (args, TRUE);
}

static void close_zone_dialog (struct zone_dialog_args *args, int do_close)
{
    struct game *game = args->game;
    struct zone *zone;
    int qty, do_shuffle;

    zone = game->player[args->pid]->zone[args->znum];
    if ((zone->flags & ZONE_PRIVATE) != 0) {
	qty = zone->top_cards_requested;
	game_inform_peeking (game, game->local_player, args->pid,
			     args->znum, qty, 0);
	send_inform_peeking (game, args->pid, args->znum, qty, 0);
    }
    zone->list = NULL;
    zone->dialog = NULL;
    do_shuffle = FALSE;
    if (do_close) {
	if (args->shuffle_box != NULL)
	    do_shuffle = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
						       (args->shuffle_box));
	gtk_widget_destroy (args->dialog);
    }
    free (args);
    if (do_shuffle) {
	/* Let's just assume this zone is the local player's library. */
	local_player_shuffle (game);
    }
}

static void do_view_card (GtkWidget *w, struct zone_dialog_args *args)
{
    view_card_selected_in_zone_list (args->game, args->pid, args->znum);
}

void view_card_selected_in_zone_list (struct game *game, int pid, int znum)
{
    struct zone *zone;
    int idx;

    zone = game->player[pid]->zone[znum];
    if (zone->list == NULL)
	return;
    idx = gtk_clist_get_selected_row (GTK_CLIST (zone->list));
    if (idx < 0)
	return;
    if ((zone->flags & ZONE_DISPLAY_HAND) == 0)
	idx = zone->size - 1 - idx;
    do_view_card_dialog (game, zone->cardlist[idx]);
}

int gtk_clist_get_selected_row (GtkCList *clist)
{
    GList *selection;

    selection = clist->selection;
    if (selection == NULL)
	return -1;
    if (selection->next != NULL)
	return -1;
    return GPOINTER_TO_INT (selection->data);
}

static void do_zone_move_card (GtkWidget *w, struct zone_dialog_args *args)
{
    struct game *game = args->game;
    struct zone *zone;
    int idx, other, cid, rows;

    if (args->pid != game->local_player)
	return;
    zone = game->player[args->pid]->zone[args->znum];
    if (zone->list == NULL)
	return;
    idx = gtk_clist_get_selected_row (GTK_CLIST (zone->list));
    if (idx < 0)
	return;
    rows = GTK_CLIST (zone->list)->rows;
    other = idx + (w == args->up_button ? -1 : +1);
    if ((other < 0) || (other >= zone->size) || (other >= rows))
	return;
    if ((zone->flags & ZONE_DISPLAY_HAND) == 0) {
	idx = zone->size - 1 - idx;
	other = zone->size - 1 - other;
    }
    cid = zone_move_card (zone, idx, other);

    game_rearrange_message (game, game->local_player, args->pid, args->znum);
    send_card_order_in_zone (game, cid, args->pid, args->znum, other);
}

static void
do_move_all_to_bottom (GtkWidget *w, struct zone_dialog_args *args)
{
    struct game *game = args->game;
    struct zone *zone;
    int idx, other, cid, rows;

    if (args->pid != game->local_player)
	return;
    zone = game->player[args->pid]->zone[args->znum];
    if ((zone->list == NULL) || (zone->top_cards_requested == 0))
	return;
    for (rows = GTK_CLIST (zone->list)->rows; rows > 0; rows--) {
	idx = 0;
	other = zone->size-1;
	if ((zone->flags & ZONE_DISPLAY_HAND) == 0) {
	    idx = zone->size - 1 - idx;
	    other = zone->size - 1 - other;
	}
	cid = zone_move_card (zone, idx, other);
	game_rearrange_message (game, game->local_player,
				args->pid, args->znum);
	send_card_order_in_zone (game, cid, args->pid, args->znum, other);
    }
}

/* ----------------------------------------------------------------- */

struct create_args {
    GtkWidget *dialog;
    GtkWidget *name_entry;
    GtkWidget *pow_entry;
    GtkWidget *color_menu;
    struct game *game;
};

static void create_card_ok (GtkWidget *dialog, struct create_args *args);
static void create_card_cancel (GtkWidget *dialog, struct create_args *args);

static char *create_card_name;
static char *create_card_power;
static int create_card_color;

void do_create_card_dialog (struct game *game)
{
    GtkWidget *table, *label, *button;
    GtkDialog *dp;
    struct create_args *args;
    GList *items;
    int i;

    args = malloc (sizeof (struct create_args));
    args->dialog = gtk_dialog_new ();
    dp = GTK_DIALOG (args->dialog);
    gtk_container_set_border_width (GTK_CONTAINER (dp->vbox), 10);
    gtk_window_set_title (GTK_WINDOW (args->dialog), "Create Card");

    table = gtk_table_new (2, 3, FALSE);

    label = gtk_label_new ("Name:");
    align_and_attach (table, label, 0, 1, 0, 1);
    label = gtk_label_new ("Power/Toughness:");
    align_and_attach (table, label, 0, 1, 1, 2);
    label = gtk_label_new ("Color:");
    align_and_attach (table, label, 0, 1, 2, 3);
    gtk_container_add (GTK_CONTAINER (dp->vbox), table);

    args->name_entry = gtk_entry_new ();
    align_and_attach (table, args->name_entry, 1, 2, 0, 1);
    args->pow_entry = gtk_entry_new ();
    align_and_attach (table, args->pow_entry, 1, 2, 1, 2);
    args->color_menu = gtk_combo_new ();
    items = NULL;
    for (i = 0; color_names[i] != NULL; i++)
	items = g_list_append (items, color_names[i]);
    gtk_combo_set_popdown_strings (GTK_COMBO (args->color_menu), items);
    align_and_attach (table, args->color_menu, 1, 2, 2, 3);
    args->game = game;

    if (create_card_name != NULL) {
	gtk_entry_set_text (GTK_ENTRY (args->name_entry), create_card_name);
	gtk_editable_select_region (GTK_EDITABLE (args->name_entry), 0,
				    strlen(create_card_name));
    }
    gtk_widget_grab_focus (GTK_WIDGET (args->name_entry));
    if (create_card_power != NULL)
	gtk_entry_set_text (GTK_ENTRY (args->pow_entry), create_card_power);
    if (create_card_color > 0)
	gtk_entry_set_text (GTK_ENTRY (GTK_COMBO (args->color_menu)->entry),
			    color_names[create_card_color-1]);

    button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (create_card_ok), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), button, TRUE, TRUE, 15);
    button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (create_card_cancel), args);
    gtk_box_pack_start (GTK_BOX (dp->action_area), button, TRUE, TRUE, 15);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 25);

    gtk_table_set_row_spacings (GTK_TABLE (table), 5);
    gtk_table_set_col_spacings (GTK_TABLE (table), 5);
    gtk_widget_show_all (args->dialog);
}

static void create_card_ok (GtkWidget *dialog, struct create_args *args)
{
    struct game *game = args->game;
    char *name, *power, *ctext;
    int color, cid, znum;
    GtkCombo *combo;
    struct deck *deck;

    name = gtk_editable_get_chars (GTK_EDITABLE (args->name_entry), 0, -1);
    power = gtk_editable_get_chars (GTK_EDITABLE (args->pow_entry), 0, -1);
    combo = GTK_COMBO (args->color_menu);
    ctext = gtk_editable_get_chars (GTK_EDITABLE (combo->entry), 0, -1);
    color = color_name_to_number (ctext);
    g_free (ctext);

    deck = game->player[game->local_player]->deck;
    znum = HAND_ZONE;
    cid = game_create_card (game, game->local_player, name, deck->tk_size,
			    znum, power, color);
    send_create_card (game, name, cid, znum, power, color);

    if (create_card_name != NULL)
	g_free (create_card_name);
    create_card_name = name;
    if (create_card_power != NULL)
	g_free (create_card_power);
    create_card_power = power;
    create_card_color = color;

    gtk_widget_destroy (args->dialog);
    free (args);
}

static void create_card_cancel (GtkWidget *dialog, struct create_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static GtkWidget *pixbuf_to_widget (GdkPixbuf *, GtkWidget *);
static void do_card_image_dialog (GtkWidget *w, struct cardinfo *);

void do_view_card_dialog (struct game *game, int cid)
{
    struct cardinfo *info;
    GtkWidget *dialog, *picwidget, *name_label, *cost_label, *text_label;
    GtkWidget *cardtype_label, *flavor_label, *pow_label;
    GtkWidget *layout, *namebox, *frame;
    GtkWidget *botbox, *close_button, *comp_button;
    GdkPixbuf *pixbuf;

    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (dialog), "View Card");

    info = game_get_cardinfo (game, cid);
    if (info->art == NULL)
	info->art = cardart_load_cardart (info->name, info->expansion);

    /*  Do the old View Card dialog.  */
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 10);

    pixbuf = cardart_get_unscaled_art (info->art);
    picwidget = NULL;
    if (pixbuf != NULL)
	picwidget = pixbuf_to_widget (pixbuf, game->playarea->darea);

    name_label = gtk_label_new (info->name);
    cost_label = gtk_label_new (info->cost);
    cardtype_label = gtk_label_new (info->cardtype);
    text_label = gtk_label_new (info->text);
    flavor_label = gtk_label_new (info->flavor);
    pow_label = gtk_label_new (info->pow_tgh);

    gtk_misc_set_alignment (GTK_MISC (name_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (cost_label), 1, 0);
    gtk_misc_set_alignment (GTK_MISC (cardtype_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (text_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (flavor_label), 0, 0);
    gtk_misc_set_alignment (GTK_MISC (pow_label), 1, 0);

    gtk_label_set_line_wrap (GTK_LABEL (text_label), TRUE);
    gtk_label_set_line_wrap (GTK_LABEL (flavor_label), TRUE);

    close_button = new_wide_button ("Close", 6);
    gtk_signal_connect_object (GTK_OBJECT (close_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (dialog));
    comp_button = new_wide_button ("Card Image", 6);
    gtk_signal_connect (GTK_OBJECT (comp_button), "clicked",
			GTK_SIGNAL_FUNC (do_card_image_dialog), info);

    layout = gtk_vbox_new (FALSE, 5);
    gtk_box_set_spacing (GTK_BOX (layout), 5);
    namebox = gtk_hbox_new (FALSE, 0);
    gtk_box_pack_start (GTK_BOX (namebox), name_label, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (namebox), cost_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (layout), namebox, FALSE, FALSE, 0);
    if (picwidget != NULL)
	gtk_box_pack_start (GTK_BOX (layout), picwidget, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (layout), cardtype_label, TRUE, FALSE, 0);
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), text_label);
    gtk_misc_set_padding (GTK_MISC (text_label), 5, 5);
    gtk_box_pack_start (GTK_BOX (layout), frame, FALSE, FALSE, 0);
    frame = gtk_frame_new (NULL);
    gtk_container_add (GTK_CONTAINER (frame), flavor_label);
    gtk_misc_set_padding (GTK_MISC (flavor_label), 5, 5);
    gtk_box_pack_start (GTK_BOX (layout), frame, FALSE, FALSE, 0);
    botbox = gtk_hbox_new (FALSE, 10);
    gtk_box_pack_start (GTK_BOX (botbox), close_button, FALSE, FALSE, 0);
    if (comp_button != NULL)
	gtk_box_pack_start (GTK_BOX (botbox), comp_button, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (botbox), pow_label, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (layout), botbox, FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (dialog), layout);
    gtk_widget_show_all (dialog);
}

static GtkWidget *pixbuf_to_widget (GdkPixbuf *pixbuf, GtkWidget *widget)
{
    GdkPixmap *pixmap;
    int width, height;
    GtkWidget *picwidget;

    width = gdk_pixbuf_get_width (pixbuf);
    height = gdk_pixbuf_get_height (pixbuf);
    pixmap = gdk_pixmap_new (widget->window, width, height, -1);
    gdk_pixbuf_render_to_drawable (pixbuf, pixmap, widget->style->white_gc,
				   0, 0, 0, 0, width, height,
				   GDK_RGB_DITHER_NORMAL, 0, 0);
    picwidget = gtk_pixmap_new (pixmap, NULL);
    gdk_pixmap_unref (pixmap);
    return (picwidget);
}

static void do_card_image_dialog (GtkWidget *w, struct cardinfo *info)
{
    GdkPixbuf *pixbuf;
    GtkWidget *dialog, *picwidget;

    if (info->art == NULL)
	info->art = cardart_load_cardart3 (info->name, info->expansion, TRUE);
    pixbuf = cardart_get_card_image (info);
    if (pixbuf == NULL)
	return;

    dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (dialog), "View Card Image");
    picwidget = pixbuf_to_widget (pixbuf, w);
    gtk_container_add (GTK_CONTAINER (dialog), picwidget);
    gtk_widget_show_all (dialog);
}

/* ----------------------------------------------------------------- */

struct preferences_args {
    struct game *game;
    GtkWidget *dialog;
    GtkWidget *download_button;
    GtkWidget *picture_button;
    GtkWidget *name_button;
    GtkWidget *both_button;
    GtkWidget *zoom_entry;
    GtkWidget *image_entry;
    GtkWidget *tile_button;
    GtkWidget *center_button;
};

static void preferences_apply (GtkWidget *w, struct preferences_args *args);
static void preferences_ok (GtkWidget *w, struct preferences_args *args);
static void preferences_cancel (GtkWidget *w, struct preferences_args *args);
static void image_browse (GtkWidget *w, struct preferences_args *args);
static void image_browse_ok (GtkWidget *dialog, gpointer data);
#define is_button_active(x) \
	gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(x))
#define toggle_button_set_active(x, val) \
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(x), val)

void do_preferences_dialog (struct game *game)
{
    struct preferences_args *args;
    int do_download, pic_type, background_type;
    GtkWidget *items, *radios, *button, *zoombox, *label;
    GtkWidget *frame, *vbox, *hbox, *alignment;
    GtkDialog *dp;
    char *text, *background;
    GSList *group;

    args = malloc (sizeof (struct preferences_args));
    args->game = game;

    args->dialog = gtk_dialog_new ();
    gtk_window_set_title (GTK_WINDOW (args->dialog), "Preferences");
    dp = GTK_DIALOG (args->dialog);
    gtk_container_set_border_width (GTK_CONTAINER (args->dialog), 10);
    gtk_box_set_spacing (GTK_BOX (dp->vbox), 10);
    gtk_container_set_border_width (GTK_CONTAINER (dp->action_area), 0);
    gtk_box_set_spacing (GTK_BOX (dp->action_area), 15);

    /* frame = gtk_frame_new ("Card Display"); */
    items = gtk_vbox_new (FALSE, 10);
    gtk_container_set_border_width (GTK_CONTAINER (items), 6);

    text = "Download card pictures from www.wizards.com";
    args->download_button = gtk_check_button_new_with_label (text);
    gtk_container_add (GTK_CONTAINER (items), args->download_button);

    radios = gtk_vbox_new (FALSE, 0);
    text = "Display only card picture";
    args->picture_button = gtk_radio_button_new_with_label (NULL, text);
    gtk_container_add (GTK_CONTAINER (radios), args->picture_button);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->picture_button));
    text = "Display only card name";
    args->name_button = gtk_radio_button_new_with_label (group, text);
    gtk_container_add (GTK_CONTAINER (radios), args->name_button);
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->name_button));
    text = "Display card picture and name";
    args->both_button = gtk_radio_button_new_with_label (group, text);
    gtk_container_add (GTK_CONTAINER (radios), args->both_button);
    gtk_container_add (GTK_CONTAINER (items), radios);

    zoombox = gtk_hbox_new (FALSE, 0);
    text = "Zoom ratio:";
    label = gtk_label_new (text);
    gtk_misc_set_alignment (GTK_MISC (label), 0, 0.5);
    gtk_box_pack_start (GTK_BOX (zoombox), label, TRUE, TRUE, 5);
    args->zoom_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (zoombox), args->zoom_entry, FALSE, FALSE, 5);
    gtk_container_add (GTK_CONTAINER (items), zoombox);

    gtk_container_add (GTK_CONTAINER (dp->vbox), items);

    do_download = get_int_preference (game->prefs, download_cardart_key, 0);
    if (do_download) {
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON(args->download_button),
				      TRUE);
    }

    pic_type = get_int_preference (game->prefs, card_pic_type_key,
				   PLAYAREA_ART_ONLY);
    switch (pic_type) {
    case PLAYAREA_ART_ONLY:
	toggle_button_set_active (args->picture_button, TRUE);
	break;
    case PLAYAREA_NAME_ONLY:
	toggle_button_set_active (args->name_button, TRUE);
	break;
    case PLAYAREA_ART_AND_NAME:
	toggle_button_set_active (args->both_button, TRUE);
	break;
    }

    text = get_preference (game->prefs, zoom_ratio_key);
    if (text == NULL)
	text = "1";
    gtk_entry_set_text (GTK_ENTRY (args->zoom_entry), text);

    frame = gtk_frame_new ("Play Area");
    vbox = gtk_vbox_new (FALSE, 6);
    gtk_container_set_border_width (GTK_CONTAINER (vbox), 6);
    hbox = gtk_hbox_new (FALSE, 12);
    label = gtk_label_new ("Image:");
    gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
    args->image_entry = gtk_entry_new ();
    gtk_box_pack_start (GTK_BOX (hbox), args->image_entry, TRUE, TRUE, 0);
    gtk_container_add (GTK_CONTAINER (vbox), hbox);
    hbox = gtk_hbox_new (TRUE, 0);
    args->tile_button = gtk_radio_button_new_with_label (NULL, "Tile");
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->tile_button));
    alignment = gtk_alignment_new (0.75, 0.5, 0, 0);
    gtk_container_add (GTK_CONTAINER (alignment), args->tile_button);
    gtk_container_add (GTK_CONTAINER (hbox), alignment);
    args->center_button = gtk_radio_button_new_with_label (group, "Center");
    group = gtk_radio_button_group (GTK_RADIO_BUTTON (args->center_button));
    alignment = gtk_alignment_new (0.25, 0.5, 0, 0);
    gtk_container_add (GTK_CONTAINER (alignment), args->center_button);
    gtk_container_add (GTK_CONTAINER (hbox), alignment);
    button = gtk_button_new_with_label ("Browse...");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (image_browse), args);
    gtk_container_add (GTK_CONTAINER (hbox), button);
    gtk_container_add (GTK_CONTAINER (vbox), hbox);
    gtk_container_add (GTK_CONTAINER (frame), vbox);
    gtk_container_add (GTK_CONTAINER (dp->vbox), frame);

    background = get_preference (game->prefs, background_key);
    if (background != NULL)
	gtk_entry_set_text (GTK_ENTRY (args->image_entry), background);
    background_type = get_int_preference (game->prefs, background_type_key, 0);
    switch (background_type) {
    case PLAYAREA_TILE:
	toggle_button_set_active (args->tile_button, TRUE);
	break;
    case PLAYAREA_CENTER:
	toggle_button_set_active (args->center_button, TRUE);
	break;
    }

    button = gtk_button_new_with_label ("Ok");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_ok), args);
    gtk_container_add (GTK_CONTAINER (dp->action_area), button);
    button = gtk_button_new_with_label ("Apply");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_apply), args);
    gtk_container_add (GTK_CONTAINER (dp->action_area), button);
    button = gtk_button_new_with_label ("Cancel");
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (preferences_cancel), args);
    gtk_container_add (GTK_CONTAINER (dp->action_area), button);

    gtk_widget_show_all (args->dialog);
}

static void preferences_apply (GtkWidget *w, struct preferences_args *args)
{
    struct game *game = args->game;
    int option;
    const gchar *text;
    float ratio;

    option = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON
					   (args->download_button));
    set_int_preference (game->prefs, download_cardart_key, option);
    cardart_initialize (game->prefs);

    option = -1;
    if (is_button_active (args->picture_button))
	option = PLAYAREA_ART_ONLY;
    else if (is_button_active (args->name_button))
	option = PLAYAREA_NAME_ONLY;
    else if (is_button_active (args->both_button))
	option = PLAYAREA_ART_AND_NAME;
    if (option >= 0)
	set_int_preference (game->prefs, card_pic_type_key, option);

    text = gtk_entry_get_text (GTK_ENTRY (args->zoom_entry));
    ratio = strtod (text, NULL);
    if (ratio > 0)
	set_float_preference (game->prefs, zoom_ratio_key, ratio);

    text = gtk_entry_get_text (GTK_ENTRY (args->image_entry));
    if ((text == NULL) || (*text == 0)) {
	clear_preference (game->prefs, background_key);
    } else {
	set_preference (game->prefs, background_key, text);
    }
    option = -1;
    if (is_button_active (args->tile_button))
	option = PLAYAREA_TILE;
    else if (is_button_active (args->center_button))
	option = PLAYAREA_CENTER;
    if (option >= 0)
	set_int_preference (game->prefs, background_type_key, option);

    playarea_refresh_prefs (game->playarea);
    playarea_repaint (game->playarea);
}

static void preferences_ok (GtkWidget *w, struct preferences_args *args)
{
    preferences_apply (w, args);
    gtk_widget_destroy (args->dialog);
    free (args);
}

static void preferences_cancel (GtkWidget *w, struct preferences_args *args)
{
    gtk_widget_destroy (args->dialog);
    free (args);
}

static void image_browse (GtkWidget *w, struct preferences_args *args)
{
    GtkWidget *filew;
    GtkFileSelection *fsp;

    filew = gtk_file_selection_new ("Background Image");
    gtk_object_set_user_data (GTK_OBJECT (filew), args);
    fsp = GTK_FILE_SELECTION (filew);
    gtk_signal_connect_object (GTK_OBJECT (fsp->ok_button), "clicked",
			       GTK_SIGNAL_FUNC (image_browse_ok),
			       GTK_OBJECT (filew));
    gtk_signal_connect_object (GTK_OBJECT (fsp->cancel_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (filew));
    gtk_widget_show (filew);
}

static void image_browse_ok (GtkWidget *dialog, gpointer data)
{
    struct preferences_args *args;
    const gchar *filename;

    args = gtk_object_get_user_data (GTK_OBJECT (dialog));
    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog));
    if (filename == NULL)
	return;
    gtk_entry_set_text (GTK_ENTRY (args->image_entry), filename);
    gtk_widget_destroy (dialog);
}
    
/* ----------------------------------------------------------------- */

struct sblist_line {
    int qty;
    int globalid;
};

struct sb_list {
    GtkWidget *scrollbox;
    GtkWidget *clist;
    int num_lines;
    struct sblist_line *lineinfo;
};

struct sideboard_args {
    struct game *game;
    GtkWidget *dialog;
    struct sb_list deck_list;
    struct sb_list sb_list;
    int changed;
};

static void create_sb_list (struct sb_list *, int, int *, struct game *);
static void sideboard_trade (GtkWidget *, struct sideboard_args *);
static void decrement_line (struct sb_list *listp, int idx);
static void increment_globalid (struct sb_list *, int, struct cardbase *);
static void sideboard_dialog_ok (GtkWidget *, struct sideboard_args *);
static void sblist_to_cardlist (struct sb_list *listp, int *cardlist);
static void sideboard_dialog_cancel (GtkWidget *, struct sideboard_args *);

void do_sideboard_dialog (struct game *game)
{
    struct deck *deck;
    struct sideboard_args *args;
    GtkWidget *hbox, *vbox, *button;

    deck = game->player[game->local_player]->deck;
    if (deck == NULL)
	return;
    args = malloc (sizeof (struct sideboard_args));
    args->game = game;
    args->dialog = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_window_set_title (GTK_WINDOW (args->dialog), "Sideboard");
    gtk_container_set_border_width (GTK_CONTAINER (args->dialog), 10);
    hbox = gtk_hbox_new (FALSE, 10);
    create_sb_list (&args->deck_list, deck->size, deck->cardlist, game);
    gtk_box_pack_start (GTK_BOX (hbox), args->deck_list.scrollbox,
			FALSE, FALSE, 0);
    vbox = gtk_vbox_new (FALSE, 10);
    button = new_wide_button ("Trade", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (sideboard_trade), args);
    gtk_box_pack_start (GTK_BOX (vbox), button, TRUE, FALSE, 0);
    button = new_wide_button ("Ok", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (sideboard_dialog_ok), args);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
    button = new_wide_button ("Cancel", 6);
    gtk_signal_connect (GTK_OBJECT (button), "clicked",
			GTK_SIGNAL_FUNC (sideboard_dialog_cancel), args);
    gtk_box_pack_start (GTK_BOX (vbox), button, FALSE, FALSE, 0);
    gtk_box_pack_start (GTK_BOX (hbox), vbox, FALSE, FALSE, 0);
    create_sb_list (&args->sb_list, deck->sb_size, deck->sb_cardlist, game);
    gtk_box_pack_start (GTK_BOX (hbox), args->sb_list.scrollbox,
			FALSE, FALSE, 0);
    gtk_container_add (GTK_CONTAINER (args->dialog), hbox);
    args->changed = FALSE;
    gtk_widget_show_all (args->dialog);
    display_message (game, "+ %s is sideboarding.",
		     game->player[game->local_player]->name);
    send_inform_sideboard (game);
}

static void
create_sb_list (struct sb_list *listp, int size, int *cardlist,
		struct game *game)
{
    int i, globalid, count;
    struct cardinfo *info;
    char buf[10], *sarr[2];

    listp->scrollbox = gtk_scrolled_window_new (NULL, NULL);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (listp->scrollbox),
				    GTK_POLICY_AUTOMATIC,
				    GTK_POLICY_AUTOMATIC);
    gtk_widget_set_usize (GTK_WIDGET (listp->scrollbox), 210, 250);
    listp->clist = gtk_clist_new (2);
    gtk_container_add (GTK_CONTAINER (listp->scrollbox), listp->clist);

    gtk_clist_set_selection_mode (GTK_CLIST (listp->clist),
				  GTK_SELECTION_SINGLE);
    gtk_clist_set_column_width (GTK_CLIST (listp->clist), 0, 15);
    gtk_clist_set_column_width (GTK_CLIST (listp->clist), 1, 0);

    listp->lineinfo = malloc (sizeof (struct sblist_line) * size);
    listp->num_lines = 0;

    i = 0;
    while (i < size) {
	globalid = cardlist[i];
	count = 1;
	while (cardlist[i+count] == globalid)
	    count++;
	info = cardbase_get_card (game->cardbase, globalid);
	sprintf (buf, "%d", count);
	sarr[0] = buf;
	sarr[1] = (info == NULL ? "" : info->name);
	gtk_clist_append (GTK_CLIST (listp->clist), sarr);
	listp->lineinfo[listp->num_lines].qty = count;
	listp->lineinfo[listp->num_lines].globalid = globalid;
	listp->num_lines++;
	i += count;
    }
}

static void sideboard_trade (GtkWidget *w, struct sideboard_args *args)
{
    int idx, sb_idx, left_gid, right_gid;

    idx = gtk_clist_get_selected_row (GTK_CLIST (args->deck_list.clist));
    if (idx < 0)
	return;
    sb_idx = gtk_clist_get_selected_row (GTK_CLIST (args->sb_list.clist));
    if (sb_idx < 0)
	return;
    left_gid = args->deck_list.lineinfo[idx].globalid;
    right_gid = args->sb_list.lineinfo[sb_idx].globalid;
    decrement_line (&args->deck_list, idx);
    decrement_line (&args->sb_list, sb_idx);
    increment_globalid (&args->deck_list, right_gid,
			args->game->cardbase);
    increment_globalid (&args->sb_list, left_gid,
			args->game->cardbase);
    args->changed = TRUE;
}

static void decrement_line (struct sb_list *listp, int idx)
{
    char buf[10];
    int i;

    listp->lineinfo[idx].qty--;
    if (listp->lineinfo[idx].qty > 0) {
	sprintf (buf, "%d", listp->lineinfo[idx].qty);
	gtk_clist_set_text (GTK_CLIST (listp->clist), idx, 0, buf);
	return;
    }
    for (i = idx; i < listp->num_lines; i++)
	listp->lineinfo[i] = listp->lineinfo[i+1];
    listp->num_lines--;
    gtk_clist_remove (GTK_CLIST (listp->clist), idx);
}

static void increment_globalid (struct sb_list *listp, int globalid,
				struct cardbase *cardbase)
{
    int idx;
    char buf[10], *sarr[2];
    struct cardinfo *info;

    for (idx = 0; idx < listp->num_lines; idx++) {
	if (listp->lineinfo[idx].globalid == globalid) {
	    listp->lineinfo[idx].qty++;
	    sprintf (buf, "%d", listp->lineinfo[idx].qty);
	    gtk_clist_set_text (GTK_CLIST (listp->clist), idx, 0, buf);
	    return;
	}
    }
    /* Add a new line to the list. */
    idx = listp->num_lines;
    listp->num_lines++;
    listp->lineinfo = realloc (listp->lineinfo, sizeof (struct sblist_line) *
			       listp->num_lines);
    listp->lineinfo[idx].qty = 1;
    listp->lineinfo[idx].globalid = globalid;
    info = cardbase_get_card (cardbase, globalid);
    sprintf (buf, "%d", listp->lineinfo[idx].qty);
    sarr[0] = buf;
    sarr[1] = (info == NULL ? "" : info->name);
    gtk_clist_append (GTK_CLIST (listp->clist), sarr);
}

static void sideboard_dialog_ok (GtkWidget *w, struct sideboard_args *args)
{
    struct game *game;
    struct deck *deck;

    if (args->changed) {
	game = args->game;
	if (game->new_game_dialog == NULL) {
	    do_message_dialog ("You may not modify your deck "
			       "during a game.");
	    return;
	}
	deck = game->player[game->local_player]->deck;
	sblist_to_cardlist (&args->deck_list, deck->cardlist);
	sblist_to_cardlist (&args->sb_list, deck->sb_cardlist);
    }
    sideboard_dialog_cancel (w, args);
}

static void sblist_to_cardlist (struct sb_list *listp, int *cardlist)
{
    int count, idx, x;

    for (idx = count = 0; idx < listp->num_lines; idx++)
	for (x = 0; x < listp->lineinfo[idx].qty; x++)
	    cardlist[count++] = listp->lineinfo[idx].globalid;
}

static void sideboard_dialog_cancel (GtkWidget *w, struct sideboard_args *args)
{
    free (args->deck_list.lineinfo);
    free (args->sb_list.lineinfo);
    gtk_widget_destroy (args->dialog);
    free (args);
}

/* ----------------------------------------------------------------- */

static void do_cardinfo_ok (GtkWidget *w, GtkWidget *dialog);
static int is_ascii_file (const char *filename);

void do_find_cardinfo_dialog (struct game *game)
{
    GtkWidget *filew;
    GtkFileSelection *fsp;

    filew = gtk_file_selection_new ("Load Magic Card Database");
    gtk_object_set_user_data (GTK_OBJECT (filew), game);
    fsp = GTK_FILE_SELECTION (filew);
    gtk_signal_connect (GTK_OBJECT (fsp->ok_button), "clicked",
			GTK_SIGNAL_FUNC (do_cardinfo_ok), filew);
    gtk_signal_connect_object (GTK_OBJECT (fsp->cancel_button), "clicked",
			       GTK_SIGNAL_FUNC (gtk_widget_destroy),
			       GTK_OBJECT (filew));
    gtk_widget_show (filew);
}

static void do_cardinfo_ok (GtkWidget *w, GtkWidget *dialog)
{
    struct game *game;
    const char *filename;

    game = gtk_object_get_user_data (GTK_OBJECT (dialog));
    filename = gtk_file_selection_get_filename (GTK_FILE_SELECTION (dialog));
    if (filename == NULL)
	return;
    if (is_ascii_file (filename)) {
	game->cardbase = cardbase_new_from_ort (filename);
	if (game->cardbase == NULL) {
	    display_message (game, "%s: Failed to read oracle card database.",
			     filename);
	    return;
	}
	set_preference (game->prefs, oracle_name_key, filename);
    }
    else {
	game->cardbase = cardbase_new_from_dat (filename);
	if (game->cardbase == NULL) {
	    display_message (game, "%s: Failed to read cardinfo database.",
			     filename);
	    return;
	}
	set_preference (game->prefs, cardinfo_name_key, filename);
    }
    gtk_widget_destroy (dialog);
}

/**
 *  Return true if the first line of the given file contains all plain-text
 *  characters.
 */
static int is_ascii_file (const char *filename)
{
    FILE *fp;
    char buf[256];
    int i;

    fp = fopen (filename, "r");
    if (fp == NULL)
	return (FALSE);
    fgets (buf, sizeof (buf), fp);
    fclose (fp);
    for (i = 0; i < sizeof (buf); i++) {
	if (buf[i] == '\r')
	    return ((buf[i+1] == '\n') && (buf[i+2] == 0));
	if (buf[i] == '\n')
	    return (buf[i+1] == 0);
	if (!isascii (buf[i]))
	    return (FALSE);
    }
    return (FALSE);
}
