/***************************** LICENSE START ***********************************

 Copyright 2018 ECMWF and INPE. This software is distributed under the terms
 of the Apache License version 2.0. In applying this license, ECMWF does not
 waive the privileges and immunities granted to it by virtue of its status as
 an Intergovernmental Organization or submit itself to any jurisdiction.

 ***************************** LICENSE END *************************************/

#include "MvQEcStyleHelp.h"

#include <QtGlobal>
#include <QApplication>
#include <QComboBox>
#include <QGridLayout>
#include <QLabel>
#include <QLineEdit>
#include <QPainter>
#include <QSortFilterProxyModel>
#include <QToolButton>
#include <QTreeView>
#include <QVBoxLayout>

#include "MvQStyleDb.h"
#include "MvQEcLayerDb.h"
//#include "MvQStyleLine.h"

#include "MvMiscelaneous.h"

#include "HelpFactory.h"

//=====================================================
//
//  MvQEcStyleModel
//
//=====================================================

MvQEcStyleModel::MvQEcStyleModel(QObject *parent) :
    QAbstractItemModel(parent)
{
}

MvQEcStyleModel::~MvQEcStyleModel()
{
}

int MvQEcStyleModel::columnCount( const QModelIndex& /*parent */ ) const
{
     return 3;
}

int MvQEcStyleModel::rowCount( const QModelIndex& parent) const
{
    //Parent is the root:
    if(!parent.isValid())
    {
        return MvQStyleDb::instance()->items().count();
    }

    return 0;
}

QVariant MvQEcStyleModel::data( const QModelIndex& index, int role ) const
{
    int row=index.row();
    if(row < 0 || row >= MvQStyleDb::instance()->items().count())
        return QVariant();

    if(role == Qt::DisplayRole)
    {
        if(index.column() == 0)
            return row;
        else if(index.column() == 1)
            return MvQStyleDb::instance()->items()[row]->name();
        else if(index.column() == 2)
            return MvQStyleDb::instance()->items()[row]->description();
    }
    else if(role == Qt::UserRole)
    {
        return (isFiltered(row) == true)?"1":"0";
    }
    else if(role == SortRole)
    {
        if(index.column() == 0)
            return row;
        else if(index.column() == 1)
            return MvQStyleDb::instance()->items()[row]->name();
    }

    return QVariant();
}

QVariant MvQEcStyleModel::headerData( const int section, const Qt::Orientation orient , const int role ) const
{
    if ( orient != Qt::Horizontal || (role != Qt::DisplayRole &&  role != Qt::ToolTipRole))
              return QAbstractItemModel::headerData( section, orient, role );

    if(role == Qt::DisplayRole)
    {
        switch ( section )
        {
        case 0: return "Preview";
        case 1: return "Name";
        case 2: return "Description";
        default: return QVariant();
        }
    }

    return QVariant();
}

QModelIndex MvQEcStyleModel::index( int row, int column, const QModelIndex & parent ) const
{
    if(row < 0 || column < 0)
    {
        return QModelIndex();
    }

    //When parent is the root this index refers to a node or server
    if(!parent.isValid())
    {
        return createIndex(row,column);
    }

    return QModelIndex();

}

QModelIndex MvQEcStyleModel::parent(const QModelIndex &child) const
{
    return QModelIndex();
}

void MvQEcStyleModel::setIndexFilter(QList<int> lst)
{
    indexFilter_=lst;
}

void MvQEcStyleModel::clearFilter()
{
    indexFilter_.clear();
}

bool MvQEcStyleModel::isFiltered(int row) const
{
    if(!indexFilter_.isEmpty())
    {
        if(!indexFilter_.contains(row))
            return false;
    }

    return true;
}

//==============================
//
// MvQEcStyleDelegate
//
//==============================

MvQEcStyleDelegate::MvQEcStyleDelegate(QWidget *parent) :
    QStyledItemDelegate(parent),
    borderCol_(QColor(210,210,210))
{
}

void MvQEcStyleDelegate::paint(QPainter *painter,const QStyleOptionViewItem &option,
                   const QModelIndex& index) const
{
    painter->save();

    if(index.column() == 0)
    {
        //Background
        QStyleOptionViewItem vopt(option);
        initStyleOption(&vopt, index);

        const QStyle *style = vopt.widget ? vopt.widget->style() : QApplication::style();
        const QWidget* widget = vopt.widget;

        //painter->fillRect(option.rect,QColor(238,238,238));

        //We render everything with the default method
        style->drawControl(QStyle::CE_ItemViewItem, &vopt, painter, widget);

        int row=index.data(Qt::DisplayRole).toInt();
        if(row < 0)
            return;

        QRect rect=option.rect.adjusted(2,2,-4,-2);

        MvQStyleDbItem* item=MvQStyleDb::instance()->items()[row];
        Q_ASSERT(item);
        if(item)
            item->paint(painter,rect);
    }
    else
    {
        QStyledItemDelegate::paint(painter,option,index);
    }

    //Render the horizontal border for rows. We only render the top border line.
    //With this technique we miss the bottom border line of the last row!!!
    //QRect fullRect=QRect(0,option.rect.y(),painter->device()->width(),option.rect.height());
    QRect bgRect=option.rect;
    painter->setPen(borderCol_);
    painter->drawLine(bgRect.topLeft(),bgRect.topRight());

    painter->restore();
}

QSize MvQEcStyleDelegate::sizeHint(const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    //s.height()+2
    QSize s=QStyledItemDelegate::sizeHint(option,index);
    if(index.column() == 0)
        return QSize(180,50);

    return QSize(s.width(),50);
}



//=====================================================
//
// MvQEcStyleSelectionWidget
//
//=====================================================

MvQEcStyleSelectionWidget::MvQEcStyleSelectionWidget(QWidget* parent) :
    QWidget(parent)
    //ignoreFilterChanged_(false)
{
    QVBoxLayout* vb=new QVBoxLayout(this);

    tree_=new QTreeView(this);
    tree_->setRootIsDecorated(false);
    tree_->setUniformRowHeights(true);
    tree_->setMinimumHeight(200);
    vb->addWidget(tree_,1);

    model_=new MvQEcStyleModel(this);
    tree_->setItemDelegate(new MvQEcStyleDelegate(this));

    sortModel_=new QSortFilterProxyModel(this);
    sortModel_->setFilterRegExp("1");
    sortModel_->setFilterRole(Qt::UserRole);
    sortModel_->setSortRole(MvQEcStyleModel::SortRole);
    sortModel_->setSourceModel(model_);

    tree_->setSortingEnabled(true);
    tree_->sortByColumn(1,Qt::AscendingOrder);
    tree_->setModel(sortModel_);

    connect(tree_,SIGNAL(clicked(QModelIndex)),
            this,SLOT(slotItemSelected(QModelIndex)));

    //Init the button states!!
    checkButtonState();
}


void MvQEcStyleSelectionWidget::slotItemSelected(const QModelIndex& idx)
{
    QModelIndex idxSrc=sortModel_->mapToSource(idx);
    if(idxSrc.isValid())
    {
        emit itemSelected(idxSrc.row());
    }
}


void MvQEcStyleSelectionWidget::slotClearFilter()
{
    //ignoreFilterChanged_=true;
    //nameLe_->clear();
    //keywordCb_->setCurrentIndex(0);
    //colourCb_->setCurrentIndex(0);
    //paramCb_->setCurrentIndex(0);
    //ignoreFilterChanged_=false;

    model_->clearFilter();
    sortModel_->invalidate();
    //checkButtonState();
}

void MvQEcStyleSelectionWidget::checkButtonState()
{
#if 0
    nameResetTb_->setEnabled(!nameLe_->text().isEmpty());
    keywordResetTb_->setEnabled(keywordCb_->currentText() != "ANY");
    colourResetTb_->setEnabled(colourCb_->currentText() != "ANY");
    paramResetTb_->setEnabled(paramCb_->currentText() != "ANY");

    resetTb_->setEnabled(keywordResetTb_->isEnabled() ||
                         nameResetTb_->isEnabled() || colourResetTb_->isEnabled() ||
                         paramResetTb_->isEnabled());
#endif

}

void MvQEcStyleSelectionWidget::setCurrent(const std::string& name,const std::string& layerName)
{
    if(layerName_ != layerName)
        layerName_=layerName;
    else
    {
        QModelIndex idxSrc=sortModel_->mapToSource(tree_->currentIndex());
        if(idxSrc.isValid())
        {
            int row=idxSrc.row();
            if(row >=0 && row < MvQStyleDb::instance()->items().size() &&
               MvQStyleDb::instance()->items()[row]->name().toStdString() == name)
                return;
        }
    }

    slotClearFilter();
    int row=MvQStyleDb::instance()->indexOf(name);

    if(MvQEcLayerDbItem *layerItem=MvQEcLayerDb::instance()->find(layerName_))
    {
        model_->setIndexFilter(layerItem->styleIds());
        sortModel_->invalidate();
    }

    if(row >= 0)
    {
        QModelIndex idxSrc=model_->index(row,0);
        if(idxSrc.isValid())
        {
            tree_->setCurrentIndex(sortModel_->mapFromSource(idxSrc));
        }
    }
}

//==================================================
//
//  MvQEcStyleHelp
//
//==================================================

MvQEcStyleHelp::MvQEcStyleHelp(RequestPanel& owner,const Parameter& param) :
           MvQRequestPanelHelp(owner,param)
{
    selector_=new MvQEcStyleSelectionWidget(parentWidget_);

    connect(selector_,SIGNAL(itemSelected(int)),
            this,SLOT(slotSelected(int)));

}

void MvQEcStyleHelp::slotSelected(int idx)
{
    if(idx >=0  && idx < MvQStyleDb::instance()->items().count())
    {
        if(MvQStyleDb::instance()->items()[idx]->name() != oriName_)
        {
            std::vector<string> vs;
            vs.push_back(metview::toString<int>(idx));
            emit edited(vs);
            oriName_.clear();
        }
    }
}

void MvQEcStyleHelp::refresh(const vector<string>& values)
{
    if(values.size() == 0)
        return;

    oriName_=QString::fromStdString(values[0]);

    std::string layerName;
    if(values.size() >=  2)
        layerName=values[1];

    selector_->setCurrent(values[0],layerName);
}

static HelpMaker<MvQEcStyleHelp> maker2("help_ecstyle");
