%-----------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 expandtab
%-----------------------------------------------------------------------------%
% Copyright (C) 2005-2006, 2012 The University of Melbourne.
% This file may only be copied under the terms of the GNU General
% Public License - see the file COPYING in the Mercury distribution.
%-----------------------------------------------------------------------------%
%
% Mercury dice tool.
%
% Main author: zs.
%
%-----------------------------------------------------------------------------%

:- module mdice.

:- interface.

:- import_module io.

:- pred main(io::di, io::uo) is det.

%-----------------------------------------------------------------------------%
%-----------------------------------------------------------------------------%

:- implementation.

:- import_module mdbcomp.
:- import_module mdbcomp.slice_and_dice.
:- import_module mdbcomp.shared_utilities.

:- import_module getopt.
:- import_module maybe.
:- import_module list.
:- import_module string.

%-----------------------------------------------------------------------------%

main(!IO) :-
    unlimit_stack(!IO),
    io.command_line_arguments(Args0, !IO),
    OptionOps = option_ops_multi(short_option, long_option, option_default),
    getopt.process_options(OptionOps, Args0, Args, GetoptResult),
    (
        GetoptResult = ok(OptionTable),
        (
            Args = [],
            usage(!IO)
        ;
            Args = [_],
            usage(!IO)
        ;
            Args = [PassFileName, FailFileName],
            lookup_string_option(OptionTable, sort, SortStr),
            lookup_string_option(OptionTable, modulename, Module),
            lookup_int_option(OptionTable, max_row, MaxRow),
            lookup_int_option(OptionTable, max_pred_column, MaxPredColumn),
            lookup_int_option(OptionTable, max_path_column, MaxPathColumn),
            lookup_int_option(OptionTable, max_file_column, MaxFileColumn),
            ( MaxPredColumn = 0 ->
                MaybeMaxPredColumn = no
            ;
                MaybeMaxPredColumn = yes(MaxPredColumn)
            ),
            ( MaxPathColumn = 0 ->
                MaybeMaxPathColumn = no
            ;
                MaybeMaxPathColumn = yes(MaxPathColumn)
            ),
            ( MaxFileColumn = 0 ->
                MaybeMaxFileColumn = no
            ;
                MaybeMaxFileColumn = yes(MaxFileColumn)
            ),
            read_dice_to_string(PassFileName, FailFileName, SortStr, MaxRow,
                MaybeMaxPredColumn, MaybeMaxPathColumn, MaybeMaxFileColumn,
                Module, DiceStr, Problem, !IO),
            ( Problem = "" ->
                io.write_string(DiceStr, !IO)
            ;
                io.write_string(Problem, !IO),
                io.nl(!IO),
                io.set_exit_status(1, !IO)
            )
        ;
            Args = [_, _, _ | _],
            usage(!IO)
        )
    ;
        GetoptResult = error(GetoptErrorMsg),
        io.write_string(GetoptErrorMsg, !IO),
        io.nl(!IO)
    ).

:- pred usage(io::di, io::uo) is det.

usage(!IO) :-
    io.write_string(
        "Usage: mdice [-s sortspec] [-m module] [-l N] [-n N] [-p N] [-f N] "
        ++ "passfile failfile\n",
        !IO).

%-----------------------------------------------------------------------------%

:- type option
    --->    sort
    ;       max_row
    ;       max_pred_column
    ;       max_path_column
    ;       max_file_column
    ;       modulename.

:- type option_table == option_table(option).

:- pred short_option(character::in, option::out) is semidet.
:- pred long_option(string::in, option::out) is semidet.
:- pred option_default(option::out, option_data::out) is multi.

option_default(sort,            string("S")).
option_default(max_row,         int(100)).
option_default(max_pred_column, int(35)).
option_default(max_path_column, int(12)).
option_default(max_file_column, int(20)).
option_default(modulename,      string("")).

short_option('s',               sort).
short_option('l',               max_row).
short_option('n',               max_pred_column).
short_option('p',               max_path_column).
short_option('f',               max_file_column).
short_option('m',               modulename).

long_option("sort",             sort).
long_option("limit",            max_row).
long_option("max-name-column",  max_pred_column).
long_option("max-path-column",  max_path_column).
long_option("max-file-column",  max_file_column).
long_option("module",           modulename).

%-----------------------------------------------------------------------------%
