%-----------------------------------------------------------------------------%
% vim: ft=mercury ts=4 sw=4 et
%-----------------------------------------------------------------------------%
% Copyright (C) 1994-1996, 1998-2010 The University of Melbourne.
% This file may only be copied under the terms of the GNU General
% Public License - see the file COPYING in the Mercury distribution.
%-----------------------------------------------------------------------------%
%
% File: make_tags.m.
% Main author: fjh.
%
% This module is where we determine the representation for
% discriminated union types.  Each d.u. type is represented as
% a word.  In the case of functors with arguments, we allocate
% the arguments on the heap, and the word contains a pointer to
% those arguments.
%
% For types which are just enumerations (all the constructors
% are constants), we just assign a different value for each
% constructor.
%
% For types which have only one functor of arity one, there is
% no need to store the functor, and we just store the argument
% value directly; construction and deconstruction unifications
% on these type are no-ops.
%
% For other types, we use a couple of bits of the word as a
% tag.  We split the constructors into constants and functors,
% and assign tag zero to the constants (if any).  If there is
% more than one constant, we distinguish between the different
% constants by the value of the rest of the word.  Then we
% assign one tag bit each to the first few functors.  The
% remaining functors all get the last remaining two-bit tag.
% These functors are distinguished by a secondary tag which is
% the first word of the argument vector for those functors.
%
% If there are no tag bits available, then we try using reserved
% addresses (e.g. NULL, (void *)1, (void *)2, etc.) instead.
% We split the constructors into constants and functors,
% and assign numerical reserved addresses to the first constants,
% up to the limit set by --num-reserved-addresses.
% After that, for the MLDS back-end, we assign symbolic reserved
% addresses to the remaining constants, up to the limit set by
% --num-reserved-objects; these symbolic reserved addresses
% are the addresses of global variables that we generate specially
% for this purpose.  Finally, the functors and any remaining
% constants are distinguished by a secondary tag, if there are more
% than one of them.
%
% If there is a `pragma reserve_tag' declaration for the type,
% or if the `--reserve-tag' option is set,
% then we reserve the first primary tag (for representing
% unbound variables).  This is used by HAL, for Herbrand constraints
% (i.e. Prolog-style logic variables).
% This also disables enumerations and no_tag types.
%
%-----------------------------------------------------------------------------%

:- module hlds.make_tags.
:- interface.

:- import_module hlds.hlds_data.
:- import_module libs.globals.
:- import_module parse_tree.prog_data.

:- import_module list.
:- import_module maybe.

    % assign_constructor_tags(Constructors, MaybeUserEq, TypeCtor,
    %   ReservedTagPragma, Globals, TagValues, IsEnum):
    %
    % Assign a constructor tag to each constructor for a discriminated union
    % type, and determine whether (a) the type representation uses reserved
    % addresses, and (b) the type is an enumeration or dummy type.
    % (`Globals' is passed because exact way in which this is done is
    % dependent on a compilation option.)
    %
:- pred assign_constructor_tags(list(constructor)::in,
    maybe(unify_compare)::in, type_ctor::in, uses_reserved_tag::in,
    globals::in, cons_tag_values::out,
    uses_reserved_address::out, du_type_kind::out) is det.

%-----------------------------------------------------------------------------%
%-----------------------------------------------------------------------------%

:- implementation.

:- import_module libs.globals.
:- import_module libs.options.
:- import_module parse_tree.prog_type.
:- import_module parse_tree.prog_util.

:- import_module bool.
:- import_module int.
:- import_module map.
:- import_module require.
:- import_module svmap.

%-----------------------------------------------------------------------------%

assign_constructor_tags(Ctors, UserEqCmp, TypeCtor, ReservedTagPragma, Globals,
        !:CtorTags, ReservedAddr, DuTypeKind) :-
    % Work out how many tag bits and reserved addresses we've got to play with.
    globals.lookup_int_option(Globals, num_tag_bits, NumTagBits),
    globals.lookup_int_option(Globals, num_reserved_addresses,
        NumReservedAddresses),
    globals.lookup_int_option(Globals, num_reserved_objects,
        NumReservedObjects),
    globals.lookup_bool_option(Globals, highlevel_code, HighLevelCode),

    % Determine if we need to reserve a tag for use by HAL's Herbrand
    % constraint solver. (This also disables enumerations and no_tag types.)
    (
        ReservedTagPragma = uses_reserved_tag,
        InitTag = 1
    ;
        ReservedTagPragma = does_not_use_reserved_tag,
        InitTag = 0
    ),

    % Now assign them.
    map.init(!:CtorTags),
    (
        % Try representing the type as an enumeration: all the constructors
        % must be constant, and we must be allowed to make unboxed enums.
        globals.lookup_bool_option(Globals, unboxed_enums, yes),
        ctors_are_all_constants(Ctors),
        ReservedTagPragma = does_not_use_reserved_tag
    ->
        ( Ctors = [_] ->
            DuTypeKind = du_type_kind_direct_dummy
        ;
            DuTypeKind = du_type_kind_mercury_enum
        ),
        assign_enum_constants(TypeCtor, Ctors, InitTag, !CtorTags),
        ReservedAddr = does_not_use_reserved_address
    ;
        (
            % Try representing it as a no-tag type.
            type_ctor_should_be_notag(Globals, TypeCtor, ReservedTagPragma,
                Ctors, UserEqCmp, SingleFunctorName, SingleArgType,
                MaybeSingleArgName)
        ->
            SingleConsId = cons(SingleFunctorName, 1, TypeCtor),
            svmap.det_insert(SingleConsId, no_tag, !CtorTags),
            % XXX What if SingleArgType uses reserved addresses?
            ReservedAddr = does_not_use_reserved_address,
            DuTypeKind = du_type_kind_notag(SingleFunctorName, SingleArgType,
                MaybeSingleArgName)
        ;
            DuTypeKind = du_type_kind_general,
            ( NumTagBits = 0 ->
                (
                    ReservedTagPragma = uses_reserved_tag,
                    % XXX Need to fix this.
                    % This occurs for the .NET and Java backends.
                    sorry("make_tags", "--reserve-tag with num_tag_bits = 0")
                ;
                    ReservedTagPragma = does_not_use_reserved_tag
                ),
                % Assign reserved addresses to the constants, if possible.
                separate_out_constants(Ctors, Constants, Functors),
                assign_reserved_numeric_addresses(TypeCtor, Constants,
                    LeftOverConstants0, !CtorTags, 0, NumReservedAddresses,
                    does_not_use_reserved_address, ReservedAddr1),
                (
                    HighLevelCode = yes,
                    assign_reserved_symbolic_addresses(TypeCtor,
                        LeftOverConstants0, LeftOverConstants,
                        !CtorTags, 0, NumReservedObjects,
                        ReservedAddr1, ReservedAddr)
                ;
                    HighLevelCode = no,
                    % Reserved symbolic addresses are not supported for the
                    % LLDS back-end.
                    LeftOverConstants = LeftOverConstants0,
                    ReservedAddr = ReservedAddr1
                ),
                % Assign shared_with_reserved_address(...) representations
                % for the remaining constructors.
                RemainingCtors = LeftOverConstants ++ Functors,
                GetRA = (func(reserved_address_tag(RA)) = RA is semidet),
                ReservedAddresses = list.filter_map(GetRA,
                    map.values(!.CtorTags)),
                assign_unshared_tags(TypeCtor, RemainingCtors, 0, 0,
                    ReservedAddresses, !CtorTags)
            ;
                MaxTag = max_num_tags(NumTagBits) - 1,
                separate_out_constants(Ctors, Constants, Functors),
                assign_constant_tags(TypeCtor, Constants, !CtorTags,
                    InitTag, NextTag),
                assign_unshared_tags(TypeCtor, Functors, NextTag, MaxTag,
                    [], !CtorTags),
                ReservedAddr = does_not_use_reserved_address
            )
        )
    ).

:- pred assign_enum_constants(type_ctor::in, list(constructor)::in, int::in,
    cons_tag_values::in, cons_tag_values::out) is det.

assign_enum_constants(_, [], _, !CtorTags).
assign_enum_constants(TypeCtor, [Ctor | Ctors], Val, !CtorTags) :-
    Ctor = ctor(_ExistQVars, _Constraints, Name, Args, _Ctxt),
    ConsId = cons(Name, list.length(Args), TypeCtor),
    Tag = int_tag(Val),
    % We call set instead of det_insert because we don't want types
    % that erroneously contain more than one copy of a cons_id to crash
    % the compiler.
    svmap.set(ConsId, Tag, !CtorTags),
    assign_enum_constants(TypeCtor, Ctors, Val + 1, !CtorTags).

    % Assign the representations null_pointer, small_pointer(1),
    % small_pointer(2), ..., small_pointer(N) to the constructors,
    % until N >= NumReservedAddresses.
    %
:- pred assign_reserved_numeric_addresses(type_ctor::in,
    list(constructor)::in, list(constructor)::out,
    cons_tag_values::in, cons_tag_values::out, int::in, int::in,
    uses_reserved_address::in, uses_reserved_address::out) is det.

assign_reserved_numeric_addresses(_, [], [], !CtorTags, _, _, !ReservedAddr).
assign_reserved_numeric_addresses(TypeCtor, [Ctor | Ctors], LeftOverConstants,
        !CtorTags, Address, NumReservedAddresses, !ReservedAddr) :-
    ( Address >= NumReservedAddresses ->
        LeftOverConstants = [Ctor | Ctors]
    ;
        Ctor = ctor(_ExistQVars, _Constraints, Name, Args, _Ctxt),
        ConsId = cons(Name, list.length(Args), TypeCtor),
        ( Address = 0 ->
            Tag = reserved_address_tag(null_pointer)
        ;
            Tag = reserved_address_tag(small_pointer(Address))
        ),
        % We call set instead of det_insert because we don't want types
        % that erroneously contain more than one copy of a cons_id to crash
        % the compiler.
        svmap.set(ConsId, Tag, !CtorTags),
        !:ReservedAddr = uses_reserved_address,
        assign_reserved_numeric_addresses(TypeCtor, Ctors, LeftOverConstants,
            !CtorTags, Address + 1, NumReservedAddresses, !ReservedAddr)
    ).

    % Assign reserved_object(CtorName, CtorArity) representations
    % to the specified constructors.
    %
:- pred assign_reserved_symbolic_addresses(type_ctor::in,
    list(constructor)::in, list(constructor)::out,
    cons_tag_values::in, cons_tag_values::out, int::in, int::in,
    uses_reserved_address::in, uses_reserved_address::out) is det.

assign_reserved_symbolic_addresses(_, [], [], !CtorTags, _, _, !ReservedAddr).
assign_reserved_symbolic_addresses(TypeCtor, [Ctor | Ctors], LeftOverConstants,
        !CtorTags, Num, Max, !ReservedAddr) :-
    ( Num >= Max ->
        LeftOverConstants = [Ctor | Ctors]
    ;
        Ctor = ctor(_ExistQVars, _Constraints, Name, Args, _Ctxt),
        Arity = list.length(Args),
        Tag = reserved_address_tag(reserved_object(TypeCtor, Name, Arity)),
        ConsId = cons(Name, list.length(Args), TypeCtor),
        % We call set instead of det_insert because we don't want types
        % that erroneously contain more than one copy of a cons_id to crash
        % the compiler.
        svmap.set(ConsId, Tag, !CtorTags),
        !:ReservedAddr = uses_reserved_address,
        assign_reserved_symbolic_addresses(TypeCtor, Ctors, LeftOverConstants,
            !CtorTags, Num + 1, Max, !ReservedAddr)
    ).

:- pred assign_constant_tags(type_ctor::in, list(constructor)::in,
    cons_tag_values::in, cons_tag_values::out, int::in, int::out) is det.

    % If there's no constants, don't do anything. Otherwise, allocate the
    % first tag for the constants, and give them all shared local tags
    % with that tag as the primary tag, and different secondary tags
    % starting from zero.
    %
    % Note that if there's a single constant, we still give it a
    % shared_local_tag rather than a unshared_tag. That's because
    % deconstruction of the shared_local_tag is more efficient.
    %
assign_constant_tags(TypeCtor, Constants, !CtorTags, InitTag, NextTag) :-
    (
        Constants = [],
        NextTag = InitTag
    ;
        Constants = [_ | _],
        NextTag = InitTag + 1,
        assign_shared_local_tags(TypeCtor, Constants, InitTag, 0, !CtorTags)
    ).

:- pred assign_unshared_tags(type_ctor::in, list(constructor)::in,
    int::in, int::in, list(reserved_address)::in,
    cons_tag_values::in, cons_tag_values::out) is det.

assign_unshared_tags(_, [], _, _, _, !CtorTags).
assign_unshared_tags(TypeCtor, [Ctor | Ctors], Val, MaxTag, ReservedAddresses,
        !CtorTags) :-
    Ctor = ctor(_ExistQVars, _Constraints, Name, Args, _Ctxt),
    ConsId = cons(Name, list.length(Args), TypeCtor),
    % If there's only one functor, give it the "single_functor" (untagged)
    % representation, rather than giving it unshared_tag(0).
    (
        Val = 0,
        Ctors = []
    ->
        Tag = maybe_add_reserved_addresses(ReservedAddresses,
            single_functor_tag),
        % We call set instead of det_insert because we don't want types
        % that erroneously contain more than one copy of a cons_id to crash
        % the compiler.
        svmap.set(ConsId, Tag, !CtorTags)
    ;
        % If we're about to run out of unshared tags, start assigning
        % shared remote tags instead.
        Val = MaxTag,
        Ctors = [_ | _]
    ->
        assign_shared_remote_tags(TypeCtor, [Ctor | Ctors], MaxTag, 0,
            ReservedAddresses, !CtorTags)
    ;
        Tag = maybe_add_reserved_addresses(ReservedAddresses,
            unshared_tag(Val)),
        % We call set instead of det_insert because we don't want types
        % that erroneously contain more than one copy of a cons_id to crash
        % the compiler.
        svmap.set(ConsId, Tag, !CtorTags),
        assign_unshared_tags(TypeCtor, Ctors, Val + 1, MaxTag,
            ReservedAddresses, !CtorTags)
    ).

:- pred assign_shared_remote_tags(type_ctor::in, list(constructor)::in,
    int::in, int::in, list(reserved_address)::in,
    cons_tag_values::in, cons_tag_values::out) is det.

assign_shared_remote_tags(_, [], _, _, _, !CtorTags).
assign_shared_remote_tags(TypeCtor, [Ctor | Ctors], PrimaryVal, SecondaryVal,
        ReservedAddresses, !CtorTags) :-
    Ctor = ctor(_ExistQVars, _Constraints, Name, Args, _Ctxt),
    ConsId = cons(Name, list.length(Args), TypeCtor),
    Tag = maybe_add_reserved_addresses(ReservedAddresses,
        shared_remote_tag(PrimaryVal, SecondaryVal)),
    % We call set instead of det_insert because we don't want types
    % that erroneously contain more than one copy of a cons_id to crash
    % the compiler.
    svmap.set(ConsId, Tag, !CtorTags),
    SecondaryVal1 = SecondaryVal + 1,
    assign_shared_remote_tags(TypeCtor, Ctors, PrimaryVal, SecondaryVal1,
        ReservedAddresses, !CtorTags).

:- pred assign_shared_local_tags(type_ctor::in, list(constructor)::in,
    int::in, int::in, cons_tag_values::in, cons_tag_values::out) is det.

assign_shared_local_tags(_, [], _, _, !CtorTags).
assign_shared_local_tags(TypeCtor, [Ctor | Ctors], PrimaryVal, SecondaryVal,
        !CtorTags) :-
    Ctor = ctor(_ExistQVars, _Constraints, Name, Args, _Ctxt),
    ConsId = cons(Name, list.length(Args), TypeCtor),
    Tag = shared_local_tag(PrimaryVal, SecondaryVal),
    % We call set instead of det_insert because we don't want types
    % that erroneously contain more than one copy of a cons_id to crash
    % the compiler.
    svmap.set(ConsId, Tag, !CtorTags),
    assign_shared_local_tags(TypeCtor, Ctors, PrimaryVal, SecondaryVal + 1,
        !CtorTags).

:- func maybe_add_reserved_addresses(list(reserved_address), cons_tag) =
    cons_tag.

maybe_add_reserved_addresses(ReservedAddresses, Tag0) = Tag :-
    (
        ReservedAddresses = [],
        Tag = Tag0
    ;
        ReservedAddresses = [_ | _],
        Tag = shared_with_reserved_addresses_tag(ReservedAddresses, Tag0)
    ).

%-----------------------------------------------------------------------------%

:- func max_num_tags(int) = int.

max_num_tags(NumTagBits) = MaxTags :-
    int.pow(2, NumTagBits, MaxTags).

%-----------------------------------------------------------------------------%

:- pred ctors_are_all_constants(list(constructor)::in) is semidet.

ctors_are_all_constants([]).
ctors_are_all_constants([Ctor | Rest]) :-
    Ctor = ctor(_ExistQVars, _Constraints, _Name, Args, _Ctxt),
    Args = [],
    ctors_are_all_constants(Rest).

%-----------------------------------------------------------------------------%

:- pred separate_out_constants(list(constructor)::in,
    list(constructor)::out, list(constructor)::out) is det.

separate_out_constants([], [], []).
separate_out_constants([Ctor | Ctors], Constants, Functors) :-
    separate_out_constants(Ctors, Constants0, Functors0),
    Args = Ctor ^ cons_args,
    (
        Args = [],
        Constants = [Ctor | Constants0],
        Functors = Functors0
    ;
        Args = [_ | _],
        Constants = Constants0,
        Functors = [Ctor | Functors0]
    ).

%-----------------------------------------------------------------------------%
%-----------------------------------------------------------------------------%
