<?php
/**
 * Show statistics from the cache.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 * http://www.gnu.org/copyleft/gpl.html
 *
 * @file
 * @ingroup Maintenance
 */

require_once __DIR__ . '/Maintenance.php';

/**
 * Maintenance script that shows statistics from the cache.
 *
 * @ingroup Maintenance
 */
class ShowCacheStats extends Maintenance {

	public function __construct() {
		$this->mDescription = "Show statistics from the cache";
		parent::__construct();
	}

	public function getDbType() {
		return Maintenance::DB_NONE;
	}

	public function execute() {
		global $wgMemc;

		// Can't do stats if
		if ( get_class( $wgMemc ) == 'EmptyBagOStuff' ) {
			$this->error( "You are running EmptyBagOStuff, I can not provide any statistics.", true );
		}

		$this->output( "\nParser cache\n" );
		$hits = intval( $wgMemc->get( wfMemcKey( 'stats', 'pcache_hit' ) ) );
		$expired = intval( $wgMemc->get( wfMemcKey( 'stats', 'pcache_miss_expired' ) ) );
		$absent = intval( $wgMemc->get( wfMemcKey( 'stats', 'pcache_miss_absent' ) ) );
		$stub = intval( $wgMemc->get( wfMemcKey( 'stats', 'pcache_miss_stub' ) ) );
		$total = $hits + $expired + $absent + $stub;
		if ( $total ) {
			$this->output( sprintf( "hits:              %-10d %6.2f%%\n", $hits, $hits / $total * 100 ) );
			$this->output( sprintf( "expired:           %-10d %6.2f%%\n", $expired, $expired / $total * 100 ) );
			$this->output( sprintf( "absent:            %-10d %6.2f%%\n", $absent, $absent / $total * 100 ) );
			$this->output( sprintf( "stub threshold:    %-10d %6.2f%%\n", $stub, $stub / $total * 100 ) );
			$this->output( sprintf( "total:             %-10d %6.2f%%\n", $total, 100 ) );
		} else {
			$this->output( "no statistics available\n" );
		}

		$this->output( "\nImage cache\n" );
		$hits = intval( $wgMemc->get( wfMemcKey( 'stats', 'image_cache_hit' ) ) );
		$misses = intval( $wgMemc->get( wfMemcKey( 'stats', 'image_cache_miss' ) ) );
		$updates = intval( $wgMemc->get( wfMemcKey( 'stats', 'image_cache_update' ) ) );
		$total = $hits + $misses;
		if ( $total ) {
			$this->output( sprintf( "hits:              %-10d %6.2f%%\n", $hits, $hits / $total * 100 ) );
			$this->output( sprintf( "misses:            %-10d %6.2f%%\n", $misses, $misses / $total * 100 ) );
			$this->output( sprintf( "updates:           %-10d\n", $updates ) );
		} else {
			$this->output( "no statistics available\n" );
		}

		$this->output( "\nDiff cache\n" );
		$hits = intval( $wgMemc->get( wfMemcKey( 'stats', 'diff_cache_hit' ) ) );
		$misses = intval( $wgMemc->get( wfMemcKey( 'stats', 'diff_cache_miss' ) ) );
		$uncacheable = intval( $wgMemc->get( wfMemcKey( 'stats', 'diff_uncacheable' ) ) );
		$total = $hits + $misses + $uncacheable;
		if ( $total ) {
			$this->output( sprintf( "hits:              %-10d %6.2f%%\n", $hits, $hits / $total * 100 ) );
			$this->output( sprintf( "misses:            %-10d %6.2f%%\n", $misses, $misses / $total * 100 ) );
			$this->output( sprintf( "uncacheable:       %-10d %6.2f%%\n", $uncacheable, $uncacheable / $total * 100 ) );
		} else {
			$this->output( "no statistics available\n" );
		}
	}
}

$maintClass = "ShowCacheStats";
require_once RUN_MAINTENANCE_IF_MAIN;
