// license:BSD-3-Clause
// copyright-holders: David Haywood, ElSemi

/* IGS 027 Encryptions

  the IGS027 chips are typically used with encrypted ROMs.

  Games with an External ARM Rom usually have that ROM encrypted
  otherwise (in the case of PGM) the 68k ROM ends up encrypted
  instead.  The encryption is controlled by a set of registers
  mapped to ARM space where the table and some additional values
  are uploaded.  We don't currently emulate this in realtime
  even for cases where the internal ROM is dumped.

*/



#include "emu.h"
#include "pgmcrypt.h"

// a common set of these are used, selectable in the real scheme by an uploaded bitfield
// some may be duplicates simply due to differing ROM sizes

#define IGS27_CRYPT1 \
	if ((i & 0x040480) != 0x000080) x ^= 0x0001;
#define IGS27_CRYPT1_ALT \
	if ((i & 0x040080) != 0x000080) x ^= 0x0001;
#define IGS27_CRYPT1_ALT2 \
	if ((i & 0x0480) != 0x0080) x ^= 0x0001;
#define IGS27_CRYPT2 \
	if ((i & 0x104008) == 0x104008) x ^= 0x0002;
#define IGS27_CRYPT2_ALT \
	if ((i & 0x004008) == 0x004008) x ^= 0x0002;
#define IGS27_CRYPT2_ALT2 \
	if((i & 0x004008) == 0x004008 && (i & 0x180000) != 0x000000) x ^= 0x0002;
#define IGS27_CRYPT2_ALT3 \
	if((i & 0x84008) == 0x84008) x ^= 0x0002;
#define IGS27_CRYPT3 \
	if ((i & 0x080030) == 0x080010) x ^= 0x0004;
#define IGS27_CRYPT3_ALT \
	if((i & 0x000030) == 0x000010 && (i & 0x180000) != 0x080000) x ^= 0x0004;
#define IGS27_CRYPT3_ALT2 \
	if ((i & 0x000030) == 0x000010) x ^= 0x0004;
// ket - due to address starting at 0 and not 100000/2!
#define IGS27_CRYPT3_ALT3 \
	if ((i & 0x080030) == 0x000010) x ^= 0x0004;
#define IGS27_CRYPT4 \
	if ((i & 0x000242) != 0x000042) x ^= 0x0008;
#define IGS27_CRYPT4_ALT \
	if ((i & 0x000042) != 0x000042) x ^= 0x0008;
#define IGS27_CRYPT5 \
	if ((i & 0x008100) == 0x008000) x ^= 0x0010;
#define IGS27_CRYPT5_ALT \
	if ((i & 0x48100) == 0x48000) x ^= 0x0010;
#define IGS27_CRYPT6 \
	if ((i & 0x002004) != 0x000004) x ^= 0x0020;
#define IGS27_CRYPT6_ALT \
	if ((i & 0x022004) != 0x000004) x ^= 0x0020;
#define IGS27_CRYPT7 \
	if ((i & 0x011800) != 0x010000) x ^= 0x0040;
#define IGS27_CRYPT7_ALT \
	if ((i & 0x01800) != 0x00000) x ^= 0x0040;
#define IGS27_CRYPT8 \
	if ((i & 0x004820) == 0x004820) x ^= 0x0080;
#define IGS27_CRYPT8_ALT \
	if ((i & 0x000820) == 0x000820) x ^= 0x0080;

static const uint8_t kov_tab[256] = {
	0x17, 0x1c, 0xe3, 0x02, 0x62, 0x59, 0x97, 0x4a, 0x67, 0x4d, 0x1f, 0x11, 0x76, 0x64, 0xc1, 0xe1,
	0xd2, 0x41, 0x9f, 0xfd, 0xfa, 0x04, 0xfe, 0xab, 0x89, 0xeb, 0xc0, 0xf5, 0xac, 0x2b, 0x64, 0x22,
	0x90, 0x7d, 0x88, 0xc5, 0x8c, 0xe0, 0xd9, 0x70, 0x3c, 0xf4, 0x7d, 0x31, 0x1c, 0xca, 0xe2, 0xf1,
	0x31, 0x82, 0x86, 0xb1, 0x55, 0x95, 0x77, 0x01, 0x77, 0x3b, 0xab, 0xe6, 0x88, 0xef, 0x77, 0x11,
	0x56, 0x01, 0xac, 0x55, 0xf7, 0x6d, 0x9b, 0x6d, 0x92, 0x14, 0x23, 0xae, 0x4b, 0x80, 0xae, 0x6a,
	0x43, 0xcc, 0x35, 0xfe, 0xa1, 0x0d, 0xb3, 0x21, 0x4e, 0x4c, 0x99, 0x80, 0xc2, 0x3d, 0xce, 0x46,
	0x9b, 0x5d, 0x68, 0x75, 0xfe, 0x1e, 0x25, 0x41, 0x24, 0xa0, 0x79, 0xfd, 0xb5, 0x67, 0x93, 0x07,
	0x3a, 0x78, 0x24, 0x64, 0xe1, 0xa3, 0x62, 0x75, 0x38, 0x65, 0x8a, 0xbf, 0xf9, 0x7c, 0x00, 0xa0,
	0x6d, 0xdb, 0x1f, 0x80, 0x37, 0x37, 0x8e, 0x97, 0x1a, 0x45, 0x61, 0x0e, 0x10, 0x24, 0x8a, 0x27,
	0xf2, 0x44, 0x91, 0x3e, 0x62, 0x44, 0xc5, 0x55, 0xe6, 0x8e, 0x5a, 0x25, 0x8a, 0x90, 0x25, 0x74,
	0xa0, 0x95, 0x33, 0xf7, 0x51, 0xce, 0xe4, 0xa0, 0x13, 0xcf, 0x33, 0x1e, 0x59, 0x5b, 0xec, 0x42,
	0xc5, 0xb8, 0xe4, 0xc5, 0x71, 0x38, 0xc5, 0x6b, 0x8d, 0x1d, 0x84, 0xf8, 0x4e, 0x21, 0x6d, 0xdc,
	0x2c, 0xf1, 0xae, 0xad, 0x19, 0xc5, 0xed, 0x8e, 0x36, 0xb5, 0x81, 0x94, 0xfe, 0x62, 0x3a, 0xe8,
	0xc9, 0x95, 0x84, 0xbd, 0x65, 0x15, 0x16, 0x15, 0xd2, 0xe7, 0x16, 0xd7, 0x9c, 0xd3, 0xd2, 0x66,
	0xf6, 0x46, 0xe3, 0x32, 0x62, 0x51, 0x86, 0x4a, 0x67, 0xcc, 0x4d, 0xea, 0x37, 0x45, 0xd5, 0xa6,
	0x80, 0xe6, 0xba, 0xb3, 0x08, 0xd8, 0x30, 0x5b, 0x5f, 0xf2, 0x5a, 0xfb, 0x63, 0xb0, 0xa4, 0x41
};

void pgm_kov_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= kov_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t kovsh_tab[256] = {
	0xe7, 0x06, 0xa3, 0x70, 0xf2, 0x58, 0xe6, 0x59, 0xe4, 0xcf, 0xc2, 0x79, 0x1d, 0xe3, 0x71, 0x0e,
	0xb6, 0x90, 0x9a, 0x2a, 0x8c, 0x41, 0xf7, 0x82, 0x9b, 0xef, 0x99, 0x0c, 0xfa, 0x2f, 0xf1, 0xfe,
	0x8f, 0x70, 0xf4, 0xc1, 0xb5, 0x3d, 0x7c, 0x60, 0x4c, 0x09, 0xf4, 0x2e, 0x7c, 0x87, 0x63, 0x5f,
	0xce, 0x99, 0x84, 0x95, 0x06, 0x9a, 0x20, 0x23, 0x5a, 0xb9, 0x52, 0x95, 0x48, 0x2c, 0x84, 0x60,
	0x69, 0xe3, 0x93, 0x49, 0xb9, 0xd6, 0xbb, 0xd6, 0x9e, 0xdc, 0x96, 0x12, 0xfa, 0x60, 0xda, 0x5f,
	0x55, 0x5d, 0x5b, 0x20, 0x07, 0x1e, 0x97, 0x42, 0x77, 0xea, 0x1d, 0xe0, 0x70, 0xfb, 0x6a, 0x00,
	0x77, 0x9a, 0xef, 0x1b, 0xe0, 0xf9, 0x0d, 0xc1, 0x2e, 0x2f, 0xef, 0x25, 0x29, 0xe5, 0xd8, 0x2c,
	0xaf, 0x01, 0xd9, 0x6c, 0x31, 0xce, 0x5c, 0xea, 0xab, 0x1c, 0x92, 0x16, 0x61, 0xbc, 0xe4, 0x7c,
	0x5a, 0x76, 0xe9, 0x92, 0x39, 0x5b, 0x97, 0x60, 0xea, 0x57, 0x83, 0x9c, 0x92, 0x29, 0xa7, 0x12,
	0xa9, 0x71, 0x7a, 0xf9, 0x07, 0x68, 0xa7, 0x45, 0x88, 0x10, 0x81, 0x12, 0x2c, 0x67, 0x4d, 0x55,
	0x33, 0xf0, 0xfa, 0xd7, 0x1d, 0x4d, 0x0e, 0x63, 0x03, 0x34, 0x65, 0xe2, 0x76, 0x0f, 0x98, 0xa9,
	0x5f, 0x9a, 0xd3, 0xca, 0xdd, 0xc1, 0x5b, 0x3d, 0x4d, 0xf8, 0x40, 0x08, 0xdc, 0x05, 0x38, 0x00,
	0xcb, 0x24, 0x02, 0xff, 0x39, 0xe2, 0x9e, 0x04, 0x9a, 0x08, 0x63, 0xc8, 0x2b, 0x5a, 0x34, 0x06,
	0x62, 0xc1, 0xbb, 0x8a, 0xd0, 0x54, 0x4c, 0x43, 0x21, 0x4e, 0x4c, 0x99, 0x80, 0xc2, 0x3d, 0xce,
	0x2a, 0x7b, 0x09, 0x62, 0x1a, 0x91, 0x9b, 0xc3, 0x41, 0x24, 0xa0, 0xfd, 0xb5, 0x67, 0x93, 0x07,
	0xa7, 0xb8, 0x85, 0x8a, 0xa1, 0x1e, 0x4f, 0xb6, 0x75, 0x38, 0x65, 0x8a, 0xf9, 0x7c, 0x00, 0xa0
};


void pgm_kovsh_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= kovsh_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t photoy2k_tab[256] = {
	0xd9, 0x92, 0xb2, 0xbc, 0xa5, 0x88, 0xe3, 0x48, 0x7d, 0xeb, 0xc5, 0x4d, 0x31, 0xe4, 0x82, 0xbc,
	0x82, 0xcf, 0xe7, 0xf3, 0x15, 0xde, 0x8f, 0x91, 0xef, 0xc6, 0xb8, 0x81, 0x97, 0xe3, 0xdf, 0x4d,
	0x88, 0xbf, 0xe4, 0x05, 0x25, 0x73, 0x1e, 0xd0, 0xcf, 0x1e, 0xeb, 0x4d, 0x18, 0x4e, 0x6f, 0x9f,
	0x00, 0x72, 0xc3, 0x74, 0xbe, 0x02, 0x09, 0x0a, 0xb0, 0xb1, 0x8e, 0x9b, 0x08, 0xed, 0x68, 0x6d,
	0x25, 0xe8, 0x28, 0x94, 0xa6, 0x44, 0xa6, 0xfa, 0x95, 0x69, 0x72, 0xd3, 0x6d, 0xb6, 0xff, 0xf3,
	0x45, 0x4e, 0xa3, 0x60, 0xf2, 0x58, 0xe7, 0x59, 0xe4, 0x4f, 0x70, 0xd2, 0xdd, 0xc0, 0x6e, 0xf3,
	0xd7, 0xb2, 0xdc, 0x1e, 0xa8, 0x41, 0x07, 0x5d, 0x60, 0x15, 0xea, 0xcf, 0xdb, 0xc1, 0x1d, 0x4d,
	0xb7, 0x42, 0xec, 0xc4, 0xca, 0xa9, 0x40, 0x30, 0x0f, 0x3c, 0xe2, 0x81, 0xe0, 0x5c, 0x51, 0x07,
	0xb0, 0x1e, 0x4a, 0xb3, 0x64, 0x3e, 0x1c, 0x62, 0x17, 0xcd, 0xf2, 0xe4, 0x14, 0x9d, 0xa6, 0xd4,
	0x64, 0x36, 0xa5, 0xe8, 0x7e, 0x84, 0x0e, 0xb3, 0x5d, 0x79, 0x57, 0xea, 0xd7, 0xad, 0xbc, 0x9e,
	0x2d, 0x90, 0x03, 0x9e, 0x0e, 0xc6, 0x98, 0xdb, 0xe3, 0xb6, 0x9f, 0x9b, 0xf6, 0x21, 0xe6, 0x98,
	0x94, 0x77, 0xb7, 0x2b, 0xaa, 0xc9, 0xff, 0xef, 0x7a, 0xf2, 0x71, 0x4e, 0x52, 0x06, 0x85, 0x37,
	0x81, 0x8e, 0x86, 0x64, 0x39, 0x92, 0x2a, 0xca, 0xf3, 0x3e, 0x87, 0xb5, 0x0c, 0x7b, 0x42, 0x5e,
	0x04, 0xa7, 0xfb, 0xd7, 0x13, 0x7f, 0x83, 0x6a, 0x77, 0x0f, 0xa7, 0x34, 0x51, 0x88, 0x9c, 0xac,
	0x23, 0x90, 0x4d, 0x4d, 0x72, 0x4e, 0xa3, 0x26, 0x1a, 0x45, 0x61, 0x0e, 0x10, 0x24, 0x8a, 0x27,
	0x92, 0x14, 0x23, 0xae, 0x4b, 0x80, 0xae, 0x6a, 0x56, 0x01, 0xac, 0x55, 0xf7, 0x6d, 0x9b, 0x6d
};

void pgm_photoy2k_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT3
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		x ^= photoy2k_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t pstar[256] = {
	0x62, 0x59, 0x17, 0xe3, 0xe1, 0x11, 0x02, 0x97, 0x67, 0x4d, 0x4a, 0x1c, 0x1f, 0x76, 0x64, 0xc1,
	0xfa, 0x04, 0xd2, 0x9f, 0x22, 0xf5, 0xfd, 0xfe, 0x89, 0xeb, 0xab, 0x41, 0xc0, 0xac, 0x2b, 0x64,
	0xfe, 0x1e, 0x9b, 0x68, 0x07, 0xfd, 0x75, 0x25, 0x24, 0xa0, 0x41, 0x5d, 0x79, 0xb5, 0x67, 0x93,
	0xe1, 0xa3, 0x3a, 0x24, 0xa0, 0xbf, 0x64, 0x62, 0x38, 0x65, 0x75, 0x78, 0x8a, 0xf9, 0x7c, 0x00,
	0x71, 0x38, 0xc5, 0xe4, 0xdc, 0xf8, 0xc5, 0xc5, 0x8d, 0x1d, 0x6b, 0xb8, 0x84, 0x4e, 0x21, 0x6d,
	0x55, 0x95, 0x31, 0x86, 0x11, 0xe6, 0xb1, 0x77, 0x77, 0x3b, 0x01, 0x82, 0xab, 0x88, 0xef, 0x77,
	0x08, 0xd8, 0x80, 0xba, 0x41, 0xfb, 0xb3, 0x30, 0x5f, 0xf2, 0x5b, 0xe6, 0x5a, 0x63, 0xb0, 0xa4,
	0x37, 0x37, 0x6d, 0x1f, 0x27, 0x0e, 0x80, 0x8e, 0x1a, 0x45, 0x97, 0xdb, 0x61, 0x10, 0x24, 0x8a,
	0x62, 0x44, 0xf2, 0x91, 0x74, 0x25, 0x3e, 0xc5, 0xe6, 0x8e, 0x55, 0x44, 0x5a, 0x8a, 0x90, 0x25,
	0xa1, 0x0d, 0x43, 0x35, 0x46, 0x80, 0xfe, 0xb3, 0x4e, 0x4c, 0x21, 0xcc, 0x99, 0xc2, 0x3d, 0xce,
	0x19, 0xc5, 0x2c, 0xae, 0xe8, 0x94, 0xad, 0xed, 0x36, 0xb5, 0x8e, 0xf1, 0x81, 0xfe, 0x62, 0x3a,
	0x8c, 0xe0, 0x90, 0x88, 0xf1, 0x31, 0xc5, 0xd9, 0x3c, 0xf4, 0x70, 0x7d, 0x7d, 0x1c, 0xca, 0xe2,
	0x51, 0xce, 0xa0, 0x33, 0x42, 0x1e, 0xf7, 0xe4, 0x13, 0xcf, 0xa0, 0x95, 0x33, 0x59, 0x5b, 0xec,
	0xf7, 0x6d, 0x56, 0xac, 0x6a, 0xae, 0x55, 0x9b, 0x92, 0x14, 0x6d, 0x01, 0x23, 0x4b, 0x80, 0xae,
	0x65, 0x15, 0xc9, 0x84, 0x66, 0xd7, 0xbd, 0x16, 0xd2, 0xe7, 0x15, 0x95, 0x16, 0x9c, 0xd3, 0xd2,
	0x62, 0x51, 0xf6, 0xe3, 0xa6, 0xea, 0x32, 0x86, 0x67, 0xcc, 0x4a, 0x46, 0x4d, 0x37, 0x45, 0xd5
};

void pgm_pstar_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x100000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x = x ^ pstar[(i & 255)] << 8;
		src[i] = x ;
	}
}

static const uint8_t dfront_tab[256] = {
	0x51, 0xc4, 0xe3, 0x10, 0x1c, 0xad, 0x8a, 0x39, 0x8c, 0xe0, 0xa5, 0x04, 0x0f, 0xe4, 0x35, 0xc3,
	0x2d, 0x6b, 0x32, 0xe2, 0x60, 0x54, 0x63, 0x06, 0xa3, 0xf1, 0x0b, 0x5f, 0x6c, 0x5c, 0xb3, 0xec,
	0x77, 0x61, 0x69, 0xe7, 0x3c, 0xb7, 0x42, 0x72, 0x1a, 0x70, 0xb0, 0x96, 0xa4, 0x28, 0xc0, 0xfb,
	0x0a, 0x00, 0xcb, 0x15, 0x49, 0x48, 0xd3, 0x94, 0x58, 0xcf, 0x41, 0x86, 0x17, 0x71, 0xb1, 0xbd,
	0x21, 0x01, 0x37, 0x1e, 0xba, 0xeb, 0xf3, 0x59, 0xf6, 0xa7, 0x29, 0x4f, 0xb5, 0xca, 0x4c, 0x34,
	0x20, 0xa2, 0x62, 0x4b, 0x93, 0x9e, 0x47, 0x9f, 0x8d, 0x0e, 0x1b, 0xb6, 0x4d, 0x82, 0xd5, 0xf4,
	0x85, 0x79, 0x53, 0x92, 0x9b, 0xf7, 0xea, 0x44, 0x76, 0x1f, 0x22, 0x45, 0xed, 0xbe, 0x11, 0x55,
	0xaf, 0xf5, 0xf8, 0x50, 0x07, 0xe6, 0xc7, 0x5e, 0xd7, 0xde, 0xe5, 0x26, 0x2b, 0xf2, 0x6a, 0x8b,
	0xb8, 0x98, 0x89, 0xdb, 0x14, 0x5b, 0xc5, 0x78, 0xdc, 0xd0, 0x87, 0x5d, 0xc1, 0x0d, 0x95, 0x97,
	0x7e, 0xa8, 0x24, 0x3d, 0xe1, 0xd1, 0x19, 0xa6, 0x99, 0xd8, 0x83, 0x1d, 0xff, 0x30, 0x9d, 0x05,
	0xd4, 0x02, 0x27, 0x7b, 0x13, 0xb2, 0x7f, 0x40, 0x12, 0xa0, 0x68, 0x67, 0x4e, 0x3a, 0x46, 0xb9,
	0xee, 0xdf, 0x66, 0xd6, 0x8f, 0xa9, 0x0c, 0x91, 0x65, 0x18, 0x52, 0x56, 0xd9, 0x74, 0x09, 0x6e,
	0xc6, 0x73, 0xc9, 0xfc, 0x03, 0x43, 0xef, 0xaa, 0x7c, 0xbb, 0x2c, 0x90, 0xcc, 0xce, 0xe8, 0xae,
	0x2a, 0xf9, 0x57, 0x88, 0xc8, 0xe9, 0x5a, 0xdd, 0x2e, 0x7d, 0x64, 0xc2, 0x6d, 0x3e, 0xfa, 0x80,
	0x16, 0xcd, 0x6f, 0x84, 0x8e, 0x9c, 0xf0, 0xac, 0xb4, 0x9a, 0x2f, 0xbc, 0x31, 0x23, 0xfe, 0x38,
	0x08, 0x75, 0xa1, 0x33, 0xab, 0xd2, 0xda, 0x81, 0xbf, 0x7a, 0x3b, 0x3f, 0x4a, 0xfd, 0x25, 0x36
};

void pgm_dfront_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= dfront_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t ddp2_tab[256] = {
	0x2a, 0x4a, 0x39, 0x98, 0xac, 0x39, 0xb2, 0x55, 0x72, 0xf3, 0x7b, 0x3c, 0xee, 0x94, 0x6e, 0xd5,
	0xcd, 0xbc, 0x9a, 0xd0, 0x45, 0x7d, 0x49, 0x68, 0xb1, 0x61, 0x54, 0xef, 0xa2, 0x84, 0x29, 0x20,
	0x32, 0x52, 0x82, 0x04, 0x38, 0x69, 0x9f, 0x24, 0x46, 0xf4, 0x3f, 0xc2, 0xf1, 0x25, 0xac, 0x2d,
	0xdf, 0x2d, 0xb4, 0x51, 0xc7, 0xb5, 0xe5, 0x88, 0xbd, 0x3b, 0x5a, 0x25, 0x5b, 0xc7, 0xae, 0x5f,
	0x43, 0xcf, 0x89, 0xd9, 0xe2, 0x63, 0xc6, 0x76, 0x21, 0x2b, 0x77, 0xc0, 0x27, 0x98, 0xfd, 0x09,
	0xe1, 0x8c, 0x26, 0x2e, 0x92, 0x99, 0xbc, 0xbe, 0x0e, 0xba, 0xbf, 0x70, 0xe7, 0xb7, 0xe9, 0x37,
	0x5c, 0xd1, 0x5e, 0xad, 0x22, 0x17, 0xc5, 0x67, 0x9d, 0xc6, 0xfb, 0x53, 0xc7, 0x4d, 0x32, 0xb4,
	0xf2, 0x43, 0x53, 0x7c, 0x01, 0xfe, 0xd2, 0x91, 0x40, 0x85, 0xa3, 0xe8, 0xdf, 0xdb, 0xff, 0x6c,
	0x64, 0x15, 0xcd, 0x8e, 0x07, 0x82, 0x78, 0x8d, 0x4e, 0x2d, 0x66, 0x8a, 0x62, 0x6f, 0xd3, 0x6a,
	0xae, 0x16, 0x44, 0x1e, 0xed, 0xc4, 0x12, 0x7a, 0xbe, 0x05, 0x06, 0xce, 0x9b, 0x8a, 0xf7, 0xf8,
	0x74, 0x23, 0x73, 0x74, 0xb8, 0x13, 0xc2, 0x42, 0xea, 0xf9, 0x7f, 0xa9, 0xaf, 0x56, 0xd6, 0xb3,
	0xb7, 0xc4, 0x47, 0x31, 0x67, 0xaa, 0x58, 0x8b, 0x47, 0x1b, 0xf5, 0x75, 0x95, 0x8f, 0xf0, 0x3a,
	0x85, 0x76, 0x59, 0x24, 0x0c, 0xd7, 0x00, 0xb3, 0xdc, 0xfc, 0x65, 0x34, 0xde, 0xfa, 0xd8, 0xc3,
	0xc3, 0x5e, 0xe3, 0x9e, 0x02, 0x28, 0x50, 0x81, 0x95, 0x2f, 0xe4, 0xb5, 0xa0, 0x4d, 0xa1, 0x36,
	0x9d, 0x18, 0x6d, 0x79, 0x19, 0x3b, 0x1d, 0xb8, 0xe1, 0xcc, 0x61, 0x1a, 0xe2, 0x31, 0x4c, 0x3f,
	0xdc, 0xca, 0xd4, 0xda, 0xcd, 0xd2, 0x83, 0xca, 0xeb, 0x4f, 0xf2, 0x2f, 0x2d, 0x2a, 0xec, 0x1f
};

void pgm_ddp2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT2
		// NO CRYPT2
		// NO CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8_ALT

		x ^= ddp2_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t mm_tab[256] = {
	0xd0, 0x45, 0xbc, 0x84, 0x93, 0x60, 0x7d, 0x49, 0x68, 0xb1, 0x54, 0xa2, 0x05, 0x29, 0x41, 0x20,
	0x04, 0x08, 0x52, 0x25, 0x89, 0xf4, 0x69, 0x9f, 0x24, 0x46, 0x3d, 0xf1, 0xf9, 0xab, 0xa6, 0x2d,
	0x18, 0x19, 0x6d, 0x33, 0x79, 0x23, 0x3b, 0x1d, 0xe0, 0xb8, 0x61, 0x1a, 0xe1, 0x4c, 0x5d, 0x3f,
	0x5e, 0x02, 0xe3, 0x4d, 0x9e, 0x80, 0x28, 0x50, 0xa0, 0x81, 0xe4, 0xa5, 0x97, 0xa1, 0x86, 0x36,
	0x1e, 0xed, 0x16, 0x8a, 0x44, 0x06, 0x64, 0x12, 0x9a, 0x7e, 0xce, 0x9b, 0xef, 0xf7, 0x3e, 0xf8,
	0x15, 0x07, 0xcb, 0x6f, 0x8e, 0x3c, 0x82, 0x70, 0x62, 0x8d, 0x66, 0x7a, 0x4e, 0xd3, 0xb6, 0x6a,
	0x51, 0xa7, 0x2c, 0xc7, 0xa4, 0x0b, 0xb5, 0xe5, 0x88, 0xbd, 0x5a, 0x5b, 0x1b, 0xae, 0xe6, 0x5f,
	0x2e, 0x92, 0x8c, 0xb7, 0x96, 0xba, 0x99, 0xbb, 0xbe, 0x0e, 0xbf, 0xe7, 0x2f, 0xe9, 0x30, 0x37,
	0x98, 0xac, 0x4a, 0x94, 0x38, 0xf3, 0x39, 0xb2, 0x55, 0x72, 0x7b, 0xee, 0xdd, 0x6e, 0x11, 0xd5,
	0x26, 0xa8, 0x71, 0xd6, 0x74, 0x7f, 0x13, 0xc2, 0x56, 0xea, 0xa9, 0xaf, 0xc3, 0x42, 0x03, 0xb3,
	0xc4, 0x6b, 0x47, 0xf0, 0x31, 0xf5, 0xaa, 0x58, 0x8f, 0x48, 0x75, 0x95, 0x35, 0x8b, 0x57, 0x3a,
	0x73, 0x0c, 0x59, 0xd8, 0x14, 0x65, 0xd7, 0x00, 0xfa, 0xdc, 0x34, 0xde, 0xc0, 0xb0, 0x87, 0xc1,
	0xc8, 0xcd, 0xd4, 0x2a, 0xda, 0xe8, 0xd2, 0x83, 0x0d, 0xca, 0xf2, 0x0f, 0xeb, 0xec, 0x9c, 0x1f,
	0xad, 0x22, 0xd1, 0x4b, 0x5c, 0xf6, 0x17, 0xc5, 0x67, 0x9d, 0xfb, 0xc9, 0xcc, 0x32, 0x1c, 0xb4,
	0xd9, 0xe2, 0xcf, 0x90, 0xb9, 0x2b, 0x63, 0xc6, 0x76, 0x21, 0x77, 0x27, 0xfc, 0xfd, 0x0a, 0x09,
	0x7c, 0x01, 0x43, 0xdb, 0x53, 0x85, 0xfe, 0x78, 0x91, 0x40, 0xa3, 0xdf, 0x4f, 0xff, 0x10, 0x6c
};

void pgm_mm_decrypt(running_machine &machine) // and dw2001
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= mm_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t kov2_tab[256] = {
	0x11, 0x4a, 0x38, 0x98, 0xac, 0x39, 0xb2, 0x55, 0x72, 0xf3, 0x7b, 0x3c, 0xee, 0x94, 0x6e, 0xd5,
	0x41, 0xbc, 0x93, 0xd0, 0x45, 0x7d, 0x49, 0x68, 0xb1, 0x60, 0x54, 0xef, 0xa2, 0x84, 0x29, 0x20,
	0xa6, 0x52, 0x89, 0x04, 0x08, 0x69, 0x9f, 0x24, 0x46, 0xf4, 0x3d, 0xc3, 0xf1, 0x25, 0xab, 0x2d,
	0xe6, 0x2c, 0xa4, 0x51, 0xa7, 0xb5, 0xe5, 0x88, 0xbd, 0x0b, 0x5a, 0x35, 0x5b, 0xc7, 0xae, 0x5f,
	0x0a, 0xcf, 0xb9, 0xd9, 0xe2, 0x63, 0xc6, 0x76, 0x21, 0x2b, 0x77, 0xc0, 0x27, 0x90, 0xfd, 0x09,
	0x30, 0x8c, 0x96, 0x2e, 0x92, 0x99, 0xbb, 0xbe, 0x0e, 0xba, 0xbf, 0x80, 0xe7, 0xb7, 0xe9, 0x37,
	0x1c, 0xd1, 0x5c, 0xad, 0x22, 0x17, 0xc5, 0x67, 0x9d, 0xf6, 0xfb, 0x23, 0xc9, 0x4b, 0x32, 0xb4,
	0x10, 0x43, 0x53, 0x7c, 0x01, 0xfe, 0x78, 0x91, 0x40, 0x85, 0xa3, 0xe8, 0xdf, 0xdb, 0xff, 0x6c,
	0xb6, 0x15, 0xcb, 0x8e, 0x07, 0x82, 0x70, 0x8d, 0x4e, 0xdd, 0x66, 0x7a, 0x62, 0x6f, 0xd3, 0x6a,
	0x3e, 0x16, 0x44, 0x1e, 0xed, 0x64, 0x12, 0x9a, 0x7e, 0x05, 0x06, 0xce, 0x9b, 0x8a, 0xf7, 0xf8,
	0x03, 0x26, 0x71, 0x74, 0xa8, 0x13, 0xc2, 0x42, 0xea, 0xf9, 0x7f, 0xa9, 0xaf, 0x56, 0xd6, 0xb3,
	0x57, 0xc4, 0x47, 0x31, 0x6b, 0xaa, 0x58, 0x8b, 0x48, 0x1b, 0xf5, 0x75, 0x95, 0x8f, 0xf0, 0x3a,
	0x87, 0x73, 0x59, 0x14, 0x0c, 0xd7, 0x00, 0xb0, 0xdc, 0xfc, 0x65, 0x34, 0xde, 0xfa, 0xd8, 0xc1,
	0x86, 0x5e, 0xe3, 0x9e, 0x02, 0x28, 0x50, 0x81, 0x97, 0x2f, 0xe4, 0xa5, 0xa0, 0x4d, 0xa1, 0x36,
	0x5d, 0x18, 0x6d, 0x79, 0x19, 0x3b, 0x1d, 0xb8, 0xe1, 0xcc, 0x61, 0x1a, 0xe0, 0x33, 0x4c, 0x3f,
	0x9c, 0xc8, 0xd4, 0xda, 0xcd, 0xd2, 0x83, 0xca, 0xeb, 0x4f, 0xf2, 0x0f, 0x0d, 0x2a, 0xec, 0x1f
};

void pgm_kov2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		// NO CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8_ALT

		x ^= kov2_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t kov2p_tab[256] = {
	0x44, 0x47, 0xb8, 0x28, 0x03, 0xa2, 0x21, 0xbc, 0x17, 0x32, 0x4e, 0xe2, 0xdf, 0x69, 0x35, 0xc7,
	0xa2, 0x06, 0xec, 0x36, 0xd2, 0x44, 0x12, 0x6a, 0x8d, 0x51, 0x6b, 0x20, 0x69, 0x01, 0xca, 0xf0,
	0x71, 0xc4, 0x34, 0xdc, 0x6b, 0xd6, 0x42, 0x2a, 0x5d, 0xb5, 0xc7, 0x6f, 0x4f, 0xd8, 0xb3, 0xed,
	0x51, 0x9e, 0x37, 0x1e, 0xc0, 0x85, 0x2a, 0x91, 0xc6, 0x9c, 0xac, 0xf5, 0x20, 0x3b, 0x09, 0x74,
	0x24, 0xf1, 0xe0, 0x42, 0x02, 0xbe, 0x84, 0x75, 0x4a, 0x82, 0xa2, 0x17, 0xae, 0xb6, 0x24, 0x79,
	0x0a, 0x5a, 0x56, 0xcb, 0xa1, 0x2e, 0x47, 0xea, 0xa9, 0x25, 0x73, 0x79, 0x0b, 0x17, 0x9e, 0x33,
	0x64, 0xb6, 0x03, 0x7f, 0x4f, 0xc3, 0xae, 0x45, 0xe6, 0x82, 0x27, 0x01, 0x86, 0x6b, 0x50, 0x16,
	0xd3, 0x22, 0x90, 0x64, 0xfc, 0xa9, 0x31, 0x1c, 0x41, 0xd5, 0x07, 0xd3, 0xb2, 0xfe, 0x53, 0xd6,
	0x39, 0xfb, 0xe6, 0xbe, 0xda, 0x4d, 0x8a, 0x44, 0x3a, 0x9b, 0x9d, 0x56, 0x5e, 0x5f, 0xff, 0x6a,
	0xb6, 0xde, 0x2f, 0x12, 0x5a, 0x5d, 0xb0, 0xd0, 0x93, 0x92, 0xb2, 0x2c, 0x9d, 0x59, 0xee, 0x05,
	0xab, 0xa8, 0xd2, 0x25, 0x2c, 0xc5, 0xde, 0x18, 0x4d, 0xb6, 0x4e, 0x3d, 0xbf, 0xfa, 0xf9, 0x1d,
	0xba, 0x76, 0x79, 0xfc, 0x42, 0xb2, 0x8c, 0xae, 0xa9, 0x45, 0xba, 0xac, 0x55, 0x8e, 0x38, 0x67,
	0xc3, 0xa5, 0x0d, 0xdc, 0xcc, 0x91, 0x73, 0x69, 0x27, 0xbc, 0x80, 0xdf, 0x30, 0xa4, 0x05, 0xd8,
	0xe7, 0xd2, 0xb7, 0x4b, 0x3c, 0x10, 0x8c, 0x5d, 0x8a, 0xd7, 0x68, 0x7a, 0x61, 0x07, 0xf9, 0xa5,
	0x88, 0xda, 0xdf, 0x0c, 0x42, 0x1b, 0x11, 0xe0, 0xd1, 0x93, 0x7c, 0x63, 0x39, 0xc5, 0xed, 0x43,
	0x46, 0xdb, 0x30, 0x26, 0xd0, 0xdf, 0x7a, 0x86, 0x3e, 0x2e, 0x04, 0xbf, 0x49, 0x2a, 0xf9, 0x66
};

void pgm_kov2p_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= kov2p_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t puzzli2_tab[256] = {
	0xb7, 0x66, 0xa3, 0xc0, 0x51, 0x55, 0x6d, 0x63, 0x86, 0x60, 0x64, 0x6c, 0x67, 0x18, 0x0b, 0x05,
	0x62, 0xff, 0xe0, 0x1e, 0x30, 0x21, 0x2e, 0x40, 0x41, 0xb9, 0x60, 0x38, 0xd1, 0x24, 0x7e, 0x36,
	0x7a, 0x0b, 0x1c, 0x69, 0x4f, 0x09, 0xe1, 0x9e, 0xcf, 0xcd, 0x7c, 0x00, 0x73, 0x08, 0x77, 0x37,
	0x5f, 0x50, 0x32, 0x3e, 0xd3, 0x54, 0x77, 0x6b, 0x60, 0x60, 0x74, 0x7c, 0x55, 0x4f, 0x44, 0x5e,
	0x66, 0x5c, 0x58, 0x26, 0x35, 0x29, 0x3f, 0x35, 0x3f, 0x1c, 0x0b, 0x0d, 0x08, 0x5b, 0x59, 0x5c,
	0xa0, 0xa5, 0x87, 0x85, 0x24, 0x75, 0x5f, 0x42, 0x1b, 0xf3, 0x1a, 0x58, 0x17, 0x58, 0x71, 0x6b,
	0x69, 0x89, 0x7d, 0x3a, 0xf3, 0xc4, 0x5d, 0xa0, 0x4f, 0x27, 0x58, 0xc4, 0xa8, 0xdd, 0xa8, 0xfb,
	0xbe, 0xa4, 0xe2, 0xee, 0x07, 0x10, 0x90, 0x72, 0x99, 0x08, 0x68, 0x6d, 0x5c, 0x5c, 0x6d, 0x58,
	0x2f, 0xdc, 0x15, 0xd5, 0xd6, 0xd6, 0x3b, 0x3b, 0xf9, 0x32, 0xcc, 0xdd, 0xd4, 0xf1, 0xea, 0xed,
	0xe4, 0xf6, 0xf2, 0x91, 0xca, 0xc1, 0xed, 0xf2, 0xf6, 0xfb, 0xc0, 0xe8, 0xe3, 0xe7, 0xfa, 0xf1,
	0xf5, 0x08, 0x26, 0x2b, 0x2f, 0x34, 0x39, 0x13, 0x28, 0x07, 0x88, 0x5b, 0x8f, 0x94, 0x9b, 0x2e,
	0xf5, 0xab, 0x72, 0x76, 0x7a, 0x40, 0xb9, 0x09, 0xd8, 0x3b, 0xcd, 0x31, 0x3d, 0x42, 0xab, 0xb1,
	0xb5, 0xb9, 0x3b, 0xe3, 0x0b, 0x65, 0x18, 0xfb, 0x1f, 0x12, 0xe4, 0xe8, 0xec, 0xf2, 0xf7, 0xfc,
	0xc0, 0xe8, 0xe0, 0xe6, 0xfa, 0xf1, 0xf4, 0x0b, 0x26, 0x2b, 0x30, 0x35, 0x39, 0x13, 0x29, 0x21,
	0x0c, 0x11, 0x16, 0x1b, 0x1f, 0x64, 0x0e, 0x60, 0x05, 0x79, 0x7c, 0x37, 0x00, 0x0f, 0x4f, 0x38,
	0x1d, 0x18, 0xa2, 0xb6, 0xb2, 0xa9, 0xac, 0xab, 0xae, 0x91, 0x98, 0x8d, 0x91, 0xbb, 0xb1, 0xc0
};

void pgm_puzzli2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x100000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= puzzli2_tab[i & 0xff] << 8;
		src[i] = x;
	}
}


static const uint8_t theglad_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x35, 0x52, 0x44, 0x31, 0x30, 0x32, 0x31, 0x32, 0x30, 0x33,
	0xc4, 0xa3, 0x46, 0x78, 0x30, 0xb3, 0x8b, 0xd5, 0x2f, 0xc4, 0x44, 0xbf, 0xdb, 0x76, 0xdb, 0xea,
	0xb4, 0xeb, 0x95, 0x4d, 0x15, 0x21, 0x99, 0xa1, 0xd7, 0x8c, 0x40, 0x1d, 0x43, 0xf3, 0x9f, 0x71,
	0x3d, 0x8c, 0x52, 0x01, 0xaf, 0x5b, 0x8b, 0x63, 0x34, 0xc8, 0x5c, 0x1b, 0x06, 0x7f, 0x41, 0x96,
	0x2a, 0x8d, 0xf1, 0x64, 0xda, 0xb8, 0x67, 0xba, 0x33, 0x1f, 0x2b, 0x28, 0x20, 0x13, 0xe6, 0x96,
	0x86, 0x34, 0x25, 0x85, 0xb0, 0xd0, 0x6d, 0x85, 0xfe, 0x78, 0x81, 0xf1, 0xca, 0xe4, 0xef, 0xf2,
	0x9b, 0x09, 0xe1, 0xb4, 0x8d, 0x79, 0x22, 0xe2, 0x00, 0xfb, 0x6f, 0x68, 0x80, 0x6a, 0x00, 0x69,
	0xf5, 0xd3, 0x57, 0x7e, 0x0c, 0xca, 0x48, 0x31, 0xe5, 0x0d, 0x4a, 0xb9, 0xfd, 0x5c, 0xfd, 0xf8,
	0x5f, 0x98, 0xfb, 0xb3, 0x07, 0x1a, 0xe3, 0x10, 0x96, 0x56, 0xa3, 0x56, 0x3d, 0xb1, 0x07, 0xe0,
	0xe3, 0x9f, 0x7f, 0x62, 0x99, 0x01, 0x35, 0x60, 0x40, 0xbe, 0x4f, 0xeb, 0x79, 0xa0, 0x82, 0x9f,
	0xcd, 0x71, 0xd8, 0xda, 0x1e, 0x56, 0xc2, 0x3e, 0x4e, 0x6b, 0x60, 0x69, 0x2d, 0x9f, 0x10, 0xf4,
	0xa9, 0xd3, 0x36, 0xaa, 0x31, 0x2e, 0x4c, 0x0a, 0x69, 0xc3, 0x2a, 0xff, 0x15, 0x67, 0x96, 0xde,
	0x3f, 0xcc, 0x0f, 0xa1, 0xac, 0xe2, 0xd6, 0x62, 0x7e, 0x6f, 0x3e, 0x1b, 0x2a, 0xed, 0x36, 0x9c,
	0x9d, 0xa4, 0x14, 0xcd, 0xaa, 0x08, 0xa4, 0x26, 0xb7, 0x55, 0x70, 0x6c, 0xa9, 0x69, 0x52, 0xae,
	0x0c, 0xe1, 0x38, 0x7f, 0x87, 0x78, 0x38, 0x75, 0x80, 0x9c, 0xd4, 0xe2, 0x0b, 0x52, 0x8f, 0xd2,
	0x19, 0x4c, 0xb0, 0x45, 0xde, 0x48, 0x55, 0xae, 0x82, 0xab, 0xbc, 0xab, 0x0c, 0x5e, 0xce, 0x07
};

void pgm_theglad_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= theglad_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t oldsplus_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x31, 0x33, 0x52, 0x44, 0x31, 0x30, 0x34, 0x30, 0x37, 0x32, 0x37,
	0xf5, 0x79, 0x6d, 0xab, 0x04, 0x22, 0x51, 0x96, 0xf2, 0x72, 0xe8, 0x3a, 0x96, 0xd2, 0x9a, 0xcc,
	0x3f, 0x47, 0x3c, 0x09, 0xf2, 0xd9, 0x72, 0x41, 0xe6, 0x44, 0x43, 0xa7, 0x3e, 0xe2, 0xfd, 0xd8,
	0x06, 0xd8, 0x4c, 0xa9, 0x70, 0x80, 0x95, 0x35, 0x50, 0x17, 0x99, 0x27, 0xd5, 0xa8, 0x47, 0x45,
	0x89, 0x38, 0xe1, 0x3d, 0x8c, 0x33, 0x53, 0xb4, 0x0d, 0x17, 0xd1, 0x8d, 0x09, 0x5f, 0xaf, 0x76,
	0x48, 0xb2, 0x85, 0xb9, 0x95, 0x4c, 0x83, 0x42, 0x3d, 0xad, 0x11, 0xec, 0xca, 0x82, 0xac, 0x10,
	0x01, 0xd0, 0xfd, 0x50, 0x19, 0x67, 0x3b, 0xa0, 0x3e, 0x86, 0xc2, 0x97, 0x46, 0xcb, 0xf4, 0xf5,
	0xb3, 0x5f, 0x50, 0x74, 0xe9, 0x5f, 0xd2, 0xd4, 0xb0, 0x8d, 0x8a, 0x21, 0xed, 0x37, 0x80, 0x47,
	0x9d, 0x68, 0xc7, 0xd9, 0x12, 0x4e, 0xdf, 0x1e, 0x72, 0xeb, 0x50, 0x5e, 0x6d, 0x00, 0x85, 0x6b,
	0x3e, 0x37, 0xe6, 0x72, 0xe5, 0x8f, 0x3a, 0x03, 0xa3, 0x0d, 0x3b, 0x5f, 0xb6, 0xa1, 0x7b, 0x02,
	0x56, 0x56, 0x77, 0x71, 0xef, 0xbe, 0xf9, 0x46, 0xa1, 0x9d, 0xb3, 0x79, 0xf6, 0xd5, 0x19, 0xf0,
	0xe2, 0x91, 0x7e, 0x4a, 0x01, 0xb6, 0x73, 0xe8, 0x0c, 0x86, 0x5d, 0x3e, 0x9c, 0x97, 0x55, 0x58,
	0x23, 0xf4, 0x45, 0xb0, 0x28, 0x91, 0x40, 0x2f, 0xc2, 0xf4, 0x21, 0x81, 0x58, 0x22, 0x68, 0x9d,
	0x97, 0xc7, 0x51, 0x95, 0xb4, 0xaa, 0x36, 0x9b, 0xe4, 0x51, 0x27, 0x55, 0x18, 0xf0, 0xc7, 0x62,
	0xfe, 0x98, 0x6a, 0x2d, 0x35, 0x9d, 0x6c, 0xf1, 0xcf, 0x48, 0xd4, 0x0d, 0x0c, 0xbe, 0x2a, 0x8a,
	0x55, 0x31, 0x96, 0xea, 0x78, 0x45, 0x3a, 0x33, 0x23, 0xc5, 0xd1, 0x3c, 0xa3, 0x86, 0x88, 0x38
};

void pgm_oldsplus_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		unsigned short x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= oldsplus_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t kovshp_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x39, 0x72, 0x64, 0x31, 0x30, 0x34, 0x30, 0x32, 0x31, 0x39,
	0xf9, 0x8c, 0xbd, 0x87, 0x16, 0x07, 0x39, 0xeb, 0x29, 0x9e, 0x17, 0xef, 0x4f, 0x64, 0x7c, 0xe0,
	0x5f, 0x73, 0x5b, 0xa1, 0x5e, 0x95, 0x0d, 0xf1, 0x40, 0x36, 0x2f, 0x00, 0xe2, 0x8a, 0xbc, 0x32,
	0x44, 0xfa, 0x6c, 0x33, 0x0b, 0xd5, 0x4c, 0x3b, 0x36, 0x34, 0x9e, 0xa3, 0x20, 0x2e, 0xf3, 0xa9,
	0xb7, 0x3e, 0x87, 0x80, 0xfb, 0xf1, 0xdd, 0x9c, 0xba, 0xd3, 0x9b, 0x3b, 0x8a, 0x9c, 0xa8, 0x37,
	0x07, 0x97, 0x84, 0x0c, 0x4e, 0x54, 0xe7, 0x25, 0xba, 0x8e, 0x9d, 0x6b, 0xde, 0x5f, 0xa1, 0x10,
	0xc3, 0xa2, 0x79, 0x99, 0x63, 0xa9, 0xd1, 0x2a, 0x65, 0x20, 0x5b, 0x16, 0x1b, 0x41, 0xe6, 0xa7,
	0xba, 0x3a, 0xbd, 0x2a, 0xd8, 0xdb, 0x43, 0x3f, 0x2b, 0x85, 0xcc, 0x5f, 0x80, 0x4f, 0xbe, 0xae,
	0xfa, 0x79, 0xe8, 0x03, 0x8d, 0x16, 0x22, 0x35, 0xbb, 0xf6, 0x26, 0xa9, 0x8d, 0xd2, 0xaf, 0x19,
	0xd4, 0xbb, 0xd0, 0xa6, 0xa1, 0xc4, 0x96, 0x21, 0x02, 0xef, 0xe1, 0x96, 0x00, 0x56, 0x80, 0x1b,
	0xd6, 0x9a, 0x8c, 0xd7, 0x73, 0x91, 0x07, 0x55, 0x32, 0x2b, 0xb5, 0x0b, 0xd8, 0xa5, 0x39, 0x26,
	0xce, 0xf2, 0x74, 0x98, 0xa1, 0x66, 0x1a, 0x64, 0xb8, 0xa5, 0x96, 0x29, 0x54, 0xcb, 0x21, 0xed,
	0xcd, 0xdd, 0x1e, 0x2c, 0x0b, 0x70, 0xb8, 0x22, 0x43, 0x98, 0xbe, 0x54, 0xf3, 0x14, 0xbe, 0x65,
	0x21, 0xb7, 0x61, 0x17, 0xcf, 0x19, 0x07, 0xa0, 0xc2, 0x7f, 0xa3, 0x30, 0x75, 0x08, 0xd8, 0xbf,
	0x58, 0x1a, 0x55, 0x1b, 0x4e, 0x0d, 0x6d, 0x32, 0x65, 0x15, 0xfb, 0x9e, 0xd8, 0x75, 0x76, 0x6f,
	0x42, 0xe2, 0x4f, 0x3c, 0x25, 0x35, 0x93, 0x6c, 0x9b, 0x56, 0xbe, 0xc1, 0x5b, 0x65, 0xde, 0x27
};

void pgm_kovshp_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		unsigned short x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= kovshp_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t killbldp_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x32, 0x34, 0x52, 0x44, 0x31, 0x30, 0x35, 0x30, 0x39, 0x30, 0x38,
	0x12, 0xa0, 0xd1, 0x9e, 0xb1, 0x8a, 0xfb, 0x1f, 0x50, 0x51, 0x4b, 0x81, 0x28, 0xda, 0x5f, 0x41,
	0x78, 0x6c, 0x7a, 0xf0, 0xcd, 0x6b, 0x69, 0x14, 0x94, 0x55, 0xb6, 0x42, 0xdf, 0xfe, 0x10, 0x79,
	0x74, 0x08, 0xfa, 0xc0, 0x1c, 0xa5, 0xb4, 0x03, 0x2a, 0x91, 0x67, 0x2b, 0x49, 0x4a, 0x94, 0x7d,
	0x8b, 0x92, 0xbe, 0x35, 0xaf, 0x28, 0x56, 0x63, 0xb3, 0xc2, 0xe8, 0x06, 0x9b, 0x4e, 0x85, 0x66,
	0x7f, 0x6b, 0x70, 0xb7, 0xdb, 0x22, 0x0c, 0xeb, 0x13, 0xe9, 0x06, 0xd7, 0x45, 0xda, 0xbe, 0x8b,
	0x54, 0x30, 0xfc, 0xeb, 0x32, 0x02, 0xd0, 0x92, 0x6d, 0x44, 0xca, 0xe8, 0xfd, 0xfb, 0x5b, 0x81,
	0x4c, 0xc0, 0x8b, 0xb9, 0x87, 0x78, 0xdd, 0x8e, 0x24, 0x52, 0x80, 0xbe, 0xb4, 0x01, 0xb7, 0x21,
	0xeb, 0x3c, 0x8a, 0x49, 0xed, 0x73, 0xae, 0x58, 0xdb, 0xd2, 0xb2, 0x21, 0x9e, 0x7c, 0x6c, 0x82,
	0xf3, 0x01, 0xa3, 0x00, 0xb7, 0x21, 0xfe, 0xa5, 0x75, 0xc4, 0x2d, 0x17, 0x2d, 0x39, 0x56, 0xf9,
	0x67, 0xae, 0xc2, 0x87, 0x79, 0xf1, 0xc8, 0x6d, 0x15, 0x66, 0xfa, 0xe8, 0x16, 0x48, 0x8f, 0x1f,
	0x8b, 0x24, 0x10, 0xc4, 0x04, 0x93, 0x47, 0xe6, 0x1d, 0x37, 0x65, 0x1a, 0x49, 0xf8, 0x72, 0xcb,
	0xe1, 0x80, 0xfa, 0xdd, 0x6d, 0xf5, 0xf6, 0x89, 0x32, 0xf6, 0xf8, 0x75, 0xfc, 0xd8, 0x9b, 0x12,
	0x2d, 0x22, 0x2a, 0x3b, 0x06, 0x46, 0x90, 0x0c, 0x35, 0xa2, 0x80, 0xff, 0xa0, 0xb7, 0xe5, 0x4d,
	0x71, 0xa9, 0x8c, 0x84, 0x62, 0xf7, 0x10, 0x65, 0x4a, 0x7b, 0x06, 0x00, 0xe8, 0xa4, 0x6a, 0x13,
	0xf0, 0xf3, 0x4a, 0x9f, 0x54, 0xb4, 0xb1, 0xcc, 0xd4, 0xff, 0xd6, 0xff, 0xc9, 0xee, 0x86, 0x39
};

void pgm_killbldp_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= killbldp_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


void pgm_svg_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x800000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		src[i] = x;
	}
}

static const unsigned char svgpcb_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x31, 0x37, 0x52, 0x44, 0x31, 0x30, 0x35, 0x30, 0x35, 0x30, 0x34,
	0x75, 0x0b, 0xf1, 0x6b, 0x6d, 0xd7, 0xa8, 0xe7, 0x0c, 0xc5, 0x28, 0x81, 0x1f, 0xcf, 0x30, 0x15,
	0xa8, 0x0d, 0xda, 0x76, 0xf8, 0x7d, 0xd6, 0xe1, 0x0a, 0x11, 0xe3, 0xa4, 0x23, 0xff, 0x8e, 0x0b,
	0xa8, 0x44, 0x2f, 0x8b, 0x3f, 0x7a, 0x21, 0x32, 0x2a, 0xdc, 0x41, 0x4e, 0xe0, 0x97, 0xa9, 0x5d,
	0xed, 0x53, 0xae, 0x35, 0x0b, 0x02, 0x18, 0x74, 0x82, 0xe8, 0xa1, 0x2a, 0xbd, 0xeb, 0xb0, 0xc6,
	0x2e, 0x1d, 0x56, 0x3e, 0x63, 0x87, 0x8a, 0x83, 0x69, 0x38, 0xa1, 0x24, 0x61, 0x8f, 0x11, 0x41,
	0x61, 0xc5, 0x67, 0xb3, 0x8e, 0xbe, 0x85, 0x79, 0x77, 0x10, 0x21, 0x66, 0xb4, 0x54, 0x7b, 0x09,
	0xbf, 0xad, 0x5e, 0xdd, 0x12, 0x97, 0x5a, 0xb2, 0x82, 0xf3, 0x40, 0x5b, 0xdb, 0x4f, 0xde, 0x99,
	0xbd, 0x7a, 0xfc, 0x48, 0xb6, 0x48, 0x97, 0xc8, 0xa1, 0xa2, 0x5c, 0xae, 0x3e, 0xd2, 0x68, 0xac,
	0x13, 0x0d, 0x3f, 0xbe, 0x82, 0x42, 0x0a, 0x97, 0x2c, 0x22, 0x16, 0x4b, 0x85, 0x70, 0x89, 0x3d,
	0xb8, 0x8b, 0x66, 0x4c, 0xbd, 0x39, 0xc4, 0x39, 0xb9, 0xb6, 0x4b, 0x5c, 0x96, 0xfc, 0xef, 0x87,
	0xe3, 0x55, 0xf1, 0x3b, 0xed, 0x1f, 0x13, 0x0a, 0x1f, 0xdf, 0x1a, 0x4c, 0x97, 0x8a, 0x8a, 0x06,
	0x0a, 0x0f, 0x9d, 0x17, 0xda, 0x28, 0x85, 0xa4, 0x75, 0x63, 0xe4, 0xc6, 0xf1, 0x6b, 0x88, 0x73,
	0xe6, 0x9c, 0x6b, 0xaa, 0x8b, 0xc7, 0xea, 0xe3, 0x13, 0x42, 0x46, 0xb6, 0x4a, 0x34, 0x59, 0xca,
	0x6c, 0x1f, 0x99, 0x01, 0x46, 0xae, 0x52, 0xe1, 0x90, 0xc1, 0x1f, 0x46, 0x89, 0xb7, 0xab, 0x46,
	0xd4, 0xfb, 0xa6, 0x65, 0x92, 0xd1, 0x0a, 0xfa, 0xc1, 0x63, 0x90, 0xe1, 0xd5, 0x07, 0x6d, 0x62
};

void pgm_svgpcb_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x800000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT; // ok?
		IGS27_CRYPT2 // ok?
		IGS27_CRYPT3
		IGS27_CRYPT4 // ok?
		IGS27_CRYPT5  // ok?
		IGS27_CRYPT6_ALT // ok?
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= svgpcb_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t py2k2_tab[256] = {
	0x74, 0xe8, 0xa8, 0x64, 0x26, 0x44, 0xa6, 0x9a, 0xa5, 0x69, 0xa2, 0xd3, 0x6d, 0xba, 0xff, 0xf3,
	0xeb, 0x6e, 0xe3, 0x70, 0x72, 0x58, 0x27, 0xd9, 0xe4, 0x9f, 0x50, 0xa2, 0xdd, 0xce, 0x6e, 0xf6,
	0x44, 0x72, 0x0c, 0x7e, 0x4d, 0x41, 0x77, 0x2d, 0x00, 0xad, 0x1a, 0x5f, 0x6b, 0xc0, 0x1d, 0x4e,
	0x4c, 0x72, 0x62, 0x3c, 0x32, 0x28, 0x43, 0xf8, 0x9d, 0x52, 0x05, 0x7e, 0xd1, 0xee, 0x82, 0x61,
	0x3b, 0x3f, 0x77, 0xf3, 0x8f, 0x7e, 0x3f, 0xf1, 0xdf, 0x8f, 0x68, 0x43, 0xd7, 0x68, 0xdf, 0x19,
	0x87, 0xff, 0x74, 0xe5, 0x3f, 0x43, 0x8e, 0x80, 0x0f, 0x7e, 0xdb, 0x32, 0xe8, 0xd1, 0x66, 0x8f,
	0xbe, 0xe2, 0x33, 0x94, 0xc8, 0x32, 0x39, 0xfa, 0xf0, 0x43, 0xde, 0x84, 0x18, 0xd0, 0x6d, 0xd5,
	0x74, 0x98, 0xf8, 0x64, 0xcf, 0x84, 0xc6, 0xea, 0x55, 0x32, 0xe2, 0x38, 0xdd, 0xea, 0xfd, 0x6c,
	0xeb, 0x6e, 0xe3, 0x70, 0xae, 0x38, 0xc7, 0xd9, 0x54, 0x84, 0x10, 0xc1, 0xfd, 0x1e, 0x6e, 0x6d,
	0x37, 0xe0, 0x03, 0x9e, 0x06, 0x36, 0x68, 0x5b, 0xe3, 0xf6, 0x7f, 0x0b, 0x56, 0x79, 0xe0, 0xa8,
	0x98, 0x77, 0xc7, 0x2b, 0xa5, 0x79, 0xff, 0x2f, 0xca, 0x15, 0x71, 0x7e, 0x02, 0xbf, 0x87, 0xb7,
	0x7a, 0x8e, 0xe6, 0x64, 0x32, 0x62, 0x2a, 0xca, 0x23, 0x72, 0x87, 0xb5, 0x0c, 0x02, 0x4b, 0xee,
	0x44, 0x72, 0x9c, 0x7e, 0x5d, 0xc1, 0xa7, 0x1d, 0x30, 0x38, 0xda, 0xc9, 0x5b, 0xd0, 0x11, 0xf9,
	0xb1, 0x72, 0x6c, 0x04, 0x31, 0xc9, 0x50, 0x60, 0x6f, 0xc1, 0xf2, 0xae, 0x00, 0xf4, 0x5d, 0x66,
	0x43, 0x0e, 0x7a, 0xc3, 0x76, 0xae, 0x3c, 0xc2, 0xb7, 0xc9, 0x52, 0xf4, 0x74, 0x51, 0xaf, 0x12,
	0x19, 0xc6, 0x75, 0xe8, 0x6c, 0x54, 0x7e, 0x63, 0xdd, 0xae, 0x07, 0x5a, 0xb7, 0x00, 0xb5, 0x5e
};

void pgm_py2k2_decrypt(running_machine &machine) // and ddpdoj/ddpdojbl
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT3
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= py2k2_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


static const unsigned char ket_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x34, 0x52, 0x44, 0x31, 0x30, 0x32, 0x31, 0x30, 0x31, 0x35,
	0x7c, 0x49, 0x27, 0xa5, 0xff, 0xf6, 0x98, 0x2d, 0x0f, 0x3d, 0x12, 0x23, 0xe2, 0x30, 0x50, 0xcf,
	0xf1, 0x82, 0xf0, 0xce, 0x48, 0x44, 0x5b, 0xf3, 0x0d, 0xdf, 0xf8, 0x5d, 0x50, 0x53, 0x91, 0xd9,
	0x12, 0xaf, 0x05, 0x7a, 0x98, 0xd0, 0x2f, 0x76, 0xf1, 0x5d, 0x17, 0x44, 0xc5, 0x03, 0x58, 0xf4,
	0x61, 0xee, 0xd1, 0xce, 0x00, 0x88, 0x90, 0x2e, 0x5c, 0x76, 0xfb, 0x9f, 0x75, 0xcf, 0x40, 0x37,
	0xa1, 0x9f, 0x00, 0x32, 0xd5, 0x9c, 0x37, 0xd2, 0x32, 0x27, 0x6f, 0x76, 0xd3, 0x86, 0x25, 0xf9,
	0xd6, 0x60, 0x7b, 0x4e, 0xa9, 0x7a, 0x20, 0x59, 0x96, 0xb1, 0x7d, 0x10, 0x92, 0x37, 0x22, 0xd2,
	0x42, 0x12, 0x6f, 0x07, 0x4f, 0xd2, 0x87, 0xfa, 0xeb, 0x92, 0x71, 0xf3, 0xa4, 0x31, 0x91, 0x98,
	0x68, 0xd2, 0x47, 0x86, 0xda, 0x92, 0xe5, 0x2b, 0xd4, 0x89, 0xd7, 0xe7, 0x3d, 0x03, 0x0d, 0x63,
	0x0c, 0x00, 0xac, 0x31, 0x9d, 0xe9, 0xf6, 0xa5, 0x34, 0x95, 0x77, 0xf2, 0xcf, 0x7c, 0x72, 0x89,
	0x31, 0x3a, 0x8b, 0xae, 0x2b, 0x47, 0xb6, 0x5d, 0x2d, 0xf5, 0x5f, 0x5c, 0x0e, 0xab, 0xdb, 0xa1,
	0x18, 0x60, 0x0e, 0xe6, 0x58, 0x5b, 0x5e, 0x8b, 0x24, 0x29, 0xd8, 0xac, 0xed, 0xdf, 0xa2, 0x83,
	0x46, 0x91, 0xa1, 0xff, 0x35, 0x13, 0x6a, 0xa5, 0xba, 0xef, 0x6e, 0xa8, 0x9e, 0xa6, 0x62, 0x44,
	0x7e, 0x2c, 0xed, 0x60, 0x17, 0x9e, 0x96, 0x64, 0xd3, 0x46, 0xec, 0x58, 0x95, 0xd1, 0xf7, 0x3e,
	0xc2, 0xcf, 0xdf, 0xb0, 0x90, 0x6c, 0xdb, 0xbe, 0x93, 0x6d, 0x5d, 0x02, 0x85, 0x6e, 0x7c, 0x05,
	0x55, 0x5a, 0xa1, 0xd7, 0x73, 0x2b, 0x76, 0xe9, 0x5b, 0xe4, 0x0c, 0x2e, 0x60, 0xcb, 0x4b, 0x72
};

void pgm_ket_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT


		x ^= ket_tab[i & 0xff] << 8;

		src[i] = x;
	}
}

static const unsigned char espgal_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x37, 0x52, 0x44, 0x31, 0x30, 0x33, 0x30, 0x39, 0x30, 0x39,
	0xa7, 0xf1, 0x0a, 0xca, 0x69, 0xb2, 0xce, 0x86, 0xec, 0x3d, 0xa2, 0x5a, 0x03, 0xe9, 0xbf, 0xba,
	0xf7, 0xd5, 0xec, 0x68, 0x03, 0x90, 0x15, 0xcc, 0x0d, 0x08, 0x2d, 0x76, 0xa5, 0xb5, 0x41, 0xf1,
	0x43, 0x06, 0xdd, 0xcb, 0xbd, 0x0c, 0xa4, 0xe2, 0x08, 0x65, 0x2a, 0xf0, 0x30, 0x6b, 0x15, 0x59,
	0x99, 0x9e, 0x75, 0x35, 0x77, 0x4f, 0x60, 0x99, 0x8c, 0x8f, 0xd2, 0x2b, 0x21, 0x57, 0xc3, 0xe5,
	0x48, 0xf9, 0x8a, 0x29, 0x50, 0xc6, 0x71, 0x06, 0x89, 0x01, 0x9a, 0xc9, 0x39, 0x04, 0x12, 0xc8,
	0xdf, 0xb1, 0x33, 0x6b, 0xa7, 0x1c, 0x3f, 0x7b, 0x2d, 0x76, 0x3a, 0xaf, 0x76, 0x3d, 0x08, 0x74,
	0x2c, 0xa2, 0xc8, 0xfd, 0x1a, 0x3a, 0x6f, 0x8b, 0xe8, 0xe9, 0xa9, 0xfe, 0x17, 0x0c, 0xed, 0x9d,
	0x40, 0xe6, 0xdf, 0x22, 0x89, 0x4d, 0xea, 0x09, 0x68, 0x96, 0x1e, 0x1a, 0x9c, 0xbd, 0x47, 0x35,
	0x68, 0xd9, 0x4f, 0x5e, 0x12, 0xbf, 0xd6, 0x09, 0x9d, 0xf6, 0x0f, 0xa7, 0xc2, 0xdb, 0xde, 0x70,
	0x35, 0x15, 0x2f, 0x73, 0x16, 0x3c, 0x9a, 0xdc, 0xb5, 0xc5, 0x35, 0x86, 0x8a, 0x31, 0xb8, 0xc1,
	0x74, 0x76, 0xd7, 0x65, 0x32, 0xad, 0xdc, 0x17, 0x1f, 0xfe, 0x85, 0xda, 0x32, 0xc9, 0x1d, 0xda,
	0x36, 0x16, 0xde, 0x76, 0x45, 0x3f, 0x85, 0x8c, 0x8b, 0xdc, 0x37, 0x08, 0x39, 0xef, 0x94, 0xaf,
	0xc8, 0x51, 0x19, 0x29, 0x70, 0x5d, 0xbb, 0x4e, 0xe8, 0xdb, 0xc2, 0xb2, 0x5f, 0x2e, 0xe3, 0x73,
	0xba, 0xc2, 0xa1, 0x42, 0x10, 0xb0, 0xe5, 0xb0, 0x64, 0xb4, 0xdc, 0xbb, 0xa1, 0x51, 0x12, 0x98,
	0xdc, 0x43, 0xcc, 0xc3, 0xc5, 0x25, 0xab, 0x45, 0x6e, 0x63, 0x7e, 0x45, 0x40, 0x63, 0x67, 0xd2
};

void pgm_espgal_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT3
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= espgal_tab[i & 0xff] << 8;

		src[i] = x;
	}
}



static const uint8_t happy6in1_tab[256] = { // IGS0008RD1031215
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x38, 0x52, 0x44, 0x31, 0x30, 0x33, 0x31, 0x32, 0x31, 0x35,
	0x14, 0xd6, 0x37, 0x5c, 0x5e, 0xc3, 0xd3, 0x62, 0x96, 0x3d, 0xfb, 0x47, 0xf0, 0xcb, 0xbf, 0xb0,
	0x60, 0xa1, 0xc2, 0x3d, 0x90, 0xd0, 0x58, 0x56, 0x22, 0xac, 0xdd, 0x39, 0x27, 0x7e, 0x58, 0x44,
	0xe0, 0x6b, 0x51, 0x80, 0xb4, 0xa4, 0xf0, 0x6f, 0x71, 0xd0, 0x57, 0x18, 0xc7, 0xb6, 0x41, 0x50,
	0x02, 0x2f, 0xdb, 0x4a, 0x08, 0x4b, 0xe3, 0x62, 0x92, 0xc3, 0xff, 0x26, 0xaf, 0x9f, 0x60, 0xa5,
	0x76, 0x28, 0x97, 0xfd, 0x0b, 0x10, 0xb7, 0x1f, 0xd5, 0xe0, 0xac, 0xe6, 0xfd, 0xa3, 0xdb, 0x58,
	0x2a, 0xd1, 0xfc, 0x3b, 0x7c, 0x7e, 0x34, 0xdc, 0xc7, 0xc4, 0x76, 0x1b, 0x11, 0x6d, 0x1b, 0xbb,
	0x4e, 0xe5, 0xc0, 0xe8, 0x5a, 0x60, 0x60, 0x0a, 0x38, 0x47, 0xb3, 0xc9, 0x89, 0xe9, 0xc6, 0x61,
	0x50, 0x5f, 0xdb, 0x28, 0xe5, 0xc0, 0x83, 0x5c, 0x37, 0x86, 0xfa, 0x32, 0x46, 0x40, 0xc3, 0x1d,
	0xdf, 0x7a, 0x85, 0x5c, 0x9a, 0xea, 0x24, 0xc7, 0x12, 0xdc, 0x23, 0xda, 0x65, 0xdf, 0x39, 0x02,
	0xeb, 0xb1, 0x32, 0x28, 0x3a, 0x69, 0x09, 0x7c, 0x5a, 0xe3, 0x44, 0x83, 0x45, 0x71, 0x8f, 0x64,
	0xa3, 0xbf, 0x9c, 0x6f, 0xc4, 0x07, 0x3a, 0xee, 0xdd, 0x77, 0xb4, 0x31, 0x87, 0xdf, 0x6d, 0xd4,
	0x75, 0x9f, 0xb9, 0x53, 0x75, 0xd0, 0xfe, 0xd1, 0xaa, 0xb2, 0x0b, 0x25, 0x08, 0x56, 0xb8, 0x27,
	0x10, 0x8c, 0xbf, 0x39, 0xce, 0x0f, 0xdb, 0x18, 0x10, 0xf0, 0x1f, 0xe5, 0xe8, 0x40, 0x98, 0x6f,
	0x64, 0x02, 0x27, 0xc3, 0x8c, 0x4f, 0x98, 0xf6, 0x9d, 0xcb, 0x07, 0x31, 0x85, 0x48, 0x75, 0xff,
	0x9f, 0xba, 0xa6, 0xd3, 0xb0, 0x5b, 0x3d, 0xdd, 0x22, 0x1f, 0x1b, 0x0e, 0x7f, 0x5a, 0xf4, 0x6a
};

void pgm_happy6_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= happy6in1_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t sdwx_tab[] =
{
	0x49, 0x47, 0x53, 0x30, 0x30, 0x35, 0x35, 0x52, 0x44, 0x34, 0x30, 0x32, 0x30, 0x36, 0x32, 0x31,
	0x8a, 0xbb, 0x20, 0x67, 0x97, 0xa5, 0x20, 0x45, 0x6b, 0xc0, 0xe8, 0x0c, 0x80, 0xfb, 0x49, 0xaa,
	0x1e, 0xac, 0x29, 0xf2, 0xb9, 0x9f, 0x01, 0x4a, 0x8d, 0x5f, 0x95, 0x96, 0x78, 0xc3, 0xf6, 0x65,
	0x17, 0xbd, 0xb6, 0x5b, 0x25, 0x5f, 0x6b, 0xde, 0x10, 0x2e, 0x67, 0x05, 0xdc, 0xac, 0xb6, 0xbd,
	0x3d, 0x20, 0x58, 0x3d, 0xf0, 0xa8, 0xc0, 0xad, 0x5b, 0x82, 0x8d, 0x12, 0x65, 0x97, 0x87, 0x7d,
	0x97, 0x49, 0xdd, 0x74, 0x74, 0x7e, 0x9d, 0xa1, 0x15, 0xed, 0x75, 0xb9, 0x09, 0xa8, 0xa8, 0xb0,
	0x6b, 0xea, 0x54, 0x1b, 0x45, 0x23, 0xe2, 0xe5, 0x25, 0x42, 0xce, 0x36, 0xfe, 0x42, 0x99, 0xa0,
	0x41, 0xf8, 0x0b, 0x8c, 0x3c, 0x1b, 0xae, 0xe4, 0xb2, 0x94, 0x87, 0x02, 0xbc, 0x08, 0x17, 0xd9,
	0xe0, 0xa4, 0x93, 0x63, 0x6f, 0x28, 0x5f, 0x4a, 0x24, 0x36, 0xd1, 0xda, 0xfa, 0xdd, 0x23, 0x26,
	0x4e, 0x61, 0xb9, 0x7a, 0x36, 0x4d, 0x95, 0x01, 0x20, 0xbc, 0x18, 0xb7, 0xaf, 0xe4, 0xfb, 0x92,
	0xd2, 0xe3, 0x8e, 0xec, 0x26, 0xce, 0x2f, 0x34, 0x8f, 0xf7, 0x0d, 0xd6, 0x11, 0x7f, 0x1f, 0x68,
	0xf4, 0x1d, 0x5f, 0x16, 0x19, 0x2d, 0x4c, 0x4f, 0x96, 0xfc, 0x9f, 0xb0, 0x99, 0x53, 0x4c, 0x32,
	0x7b, 0x41, 0xbc, 0x90, 0x23, 0x2e, 0x4a, 0xfc, 0x9e, 0x1d, 0xfc, 0x02, 0xfc, 0x41, 0x83, 0xbc,
	0x6d, 0xc4, 0x75, 0x37, 0x9d, 0xd3, 0xc9, 0x26, 0x4d, 0xed, 0x93, 0xc6, 0x32, 0x6d, 0x02, 0x11,
	0x12, 0x56, 0x97, 0x26, 0x1d, 0x5f, 0xa7, 0xf8, 0x89, 0x3f, 0x14, 0x36, 0x72, 0x3b, 0x48, 0x7b,
	0xf1, 0xed, 0x72, 0xb7, 0x7a, 0x56, 0x05, 0xde, 0x7b, 0x27, 0x6d, 0xcf, 0x33, 0x4c, 0x14, 0x86
};


void sdwx_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT2
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= sdwx_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}


//IGS FOR V101US 2006 08 23-001

static const uint8_t hauntedh_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x32, 0x35, 0x34, 0x52, 0x44, 0x34, 0x30, 0x36, 0x30, 0x35, 0x32, 0x36,
	0x6c, 0x65, 0x33, 0xfd, 0x7a, 0x71, 0x3d, 0xb8, 0x07, 0xf1, 0x86, 0x96, 0x19, 0x5a, 0xa2, 0x05,
	0x49, 0xb1, 0xed, 0x2e, 0x7c, 0x7a, 0x65, 0x8b, 0xe1, 0xe3, 0xc8, 0xaa, 0x2b, 0x32, 0xee, 0x3f,
	0x10, 0x6c, 0x69, 0x70, 0x02, 0x47, 0x5b, 0x5d, 0x2d, 0x52, 0x97, 0xef, 0xb1, 0x63, 0xfb, 0xe3,
	0x21, 0x41, 0x0c, 0x17, 0x3c, 0x93, 0xd4, 0x13, 0xeb, 0x08, 0xf9, 0xdb, 0x7a, 0xc8, 0x1e, 0xf4,
	0x1b, 0x1b, 0x7f, 0xb4, 0x98, 0x59, 0xc8, 0xcf, 0x58, 0x12, 0x36, 0x1f, 0x96, 0x7d, 0xf0, 0xb3,
	0xdc, 0x26, 0xa8, 0x1c, 0xc6, 0xd4, 0x6e, 0xf3, 0xf5, 0xb9, 0xd4, 0xaf, 0x52, 0xdd, 0x48, 0xa5,
	0x85, 0xcc, 0xad, 0x60, 0xb4, 0x7f, 0x3c, 0x24, 0x80, 0x88, 0x9b, 0xbd, 0x3e, 0x82, 0x3b, 0x8d,
	0x73, 0xb8, 0xf7, 0xd5, 0x92, 0x15, 0xeb, 0x43, 0xf9, 0x4c, 0x91, 0xbd, 0x29, 0x48, 0x22, 0x6d,
	0x45, 0xd6, 0x2c, 0x0d, 0xce, 0x91, 0x70, 0x74, 0x9d, 0x0e, 0xfe, 0x62, 0x22, 0x49, 0x94, 0x88,
	0xdb, 0x50, 0x33, 0xdb, 0x18, 0x2e, 0x03, 0x1b, 0xed, 0x1a, 0x69, 0x9e, 0x78, 0xe1, 0x66, 0x62,
	0x54, 0x91, 0x33, 0x52, 0x5e, 0x67, 0x1b, 0xd9, 0xa7, 0xfb, 0x98, 0xa5, 0xba, 0xaa, 0xb1, 0xbd,
	0x0f, 0x44, 0x93, 0xc6, 0xcf, 0xf7, 0x6f, 0x91, 0xca, 0x7b, 0x93, 0xea, 0xb6, 0x7f, 0xcc, 0x9c,
	0xab, 0x54, 0xfb, 0xc8, 0xdb, 0xd9, 0xf5, 0x68, 0x96, 0xa7, 0xa1, 0x1f, 0x7d, 0x7d, 0x4c, 0x43,
	0x06, 0xed, 0x50, 0x2d, 0x30, 0x48, 0xe6, 0xc0, 0x88, 0xc8, 0x48, 0x38, 0x5d, 0xfc, 0x0a, 0x35,
	0x3f, 0x79, 0xba, 0x07, 0xbe, 0xbf, 0xb7, 0x3b, 0x61, 0x69, 0x4f, 0x67, 0xe5, 0x9a, 0x1d, 0x33
};

void hauntedh_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x080000;

	for (int i = 0; i < rom_size / 2; i++)
	{
			uint16_t x = src[i];

		IGS27_CRYPT1
	//  IGS27_CRYPT2
	//  IGS27_CRYPT3
		IGS27_CRYPT4_ALT
	//  IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		x ^= hauntedh_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}


static const uint8_t chessc2_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x38, 0x32, 0x52, 0x44, 0x34, 0x30, 0x32, 0x31, 0x32, 0x31, 0x31,
	0x28, 0xca, 0x9c, 0xad, 0xbb, 0x2d, 0xf0, 0x41, 0x6e, 0xce, 0xad, 0x73, 0xae, 0x1c, 0xd1, 0x14,
	0x6f, 0x9a, 0x75, 0x18, 0xa8, 0x91, 0x68, 0xe4, 0x09, 0xf4, 0x0f, 0xd7, 0xff, 0x93, 0x7d, 0x1b,
	0xeb, 0x84, 0xce, 0xad, 0x9e, 0xcf, 0xc9, 0xab, 0x18, 0x59, 0xb6, 0xde, 0x82, 0x13, 0x7c, 0x88,
	0x69, 0x63, 0xff, 0x6f, 0x3c, 0xd2, 0xb9, 0x29, 0x09, 0xf8, 0x97, 0xaa, 0x74, 0xa5, 0x16, 0x0d,
	0xf9, 0x51, 0x9e, 0x9f, 0x63, 0xc6, 0x1e, 0x32, 0x8c, 0x0c, 0xe9, 0xa0, 0x56, 0x95, 0xd1, 0x9d,
	0xea, 0xa9, 0x82, 0xc3, 0x30, 0x15, 0x21, 0xd8, 0x8f, 0x10, 0x25, 0x61, 0xe6, 0x6d, 0x75, 0x6d,
	0xcb, 0x08, 0xc3, 0x9b, 0x03, 0x6a, 0x28, 0x6d, 0x42, 0xbf, 0x00, 0xd2, 0x24, 0xfa, 0x08, 0xee,
	0x6b, 0x46, 0xb7, 0x2c, 0x7b, 0xb0, 0xda, 0x86, 0x13, 0x14, 0x73, 0x14, 0x4d, 0x45, 0xd3, 0xd4,
	0xd9, 0x80, 0xf5, 0xb8, 0x76, 0x13, 0x1e, 0xf6, 0xb1, 0x4a, 0xb3, 0x8b, 0xe2, 0x9a, 0x5a, 0x11,
	0x64, 0x11, 0x55, 0xc3, 0x14, 0xfd, 0x1b, 0xce, 0x0c, 0xdc, 0x38, 0xda, 0xa1, 0x84, 0x66, 0xd9,
	0x9b, 0x93, 0xed, 0x0f, 0xb4, 0x19, 0x38, 0x62, 0x53, 0x85, 0xb9, 0xe5, 0x89, 0xcd, 0xfe, 0x9e,
	0x4d, 0xe2, 0x14, 0x9f, 0xf4, 0x53, 0x1c, 0x46, 0xf4, 0x40, 0x2c, 0xcc, 0xda, 0x82, 0x69, 0x15,
	0x88, 0x18, 0x62, 0xb7, 0xb4, 0xd5, 0xaf, 0x4b, 0x9e, 0x48, 0xca, 0xf4, 0x11, 0xec, 0x2d, 0x2c,
	0x9d, 0x91, 0xad, 0xda, 0x13, 0x0a, 0x16, 0x86, 0x41, 0x18, 0x08, 0x01, 0xef, 0x97, 0x11, 0x1f,
	0x1a, 0xe7, 0x0c, 0xc9, 0x6d, 0x9d, 0xb9, 0x49, 0x0b, 0x6b, 0x9e, 0xd4, 0x72, 0x4d, 0x1d, 0x59
};

void chessc2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6 // correct??
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= chessc2_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}


// IGS TARZAN  CHINA ONLINE
//V104CN
static const uint8_t klxyj_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x30, 0x38, 0x52, 0x44, 0x34, 0x30, 0x31, 0x30, 0x39, 0x32, 0x34, // IGS0008RD4010924
	0x3f, 0x0f, 0x66, 0x9a, 0xbf, 0x0d, 0x06, 0x55, 0x09, 0x01, 0xeb, 0x72, 0xeb, 0x9b, 0x89, 0xfa,
	0x24, 0xd1, 0x5d, 0xca, 0xe6, 0x8a, 0x8c, 0xe0, 0x92, 0x8d, 0xbf, 0xe4, 0xaf, 0xaa, 0x3e, 0xfa,
	0x2b, 0x27, 0x4b, 0xc7, 0xd6, 0x6d, 0xc1, 0xc2, 0x1c, 0xf4, 0xed, 0xbd, 0x03, 0x6c, 0xad, 0xb3,
	0x65, 0x2d, 0xc7, 0xd3, 0x6e, 0xe0, 0x8c, 0xce, 0x59, 0x6f, 0xae, 0x5e, 0x66, 0x2b, 0x5e, 0x17,
	0x20, 0x3d, 0xa9, 0x72, 0xcd, 0x4f, 0x14, 0x17, 0x35, 0x7b, 0x77, 0x6b, 0x98, 0x73, 0x17, 0x5a,
	0xea, 0xf2, 0x07, 0x66, 0x51, 0x64, 0xc1, 0xf0, 0xe2, 0xd1, 0x00, 0xc6, 0x97, 0x0f, 0xe0, 0xee,
	0x94, 0x28, 0x39, 0xb2, 0x9b, 0x0a, 0x38, 0xed, 0xcc, 0x6e, 0x40, 0x94, 0xa2, 0x0a, 0x00, 0x88,
	0x2b, 0xfa, 0xd5, 0x9a, 0x87, 0x6c, 0x62, 0xdf, 0xa4, 0x8b, 0x6d, 0x37, 0x38, 0xae, 0xfd, 0x18,
	0xff, 0xc2, 0xb2, 0xa0, 0x37, 0xf5, 0x64, 0xdb, 0x59, 0xa5, 0x00, 0x51, 0x19, 0x88, 0x9f, 0xd4,
	0xa0, 0x1c, 0xe7, 0x88, 0x08, 0x51, 0xa7, 0x33, 0x19, 0x75, 0xae, 0xc7, 0x42, 0x61, 0xec, 0x2d,
	0xdb, 0xe2, 0xcc, 0x54, 0x9a, 0x6a, 0xd1, 0x7a, 0x53, 0xf8, 0x6f, 0xba, 0xf4, 0x45, 0x2c, 0xd7,
	0xc0, 0x30, 0xf7, 0x47, 0xcc, 0x6b, 0xc8, 0x83, 0xb7, 0x67, 0x7a, 0x8e, 0xad, 0x7e, 0xe5, 0xc4,
	0x9f, 0x60, 0x40, 0xe5, 0xbc, 0xc0, 0xb5, 0x61, 0x33, 0x3f, 0x46, 0xe6, 0x2d, 0x98, 0xdf, 0x28,
	0x05, 0x0e, 0xbc, 0xf0, 0xca, 0x13, 0xfe, 0x68, 0xf7, 0x3a, 0x89, 0xa5, 0x71, 0x5f, 0x21, 0x76,
	0xc2, 0x14, 0xc5, 0x6c, 0x95, 0x4f, 0x4a, 0x2a, 0x71, 0x52, 0x3c, 0xee, 0xba, 0xdb, 0xf1, 0x60
};

void klxyj_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= klxyj_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t zhongguo_tab[0x100] = {
	0x68, 0x56, 0xc2, 0x54, 0xa2, 0x8c, 0x7b, 0x4f, 0x37, 0xac, 0x60, 0xf8, 0x24, 0xdf, 0x3e, 0x6b,
	0xe2, 0x89, 0x3d, 0xf3, 0x31, 0x83, 0x4a, 0x65, 0x27, 0x98, 0xc5, 0xbf, 0x78, 0x3e, 0x6c, 0x02,
	0x07, 0x96, 0x88, 0x4d, 0xae, 0xa6, 0x56, 0x3a, 0x4a, 0xd5, 0xb8, 0x7e, 0x0b, 0xa7, 0x1d, 0xbc,
	0xfa, 0xba, 0xad, 0xa9, 0xcb, 0x02, 0xba, 0x66, 0xe5, 0x41, 0x63, 0x1a, 0xb0, 0xca, 0x8a, 0xcf,
	0x1e, 0x76, 0xf9, 0x8f, 0x7c, 0xe7, 0xd0, 0xc0, 0x7b, 0xfc, 0x32, 0xbc, 0x7a, 0x95, 0x2f, 0xb4,
	0x16, 0x88, 0xf5, 0xc6, 0xf4, 0xe3, 0x33, 0x5d, 0xce, 0x65, 0xce, 0xca, 0xbc, 0x37, 0xc8, 0x20,
	0xc5, 0xef, 0x6d, 0x55, 0xa6, 0xc7, 0xbf, 0x96, 0xe1, 0x1a, 0x24, 0xea, 0x09, 0x20, 0x4e, 0x0b,
	0x4d, 0xeb, 0x6b, 0x82, 0x44, 0xa1, 0x8f, 0x01, 0xf8, 0xfb, 0x5e, 0x05, 0x35, 0xff, 0xfe, 0xac,
	0x13, 0xf9, 0x3c, 0xd4, 0xc1, 0xc0, 0xfd, 0x76, 0x95, 0x27, 0xe7, 0x41, 0x52, 0xc1, 0x51, 0x7a,
	0xb8, 0xda, 0x69, 0x13, 0x52, 0xb3, 0xa4, 0x0b, 0x7b, 0xfd, 0x6b, 0x05, 0xb2, 0x98, 0x04, 0x2c,
	0x20, 0x8c, 0xbe, 0x46, 0x68, 0x48, 0x60, 0x17, 0xae, 0x1b, 0xd4, 0xf8, 0xea, 0xf1, 0x10, 0xb8,
	0x6f, 0x4f, 0x45, 0xb3, 0xb6, 0x90, 0x4c, 0x31, 0x70, 0x61, 0x4d, 0x02, 0xcc, 0x7b, 0xb1, 0x57,
	0x06, 0xa0, 0x4b, 0xe2, 0x31, 0xd9, 0xc2, 0x31, 0x45, 0xee, 0x42, 0x48, 0x6b, 0x26, 0x63, 0x7e,
	0x89, 0x40, 0x59, 0x9a, 0x09, 0xb1, 0x5e, 0x2d, 0xef, 0x20, 0x5c, 0x32, 0x1b, 0x20, 0xdf, 0xe5,
	0xda, 0x2d, 0x3b, 0xe1, 0xb4, 0xe9, 0xfa, 0x7d, 0x71, 0x97, 0x88, 0x68, 0x6d, 0xd8, 0x22, 0x82,
	0x1e, 0xa6, 0xfc, 0xfe, 0xe3, 0x8e, 0xb1, 0xb7, 0x0f, 0x32, 0xf1, 0xcf, 0x36, 0xfe, 0x65, 0x8e
};

void zhongguo_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= zhongguo_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// these titles are listed in the ROM
// IGS CASTLE KING 2001 USA ONLINE
// IGS GONE FISHIN II USA

static const uint8_t gonefsh2_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x38, 0x32, 0x52, 0x44, 0x34, 0x30, 0x32, 0x31, 0x32, 0x31, 0x31,
	0x28, 0xca, 0x9c, 0xad, 0xbb, 0x2d, 0xf0, 0x41, 0x6e, 0xce, 0xad, 0x73, 0xae, 0x1c, 0xd1, 0x14,
	0x6f, 0x9a, 0x75, 0x18, 0xa8, 0x91, 0x68, 0xe4, 0x09, 0xf4, 0x0f, 0xd7, 0xff, 0x93, 0x7d, 0x1b,
	0xeb, 0x84, 0xce, 0xad, 0x9e, 0xcf, 0xc9, 0xab, 0x18, 0x59, 0xb6, 0xde, 0x82, 0x13, 0x7c, 0x88,
	0x69, 0x63, 0xff, 0x6f, 0x3c, 0xd2, 0xb9, 0x29, 0x09, 0xf8, 0x97, 0xaa, 0x74, 0xa5, 0x16, 0x0d,
	0xf9, 0x51, 0x9e, 0x9f, 0x63, 0xc6, 0x1e, 0x32, 0x8c, 0x0c, 0xe9, 0xa0, 0x56, 0x95, 0xd1, 0x9d,
	0xea, 0xa9, 0x82, 0xc3, 0x30, 0x15, 0x21, 0xd8, 0x8f, 0x10, 0x25, 0x61, 0xe6, 0x6d, 0x75, 0x6d,
	0xcb, 0x08, 0xc3, 0x9b, 0x03, 0x6a, 0x28, 0x6d, 0x42, 0xbf, 0x00, 0xd2, 0x24, 0xfa, 0x08, 0xee,
	0x6b, 0x46, 0xb7, 0x2c, 0x7b, 0xb0, 0xda, 0x86, 0x13, 0x14, 0x73, 0x14, 0x4d, 0x45, 0xd3, 0xd4,
	0xd9, 0x80, 0xf5, 0xb8, 0x76, 0x13, 0x1e, 0xf6, 0xb1, 0x4a, 0xb3, 0x8b, 0xe2, 0x9a, 0x5a, 0x11,
	0x64, 0x11, 0x55, 0xc3, 0x14, 0xfd, 0x1b, 0xce, 0x0c, 0xdc, 0x38, 0xda, 0xa1, 0x84, 0x66, 0xd9,
	0x9b, 0x93, 0xed, 0x0f, 0xb4, 0x19, 0x38, 0x62, 0x53, 0x85, 0xb9, 0xe4, 0x89, 0xcd, 0xfe, 0x9e,
	0x4d, 0xe2, 0x14, 0x9f, 0xf4, 0x53, 0x1c, 0x46, 0xf4, 0x40, 0x2c, 0xcc, 0xda, 0x82, 0x69, 0x15,
	0x88, 0x18, 0x62, 0xb7, 0xb4, 0xd5, 0xaf, 0x4b, 0x9e, 0x48, 0xca, 0xf4, 0x11, 0xec, 0x2d, 0x2e,
	0x9d, 0x91, 0xad, 0xda, 0x13, 0x0a, 0x16, 0x86, 0x41, 0x18, 0x08, 0x01, 0xef, 0x97, 0x11, 0x1f,
	0x1a, 0xe7, 0x0c, 0xc9, 0x6f, 0x9d, 0xb9, 0x49, 0x0b, 0x6b, 0x9e, 0xd4, 0x72, 0x4d, 0x1d, 0x59
};

void gonefsh2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= gonefsh2_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t sddz_tab[0x100] = {
	0xd7, 0x5f, 0xc8, 0x5b, 0xbf, 0xfa, 0x61, 0x0c, 0xdd, 0xab, 0xc3, 0x26, 0x3b, 0xd8, 0xbc, 0x1e,
	0xa6, 0x4b, 0xcf, 0xdc, 0x12, 0x61, 0x70, 0xd2, 0x76, 0xef, 0x44, 0xe8, 0x9a, 0x65, 0xf5, 0x13,
	0xb3, 0x1e, 0xcd, 0x00, 0xdf, 0x30, 0xa7, 0x9d, 0xb7, 0x49, 0xdf, 0xf6, 0xa7, 0x0b, 0x1a, 0x58,
	0x2a, 0xea, 0x0e, 0x92, 0x39, 0x96, 0x25, 0x5d, 0x1b, 0xda, 0x27, 0x63, 0x4f, 0xe0, 0xda, 0x04,
	0x7e, 0x10, 0x16, 0xbc, 0xff, 0x52, 0x73, 0x4a, 0x82, 0xd4, 0xba, 0xd8, 0x13, 0xb3, 0xe9, 0xfd,
	0x9a, 0x6c, 0x6c, 0xd6, 0xef, 0x63, 0x9d, 0xa2, 0xf6, 0x26, 0x94, 0x55, 0x25, 0x93, 0x21, 0x26,
	0x2a, 0xbf, 0x4b, 0xc8, 0x22, 0x18, 0x2d, 0x5d, 0xb6, 0x02, 0xba, 0x70, 0xcd, 0xae, 0xd5, 0xed,
	0x5f, 0x48, 0xca, 0xd5, 0x3d, 0x41, 0xd0, 0xee, 0xd7, 0xd9, 0xba, 0x01, 0x6c, 0x34, 0x58, 0x33,
	0xde, 0x44, 0x6d, 0xee, 0x95, 0x5c, 0x0f, 0x97, 0x98, 0xdc, 0xf0, 0x76, 0xc5, 0x73, 0x46, 0x51,
	0x1c, 0xe5, 0xe0, 0xda, 0x4b, 0x38, 0xb6, 0x08, 0x20, 0xfd, 0xa2, 0xbc, 0x0e, 0xcc, 0xa9, 0xac,
	0x84, 0xb8, 0x42, 0x5c, 0x5b, 0x14, 0x4f, 0xc3, 0xb0, 0x2b, 0xbd, 0x68, 0x42, 0xbc, 0x9c, 0x01,
	0x7e, 0xec, 0xd6, 0x3b, 0x20, 0x10, 0x07, 0x78, 0xdf, 0x18, 0xa5, 0xa2, 0x15, 0xa2, 0x7a, 0xa0,
	0x7c, 0xf1, 0xcb, 0x1a, 0x7d, 0x69, 0x47, 0xe9, 0xd9, 0xf6, 0xf7, 0x78, 0x41, 0x8f, 0x0f, 0x24,
	0xed, 0xb5, 0x04, 0x39, 0x09, 0xf0, 0x7b, 0x06, 0x14, 0xb4, 0xc1, 0xe2, 0xc5, 0xa0, 0x2d, 0x75,
	0xb9, 0xb7, 0xa8, 0x81, 0x03, 0x13, 0xce, 0x0e, 0x47, 0x45, 0x0f, 0xfe, 0xe5, 0x24, 0xf7, 0x77,
	0x4c, 0x87, 0xd2, 0x04, 0xe9, 0xc1, 0xea, 0x7b, 0xbe, 0x58, 0x4f, 0x35, 0x27, 0xfc, 0xf1, 0x11
};

void sddz_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT;
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= sddz_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t lhzb3_tab[0x100] = {
	0x13, 0x45, 0x21, 0xa1, 0x69, 0x9a, 0x05, 0xda, 0x7d, 0x10, 0xda, 0x7f, 0x34, 0x9f, 0xf3, 0x64,
	0x35, 0xf9, 0x16, 0x1c, 0xd4, 0x60, 0x02, 0xf3, 0x42, 0xc7, 0x42, 0x29, 0xf3, 0x2c, 0x31, 0x71,
	0x50, 0x35, 0x4e, 0xaa, 0x9f, 0x09, 0xc3, 0xdd, 0x2f, 0x72, 0x30, 0x77, 0xc7, 0x30, 0xbc, 0xc8,
	0x51, 0xa0, 0x5e, 0xbb, 0xc3, 0x8e, 0x69, 0xd7, 0x4f, 0x57, 0x56, 0x77, 0xcb, 0x43, 0xd6, 0x46,
	0x62, 0x21, 0x18, 0xfd, 0x7d, 0x24, 0x58, 0x61, 0xec, 0xfe, 0xa9, 0x77, 0x59, 0x3b, 0x03, 0x0a,
	0xef, 0xdf, 0x90, 0x60, 0x46, 0x43, 0x33, 0x38, 0x92, 0x2d, 0x5a, 0x08, 0x0d, 0x2f, 0x05, 0x75,
	0x3e, 0x60, 0x87, 0x22, 0xb7, 0xbf, 0xd6, 0xf9, 0x17, 0x86, 0xea, 0x02, 0xbe, 0x23, 0xba, 0xd3,
	0xdd, 0x0d, 0x3e, 0x8c, 0x65, 0xa0, 0xf8, 0xd8, 0x2f, 0x35, 0xc6, 0x26, 0x6c, 0x81, 0xe6, 0x29,
	0x50, 0x30, 0x4a, 0x8e, 0xfa, 0xc2, 0x1e, 0xfd, 0xa7, 0xa5, 0x98, 0x53, 0x18, 0x94, 0xff, 0x1d,
	0x41, 0x2f, 0xff, 0x58, 0x33, 0xdc, 0x2b, 0x67, 0x4b, 0xdd, 0xd3, 0x56, 0x9c, 0xb2, 0x09, 0x4e,
	0x9b, 0xb1, 0xee, 0x58, 0x0a, 0xe4, 0x42, 0x56, 0x26, 0x23, 0x2c, 0x3f, 0x14, 0x73, 0x46, 0x9a,
	0xa1, 0x42, 0x17, 0x12, 0xdb, 0xa2, 0xdd, 0x5d, 0x0c, 0xec, 0xdc, 0xf7, 0xc1, 0x76, 0xe0, 0x24,
	0x65, 0xef, 0x41, 0x83, 0x35, 0x38, 0x78, 0x0e, 0x65, 0x82, 0xe3, 0x55, 0x90, 0xa8, 0xd5, 0xf7,
	0x66, 0xcf, 0xe2, 0x61, 0x91, 0x3c, 0x69, 0xcb, 0xe7, 0x75, 0x62, 0x6f, 0xd7, 0x9b, 0x69, 0x0c,
	0x0d, 0x07, 0x0c, 0x9c, 0x68, 0x24, 0x51, 0x51, 0x1f, 0x8d, 0x8b, 0xd6, 0x2e, 0x67, 0x5f, 0xc3,
	0x07, 0x00, 0x12, 0x61, 0x77, 0xa8, 0x15, 0xa1, 0xd6, 0xd0, 0xd3, 0x57, 0x73, 0x62, 0xb9, 0xbb
};

void lhzb3_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//  IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= lhzb3_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static const uint8_t mgfx_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x38, 0x33, 0x52, 0x44, 0x34, 0x30, 0x32, 0x31, 0x32, 0x31, 0x32,
	0x82, 0x6b, 0xca, 0xbe, 0x9b, 0x9f, 0xc3, 0xa5, 0x8f, 0x2a, 0x9f, 0x0e, 0x26, 0x28, 0x4a, 0x9d,
	0xa2, 0x57, 0xfc, 0x43, 0xf3, 0x34, 0x05, 0x72, 0x1e, 0x59, 0xd9, 0xa0, 0xe7, 0x16, 0x5b, 0xff,
	0xc6, 0x4f, 0x6e, 0x7a, 0x09, 0x96, 0xba, 0xf3, 0x46, 0x89, 0xbb, 0xbc, 0x04, 0x6d, 0x11, 0x54,
	0xa9, 0x0b, 0x03, 0x63, 0xa6, 0xcb, 0x54, 0xf7, 0xe2, 0x0f, 0x4b, 0x01, 0x65, 0xd3, 0xfd, 0x0b,
	0x46, 0x82, 0xde, 0x3c, 0xd7, 0x1b, 0x87, 0x8e, 0x0e, 0x84, 0xca, 0x4d, 0x37, 0x2e, 0xf4, 0xd4,
	0xd8, 0xeb, 0x61, 0x84, 0xe5, 0x0c, 0x46, 0x05, 0x25, 0xbf, 0xbb, 0xc0, 0xe4, 0xa7, 0x07, 0x9d,
	0xda, 0xbd, 0x31, 0xfa, 0x5b, 0x66, 0xc3, 0xec, 0xc1, 0xd5, 0xe1, 0xb9, 0x17, 0xa4, 0x8b, 0x96,
	0x07, 0xae, 0x2f, 0x9d, 0x06, 0x30, 0x73, 0x12, 0xbe, 0x1f, 0x40, 0xd7, 0xbb, 0xcc, 0x12, 0x2d,
	0x5a, 0xb6, 0x7f, 0xad, 0xef, 0xb0, 0x06, 0x86, 0x36, 0x34, 0x1a, 0xf8, 0xfc, 0x06, 0x6f, 0x11,
	0x0e, 0x0c, 0x84, 0xa7, 0x62, 0x6d, 0x71, 0x97, 0x84, 0xe9, 0xf3, 0x3c, 0x44, 0x79, 0xb5, 0x32,
	0x9e, 0x27, 0xe1, 0x4c, 0xea, 0x30, 0xe7, 0xd4, 0x45, 0x57, 0x8d, 0x02, 0x3e, 0x8c, 0x38, 0xbf,
	0xc5, 0xbe, 0x79, 0x9b, 0x51, 0xfd, 0xdb, 0x0b, 0x51, 0xd5, 0xec, 0xe8, 0xd6, 0xe6, 0x89, 0x26,
	0x7e, 0xc8, 0x6f, 0xd1, 0xa4, 0x1e, 0xff, 0x4d, 0xc5, 0xf8, 0x51, 0xce, 0x36, 0x6f, 0x7d, 0x16,
	0x04, 0x7b, 0x26, 0x6f, 0x2c, 0x18, 0x47, 0xe8, 0xfc, 0x99, 0x42, 0xd3, 0xc9, 0x4c, 0x26, 0x7f,
	0xd2, 0x4f, 0x40, 0x32, 0x74, 0xb2, 0xe6, 0x6b, 0x90, 0xcf, 0x7f, 0x56, 0x3a, 0xe5, 0xd7, 0x8f
};

void mgfx_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
	//  IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
	//  IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= mgfx_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// IGS FOR V000CN 2004 12 01

static const uint8_t lhzb4_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x31, 0x36, 0x39, 0x52, 0x44, 0x32, 0x30, 0x34, 0x30, 0x37, 0x33, 0x30,
	0x87, 0xa5, 0x22, 0x6e, 0x2f, 0x89, 0xc6, 0x3b, 0xf3, 0x4d, 0x29, 0xd5, 0x46, 0x17, 0x9c, 0x38,
	0xc2, 0xe4, 0x16, 0x4b, 0x36, 0xfd, 0xe0, 0x7f, 0xf2, 0xbe, 0x83, 0xa6, 0x52, 0x87, 0xf2, 0x11,
	0x88, 0x03, 0xf4, 0xee, 0xaf, 0x98, 0xd5, 0xe0, 0x0e, 0x2f, 0x4d, 0xdf, 0xa9, 0x26, 0xb1, 0x53,
	0x0e, 0x92, 0x58, 0x6c, 0x5e, 0xb9, 0x50, 0xc5, 0x99, 0xcf, 0x42, 0x39, 0x2f, 0xf1, 0xa3, 0x04,
	0xc5, 0x60, 0x1d, 0x1e, 0x44, 0xff, 0x3d, 0xd5, 0x28, 0x0e, 0x5d, 0xa9, 0x08, 0x29, 0xd4, 0x6c,
	0x61, 0x7c, 0x5d, 0x9b, 0xa7, 0x48, 0xc5, 0xf7, 0x8e, 0x9b, 0xd9, 0x67, 0x95, 0x4b, 0x8f, 0x12,
	0xd4, 0x35, 0x74, 0xb8, 0x07, 0xb4, 0x55, 0x52, 0xdc, 0x64, 0x32, 0xea, 0x7a, 0x18, 0x5f, 0xbc,
	0x52, 0x19, 0xfc, 0x8e, 0x2a, 0xa2, 0x98, 0x4d, 0x66, 0x99, 0x21, 0xe9, 0x9a, 0x8e, 0x0f, 0x72,
	0x4d, 0xf9, 0xd1, 0x74, 0x10, 0xb0, 0x77, 0x8f, 0xc0, 0xa7, 0xa2, 0x5b, 0x17, 0xed, 0xa9, 0x7b,
	0x7a, 0xe3, 0x0d, 0xff, 0xfe, 0xbe, 0x1f, 0xff, 0xbb, 0x40, 0xf0, 0x76, 0x56, 0xb2, 0x79, 0x5d,
	0xc9, 0x26, 0x0c, 0x08, 0x76, 0xeb, 0xfa, 0xc5, 0x6c, 0x51, 0x86, 0xb2, 0xf9, 0x9e, 0x0a, 0xdf,
	0x70, 0x50, 0x68, 0xa5, 0x3c, 0x96, 0xb4, 0x46, 0x25, 0x09, 0x1f, 0xc6, 0xe2, 0xaf, 0x26, 0x09,
	0xe0, 0x32, 0xfd, 0x2e, 0x52, 0x5d, 0x36, 0x2b, 0x79, 0xd8, 0xb6, 0xa9, 0x35, 0x24, 0xda, 0x22,
	0xcd, 0xda, 0xe5, 0x39, 0xfb, 0x20, 0xad, 0x59, 0x3c, 0x6c, 0x86, 0x92, 0x56, 0x7d, 0x6f, 0xb0,
	0x29, 0x96, 0x7c, 0x9d, 0xbb, 0xfd, 0x83, 0xf9, 0x7f, 0xb5, 0x0a, 0xf8, 0xe6, 0x77, 0x71, 0x7b
};

void lhzb4_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
	//  IGS27_CRYPT2
	//  IGS27_CRYPT3
		IGS27_CRYPT4_ALT
	//  IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= lhzb4_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}



//////////////////////////////////////////////////////////////////////
// should fearless and superkds be using the same decrypt? fearless will fail ROM check with superkds function at the moment

static const uint8_t superkds_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x32, 0x30, 0x32, 0x52, 0x44, 0x32, 0x30, 0x35, 0x30, 0x31, 0x30, 0x33,
	0x0a, 0x68, 0x3c, 0x24, 0x56, 0x67, 0xed, 0xe3, 0x3a, 0x99, 0x20, 0x24, 0x09, 0x4d, 0x0c, 0xb6,
	0x0d, 0xbb, 0xe0, 0xe4, 0x93, 0x79, 0x6c, 0x10, 0x3b, 0x3b, 0x10, 0x91, 0x7e, 0xcf, 0xe5, 0xc9,
	0x80, 0x58, 0x2b, 0x8b, 0x18, 0xca, 0xf3, 0xac, 0x1d, 0xa6, 0x0d, 0xe7, 0xf4, 0xf7, 0x34, 0xf6,
	0x57, 0x8d, 0x79, 0xef, 0x67, 0x79, 0xed, 0xdc, 0xf5, 0xca, 0x92, 0x9e, 0x0d, 0x86, 0x84, 0x04,
	0xc5, 0xeb, 0x64, 0x27, 0x44, 0xe7, 0x05, 0x09, 0x15, 0xd5, 0x5a, 0x6d, 0xae, 0x7a, 0xa0, 0xf9,
	0x3c, 0x40, 0xc8, 0x8b, 0xb1, 0xb1, 0x27, 0xd8, 0x10, 0x37, 0x61, 0x4b, 0xf8, 0x12, 0x92, 0x1d,
	0x70, 0x9a, 0xbf, 0xb0, 0xf2, 0xb7, 0x7d, 0x32, 0xba, 0x9e, 0xe0, 0x6f, 0x4f, 0xce, 0xa6, 0xf6,
	0x53, 0x4a, 0xa5, 0x70, 0x88, 0x18, 0x72, 0x3c, 0x24, 0xfa, 0x53, 0x50, 0x57, 0x6b, 0x67, 0x4c,
	0x19, 0xdd, 0x14, 0xdf, 0x39, 0x32, 0xb1, 0x5f, 0xa2, 0x79, 0x75, 0xa4, 0xf1, 0xea, 0xa0, 0x25,
	0x35, 0x24, 0xe8, 0x56, 0x05, 0xa5, 0x26, 0x41, 0xc8, 0x8a, 0x41, 0x63, 0x41, 0x89, 0x5b, 0xc9,
	0x59, 0x2c, 0x3b, 0x6c, 0x31, 0x50, 0xfb, 0xc9, 0x68, 0xdd, 0xf2, 0xc4, 0xa9, 0xc7, 0xe4, 0xbf,
	0x79, 0x46, 0x69, 0xf6, 0x3f, 0x51, 0x9b, 0x1e, 0x95, 0x61, 0x04, 0x3e, 0xce, 0x64, 0xc6, 0xcd,
	0xc8, 0xff, 0x0c, 0x0d, 0xf3, 0x08, 0xb1, 0xa7, 0xa2, 0x44, 0x30, 0x88, 0x92, 0x5d, 0xcc, 0xfb,
	0xb8, 0x27, 0x00, 0x07, 0x4f, 0x14, 0x29, 0x0b, 0x22, 0xf5, 0x72, 0x98, 0x17, 0xf3, 0x00, 0x8f,
	0xfd, 0xcd, 0x60, 0x7c, 0x97, 0x54, 0x2d, 0x32, 0xe9, 0x24, 0x05, 0xa7, 0xc1, 0xa4, 0xaf, 0x11
};

void superkds_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		// IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= superkds_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

void fearless_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// might not be 100% correct...
		IGS27_CRYPT1
		// IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		// IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= superkds_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}



static unsigned char pgm3in1_tab[256] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x31, 0x31, 0x72, 0x64, 0x31, 0x30, 0x34, 0x30, 0x36, 0x33, 0x30,
	0xc2, 0xbe, 0xde, 0x8b, 0x80, 0x8f, 0x04, 0xbb, 0xd2, 0xd6, 0x11, 0x4a, 0x54, 0xa9, 0x8d, 0x27,
	0xf1, 0x74, 0x88, 0xc2, 0xa3, 0xdd, 0x73, 0x97, 0x6c, 0xf8, 0x41, 0xf9, 0x95, 0xac, 0xbe, 0x20,
	0x26, 0x65, 0x2f, 0x76, 0xa6, 0xfc, 0x2b, 0x1f, 0x15, 0xe3, 0xb9, 0xc7, 0x27, 0x95, 0xab, 0xaf,
	0xc2, 0x3d, 0x3a, 0xfb, 0xba, 0xa5, 0xe3, 0x35, 0xcb, 0x62, 0x81, 0x26, 0xd7, 0x3e, 0xac, 0xd7,
	0x62, 0xe6, 0x51, 0xe2, 0x4b, 0xd5, 0x92, 0xfe, 0xcd, 0x80, 0xe0, 0xca, 0xb6, 0xc3, 0x57, 0xda,
	0xe6, 0x8b, 0x59, 0x00, 0x0a, 0xc5, 0x6f, 0xdc, 0x9a, 0x87, 0x5c, 0xa6, 0x11, 0x7e, 0x83, 0x3b,
	0x6c, 0x98, 0x7b, 0x66, 0xe6, 0xf2, 0xf0, 0x71, 0xf2, 0x04, 0xbd, 0xed, 0x78, 0x0b, 0x47, 0xbd,
	0x54, 0xb6, 0x1d, 0x69, 0x0d, 0x15, 0xcd, 0xa2, 0xd2, 0xc0, 0x09, 0x11, 0xbb, 0x44, 0xfb, 0x64,
	0x3d, 0xd3, 0xe5, 0x9b, 0xee, 0x2b, 0xfd, 0x90, 0x7b, 0xc6, 0x88, 0xc5, 0xe7, 0x44, 0x35, 0x73,
	0x06, 0x17, 0xbc, 0xcf, 0x39, 0x6e, 0xb7, 0x9f, 0x6b, 0x63, 0xc1, 0xfe, 0x4c, 0x68, 0xcd, 0x6c,
	0xcd, 0xee, 0xc8, 0x19, 0xdc, 0x58, 0x72, 0x72, 0x62, 0x20, 0x7a, 0xed, 0x79, 0x48, 0xd8, 0x12,
	0xf2, 0x04, 0x70, 0xca, 0x07, 0xa6, 0xe5, 0xec, 0x5d, 0xc9, 0xba, 0x05, 0x3e, 0xc2, 0xb0, 0x68,
	0x14, 0x43, 0x5b, 0x77, 0x28, 0x51, 0x06, 0x30, 0x9d, 0x69, 0xc9, 0xfb, 0xa8, 0xee, 0xe9, 0xb2,
	0x11, 0xd7, 0x70, 0xf2, 0xef, 0x96, 0x0d, 0xa1, 0xa0, 0x4b, 0x2e, 0xc0, 0x07, 0x2a, 0x5c, 0x72,
	0x09, 0x29, 0xd6, 0x4d, 0x4a, 0xef, 0x71, 0xe1, 0x25, 0xf9, 0xb0, 0x87, 0xea, 0x0f, 0x1f, 0x6c
};

void pgm_decrypt_pgm3in1(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("maincpu")->base() + 0x100000);

	int const rom_size = 0x400000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3_ALT
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8_ALT

		x ^= pgm3in1_tab[i & 0xff] << 8;

		src[i] = x;
	}
}


// IGS MAHJONG CHINA VS107C
static const uint8_t slqz3_tab[0x100] = {
	0xef, 0xdf, 0x90, 0x60, 0x46, 0x43, 0x33, 0x38, 0x92, 0x2d, 0x5a, 0x08, 0x0d, 0x2f, 0x05, 0x75,
	0xa1, 0x42, 0x17, 0x12, 0xdb, 0xa2, 0xdd, 0x5d, 0x0c, 0xec, 0xdc, 0xf7, 0xc1, 0x76, 0xe0, 0x24,
	0x65, 0xef, 0x41, 0x83, 0x35, 0x38, 0x78, 0x0e, 0x65, 0x82, 0xe3, 0x55, 0x90, 0xa8, 0xd5, 0xf7,
	0x66, 0xcf, 0xe2, 0x61, 0x91, 0x3c, 0x69, 0xcb, 0xe7, 0x75, 0x62, 0x6f, 0xd7, 0x9b, 0x69, 0x0c,
	0x0d, 0x07, 0x0c, 0x9c, 0x68, 0x24, 0x51, 0x51, 0x1f, 0x8d, 0x8b, 0xd6, 0x2e, 0x67, 0x5f, 0xc3,
	0x13, 0x45, 0x21, 0xa1, 0x69, 0x9a, 0x05, 0xda, 0x7d, 0x10, 0xda, 0x7f, 0x34, 0x9f, 0xf3, 0x64,
	0x35, 0xf9, 0x16, 0x1c, 0xd4, 0x60, 0x02, 0xf3, 0x42, 0xc7, 0x42, 0x29, 0xf3, 0x2c, 0x31, 0x71,
	0x50, 0x35, 0x4e, 0xaa, 0x9f, 0x09, 0xc3, 0xdd, 0x2f, 0x72, 0x30, 0x77, 0xc7, 0x30, 0xbc, 0xc8,
	0x51, 0xa0, 0x5e, 0xbb, 0xc3, 0x8e, 0x69, 0xd7, 0x4f, 0x57, 0x56, 0x77, 0xcb, 0x43, 0xd6, 0x46,
	0x62, 0x21, 0x18, 0xfd, 0x7d, 0x24, 0x58, 0x61, 0xec, 0xfe, 0xa9, 0x77, 0x59, 0x3b, 0x03, 0x0a,
	0x07, 0x00, 0x12, 0x61, 0x77, 0xa8, 0x15, 0xa1, 0xd6, 0xd0, 0xd3, 0x57, 0x73, 0x62, 0xb9, 0xbb,
	0x3e, 0x60, 0x87, 0x22, 0xb7, 0xbf, 0xd6, 0xf9, 0x17, 0x86, 0xea, 0x02, 0xbe, 0x23, 0xba, 0xd3,
	0xdd, 0x0d, 0x3e, 0x8c, 0x65, 0xa0, 0xf8, 0xd8, 0x2f, 0x35, 0xc6, 0x26, 0x6c, 0x81, 0xe6, 0x29,
	0x50, 0x30, 0x4a, 0x8e, 0xfa, 0xc2, 0x1e, 0xfd, 0xa7, 0xa5, 0x98, 0x53, 0x18, 0x94, 0xff, 0x1d,
	0x41, 0x2f, 0xff, 0x58, 0x33, 0xdc, 0x2b, 0x67, 0x4b, 0xdd, 0xd3, 0x56, 0x9c, 0xb2, 0x09, 0x4e,
	0x9b, 0xb1, 0xee, 0x58, 0x0a, 0xe4, 0x42, 0x56, 0x26, 0x23, 0x2c, 0x3f, 0x14, 0x73, 0x46, 0x9a
};

void slqz3_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x200000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// not 100% verified
		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= slqz3_tab[(i/2) & 0xff] << 8;

		src[i] = x;
	}
}

//IGS COPYRIGHT 1999.99.99
// V214US
static const uint8_t fruitpar_tab[0x100] = {
	0xe9, 0x0b, 0x95, 0x7e, 0x0d, 0x7d, 0x5c, 0x1e, 0x81, 0x0e, 0xa6, 0xd4, 0x8e, 0x90, 0xd8, 0x54,
	0x27, 0x65, 0x51, 0x08, 0x1c, 0xa0, 0x3b, 0x51, 0x83, 0x60, 0x93, 0x02, 0x64, 0x69, 0x77, 0x1a,
	0xa4, 0x03, 0xb0, 0xc2, 0x34, 0x18, 0x80, 0x87, 0x7a, 0x88, 0xad, 0xd9, 0xff, 0xd0, 0xce, 0xc4,
	0x5b, 0xdc, 0xd5, 0xed, 0x5e, 0x29, 0xdd, 0xcf, 0x80, 0x1f, 0x36, 0x38, 0x8b, 0xae, 0xae, 0xfe,
	0x87, 0x27, 0x22, 0x07, 0xe6, 0x5d, 0x46, 0x79, 0xf1, 0xfc, 0xb1, 0x3d, 0x72, 0x29, 0x2c, 0xba,
	0xa3, 0x5b, 0x3c, 0xcf, 0x57, 0x79, 0xed, 0x12, 0x67, 0x34, 0xe1, 0x48, 0x5f, 0xa7, 0x9a, 0x24,
	0x6a, 0x2e, 0x04, 0x44, 0x7b, 0x84, 0x46, 0x6a, 0xbd, 0x20, 0xca, 0xf7, 0x3e, 0xd1, 0x8b, 0xad,
	0xd7, 0x98, 0x9e, 0xa6, 0x5e, 0xc6, 0x04, 0x90, 0x0f, 0x57, 0xae, 0x2b, 0x38, 0x8d, 0xd2, 0x04,
	0x25, 0xd1, 0x6d, 0x73, 0x4b, 0xc6, 0x19, 0xd3, 0xb8, 0xae, 0x11, 0x01, 0xba, 0x02, 0x82, 0x17,
	0xcf, 0x4d, 0x14, 0x6a, 0xcd, 0x4a, 0xb9, 0xc1, 0x52, 0x3e, 0xb5, 0xd8, 0x6f, 0x98, 0xee, 0x16,
	0x90, 0xc6, 0x76, 0x8a, 0xaf, 0x5a, 0x56, 0x2b, 0xb9, 0x5e, 0x9e, 0x51, 0x40, 0xf4, 0xaa, 0x6e,
	0x63, 0x32, 0xb6, 0x12, 0xfb, 0x3c, 0xa5, 0x1f, 0x07, 0xa3, 0x0d, 0x49, 0x5a, 0xfe, 0x88, 0xd1,
	0x83, 0xc7, 0x37, 0x82, 0xfd, 0x78, 0x97, 0xec, 0x98, 0xe6, 0x88, 0xe0, 0x27, 0xde, 0x9a, 0x2c,
	0x6b, 0xfd, 0x9b, 0x98, 0x40, 0xd5, 0x5f, 0x20, 0x06, 0x3e, 0xcf, 0x74, 0x52, 0xf9, 0x35, 0xae,
	0xd6, 0x8c, 0xc7, 0x53, 0x8e, 0x59, 0x71, 0x8c, 0x2d, 0x00, 0xe7, 0xa5, 0xc7, 0xf8, 0xeb, 0xc7,
	0xbf, 0x68, 0xdc, 0xf2, 0xf4, 0x4c, 0x80, 0x3e, 0x27, 0xc5, 0x13, 0x52, 0xb0, 0xc0, 0x90, 0x25
};

void fruitpar_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	// not 100% verified
	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= fruitpar_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

static uint8_t amazonia_tab[0x100] = {
	0x86, 0x00, 0xd2, 0x6b, 0x78, 0x79, 0xbc, 0x80, 0xc6, 0x8e, 0x9b, 0xd8, 0x3a, 0x52, 0x09, 0x15,
	0x05, 0x57, 0x7a, 0x1f, 0x2f, 0x08, 0x47, 0xa0, 0x44, 0x4c, 0xf7, 0xbe, 0x3d, 0x59, 0xd1, 0x18,
	0xac, 0xd5, 0xa0, 0x03, 0x2c, 0x34, 0xc6, 0x06, 0x08, 0x4e, 0x84, 0x94, 0x2b, 0x3f, 0x88, 0x08,
	0x48, 0x84, 0xb8, 0xe6, 0x4a, 0x53, 0x3e, 0x51, 0xfe, 0x3b, 0xd4, 0xc8, 0xff, 0xfd, 0x52, 0x25,
	0xe2, 0xa9, 0x75, 0xd7, 0xa3, 0xfc, 0xf1, 0x5f, 0x50, 0xfa, 0xba, 0x09, 0xf4, 0xc8, 0x90, 0xed,
	0xc6, 0xcc, 0xc9, 0x25, 0x93, 0x07, 0x63, 0x50, 0x69, 0xb2, 0x4a, 0x47, 0x85, 0x18, 0xe6, 0x1f,
	0x7f, 0xb3, 0xe8, 0x5f, 0xb4, 0x8a, 0x56, 0x83, 0xf4, 0xca, 0xd5, 0xb1, 0x6d, 0xa3, 0x37, 0xba,
	0xd7, 0x66, 0x45, 0x54, 0xe2, 0xdc, 0xce, 0x96, 0xdc, 0xe9, 0xf0, 0xb5, 0xa7, 0x62, 0xa5, 0xfd,
	0xdb, 0x2b, 0x92, 0x13, 0x37, 0x95, 0x0c, 0x69, 0x4d, 0xf7, 0x6c, 0x02, 0x6e, 0x8a, 0x95, 0x68,
	0xd4, 0x8a, 0xc3, 0xea, 0x0f, 0x8b, 0x96, 0x1b, 0xb1, 0x19, 0x5e, 0x88, 0x3d, 0x94, 0xa9, 0xb9,
	0x4e, 0x4a, 0x0a, 0x6a, 0x04, 0xd5, 0x2c, 0x0b, 0xb4, 0xb7, 0x18, 0x75, 0xcf, 0x35, 0xc3, 0xef,
	0x13, 0x71, 0xdb, 0x61, 0xf2, 0xcb, 0xd3, 0xd7, 0x3f, 0x79, 0x2d, 0x39, 0x20, 0x65, 0x07, 0x49,
	0x30, 0x47, 0xe9, 0xdd, 0xf3, 0x03, 0xce, 0x5f, 0x7f, 0x45, 0x70, 0x82, 0x69, 0x5a, 0xd9, 0x46,
	0xee, 0x52, 0x27, 0x2f, 0x63, 0x54, 0x9f, 0xc1, 0xde, 0x41, 0xf5, 0x40, 0x26, 0x8d, 0xda, 0xa6,
	0xd9, 0x5a, 0xc7, 0xe4, 0xdc, 0xd6, 0x09, 0x5e, 0x08, 0xd6, 0x0d, 0xa1, 0x12, 0xb3, 0xee, 0x67,
	0xbb, 0x65, 0x3c, 0xcd, 0x3a, 0xdf, 0x0f, 0xd3, 0xe6, 0xaa, 0x4d, 0x14, 0x28, 0xc5, 0x38, 0xc8
};

void amazonia_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= amazonia_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

//IGS FOR V201BR 2003 09 01

static const uint8_t amazoni2_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x37, 0x37, 0x52, 0x44, 0x34, 0x30, 0x32, 0x31, 0x30, 0x31, 0x36,
	0xed, 0x3f, 0x77, 0xd0, 0xc4, 0x72, 0x77, 0xe1, 0x35, 0x15, 0x10, 0xc6, 0x1b, 0xee, 0x25, 0x4a,
	0x0d, 0x06, 0x37, 0x19, 0xfa, 0xf9, 0x29, 0xb0, 0x03, 0x7e, 0x32, 0x83, 0x99, 0x32, 0x23, 0x6e,
	0xd8, 0xa9, 0xdb, 0x5f, 0x23, 0xb7, 0x08, 0x41, 0x00, 0xfb, 0x20, 0x2b, 0x36, 0xae, 0x9f, 0xc9,
	0x24, 0xaa, 0x82, 0x0b, 0x64, 0xbe, 0x03, 0x0d, 0xa2, 0x31, 0x97, 0x47, 0xb6, 0x94, 0x26, 0xf6,
	0x09, 0xcd, 0x8a, 0xca, 0x26, 0x61, 0x4a, 0xd1, 0x9f, 0x01, 0x98, 0xa3, 0x23, 0x58, 0x88, 0xcf,
	0xdd, 0x14, 0x93, 0x86, 0x0f, 0x33, 0x4c, 0x86, 0xef, 0x8f, 0x61, 0x4a, 0xc2, 0xad, 0xd5, 0x70,
	0x37, 0xc2, 0x7b, 0x6a, 0x06, 0x07, 0xb7, 0x69, 0xc9, 0x3f, 0x71, 0x87, 0x1a, 0x85, 0x5a, 0x33,
	0xee, 0x5b, 0x63, 0xe2, 0x32, 0xf0, 0x7a, 0xf4, 0xa4, 0xb2, 0x87, 0xe4, 0xf3, 0x14, 0xa7, 0xb4,
	0x1a, 0xa1, 0xa8, 0x98, 0xf9, 0x42, 0xc5, 0xe1, 0x36, 0xcc, 0xa3, 0x2d, 0x54, 0xcd, 0x8b, 0xcd,
	0x11, 0x98, 0xe9, 0x77, 0x05, 0x8e, 0x06, 0x2d, 0x78, 0xb0, 0x02, 0x6d, 0x83, 0x62, 0x15, 0x9a,
	0x6b, 0x82, 0x07, 0xaa, 0x3a, 0xa9, 0xed, 0x12, 0x9f, 0xc0, 0x25, 0xee, 0x08, 0xc7, 0x94, 0x75,
	0xfd, 0xe2, 0x1f, 0x9d, 0xc0, 0xa5, 0x68, 0x0a, 0x23, 0xa1, 0xca, 0x3d, 0xa9, 0x2f, 0x97, 0xfa,
	0xe0, 0x7c, 0x91, 0xfa, 0xfe, 0xd5, 0xa6, 0xd2, 0xbb, 0x34, 0xf1, 0x23, 0x6e, 0x0c, 0xed, 0x03,
	0x6b, 0x52, 0xfb, 0xac, 0x9c, 0xcc, 0x17, 0x64, 0x5d, 0x9e, 0xd7, 0xac, 0x9e, 0x12, 0xa5, 0xac,
	0x34, 0xa7, 0x3e, 0xde, 0x7f, 0x5d, 0x6a, 0xfa, 0x42, 0x40, 0xfd, 0x23, 0xbf, 0x33, 0x0e, 0x50
};


void amazoni2_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// should be correct
		IGS27_CRYPT1
		IGS27_CRYPT2_ALT
	//    IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6_ALT // second half is junk, not sure...
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= amazoni2_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

void pgm_dwpc_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3
		IGS27_CRYPT4_ALT
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		x ^= mm_tab[(i>> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// IGS FOR V100CN 2007 04 18 in the ROM

static const uint8_t mgcs3_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x32, 0x30, 0x37, 0x52, 0x44, 0x32, 0x30, 0x35, 0x30, 0x38, 0x30, 0x32, // IGS0207RD2050802
	0x04, 0x12, 0x46, 0xbb, 0x23, 0x66, 0xae, 0x39, 0x71, 0x44, 0x17, 0xe1, 0x0b, 0xa7, 0x41, 0x7b,
	0xb3, 0xa2, 0x38, 0xe9, 0xb7, 0x39, 0xe8, 0x2b, 0x22, 0x82, 0xe4, 0x93, 0x22, 0xfc, 0x73, 0x44,
	0xb9, 0x30, 0xdc, 0xf1, 0xa8, 0x86, 0xf4, 0xd4, 0x98, 0x6a, 0xe5, 0xed, 0x8b, 0x9f, 0x40, 0x7d,
	0x83, 0xe3, 0x46, 0xc2, 0x71, 0xc4, 0x73, 0xf4, 0x60, 0xc2, 0x4c, 0x80, 0xe2, 0xa6, 0x6b, 0x83,
	0xbb, 0x22, 0xc8, 0x8a, 0xcd, 0xaa, 0x49, 0x89, 0x43, 0x92, 0x8d, 0x19, 0x02, 0x69, 0xf5, 0xf6,
	0x4c, 0x94, 0xf5, 0xb8, 0xb7, 0x2f, 0x99, 0xd2, 0x4d, 0x20, 0x5b, 0xc8, 0x05, 0x7d, 0x23, 0xb5,
	0x62, 0x21, 0x9f, 0xfc, 0x69, 0x89, 0xc6, 0x4e, 0xc9, 0xf4, 0xa8, 0xdd, 0x46, 0xbb, 0x78, 0xde,
	0x67, 0xee, 0xdc, 0x45, 0x60, 0x30, 0x73, 0xbc, 0x42, 0xd5, 0xa8, 0xe5, 0x61, 0x3a, 0xb6, 0xd1,
	0x06, 0x64, 0xfc, 0xc1, 0x56, 0xdc, 0x83, 0x1c, 0x82, 0xc9, 0xcd, 0xb0, 0x31, 0x4f, 0xe0, 0x2d,
	0x2a, 0x28, 0x93, 0xdf, 0x46, 0x82, 0x19, 0xab, 0x96, 0x18, 0xcb, 0x4d, 0xd0, 0x93, 0x3a, 0xd1,
	0xff, 0x23, 0x75, 0x4f, 0x6b, 0x59, 0x97, 0xea, 0xc6, 0x48, 0x95, 0x0b, 0x9a, 0xdd, 0x46, 0xdb,
	0xef, 0x7b, 0xb4, 0xff, 0x40, 0xda, 0xa2, 0x98, 0xa0, 0x21, 0x5d, 0x79, 0x29, 0x42, 0xc8, 0xab,
	0xa5, 0x97, 0xa2, 0x1f, 0x80, 0xba, 0x1b, 0xb2, 0xee, 0xaa, 0x97, 0x66, 0x58, 0x1b, 0xc2, 0xdf,
	0x0d, 0x1d, 0xd4, 0x1d, 0x26, 0xf2, 0x26, 0x78, 0xba, 0x29, 0xf6, 0xe1, 0x43, 0xff, 0x77, 0x58,
	0x50, 0xf6, 0x1c, 0xa8, 0x6d, 0xb7, 0x26, 0x6a, 0x50, 0x25, 0x6c, 0x39, 0x45, 0xc3, 0x6b, 0x32
};

void mgcs3_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// TODO: probably needs work
		IGS27_CRYPT1
		IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5
		IGS27_CRYPT6
		IGS27_CRYPT7_ALT
		IGS27_CRYPT8

		x ^= mgcs3_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// IGS FOR V100CN 02-12-12 in the ROM

static const uint8_t mgzz_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x38, 0x33, 0x52, 0x44, 0x34, 0x30, 0x32, 0x31, 0x32, 0x31, 0x32, // IGS0083RD4021212
	0x82, 0x6b, 0xca, 0xbe, 0x9b, 0x9f, 0xc3, 0xa5, 0x8f, 0x2a, 0x9f, 0x0e, 0x26, 0x28, 0x4a, 0x9d,
	0xa2, 0x57, 0xfc, 0x43, 0xf3, 0x34, 0x05, 0x72, 0x1e, 0x59, 0xd9, 0xa0, 0xe7, 0x16, 0x5b, 0xff,
	0xc6, 0x4f, 0x6e, 0x7a, 0x09, 0x96, 0xba, 0xf3, 0x46, 0x89, 0xbb, 0xbc, 0x04, 0x6d, 0x11, 0x54,
	0xa9, 0x0b, 0x03, 0x63, 0xa6, 0xcb, 0x54, 0xf7, 0xe2, 0x0f, 0x4b, 0x01, 0x65, 0xd3, 0xfd, 0x0b,
	0x46, 0x82, 0xde, 0x3c, 0xd7, 0x1b, 0x87, 0x8e, 0x0e, 0x84, 0xca, 0x4d, 0x37, 0x2e, 0xf4, 0xd4,
	0xd8, 0xeb, 0x61, 0x84, 0xe5, 0x0c, 0x46, 0x05, 0x25, 0xbf, 0xbb, 0xc0, 0xe4, 0xa7, 0x07, 0x9d,
	0xda, 0xbd, 0x31, 0xfa, 0x5b, 0x66, 0xc3, 0xec, 0xc1, 0xd5, 0xe1, 0xb9, 0x17, 0xa4, 0x8b, 0x96,
	0x07, 0xae, 0x2f, 0x9d, 0x06, 0x30, 0x73, 0x12, 0xbe, 0x1f, 0x40, 0xd7, 0xbb, 0xcc, 0x12, 0x2d,
	0x5a, 0xb6, 0x7f, 0xad, 0xef, 0xb0, 0x06, 0x86, 0x36, 0x34, 0x1a, 0xf8, 0xfc, 0x06, 0x6f, 0x11,
	0x0e, 0x0c, 0x84, 0xa7, 0x62, 0x6d, 0x71, 0x97, 0x84, 0xe9, 0xf3, 0x3c, 0x44, 0x79, 0xb5, 0x32,
	0x9e, 0x27, 0xe1, 0x4c, 0xea, 0x30, 0xe7, 0xd4, 0x45, 0x57, 0x8d, 0x02, 0x3e, 0x8c, 0x38, 0xbf,
	0xc5, 0xbe, 0x79, 0x9b, 0x51, 0xfd, 0xdb, 0x0b, 0x51, 0xd5, 0xec, 0xe8, 0xd6, 0xe6, 0x89, 0x26,
	0x7e, 0xc8, 0x6f, 0xd1, 0xa4, 0x1e, 0xff, 0x4d, 0xc5, 0xf8, 0x51, 0xce, 0x36, 0x6f, 0x7d, 0x16,
	0x04, 0x7b, 0x26, 0x6f, 0x2c, 0x18, 0x47, 0xe8, 0xfc, 0x99, 0x42, 0xd3, 0xc9, 0x4c, 0x26, 0x7f,
	0xd2, 0x4f, 0x40, 0x32, 0x74, 0xb2, 0xe6, 0x6b, 0x90, 0xcf, 0x7f, 0x56, 0x3a, 0xe5, 0xd7, 0x8f
};

void mgzz_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		int x = src[i];

		// TODO: probably needs work
		IGS27_CRYPT1_ALT
		IGS27_CRYPT2
		IGS27_CRYPT3_ALT2
		IGS27_CRYPT4
		IGS27_CRYPT5_ALT
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= mgzz_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// IGS FOR V101JP 2007 06 08

static const uint8_t crzybugs_tab[0x100] = {
	0x67, 0x9b, 0x05, 0x4a, 0x32, 0xcf, 0x41, 0xf8, 0xf8, 0xd5, 0xac, 0x5e, 0xd2, 0x6d, 0xbc, 0xd9,
	0x3a, 0x15, 0x63, 0x71, 0xd6, 0x65, 0x5f, 0xce, 0xba, 0x9d, 0x1c, 0x75, 0x75, 0x25, 0x3c, 0x64,
	0xd6, 0x83, 0xc6, 0x7c, 0x7a, 0xe1, 0x13, 0x3c, 0xb9, 0xbe, 0xd4, 0x12, 0x1e, 0xcf, 0x0e, 0xa4,
	0x43, 0x07, 0xdd, 0x7f, 0x7d, 0x93, 0x8d, 0x79, 0xaf, 0xc5, 0x5e, 0x16, 0x26, 0xc6, 0x7b, 0x77,
	0xd1, 0xdf, 0x0e, 0xe3, 0x7a, 0x64, 0xfe, 0xd3, 0x60, 0xd0, 0xa3, 0x3e, 0x4e, 0x10, 0x66, 0x98,
	0x57, 0x03, 0x3d, 0xea, 0x4b, 0x0e, 0x5e, 0x83, 0xf6, 0xcf, 0xc8, 0x19, 0x4f, 0xd6, 0x0b, 0xcc,
	0xc8, 0x69, 0x21, 0x82, 0x0c, 0xdd, 0x6f, 0x6a, 0x34, 0xe1, 0x4f, 0x15, 0xe6, 0xfd, 0xa7, 0xce,
	0xbe, 0xd1, 0x38, 0x58, 0x17, 0xf2, 0xa1, 0xf7, 0xbe, 0x2e, 0x4c, 0x0e, 0xc8, 0x5d, 0xf7, 0xa1,
	0x2a, 0x71, 0xd2, 0xe6, 0x07, 0x6f, 0x56, 0xcb, 0xdd, 0xe3, 0x00, 0x0b, 0x3e, 0x3d, 0xd4, 0xbf,
	0x24, 0x41, 0x7a, 0x4e, 0xb8, 0xb4, 0x67, 0xc3, 0x2e, 0xa0, 0x57, 0xb5, 0xf1, 0xe4, 0x67, 0x29,
	0x9e, 0xf8, 0x4e, 0x81, 0x45, 0x62, 0x90, 0xff, 0x7c, 0x90, 0xad, 0xea, 0xee, 0x18, 0x02, 0x1c,
	0x6e, 0x14, 0x6a, 0x44, 0x08, 0x9f, 0x6b, 0xe9, 0xe8, 0x5a, 0x7c, 0x2e, 0xa2, 0x32, 0x60, 0x5d,
	0x6a, 0x8d, 0x9c, 0xf7, 0x9d, 0x23, 0x24, 0x39, 0x47, 0x65, 0xb7, 0x3f, 0x8e, 0xb7, 0x68, 0xe4,
	0xa6, 0x19, 0x60, 0xc2, 0xde, 0xc7, 0xb5, 0x4e, 0x23, 0x94, 0x39, 0x76, 0x6b, 0x9f, 0x16, 0xf7,
	0x39, 0xb0, 0x65, 0xb6, 0xe7, 0xe5, 0xdb, 0xbf, 0xd7, 0x95, 0xbd, 0x3b, 0x36, 0x71, 0xfe, 0x8f,
	0x15, 0x10, 0xe5, 0x63, 0x12, 0xa5, 0x1a, 0xee, 0x77, 0x49, 0x2e, 0xb4, 0x25, 0xd4, 0x83, 0x2d
};

void crzybugs_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		int x = src[i];

		// may need corrected
		IGS27_CRYPT1
		IGS27_CRYPT2_ALT // correct
		IGS27_CRYPT3_ALT // correct
		IGS27_CRYPT4 // correct
		IGS27_CRYPT5 // correct
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= crzybugs_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}

// IGS MAHJONG S500CN

static const uint8_t qlgs_tab[0x100] = {
	0x49, 0x47, 0x53, 0x30, 0x30, 0x39, 0x38, 0x52, 0x44, 0x34, 0x30, 0x36, 0x30, 0x38, 0x32, 0x38, // IGS0098RD4060828
	0x64, 0x8d, 0x4a, 0x0c, 0xd4, 0x50, 0xd6, 0xea, 0x9e, 0x5a, 0xd6, 0x31, 0xef, 0x59, 0xbf, 0x13,
	0x09, 0x4b, 0x27, 0x4e, 0x20, 0x0b, 0xbb, 0x5d, 0x79, 0x1b, 0x7e, 0xde, 0xc6, 0xb3, 0x44, 0x43,
	0xbb, 0x0b, 0x6e, 0xbd, 0xf6, 0xe6, 0xe7, 0x65, 0x0e, 0x5f, 0x5f, 0xad, 0xf9, 0xf8, 0xc7, 0x06,
	0xa3, 0xe0, 0x0d, 0xd3, 0xcc, 0x85, 0x19, 0x8e, 0x23, 0x5a, 0x08, 0x39, 0xa1, 0xea, 0xa8, 0x08,
	0x27, 0x1e, 0x34, 0x4d, 0x5b, 0xcb, 0x51, 0xa3, 0xc0, 0x7f, 0x48, 0x5f, 0x14, 0x8d, 0x86, 0x33,
	0xee, 0x56, 0x51, 0x24, 0x98, 0xdb, 0xcd, 0xaf, 0x2b, 0x80, 0x2e, 0x38, 0xe9, 0x23, 0x3f, 0xb3,
	0xdf, 0x5d, 0x14, 0x95, 0xbc, 0x17, 0x0b, 0xfd, 0xec, 0x51, 0x0a, 0x1f, 0xf8, 0x2f, 0xf3, 0xf1,
	0x22, 0x45, 0x6b, 0x1a, 0x3c, 0x24, 0xcd, 0x17, 0xc9, 0x25, 0x69, 0xb0, 0x57, 0x75, 0x01, 0x9a,
	0x1c, 0x61, 0x86, 0x6e, 0xd0, 0xe3, 0x0f, 0xca, 0xca, 0x6e, 0x1a, 0xc5, 0x5e, 0xf7, 0x08, 0x98,
	0x75, 0x44, 0xd3, 0x8b, 0x6e, 0x79, 0x11, 0x1e, 0x35, 0xe0, 0x2e, 0x7a, 0xa2, 0xf8, 0xe6, 0x17,
	0x15, 0xc1, 0x01, 0xae, 0x4f, 0x47, 0x53, 0x61, 0x92, 0x6d, 0xf3, 0x29, 0xfc, 0xfc, 0xbb, 0x81,
	0x21, 0xec, 0x00, 0x51, 0xe8, 0xf1, 0x93, 0x1d, 0xa8, 0x49, 0xf8, 0x6e, 0x83, 0xc5, 0xe5, 0x81,
	0x02, 0x16, 0xff, 0x2e, 0xf1, 0x5a, 0xd0, 0x1c, 0x7d, 0xe7, 0x0c, 0x23, 0x8d, 0x57, 0x05, 0x02,
	0x5e, 0xd4, 0x6b, 0x42, 0x61, 0xa5, 0x49, 0x6a, 0x59, 0xfa, 0x3d, 0x64, 0xb1, 0xf4, 0xf8, 0x2f,
	0x1b, 0xf8, 0xf6, 0xc7, 0x6f, 0x35, 0x7e, 0x52, 0xc3, 0x74, 0xdc, 0x8c, 0xc6, 0x20, 0xde, 0x74
};

void qlgs_decrypt(running_machine &machine)
{
	auto const src = reinterpret_cast<u16 *>(machine.root_device().memregion("user1")->base());

	int const rom_size = 0x80000;

	for (int i = 0; i < rom_size / 2; i++)
	{
		uint16_t x = src[i];

		// may need corrected
		IGS27_CRYPT1_ALT
		IGS27_CRYPT2_ALT3
		IGS27_CRYPT3
		IGS27_CRYPT4
		IGS27_CRYPT5 // correct
		IGS27_CRYPT6
		IGS27_CRYPT7
		IGS27_CRYPT8

		x ^= qlgs_tab[(i >> 1) & 0xff] << 8;

		src[i] = x;
	}
}
