/*	PrefHandler.m - Handles all preferences
 
 $Id: PrefHandler.m,v 1.14 2006/02/02 00:18:23 willthimbleby Exp $
 
 Copyright 2002 Will Thimbleby (will@thimbleby.net)
 
 This program is free software; you can redistribute it and/or modify it under the terms of the
 GNU General Public License as published by the Free Software Foundation version 2.
 
 This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
 */

/*
 changes by Manfred Bergmann
 */

#import "PrefHandler.h"
#import "AppDelegate.h"
#import "MyDocument.h"
#import "NSColor_Extensions.h"
#import "SwordManager.h"

BOOL apperanceChanged = NO;

@implementation PrefHandler

/*"	initalises default preferences and registers for notification when app has started and when it is done
 "*/
- (id)init
{
    NSMutableDictionary	*appDefaults = [NSMutableDictionary dictionaryWithCapacity:25];
    defaults = [NSUserDefaults standardUserDefaults];
	
	//setup some defaults
    NSString *relPath = @"~/Library/Application Support/Sword";
    NSString *absPath = [relPath stringByExpandingTildeInPath];
    [appDefaults setObject:absPath forKey:@"moduleLocation"];
	[appDefaults setObject:@"YES" forKey:@"cache lexicons"];
	[appDefaults setObject:@"NO" forKey:@"descriptions"];
	[appDefaults setObject:@"YES" forKey:@"tool tips"];
	[appDefaults setObject:@"YES" forKey:@"remember windows"];
	[appDefaults setObject:@"NO" forKey:@"auto devote"];
	[appDefaults setObject:@"NO" forKey:@"default one-line verses"];
	[appDefaults setObject:@"NO" forKey:@"hilite verse"];
	[appDefaults setObject:@"YES" forKey:@"inset text"];
	[appDefaults setObject:[NSDictionary dictionary] forKey:@"Module Settings"];
	[appDefaults setObject:[NSDictionary dictionary] forKey:@"Module Display Settings"];
	[appDefaults setObject:[NSArray arrayWithObject:@"Root"] forKey:@"Bookmarks"];
	
	[appDefaults setObject:[NSArray arrayWithObjects:
                            [NSDictionary dictionaryWithObjectsAndKeys:NSLocalizedString(@"Pentateuch", @"Find Range: Pentateuch"), @"name", @"Genesis - Deuteronomy", @"range", nil],
                            [NSDictionary dictionaryWithObjectsAndKeys:NSLocalizedString(@"History", @"Find Range: History"), @"name", @"Joshua - Esther", @"range", nil],
                            [NSDictionary dictionaryWithObjectsAndKeys:NSLocalizedString(@"Prophets", @"Find Range: Prophets"), @"name", @"Isaiah - Malachi", @"range", nil],
                            [NSDictionary dictionaryWithObjectsAndKeys:NSLocalizedString(@"Gospels", @"Find Range: Gospels"), @"name", @"Matt - John", @"range", nil],
                            [NSDictionary dictionaryWithObjectsAndKeys:NSLocalizedString(@"Letters/Epistles", @"Find Range: Letters/Epistles"), @"name", @"Romans - Jude", @"range", nil],
                            nil] forKey:@"custom ranges"];
	
	//style preferences
	[appDefaults setObject:[NSNumber numberWithInt:100] forKey:@"default zoom"];
	
	[appDefaults setObject:@"000000" forKey:@"style-text color"];
	[appDefaults setObject:@"FFFFFF" forKey:@"style-background color"];
	[appDefaults setObject:@"0000FF" forKey:@"style-xref color"];
	[appDefaults setObject:@"AA00AA" forKey:@"style-footnote color"];
	[appDefaults setObject:@"0000FF" forKey:@"style-morph color"];
	[appDefaults setObject:@"0000FF" forKey:@"style-strongs color"];
	[appDefaults setObject:@"FF0000" forKey:@"style-jesus color"];
	
	[appDefaults setObject:@"9999CC" forKey:@"style-heading color"];
	
	[appDefaults setObject:@"Times" forKey:@"style-font name"];
	[appDefaults setObject:@"Times" forKey:@"style-font family"];
	[appDefaults setObject:[NSMutableArray array] forKey:@"style-sheets"];
	[appDefaults setObject:[NSNumber numberWithInt:1] forKey:@"style-sheet selected"];
	[appDefaults setObject:[NSNumber numberWithInt:14] forKey:@"style-font size"];
	
    [defaults registerDefaults:appDefaults];
	
	//setup notification observers for start and end
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(start:) name:@"start" object:nil];
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateMenuTitles:) name:@"description change" object:nil];
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(moduleChange:) name:@"module change" object:nil];
	
    return self;
}

/*"	sets up popup menus so that if item stored in preferences does not exist then it chooses the first item
 and stores that in the preferences
 "*/
- (void)selectPopup:(NSPopUpButton *)menu withKey:(NSString *)key
{
	NSString *module;
	
	// get the actual module
	module = [defaults objectForKey:key];
	
	// select first item
	[menu selectItemAtIndex:0];
	
	// if title exists select it
	if(module)
	{
		// find the menu item with the represented object = key
		int	i;
		
		for(i=0; i<[menu numberOfItems]; i++)
		{
			if([[[menu itemAtIndex:i] representedObject] isEqualTo:module])
			{
				// select it
				[menu selectItemAtIndex:i];
				break;
			}
		}
	}
	else
	{
		// remember selection
		[defaults setObject:[[menu itemAtIndex:0] representedObject] forKey:key];
	}
}

/*"	sets up the interface elements in the preference window
 "*/
-(void)start:(id)info
{
	//init menus
	[self setupMenus];
	
	//cache pref
	[cacheLexiconSwitch setState:[defaults boolForKey:@"cache lexicons"]];
	
	//load up menu prefs
	[self selectPopup:serviceBiblePopup withKey:@"serviceBible"];
	[self selectPopup:strongsHebrewPopup withKey:@"strongs hebrew dictionary"];
	[self selectPopup:strongsGreekPopup withKey:@"strongs greek dictionary"];
	[self selectPopup:morphHebrewPopup withKey:@"morph hebrew dictionary"];
	[self selectPopup:morphGreekPopup withKey:@"morph greek dictionary"];
	[self selectPopup:devotionalPopup withKey:@"devotional"];
	//[self selectPopup:findModulePopup withKey:@"serviceBible"];
	
	//apperance prefs
	[descriptionSwitch setState:[defaults boolForKey:@"descriptions"]];
	[autoDevoteSwitch setState:[defaults boolForKey:@"auto devote"]];
	[hiliteSwitch setState:[defaults boolForKey:@"hilite verse"]];
	[toolTipSwitch setState:[defaults boolForKey:@"tool tips"]];
	[rememberWindowsSwitch setState:[defaults boolForKey:@"remember windows"]];
	[insetTextSwitch setState:[defaults boolForKey:@"inset text"]];
	[defaultOneLineSwitch setState:[defaults boolForKey:@"default one-line verses"]];
	
	[defaultZoomPopup selectItemAtIndex:[defaultZoomPopup indexOfItemWithTag:[defaults integerForKey:@"default zoom"]]];
	
	//style preferences
	[styleTextColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-text color"]]];
	[styleBackgroundColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-background color"]]];
	[styleXRefColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-xref color"]]];
	[styleFootnoteColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-footnote color"]]];
	[styleMorphColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-morph color"]]];
	[styleStrongsColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-strongs color"]]];
	[styleJesusColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-jesus color"]]];
	
	[styleTitleColorWell setColor:[NSColor colorWithHtmlString:[defaults objectForKey:@"style-heading color"]]];
	
	[self setFont:[NSFont fontWithName:[defaults objectForKey:@"style-font name"] size:[[defaults objectForKey:@"style-font size"] intValue]]];
	
	//module locations
	[locationTextBox setStringValue:[defaults stringForKey:@"moduleLocation"]];
	
	//style sheets
	[self createStyleSheetPopup];
	
	//make sure preferences are kept up-to date
	[self updatePreferences:self];
}

- (IBAction)selectFont:(id)sender
{
	[[NSFontManager sharedFontManager] setDelegate:self];
    [[NSFontManager sharedFontManager] setSelectedFont:[NSFont fontWithName:[defaults objectForKey:@"style-font name"] size:[[defaults objectForKey:@"style-font size"] intValue]] isMultiple:NO];
	[[NSFontManager sharedFontManager] orderFrontFontPanel:self];
}

- (void)setFont:(NSFont*)newFont
{
	[fontTextBox setStringValue:newFont ? [NSString stringWithFormat:@"%@ %g", [newFont familyName], [newFont pointSize]] : @""];
	//[fontTextBox setFont:[NSFont fontWithName:[newFont familyName] size:16]];
}

- (void)changeFont:(id)fontManager
{
	NSFont		*newFont;
	
	newFont = [fontManager convertFont:[NSFont fontWithName:[defaults objectForKey:@"style-font name"] size:[[defaults objectForKey:@"style-font size"] intValue]]];
	[defaults setObject:[newFont fontName] forKey:@"style-font name"];
	[defaults setObject:[newFont familyName] forKey:@"style-font family"];
	[defaults setObject:[NSNumber numberWithInt:[newFont pointSize]] forKey:@"style-font size"];
	
	NSLog(@"%@", newFont);
	
	apperanceChanged = YES;
	[self setFont:newFont];
}

/*"	creates a folder choice dialog for the module locations
 "*/
- (IBAction)chooseLocation:(id)sender
{
    NSArray	*fileTypes = [NSArray arrayWithObject:@""];
    NSOpenPanel *oPanel = [NSOpenPanel openPanel];
    //NSString 		*choose = NSLocalizedString(@"Choose", @"Prompt for the module location dialog");	
	
	[oPanel setCanChooseDirectories:YES];
	[oPanel setCanChooseFiles:NO];
	[oPanel setAllowsMultipleSelection:NO];
	//[oPanel setTitle:@"Choose location of modules"];
	//[oPanel setPrompt:choose];
	[oPanel beginSheetForDirectory:nil file:nil types:fileTypes modalForWindow:[sender window] modalDelegate:self didEndSelector:@selector(openPanelDidEnd:returnCode:contextInfo:) contextInfo:nil];
}

/*"	called when finished choosing the module locations
 "*/
- (void)openPanelDidEnd:(NSOpenPanel *)sheet returnCode:(int)returnCode contextInfo:(void  *)contextInfo
{
	// if click ok
    if(returnCode == NSOKButton)
	{
        NSArray *files = [sheet filenames];
		
		// save the new location
		[defaults setObject:[files lastObject] forKey:@"moduleLocation"];
		[locationTextBox setStringValue:[defaults stringForKey:@"moduleLocation"]];
    }
}

/*"	called for most UI changes, saves the new state in app defaults
 "*/
- (IBAction)updatePreferences:(id)sender
{
	// if descriptionSwitch changed then update all references to module names
	if([descriptionSwitch state] != [defaults boolForKey:@"descriptions"])
	{
		[defaults setBool:[descriptionSwitch state] forKey:@"descriptions"];
		[[NSNotificationCenter defaultCenter] postNotification:[NSNotification notificationWithName:@"description change" object:NULL]];
	}
	
	// cache
	[defaults setBool:[cacheLexiconSwitch state] forKey:@"cache lexicons"];
	
	// apperance preferences
	[defaults setBool:[autoDevoteSwitch state] forKey:@"auto devote"];
	[defaults setBool:[hiliteSwitch state] forKey:@"hilite verse"];
	[defaults setBool:[toolTipSwitch state] forKey:@"tool tips"];
	[defaults setBool:[defaultOneLineSwitch state] forKey:@"default one-line verses"];
    
	[defaults setBool:[rememberWindowsSwitch state] forKey:@"rememberWindows"];
	
	[defaults setInteger:[defaultZoomPopup selectedTag] forKey:@"default zoom"];
	
	// module locations
	[defaults setObject:[locationTextBox stringValue] forKey:@"moduleLocation"];
	
	// modules for particular uses
	[defaults setObject:[[serviceBiblePopup selectedItem] representedObject] forKey:@"serviceBible"];
	[defaults setObject:[[strongsHebrewPopup selectedItem] representedObject] forKey:@"strongs hebrew dictionary"];
	[defaults setObject:[[strongsGreekPopup selectedItem] representedObject] forKey:@"strongs greek dictionary"];
	[defaults setObject:[[morphHebrewPopup selectedItem] representedObject] forKey:@"morph hebrew dictionary"];
	[defaults setObject:[[morphGreekPopup selectedItem] representedObject] forKey:@"morph greek dictionary"];
	[defaults setObject:[[devotionalPopup selectedItem] representedObject] forKey:@"devotional"];
	
	[defaults synchronize];
}

- (IBAction)updateApperance:(id)sender
{
	// apperance preferences
	[defaults setBool:[insetTextSwitch state] forKey:@"inset text"];
	
	//style preferences
	[defaults setObject:[[styleTextColorWell color] htmlString] forKey:@"style-text color"];
	[defaults setObject:[[styleBackgroundColorWell color] htmlString] forKey:@"style-background color"];
	[defaults setObject:[[styleXRefColorWell color] htmlString] forKey:@"style-xref color"];
	[defaults setObject:[[styleFootnoteColorWell color] htmlString] forKey:@"style-footnote color"];
	[defaults setObject:[[styleMorphColorWell color] htmlString] forKey:@"style-morph color"];
	[defaults setObject:[[styleStrongsColorWell color] htmlString] forKey:@"style-strongs color"];
	[defaults setObject:[[styleJesusColorWell color] htmlString] forKey:@"style-jesus color"];
	
	[defaults setObject:[[styleTitleColorWell color] htmlString] forKey:@"style-heading color"];
	
	apperanceChanged = YES;
}

/*"	sets up a popup button, removing the first item and enabling it if menu items have been added to it
 "*/
- (void)setupPopUpButton:(NSPopUpButton*)button
{
	// if there are options then remove the inital "None" item and enable the popoup button
	if([[button menu] numberOfItems] > 1)
	{
		[[button menu] removeItemAtIndex:0];
		[button setEnabled:YES];
	}
}

/*"	bad attempt at copying a menu itme but suffices for what we need - should also be in NSMenuItem class
 "*/
- (NSMenuItem*)copyMenuItem:(NSMenuItem*)menuItem
{
	NSMenuItem		*newMenuItem;
	
	newMenuItem = [[NSMenuItem alloc] initWithTitle:[menuItem title] action:[menuItem action] keyEquivalent:[menuItem keyEquivalent]];
	[newMenuItem setRepresentedObject:[menuItem representedObject]];
	
	return [newMenuItem autorelease];
}

- (void)moduleChange:(id)sender
{
	NSMenuItem		*noneItem = [[NSMenuItem alloc] initWithTitle:@"None" action:NULL keyEquivalent:@""];
    
	[strongsHebrewPopup removeAllItems];
	[strongsGreekPopup removeAllItems];
	[morphHebrewPopup removeAllItems];
	[morphGreekPopup removeAllItems];
	[devotionalPopup removeAllItems];
	[serviceBiblePopup removeAllItems];
	//[findModulePopup removeAllItems];
    
	[[strongsHebrewPopup menu] addItem:[self copyMenuItem:noneItem]];
	[[strongsGreekPopup menu] addItem:[self copyMenuItem:noneItem]];
	[[morphHebrewPopup menu] addItem:[self copyMenuItem:noneItem]];
	[[morphGreekPopup menu] addItem:[self copyMenuItem:noneItem]];
	[[devotionalPopup menu] addItem:[self copyMenuItem:noneItem]];
	[[serviceBiblePopup menu] addItem:[self copyMenuItem:noneItem]];
	//[[findModulePopup menu] addItem:[self copyMenuItem:noneItem]];
	
	[noneItem release];
	
	[self setupMenus];
}

/*"	initalises the menus that list modules
 the actual module names are set as the represented object for each menu item
 "*/
- (void)setupMenus
{
	NSMutableArray	*bibles, *dictionaries, *commentaries;
	SwordManager	*manager = [SwordManager defaultManager];
	
	[[strongsHebrewPopup menu] addArray:[manager swordModulesForFeature:@"HebrewDef"]];
	[[strongsGreekPopup menu] addArray:[manager swordModulesForFeature:@"GreekDef"]];
	[[morphHebrewPopup menu] addArray:[manager swordModulesForFeature:@"HebrewParse"]];
	[[morphGreekPopup menu] addArray:[manager swordModulesForFeature:@"GreekParse"]];
	
	[[devotionalPopup menu] addArray:[manager swordModulesForFeature:@"DailyDevotion"]];
	
	bibles = [manager swordModulesForType:@"Biblical Texts"];
	dictionaries = [manager swordModulesForType:@"Lexicons / Dictionaries"];
	commentaries = [manager swordModulesForType:@"Commentaries"];
	
	[[serviceBiblePopup menu] addArray:bibles];
	
	//fix menus removing "None"
	[self setupPopUpButton:strongsHebrewPopup];
	[self setupPopUpButton:strongsGreekPopup];
	[self setupPopUpButton:morphHebrewPopup];
	[self setupPopUpButton:morphGreekPopup];
	[self setupPopUpButton:devotionalPopup];
	[self setupPopUpButton:serviceBiblePopup];
	//[self setupPopUpButton:findModulePopup];
	
	// make sure the titles are right
	[self updateMenuTitles:self];
}

/*"	updates a menu so that its titles are correct
 "*/
- (void)updateMenuItemsTitle:(NSArray*)menuItemArray
{
	NSString		*desc, *name;
	NSMenuItem		*menuItem;
	NSEnumerator	*menuEnum;
	
	menuEnum = [menuItemArray objectEnumerator];
	
	while((menuItem = [menuEnum nextObject]))
	{
		// get the module name
		name = [menuItem representedObject];
		
		if(name)
		{
			// get either a long description or a name
			if([defaults boolForKey:@"descriptions"])
				desc = [NSString stringWithFormat:@"%@ : %@", name, [[SwordManager defaultManager] descriptionForModuleName:name]];
			else
				desc = name;
            
			// set the title
			[menuItem setTitle:desc];
		}
	}
}

/*"	updates all menus so that they are correct
 "*/
- (void)updateMenuTitles:(id)sender
{
	[self updateMenuItemsTitle:[[strongsHebrewPopup menu] itemArray]];
	[self updateMenuItemsTitle:[[strongsGreekPopup menu] itemArray]];
	[self updateMenuItemsTitle:[[morphHebrewPopup menu] itemArray]];
	[self updateMenuItemsTitle:[[morphGreekPopup menu] itemArray]];
	[self updateMenuItemsTitle:[[devotionalPopup menu] itemArray]];
	[self updateMenuItemsTitle:[[serviceBiblePopup menu] itemArray]];
	//[self updateMenuItemsTitle:[[findModulePopup menu] itemArray]];
}

- (void)createStyleSheetPopup
{
	NSArray		*sheets;
	int			i;
    
	sheets = [defaults objectForKey:@"style-sheets"];
    
	for(i=0;i<[styleSheetPopup numberOfItems]-3;i++)
		[[styleSheetPopup menu] removeItemAtIndex:i];
	
	if([sheets count] > 0)
	{
		[[styleSheetPopup menu] insertItem:[NSMenuItem separatorItem] atIndex:0];
		
		for(i=0;i<[sheets count];i++)
		{
			NSMenuItem *mi;
			
			mi = [[NSMenuItem alloc] initWithTitle:[[sheets objectAtIndex:i] lastPathComponent] action:@selector(selectStyleSheet:) keyEquivalent:@""];
			[mi setTag:i+2];
			[[styleSheetPopup menu] insertItem:[mi autorelease] atIndex:0];
		}
	}
	[styleSheetPopup selectItemAtIndex:[styleSheetPopup indexOfItemWithTag:[[defaults objectForKey:@"style-sheet selected"] intValue]]];
}

- (IBAction)selectStyleSheet:(id)sender
{
	if([[styleSheetPopup selectedItem] tag] == 0)
		[self chooseStyleSheet:sender];
	
	[defaults setObject:[NSNumber numberWithInt:[[styleSheetPopup selectedItem] tag]] forKey:@"style-sheet selected"];
}

- (void)chooseStyleSheet:(id)sender
{
    NSArray			*fileTypes = [NSArray arrayWithObject:@"css"];
    NSOpenPanel		*oPanel = [NSOpenPanel openPanel];
    //NSString 		*choose = NSLocalizedString(@"choose", @"Select a style sheet.");	
	
	[oPanel setCanChooseDirectories:NO];
	[oPanel setCanChooseFiles:YES];
	[oPanel setAllowsMultipleSelection:NO];
	//[oPanel setTitle:@"Select a style sheet"];
	//[oPanel setPrompt:choose];
	[oPanel beginSheetForDirectory:nil file:nil types:fileTypes modalForWindow:[sender window] modalDelegate:self didEndSelector:@selector(openStylePanelDidEnd:returnCode:contextInfo:) contextInfo:nil];
}

- (void)openStylePanelDidEnd:(NSOpenPanel *)sheet returnCode:(int)returnCode contextInfo:(void  *)contextInfo
{
	// if click ok
    if (returnCode == NSOKButton)
	{
        NSArray			*files = [sheet filenames];
		NSMutableArray	*newSheets;
		
		// save the new sheet
		newSheets = [NSMutableArray arrayWithArray:[defaults objectForKey:@"style-sheets"]];
		[newSheets removeObject:[files lastObject]];
		[newSheets addObject:[files lastObject]];
		
		if([newSheets count] > 5)
			[newSheets removeObjectAtIndex:0];
		[newSheets addObject:[files lastObject]];
		
		[defaults setObject:newSheets forKey:@"style-sheets"];
		
		[self createStyleSheetPopup];
		[styleSheetPopup selectItemWithTitle:[[files lastObject] lastPathComponent]];
    }
}

- (void)saveWindows
{
	NSArray *documents;
	NSMutableArray *saveData;
	NSEnumerator*docEnum;
	MyDocument *doc;
	
	saveData = [NSMutableArray array];
	documents = [[NSDocumentController sharedDocumentController] documents];
	
	docEnum = [documents objectEnumerator];
	
	while((doc = [docEnum nextObject]))
	{
		NSDictionary *dict = [doc saveInfo];
		[saveData addObject:dict];
	}
	
	[defaults setObject:saveData forKey:@"Windows"];
}

- (void)openWindows
{
	NSArray			*saveData;
	NSEnumerator	*docEnum;
	NSDictionary	*doc;
	
	saveData = [defaults objectForKey:@"Windows"];
	
	docEnum = [saveData objectEnumerator];
	
	while((doc = [docEnum nextObject]))
	{
		[MyDocument openDocumentWithSavedDoc:doc];
	}
}

- (IBAction)emptyCache:(id)sender
{
	NSString	*appSupportPath;
	
	appSupportPath = [NSString localizedStringWithFormat:@"%@/Library/Application Support/MacSword/", NSHomeDirectory()];
	
	[[NSFileManager defaultManager] removeFileAtPath:appSupportPath handler:nil];
}

+ (NSString*)getCacheFolder
{
	NSString *appSupportPath = [NSString localizedStringWithFormat:@"%@/Library/Application Support/MacSword/", NSHomeDirectory()];
	BOOL folderExists = [[NSFileManager defaultManager] fileExistsAtPath:appSupportPath];
    
	// create support dir
	if(!folderExists)
	{
		[[NSFileManager defaultManager] createDirectoryAtPath:appSupportPath attributes:nil];
	}
	
	return appSupportPath;
}

@end

#pragma mark -

@implementation NSMenu (MenuAdditions)

- (void)addArray:(NSArray*)names
{
	NSEnumerator	*enumerator;
	NSString		*name;
	NSMenuItem		*menuItem;
	
	enumerator = [names objectEnumerator];
	
	while((name = [enumerator nextObject]))
	{
		menuItem = [[NSMenuItem alloc] initWithTitle:name action:NULL keyEquivalent:@""];
		[menuItem setRepresentedObject:name];
		
		[self addItem:menuItem];
		
		[menuItem autorelease];
	}
}

@end
