/*	ModuleList.m - Handles the listing of the indivdual modules in a outline view

	$Id: ModuleList.m,v 1.8 2006/02/02 00:18:23 willthimbleby Exp $
	
	Copyright 2003 Will Thimbleby (will@thimbleby.net)
	
	This program is free software; you can redistribute it and/or modify it under the terms of the
	GNU General Public License as published by the Free Software Foundation version 2.
	
	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
	even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
*/

/*
 changes by Manfred Bergmann
 */

#import "ModuleList.h"
#import "SwordManager.h"
#import "ModuleManage_Helper.h"
#import "AppDelegate.h"
#import "MyDocument.h"
#import "globals.h"

@interface ModuleList (privateAPI)

- (void)setModules:(NSArray *)anArray;
- (void)setCategories:(NSArray *)anArray;

@end
 
@implementation ModuleList (privateAPI)

- (void)setModules:(NSArray *)anArray
{
	[anArray retain];
	[modules release];
	modules = anArray;
}

- (void)setCategories:(NSArray *)anArray
{
	[anArray retain];
	[categories release];
	categories = anArray;	
}

@end

@implementation ModuleList

-(id) init
{
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(start:) name:@"start" object:nil];
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(updateMenuTitles:) name:@"description change" object:nil];	
	// we listen to notifications that change anything on installed modules
	// this notification is posted by the module manager
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(moduleChange:) name:ModuleChangeNotificationName object:nil];
	
	// init modules
	[self setModules:[NSArray array]];
	// init categories
	[self setCategories:[NSArray array]];
	
	return self;
}

- (void)awakeFromNib
{
	// make the module list not float
	[moduleWindow setFloatingPanel:NO];
}

- (void)moduleChange:(id)sender
{
	// get ModuleManager instance and tell him to rescan it's modules
	ModuleManageHelper *manager = [ModuleManageHelper sharedManager];
    [manager rescanInstalledModules];
	// get current set of modules from manager
	[self setModules:[manager installedModules]];

	// reload data
	[moduleList reloadData];
}

/*" Setup code run after app has started - used instead of awakeFromNib to ensure manager is loaded
"*/
- (void)start:(id)object
{
	// get available categories and modules from module manager
	ModuleManageHelper *mManager = [ModuleManageHelper sharedManager];
	[self setModules:[mManager installedModules]];
	[self setCategories:[mManager availableCategories]];

	// reload table data
	[moduleList reloadData];

	[[[moduleList tableColumns] lastObject] setEditable: NO];//setDataCell:cell];
	
	[moduleList setTarget:self];
	[moduleList setDoubleAction:@selector(doubleClick:)];
	
	
	// more complicated than I thought
	//[moduleList setAutosaveExpandedItems:YES];
	//[moduleList setAutosaveName:@"module outline"];
	{
		NSArray		*items = [[NSUserDefaults standardUserDefaults] objectForKey:@"module list expanded items"];
		int			i;
		
		for(i=0; i<[items count]; i++)
			[moduleList expandItem:[categories objectAtIndex:[[items objectAtIndex:i] intValue]]];
	}
}

- (void)outlineViewItemDidExpand:(NSNotification *)notification
{
	[self updatePrefs];
}

- (void)outlineViewItemDidCollapse:(NSNotification *)notification
{
	[self updatePrefs];
}

- (void)updatePrefs
{
	NSMutableArray		*items = [NSMutableArray array];
	int			i;
	
	for(i=0; i<[categories count]; i++)
		if([moduleList isItemExpanded:[categories objectAtIndex:i]])
			[items addObject:[NSNumber numberWithInt:i]];
	
	[[NSUserDefaults standardUserDefaults] setObject:items forKey:@"module list expanded items"];
}

- (void)updateMenuTitles:(id)sender
{
	[self reloadData];
}

- (void)reloadData
{
	[moduleList reloadData];
}

- (IBAction)doubleClick:(id)sender
{
	NSString *moduleName;
	NSString *reference = @"";
	NSString *type;
	
	Module *mModule = [moduleList itemAtRow:[moduleList selectedRow]];
	moduleName = [mModule name];
	
	type = [[SwordManager defaultManager] typeForModuleName:moduleName];
	
	//if module is a bible/commentary
	if([type isEqualTo:ModuleTypeBiblicalText] || [type isEqualTo:ModuleTypeCommentary])
	{
		if([[NSDocumentController sharedDocumentController] currentDocument])
		{
			NSURL *aUrl = nil;
			
			aUrl = [[[NSDocumentController sharedDocumentController] currentDocument] url];
			
			reference = [aUrl path];
			if([aUrl parameterString])
			{
				reference = [reference stringByAppendingFormat:@";%@", [aUrl parameterString]];
			}
		}
		[MyDocument openDocumentWithURLString:[NSString stringWithFormat:@"sword://%@%@", moduleName, reference]];

	}
	else if(moduleName)
	{
		[MyDocument openDocumentWithURLString:[NSString stringWithFormat:@"sword://%@/", moduleName]];
	}
}

// ---------------------------------------
// OutlineView DataSource delegates
// ---------------------------------------
- (id)outlineView:(NSOutlineView *)outlineView child:(int)index ofItem:(id)item
{
	// if no parent return specified category this is root then
	if(item == nil)
	{
		return [categories objectAtIndex:index];
	}
	
	// otherwise return specified module
	return [[(Category *)item content] objectAtIndex:index];
}

- (BOOL)outlineView:(NSOutlineView *)outlineView isItemExpandable:(id)item
{
	// if it is a category and category contains modules then always yes
	if([item isKindOfClass:[Category class]])
	{
		if([[(Category *)item content] count] > 0)
		{
			return YES;
		}
	}
	
	// otherwise always no
	return NO;
}

- (int)outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item
{
	// if no parent return number of categories
	if(item == nil)
	{
		return [categories count];
	}
	else
	{
		return [[(Category *)item content] count];
	}
}

- (id)outlineView:(NSOutlineView *)outlineView objectValueForTableColumn:(NSTableColumn *)tableColumn byItem:(id)item
{
	if([categories containsObject:item])
	{
		return [(Category *)item name];
	}
	else
	{
		if([[NSUserDefaults standardUserDefaults] boolForKey:@"descriptions"])
		{
			return [NSString stringWithFormat:@"%@ : %@", [(Module *)item name], [(Module *)item descr]];
		}
		else
		{
			return [(Module *)item name];
		}
	}
}

@end
