/*
 created by Manfred Bergmann in 2007
 */

#import "ModuleManage_Helper.h"
#import <SwordManager.h>

#define TableColCheck				@"check"
#define TableColName				@"name"
#define TableColStatus				@"status"
#define TableColInstalledVersion	@"installedv"
#define TableColRemoteVersion		@"remotev"


@interface ModuleManageHelper (privateAPI)

- (void)retrieveInstalledModules;

// setters
- (void)setInstalledModuleList:(NSMutableArray *)anArray;
- (void)setCategoryList:(NSMutableArray *)anArray;
// getters
- (NSMutableArray *)installedModuleList;
- (NSMutableArray *)categoryList;

@end

@implementation ModuleManageHelper (privateAPI)

/**
\brief retrieve all installed modules from SwordManager
 */
- (void)retrieveInstalledModules
{
	// get SwordManager
	SwordManager *manager = [SwordManager defaultManager];
	
	NSEnumerator *enumerator = [manager moduleNameEnumerator];
	NSString *name = nil;
	while((name = [enumerator nextObject]))
	{
		//SwordModule *swModule = [manager moduleWithName:name];
		NSString *type = [manager typeForModuleName:name];
		
		// the module we are adding to the category
		Module *module = [Module moduleForName:name andDescription:[manager descriptionForModuleName:name]];
		// add this module to our module list
		int cat = -1;
		
		if([[manager featureForModuleName:name feature:ModuleTypeUnorthodox] boolValue])
		{
			if([categoryList count] < 6)
			{
				// add this category
				[categoryList addObject:[Category categoryForName:NSLocalizedString(ModuleTypeUnorthodox, @"Module List: Unorthodox")]];
			}
			
			cat = 5;
		}
		else if([type isEqualTo:ModuleTypeBiblicalText])
		{
			cat = 0;
		}
		else if([type isEqualTo:ModuleTypeCommentary])
		{
			cat = 1;
		}
		else if([type isEqualTo:ModuleTypeDictionary])
		{
			if([[manager featureForModuleName:name feature:ModuleTypeDailyDevotion] boolValue])
			{
				cat = 4;
			}
			else
			{
				cat = 2;
			}
		}
		else if ([type isEqualTo:ModuleTypeGenBook])
		{
			cat = 3;
		}

		if(cat > -1)
		{
			Category *catBuf = [categoryList objectAtIndex:cat];
			[catBuf addModule:module];		
		}
		else
		{
			MBLOG(MBLOG_WARN, @"[ModuleManager -retrieveInstalledModules:] unknown category!");
		}
	}
}

/**
\brief init instance variable with new value
 */
- (void)setInstalledModuleList:(NSMutableArray *)anArray
{
	[anArray retain];
	[installedModuleList release];
	installedModuleList = anArray;
}

/**
\brief set new instance of category
 */
- (void)setCategoryList:(NSMutableArray *)anArray
{
	[anArray retain];
	[categoryList release];
	categoryList = anArray;	
}

/**
\brief get modules
 */
- (NSMutableArray *)installedModuleList
{
	return installedModuleList;
}

/**
\brief get categories
 */
- (NSMutableArray *)categoryList
{
	return categoryList;
}

@end

@implementation ModuleManageHelper

/**
\brief this is a singleton
 */
+ (ModuleManageHelper *)sharedManager
{
	static ModuleManageHelper *singleton;
	
	if(singleton == nil)
	{
		singleton = [[ModuleManageHelper alloc] init];
	}
	
	return singleton;	
}

/**
\brief init is called after alloc:. some initialization work can be done here.
 No GUI elements are available here. It additinally calls the init method of superclass
 @returns initialized not nil object
 */
- (id)init
{
	MBLOG(MBLOG_DEBUG,@"init of ModuleManager");
	
	self = [super initWithWindowNibName:@"ModuleManager"];
	if(self == nil)
	{
		MBLOG(MBLOG_ERR,@"cannot alloc ModuleManager!");		
	}
	else
	{
		[self setInstalledModuleList:[NSArray array]];
		categoryList = [[NSMutableArray alloc] initWithObjects:
			[Category categoryForName:NSLocalizedString(@"Bibles", @"Module List: Bibles")],
			[Category categoryForName:NSLocalizedString(@"Commentaries", @"Module List: Commentaries")],
			[Category categoryForName:NSLocalizedString(@"Lexicons", @"Module List: Lexicons")],
			[Category categoryForName:NSLocalizedString(@"Books", @"Module List: Books")],
			[Category categoryForName:NSLocalizedString(@"Devotionals", @"Module List: Devotionals")],
			nil];
		
		// scan installed modules
		[self retrieveInstalledModules];
	}
	
	return self;
}

/**
\brief dealloc of this class is called on closing this document
 */
- (void)dealloc
{
	MBLOG(MBLOG_DEBUG,@"dealloc of ModuleManager");
	
	// release some instances
	[self setInstalledModuleList:nil];
	[self setCategoryList:nil];
	
	// dealloc object
	[super dealloc];
}

/**
\brief get installed list of modules from this manager
 the NSArray is autoreleased
 */
- (NSArray *)installedModules
{
	return [NSArray arrayWithArray:installedModuleList];
}

/**
\brief get available categories from this manager
 the NSArray is autoreleased
 */
- (NSArray *)availableCategories
{
	return [NSArray arrayWithArray:categoryList];
}

/**
\brief let this manager rescan hi sinstalled modules
 */
- (void)rescanInstalledModules
{
	// remove all modules for all categories
	int len = [categoryList count];
	for(int i = 0;i < len;i++)
	{
		[[categoryList objectAtIndex:i] setContent:[NSArray array]];
	}
	
	// and retrieve new modules from sword manager
	[self retrieveInstalledModules];
}

//-------------------------------------------
// Actions
//-------------------------------------------
- (IBAction)closeButton:(id)sender
{
	[[self window] close];
}

- (IBAction)processButton:(id)sender
{
}

@end
