/*	FindControl.m - Handles the searching modules
 
 $Id: FindControl.m,v 1.9 2006/02/02 00:18:24 willthimbleby Exp $
 
 Copyright 2003 Will Thimbleby (will@thimbleby.net)
 
 This program is free software; you can redistribute it and/or modify it under the terms of the
 GNU General Public License as published by the Free Software Foundation version 2.
 
 This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
 */

#import "FindControl.h"
#import "AppDelegate.h"
#import "MyDocument.h"
#import "SwordManager.h"
#import "Highlighter.h"
#import <WebKit/WebKit.h>

NSString *CorePasteboardFlavorType_url  = @"CorePasteboardFlavorType 0x75726C20"; // 'url '
NSString *CorePasteboardFlavorType_urln = @"CorePasteboardFlavorType 0x75726C6E"; // 'urln'
NSString *WebURLsWithTitlesPboardType = @"WebURLsWithTitlesPboardType";

NSString *foundString = NULL;
NSString *foundNoneString = NULL;

@implementation FindControl

OSStatus MyGotoHelpAnchor( CFStringRef anchorName)
{
    CFBundleRef myApplicationBundle = NULL;
    CFTypeRef myBookName = NULL;
    OSStatus err = noErr;
    myApplicationBundle = CFBundleGetMainBundle(); 
	// 1
    if (myApplicationBundle == NULL) {err = fnfErr; return err;}
    myBookName = CFBundleGetValueForInfoDictionaryKey( 
                                                      // 2
                                                      myApplicationBundle,
                                                      CFSTR("CFBundleHelpBookName"));
    if(myBookName == NULL)
	{
		err = fnfErr;
		return err;
	}
    if (CFGetTypeID(myBookName) != CFStringGetTypeID())
	{ 
		// 3
        err = paramErr;
    }
    if(err == noErr)
		err = AHLookupAnchor(myBookName, anchorName); 
	// 4
    return err;
}

OSStatus MyGotoHelpPage (CFStringRef pagePath, CFStringRef anchorName)
{
    CFBundleRef myApplicationBundle = NULL;
    CFStringRef myBookName = NULL;
    OSStatus err = noErr;
    myApplicationBundle = CFBundleGetMainBundle(); 
	// 1
    if (myApplicationBundle == NULL) {err = fnfErr; return err;} 
	// 2
    myBookName = CFBundleGetValueForInfoDictionaryKey( 
                                                      // 3
                                                      myApplicationBundle,
                                                      CFSTR("CFBundleHelpBookName"));
    if (myBookName == NULL)
	{
		err = fnfErr;
		return err;
	}
    if (CFGetTypeID(myBookName) != CFStringGetTypeID())
	{ 
		// 4
        err = paramErr;
    }
    if (err == noErr)
		err = AHGotoPage (myBookName, pagePath, anchorName); 
	// 5
    return err;
}

/*"	inits and creates an empty array for results
 "*/
- (id)init
{
	//self = [super init];
    self = [self initWithWindowNibName:@"Find"];
	
	searching = NO;
	module = nil;
	searchResults = nil;
	history = [[NSMutableArray array] retain];
	ranges = [[NSMutableArray array] retain];
	incrementalSearch = NO;
	
	[self setShouldCascadeWindows:NO];
	
	//liveTimer = [NSTimer scheduledTimerWithTimeInterval:600 target:self selector:@selector(find:) userInfo:self repeats:YES];
	//[liveTimer retain];
	
	if(!foundString)
		foundString = [NSLocalizedString(@"Found %d references.", @"Search dialog: Found ? references") retain];
	if(!foundNoneString)
		foundNoneString = [NSLocalizedString(@"No references were found.", @"Search dialog: No references were found") retain];
	
	return self;
}

- (void)setupFindMenus
{
	NSMutableArray	*bibles, *dictionaries, *commentaries, *books;
	SwordManager	*manager = [SwordManager defaultManager];
	
	[moduleMenu removeAllItems];
	
	bibles = [manager swordModulesForType:@"Biblical Texts"];
	dictionaries = [manager swordModulesForType:@"Lexicons / Dictionaries"];
	commentaries = [manager swordModulesForType:@"Commentaries"];
	books = [manager swordModulesForType:@"Generic Books"];
	
	//find popup menu
	[[moduleMenu menu] addArray:bibles];
	if([bibles count] > 0) [[moduleMenu menu] addItem:[NSMenuItem separatorItem]];
	
	[[moduleMenu menu] addArray:commentaries];
	if([commentaries count] > 0) [[moduleMenu menu] addItem:[NSMenuItem separatorItem]];
	
	[[moduleMenu menu] addArray:dictionaries];
	if([dictionaries count] > 0) [[moduleMenu menu] addItem:[NSMenuItem separatorItem]];
	
	[[moduleMenu menu] addArray:books];
	
	//pick service bible or document
	NSString *mod = [[NSUserDefaults standardUserDefaults] objectForKey:@"serviceBible"];
	
	if([[NSDocumentController sharedDocumentController] currentDocument])
		mod = [[[[NSDocumentController sharedDocumentController] currentDocument] module] moduleName];
    
	[moduleMenu selectItemAtIndex:[moduleMenu indexOfItemWithRepresentedObject:mod]];
	
	[self changeModule:self];
}

/*"	sets the results table double click action to me - so that it opens new bibles on clicking
 "*/
- (void)windowDidLoad
{
	[self setupFindMenus];
    
	[resultsTable setDoubleAction:@selector(doubleClick:)];
	[resultsTable setTarget:self];
	[resultsTable setRowHeight:[resultsTable rowHeight]*3.0*[[NSUserDefaults standardUserDefaults] integerForKey:@"default zoom"]/100];
	[[[resultsTable tableColumnWithIdentifier:@"contents"] dataCell] setWraps:YES];
	[[[resultsTable tableColumnWithIdentifier:@"key"] dataCell] setWraps:YES];
	
	[(NSCell *)[[resultsTable tableColumnWithIdentifier:@"key"] dataCell] setFont:
     [NSFont userFontOfSize:12.0*[[NSUserDefaults standardUserDefaults] integerForKey:@"default zoom"]/100]];
	[(NSCell *)[[resultsTable tableColumnWithIdentifier:@"contents"] dataCell] setFont:
     [NSFont userFontOfSize:12.0*[[NSUserDefaults standardUserDefaults] integerForKey:@"default zoom"]/100]];
	
	[self nameWindow];
	
	// make sure it autosaves the table
	[resultsTable setAutosaveTableColumns:YES];
	[resultsTable setAutosaveName:@"results table"];
	
	//range popup
	ranges = [[NSMutableArray arrayWithArray:[[NSUserDefaults standardUserDefaults] objectForKey:@"custom ranges"]] retain];
	[self fixPopup];
}

- (void)nameWindow
{
	NSArray *windows = [NSApp windows];
	int i, f=0;
	BOOL found = YES;
	
	do
	{	
		f++;
		found = YES;
		for(i=0;i<[windows count];i++)
		{
			if(![[windows objectAtIndex:i] isVisible]) continue;
			
			NSString *asn = [[windows objectAtIndex:i] frameAutosaveName];
			if([asn hasPrefix:@"find window"])
			{
				if(f == [[asn substringFromIndex:11] intValue])
				{
					found = NO;
					break;
				}
			}
		}
	}
	while(!found);
	
	[findWindow setFrameAutosaveName:[NSString stringWithFormat:@"find window %d", f]];
}

- (IBAction)openFindWindow:(id)sender
{
	if([[NSDocumentController sharedDocumentController] currentDocument])
	{
		NSURL		*url = [[[NSDocumentController sharedDocumentController] currentDocument] url];
		SwordModule *mod = [[SwordManager defaultManager] moduleForURL:url];
		
		[moduleMenu selectItemAtIndex:[moduleMenu indexOfItemWithRepresentedObject:[mod moduleName]]];
	}
	
	[findWindow makeKeyAndOrderFront:self];
}

/*"	updates the progress bar - called from callback during searching
 "*/
- (void)updateProgress:(id)progress
{
	[progressIndicator setDoubleValue:[progress doubleValue]];
}

- (NSString *)urlString:(int)row
{
	// if actualy has a result
	if(row >= 0 && row < [searchResults count])
	{
		NSString *mod, *openKey;
		
		NSString *moduleName = [[moduleMenu selectedItem] representedObject];
		SwordModule *module2 = [[SwordManager defaultManager] moduleWithName:moduleName];
		// get the modules name
		mod = [module2 moduleName];//[[results objectAtIndex:row] objectForKey:@"module"];
		
		// get the reference
		openKey = [[searchResults objectAtIndex:row] objectForKey:IndexPropSwordKeyString];
		
		// tell the app to create a new document
		if([[[SwordManager defaultManager] typeForModuleName:mod] isEqualToString:@"Biblical Texts"])
		{
			return [NSString stringWithFormat:@"sword://%@/%@#%@", mod, [SwordBible context:openKey], openKey];
		}
		else
		{
			return [NSString stringWithFormat:@"sword://%@/%@", mod, openKey];
		}
	}
	
	return @"";
}

/*"	creates a new document on double clicking a link
 "*/
- (IBAction)doubleClick:(id)sender
{
	// if actualy has a result
	[MyDocument openDocumentWithURLString:[self urlString:[resultsTable clickedRow]]];
}

/**
 \brief switch incremental search ON/OFF
 */
- (IBAction)switchIncrementalSearch:(id)sender
{
	if([(NSButton *)sender state] == NSOnState)
	{
		incrementalSearch = YES;
	}
	else
	{
		incrementalSearch = NO;
	}
}

/*
 creates a new document on double clicking a link
 */
- (IBAction)changeModule:(id)sender
{
	// get the module name
	NSString *moduleName = [[moduleMenu selectedItem] representedObject];
    
	// load it up
	SwordModule *module2 = [[SwordManager defaultManager] moduleWithName:moduleName];//should use different modules to what is normally used (searchmanager?)
	
	if([module2 type] == bible || [module2 type] == commentary)
	{
		[rangePopup setEnabled:YES];
	}
	else
	{
		[rangePopup setEnabled:NO];
	}
}

-(OSStatus) RegisterMyHelpBook 
{
    CFBundleRef myApplicationBundle;
    CFURLRef myBundleURL;
    FSRef myBundleRef;
    OSStatus err = noErr;
    myApplicationBundle = NULL;
    myBundleURL = NULL;
    myApplicationBundle = CFBundleGetMainBundle(); 
    // 1
    if (myApplicationBundle == NULL) {err = fnfErr; return err;}
    myBundleURL = CFBundleCopyBundleURL(myApplicationBundle); 
    // 2
    if (myBundleURL == NULL) {err = fnfErr; return err;}
    if (!CFURLGetFSRef(myBundleURL, &myBundleRef)) err = fnfErr; 
    // 3
    if (err == noErr) err = AHRegisterHelpBook(&myBundleRef); 
    // 4
    return err;
}

- (IBAction)showHelp:(id)sender
{
	//NSString *file = [[NSBundle mainBundle] pathForResource:@"Searching" ofType:@"htm" inDirectory:@"Help"];
	//[[NSWorkspace sharedWorkspace] openURL:[NSURL URLWithString:[NSString stringWithFormat:@"help://%@",file]]];
	//MyGotoHelpAnchor(CFSTR("MacSword_SEARCHING"));
	//AHLookupAnchor (NULL, CFSTR("MacSword_SEARCHING")); 
	//[[NSHelpManager sharedHelpManager] openHelpAnchor:@"MacSword_SEARCHING" inBook:nil];
	//MyGotoHelpPage(CFSTR("Searching.htm"),NULL);
	[self RegisterMyHelpBook];
    AHGotoPage (CFSTR("MacSword Help"), CFSTR("Searching.htm"), NULL); 
}

/**
 searching loads a module and searches it
 this method is used for threaded searching
 all necasary data is passed in the dictionary
 */
- (void)searchFor:(NSDictionary *)info
{
	NSAutoreleasePool	*pool = [[NSAutoreleasePool alloc] init];
	
	// get the module name
	SwordModule *searchModule = [info objectForKey:@"module"];
	NSString *searchQuery = [info objectForKey:@"string"];
	
    // has the module an index already?
	if(![searchModule hasIndex]) {
        // if not, create one first
		[self createIndexUnthreaded:self];
	}
	
	// search
	if(searchResults != nil)
	{
        // release old search results array
		[searchResults release];
        searchResults = nil;
	}
    
    // get search range
    NSString *range = [[rangePopup selectedItem] representedObject];
    
    BOOL haveRange = NO;
    unsigned int indexA = -1;
    unsigned int indexB = -1;        
    if(![range caseInsensitiveCompare:@"Everything"] == NSOrderedSame)
    {
        MBLOG(MBLOG_DEBUG, @"Have a range...");
        
        haveRange = YES;
        
        // get range setting from popup button
        NSArray *bookNames = [(SwordBible *)searchModule engBookNames];        
        NSArray *books = [range componentsSeparatedByString:@" - "];
        int len = [books count];
        if(len > 0)
        {
            indexA = (unsigned int)[bookNames indexOfObject:[books objectAtIndex:0]];
            if(indexA != NSNotFound)
            {
                indexA++;   // array index is one less
                MBLOGV(MBLOG_DEBUG, @"Searching for booknumber: %i", indexA);
            }
        }
        if(len > 1)
        {
            indexB = (unsigned int)[bookNames indexOfObject:[books objectAtIndex:1]];
            if(indexB != NSNotFound)
            {
                indexB++;
                MBLOGV(MBLOG_DEBUG, @"Searching for booknumber: %i", indexB);
            }
        }
    }
    
    
    NSRange searchRange;
    if(haveRange == YES)
    {
        // create range for searching
        searchRange = NSMakeRange(indexA, indexB);
    }
    else
    {
        searchRange = NSMakeRange(0, 0);
    }
    
    // get max verses for this bible book
    int maxResults = 0;
    ModuleType type = [SwordModule moduleTypeForModuleTypeString:[searchModule moduleType]];
    if(type == bible)
    {
        maxResults = [(SwordBible *)searchModule maxVersesForBible];
    }
    else if(type == dictionary)
    {
        maxResults = [[(SwordDictionary *)searchModule getEntries] count];
    }
    else if(type == book)
    {
        maxResults = 0;
    }
    else
    {
        MBLOG(MBLOG_WARN, @"Module type for searching not yet supported!");
    }
    
    // get new search results
    //Indexer *indexer = [IndexingManager indexerForModule:searchModule];
    Indexer *indexer = [Indexer indexerWithModuleName:[searchModule moduleName] moduleType:[SwordModule moduleTypeForModuleTypeString:[searchModule moduleType]]];
    if(indexer == nil)
    {
        MBLOG(MBLOG_ERR, @"Could not get indexer for searching!");
    }
    else
    {
        NSArray *tempResults = [indexer performSearchOperation:searchQuery range:searchRange maxResults:maxResults];
        searchResults = [tempResults mutableCopy];
        
        // close indexer
        [indexer close];
        
        // create out own SortDescriptors according to whome we sort
        NSArray *sortDescriptors = [NSArray arrayWithObject:
                                    [[[NSSortDescriptor alloc] initWithKey:@"documentName" 
                                                                 ascending:YES 
                                                                  selector:@selector(caseInsensitiveCompare:)] autorelease]];
        // sort
        [searchResults sortUsingDescriptors:sortDescriptors];
        
        // show how many references we found
        NSString *text;
        
        int resultCount = [searchResults count];
        // print all search results
        /*
         for(int i = 0;i < resultCount;i++)
         {
         SearchResultEntry *entry = [searchResults objectAtIndex:i];
         MBLOGV(MBLOG_DEBUG, @"DocName = %@", [entry documentName]);
         }
         */
        
        if(resultCount > 0)
        {
            text = [NSString stringWithFormat:foundString, resultCount];
        }
        else 
        {
            text = foundNoneString;
        }
        
        [statusText performSelectorOnMainThread:@selector(setStringValue:) withObject:text waitUntilDone:NO];
        
        // reset the find button
        [findButton performSelectorOnMainThread:@selector(setTitle:) withObject:NSLocalizedString(@"Find", @"Search dialog: Find (button title)") waitUntilDone:NO];
        
        // release and return
        // update the results table
        [resultsTable performSelectorOnMainThread:@selector(reloadData) withObject:NULL waitUntilDone:NO];
        
        [info release];
        [pool release];
        
        searching = NO;
    }
	
    return;
}

- (void)controlTextDidChange:(NSNotification *)aNotification
{
	MBLOG(MBLOG_DEBUG, @"text did change in combo box!");
	
	if(incrementalSearch == YES)
	{
		// start new search
		[self find:findButton];
	}
}

/**
 button action - if finding stop otherwise find
 */
- (IBAction)find:(id)sender
{
	MBLOG(MBLOG_DEBUG, @"search: click");
	
	if(![findWindow isVisible])
	{
		//[liveTimer setFireDate:[NSDate dateWithTimeIntervalSinceNow:600]];
		return;
	}
	
    // are we searching at the moment?
    // beware, we have multithreading here
	if(searching == YES)
	{
		if(module != nil)
		{
            // stop searching
			[module stopSearch];
		}
		return;
	}
    
    // set searching flag
	searching = YES;
	
	// handle the find history
	[history removeObject:[inputTextBox stringValue]];
	[history addObject:[inputTextBox stringValue]];
	
	if([inputTextBox numberOfItems] < 10)
	{
		[inputTextBox setNumberOfVisibleItems:[inputTextBox numberOfItems]];
	}
	else
	{
		[inputTextBox setNumberOfVisibleItems:10];
	}
	
    // reset results table
	[resultsTable reloadData];
    
	// change search button text
	[findButton setTitle:NSLocalizedString(@"Stop", @"Search dialog: Stop (button title)")];
    
	// get sword module from the selected module to search for
	NSString *moduleName = [[moduleMenu selectedItem] representedObject];
	module = [[SwordManager defaultManager] moduleWithName:moduleName];
	if(module == nil)
	{
        MBLOGV(MBLOG_ERR, @"SwordManager could not get module with name: %@", moduleName);
	}
    else
    {
        // get search query
        NSString *searchedFor2 = [[inputTextBox stringValue] retain];
        
        // construct dictionary with all information needed for searching
        NSDictionary *dict = [NSDictionary dictionaryWithObjectsAndKeys:module, @"module", searchedFor2, @"string", NULL];
        
        //do the search
        [NSThread detachNewThreadSelector:@selector(searchFor:) toTarget:self withObject:[dict retain]];
        
        // change search button title again
        [findWindow setTitle: [NSString stringWithFormat:NSLocalizedString(@"Find: %@", @"Find window name"), [inputTextBox stringValue]]];
    }
}

- (void)startIndex
{
	indexing = YES;
	[progressIndicator setDoubleValue:0];
	[indexProgress startAnimation:self];
	[indexInfo setStringValue:[NSString stringWithFormat:NSLocalizedString(@"Indexing %@.", @"Indexing information"), [module moduleName]]];
	
	[NSApp beginSheet:indexView modalForWindow:findWindow modalDelegate:self didEndSelector:NULL contextInfo:NULL];
}

- (void)endIndex
{
	[NSApp endSheet:indexView];
	[indexView close];
	
	[indexProgress stopAnimation:self];
	indexing = NO;
}

- (void)cancelIndex:(id)sender
{
	[module stopIndex];
}

- (void)createIndexUnthreaded:(id)sender
{
	[self performSelectorOnMainThread:@selector(startIndex) withObject:NULL waitUntilDone:YES];
    
	[[[SwordManager defaultManager] moduleWithName:[[moduleMenu selectedItem] representedObject]] createIndex:indexProgress];
	
	[self performSelectorOnMainThread:@selector(endIndex) withObject:NULL waitUntilDone:YES];
}

- (IBAction)editRanges:(id)sender
{
	[NSApp beginSheet:rangeEditView modalForWindow:findWindow modalDelegate:self didEndSelector:NULL contextInfo:NULL];
}

- (IBAction)addRange:(id)sender
{
	SwordBible *defaultModule;
	NSArray *books;
	
	defaultModule = (SwordBible*)[[SwordManager defaultManager] moduleForURL:[NSURL URLWithString:@"sword://Bible/Fake" relativeToURL:NULL]];
	books = [defaultModule bookNames];
	
	[ranges addObject:[NSDictionary dictionaryWithObjectsAndKeys:@"?", @"name", [NSString stringWithFormat:@"%@ - %@",[books objectAtIndex:0], [books lastObject]], @"range", nil]];
	[rangeTable reloadData];
}

- (IBAction)deleteRange:(id)sender
{
	if([rangeTable selectedRow] >= 0)
	{
		[ranges removeObjectAtIndex:[rangeTable selectedRow]];
		[rangeTable reloadData];
	}
}

- (IBAction)endEditRanges:(id)sender
{
	[NSApp endSheet:rangeEditView];
	[rangeEditView close];
	
	[rangeTable abortEditing];
	[self fixPopup];
}

- (void)fixPopup
{
	NSMenuItem *mi = [[NSMenuItem alloc] initWithTitle:NSLocalizedString(@"Everything", @"Find range:Everything") action:NULL keyEquivalent:@""];
	NSMenuItem *mib = [[NSMenuItem alloc] initWithTitle:NSLocalizedString(@"Last Result", @"Find range:Last Result") action:NULL keyEquivalent:@""];
	
	[rangePopup removeAllItems];
	
	[mi setRepresentedObject:@"Everything"];
	[[rangePopup menu] addItem:mi];
	[mib setRepresentedObject:@"Last Result"];
	[[rangePopup menu] addItem:mib];
	
	if([ranges count] > 0)
	{
		int i;
		
		[[rangePopup menu] addItem:[NSMenuItem separatorItem]];
        
		for(i=0;i<[ranges count]; i++)
		{
			mi = [[NSMenuItem alloc] initWithTitle:[[ranges objectAtIndex:i] objectForKey:@"name"] action:NULL keyEquivalent:@""];
			[mi setRepresentedObject:[[ranges objectAtIndex:i] objectForKey:@"range"]];
			[[rangePopup menu] addItem:mi];
		}
	}
	
	[[rangePopup menu] addItem:[NSMenuItem separatorItem]];
	mi = [[NSMenuItem alloc] initWithTitle:NSLocalizedString(@"Edit ranges...", @"Find range:Edit ranges...") action:@selector(editRanges:) keyEquivalent:@""];
	[mi setTarget:self];
	[[rangePopup menu] addItem:mi];
	
	[[NSUserDefaults standardUserDefaults] setObject:ranges forKey:@"custom ranges"];
}

// Data Source methods for the NSTableView

/**
 \brief sort descriptors have changed, that means that the user has clicked on table header
 */
- (void)tableView:(NSTableView *)aTableView sortDescriptorsDidChange:(NSArray *)oldDescriptors
{
	MBLOG(MBLOG_DEBUG,@"-tableView:sortDescriptorsDidChange:]");
    
	NSArray *newDescriptors = [aTableView sortDescriptors];
    
    // sort the current rtesult set
    NSMutableArray *temp = [searchResults mutableCopy];
    [searchResults release];
    
    [temp sortUsingDescriptors:newDescriptors];
    searchResults = [temp retain];
    [temp release];
    
    [resultsTable reloadData];
}

/*	
 returns the data for the row and column
 hilights the searched for words in the verse
 */
- (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	NSString *ret = @"";
	
	if(aTableView == rangeTable)
	{
		ret = [[ranges objectAtIndex:rowIndex] objectForKey:[aTableColumn identifier]];
		return ret;
	}
	
	if(searchResults == nil)
	{
		ret = @"";
		return ret;
	}
	
	if([[aTableColumn identifier] isEqualToString:@"contents"])
	{
        NSString *text = [[searchResults objectAtIndex:rowIndex] keyContent];
		NSAttributedString *attrString = [Highlighter highlightText:text forTokens:[inputTextBox stringValue]];
		return attrString;
	}
	else
	{
		ret = [[searchResults objectAtIndex:rowIndex] keyString];
		return ret;
	}
}

- (int)numberOfRowsInTableView:(NSTableView *)aTableView
{
	int ret = 0;
	
	if(aTableView == rangeTable)
	{
		ret = [ranges count];
		return ret;
	}
	
	if(searchResults == nil)
	{
		ret = 0;
		return ret;
	}
	
	ret = [searchResults count];	
	return ret;
}

/*
 - (id)tableView:(NSTableView *)aTableView objectValueForTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
 {
 if(aTableView == rangeTable)
 return [[ranges objectAtIndex:rowIndex] objectForKey:[aTableColumn identifier]];
 
 //if(searching)
 //	return @"";
 
 if([[aTableColumn identifier] isEqualToString:@"contents"])
 {
 NSDictionary *d = [results objectAtIndex:rowIndex];
 NSString *contents = [d objectForKey:@"contents"];
 
 if(contents) return contents;
 
 contents = [module textForIndex:[[d objectForKey: @"verse"] intValue]];
 [d setObject:contents forKey:@"contents"];
 
 return contents;
 }
 else
 return [module keyForIndex:[[[results objectAtIndex:rowIndex] objectForKey: @"verse"] intValue]];
 //	return [[results objectAtIndex:rowIndex] objectForKey:[aTableColumn identifier]];
 }
 */

/*"	returns the number of references found"
 */
/*
 - (int)numberOfRowsInTableView:(NSTableView *)aTableView
 {
 if(aTableView == rangeTable)
 return [ranges count];
 
 //if(searching)
 //	return 0;
 
 return [results count];
 }
 */
- (BOOL)tableView:(NSTableView *)tableView writeRows:(NSArray*)rows toPasteboard:(NSPasteboard*)pboard
{
	NSMutableString		*verseList = [NSMutableString string];
	NSMutableString		*verseListString = [NSMutableString string];
	NSMutableString		*relativeURL = [NSMutableString string];
	NSEnumerator		*rowEnum;
	id					row;
	NSString			*urlString;
	
	[relativeURL appendString:@"/"];
    
	// Provide data for our custom type, and simple NSStrings.
	[pboard declareTypes:[NSArray arrayWithObjects: NSStringPboardType, @"relative url", DragDropVerseListPboardType, nil] owner:self];
    
	urlString = [self urlString:[[rows objectAtIndex:0] intValue]];
	
	rowEnum = [rows objectEnumerator];
	while((row = [rowEnum nextObject]))
	{
        NSString *verseKeyContent = [[searchResults objectAtIndex:[row intValue]] objectForKey:IndexPropSwordKeyContent];
        NSString *verseKeyString = [[searchResults objectAtIndex:[row intValue]] objectForKey:IndexPropSwordKeyString];
        
		if([verseListString length] > 0) 
		{
			[verseListString appendString:@"\n"];
		}
		[verseListString appendString:[NSString stringWithFormat:@"%@\n(%@)\n",
                                       verseKeyContent,
                                       verseKeyString]];
        
		if([verseList length] > 0)
		{
			[verseList appendString:@"	"];
		}
		[verseList appendString:verseKeyString];
		
		[relativeURL appendString:verseKeyString];
		[relativeURL appendString:@";"];
	}
	
	//dock drag and drop
	[pboard setString:verseListString forType:NSStringPboardType];
	
	//MBLOG(MBLOG_DEBUG, @"%@",urlString);
	//MBLOG(MBLOG_DEBUG, @"%@",[NSURL URLWithString:[urlString addPercentEscapes]]);
	[[NSURL URLWithString:[urlString addPercentEscapes]] writeToPasteboard:pboard];
	
	[pboard setString:relativeURL forType:@"relative url"];
	
	//[pboard setString:urlString forType:CorePasteboardFlavorType_url];
	//[pboard setString:urlString forType:CorePasteboardFlavorType_urln];
	//[pboard setString:verseListString forType:NSURLPboardType];
	[pboard setString:verseList forType:DragDropVerseListPboardType];
	
	return YES;
}

- (void)tableView:(NSTableView *)aTableView setObjectValue:(id)anObject forTableColumn:(NSTableColumn *)aTableColumn row:(int)rowIndex
{
	if(aTableView == rangeTable)
	{
		NSMutableDictionary *md = [NSMutableDictionary dictionaryWithDictionary:[ranges objectAtIndex:rowIndex]];
		[md setObject:anObject forKey:[aTableColumn identifier]];
		[ranges replaceObjectAtIndex:rowIndex withObject:md];
	}
}

- (int)numberOfItemsInComboBox:(NSComboBox *)aComboBox
{
	return [history count];
}

- (id)comboBox:(NSComboBox *)aComboBox objectValueForItemAtIndex:(int)index
{
	return [history objectAtIndex:[history count]-1-index];
}

// ---------------------------------------
// NSComboBox / NSTextField delegate methodss
- (void)textDidChange:(NSNotification *)aNotification
{
	MBLOG(MBLOG_DEBUG, @"text did change in combo box!");
}

@end

@implementation FindTableColumn

- (id)dataCellForRow:(int)row
{
    if(!greyCell);
    {
		greyCell = [[NSTextFieldCell alloc] init];
		[greyCell setBackgroundColor:[[NSColor colorWithDeviceWhite:0.8 alpha:1] retain]]; 
		[greyCell setDrawsBackground:YES];
		[greyCell setEditable:NO];
		[greyCell setSelectable:YES];
		[greyCell setFont:[NSFont userFontOfSize:12.0*[[NSUserDefaults standardUserDefaults] integerForKey:@"default zoom"]/100]];
    }
    
	if(row % 2)
		return greyCell;
    
    return [self dataCell];
}

@end

@implementation FindTable

- (NSDragOperation)draggingSourceOperationMaskForLocal:(BOOL)isLocal
{
	/*if (isLocal)
     return NSDragOperationEvery;
     else*/
    return NSDragOperationCopy;
}


@end


