/*	MyDocument.m - Basic document class

	$Id: MyDocument.m,v 1.19 2006/02/02 00:18:24 willthimbleby Exp $
	
	Copyright 2002 Will Thimbleby (will@thimbleby.net)
	
	This program is free software; you can redistribute it and/or modify it under the terms of the
	GNU General Public License as published by the Free Software Foundation version 2.
	
	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
	even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
*/

#import "MyDocument.h"
#import "ToolbarDelegateCategory.h"
#import "BibleService.h"
#import "ToolTip.h"
#import "SwordServer.h"
#import "SwordServer.h"
#import "NSColor_Extensions.h"
#import "NSString_Extensions.h"
#import "SwordDictionary.h"
#import "AppKit/NSAccessibility.h"
#import "ModuleToolbarView.h"
#import "CocoLogger/CocoLogger.h"

extern NSArray *daysPerMonth;

@implementation MyDocument

+ (MyDocument *)openDocumentWithSavedDoc:(NSDictionary*)doc
{
	MyDocument *mydoc;
	
	mydoc = [[MyDocument alloc] initWithURL:[NSURL URLWithString:[doc objectForKey:@"URL"]]];
	
	if(mydoc)
	{
		NSString *tagBuf = [doc objectForKey:@"Tag"];
		if(tagBuf)
		{
			[mydoc setTag:tagBuf];
		}        
        [mydoc setDisplayOptions:[doc objectForKey:@"DisplayOptions"]];
		
		[[NSDocumentController sharedDocumentController] addDocument:mydoc];
		[mydoc makeWindowControllers];
		[[mydoc windowControllers] makeObjectsPerformSelector:@selector(window)];
		[[mydoc window] setFrameFromString:[doc objectForKey:@"Frame"]];
		
		id zoom = [doc objectForKey:@"Zoom"];
		if(zoom)
		{
			[[mydoc webView] setTextSizeMultiplier:[zoom floatValue]];
		}
		
		[mydoc showWindows];
		[mydoc release];
	}
	
	return mydoc;
}

+ (MyDocument *)openDocumentWithURLString:(NSString*)aURLString
{
	return [self openDocumentWithURL:[NSURL URLWithString:[aURLString addPercentEscapes]]];
}
	
+ (MyDocument *)openDocumentWithURL:(NSURL*)aURL
{
	MyDocument *doc;
		
	doc = [[MyDocument alloc] initWithURL:aURL];	
	if(doc)
	{
		[[NSDocumentController sharedDocumentController] addDocument:doc];
		[doc makeWindowControllers];
		[doc showWindows];
		//[doc release];
	}
	
	return doc;
}

- (id)initWithContentsOfURL:(NSURL *)aURL ofType:(NSString *)docType
{
	MBLOG(MBLOG_DEBUG, @"initWithContentsOfURL -Error");
	return nil;
}

- (id)initWithURL:(NSURL*)aURL
{
	self = [super init];
	MBLOGV(MBLOG_DEBUG, @"initWithURL:%@",aURL);
	module = [[SwordManager defaultManager] moduleForURL:aURL];
	editView = nil;
	
	displayOptions = [[NSMutableDictionary dictionaryWithDictionary:[module displayOptions]] retain];
	
	// init variables
	[self setTag:@""];
	windowClosing = NO;
	
	if(module) {
		url = [aURL retain];
		type = [module type];
		
		return self;
	}
	
	return NULL;
}

- (NSString *)windowNibName {
    // Override returning the nib file name of the document
    // If you need to use a subclass of NSWindowController or if your document supports multiple NSWindowControllers, you should remove this method and override -makeWindowControllers instead.
    return @"MyDocument";
}

- (void)windowControllerDidLoadNib:(NSWindowController *) aController {
    [super windowControllerDidLoadNib:aController];
	
	[urlBar setNextResponder:webView];
    // Set the WebView delegates
    [webView setFrameLoadDelegate:self];
    [webView setPolicyDelegate:self];
    [webView setUIDelegate:self];
	// [webView setResourceLoadDelegate:self];

	[[webView mainFrame] loadRequest:[NSURLRequest requestWithURL:url]];

	//accessibility
	//[myWindow accessibilitySetValue:url forAttribute:NSAccessibilityDocumentAttribute];
	
	// setup the toolbar
	[self setupToolbar];
	
	//initalise module menu for toolbar
	if(type == dictionary) {
		[dictCombo addItemWithObjectValue:aboutEntrySpecifier];
		[dictCombo addItemsWithObjectValues:[(SwordDictionary*)module getEntries]];
	} else if(type == devotional) {
        [monthCombo reloadData];
    }
	
	//setup tooltip tracking rect
	[self setupTrackingRect];
	
	[webView setTextSizeMultiplier:(float)[[NSUserDefaults standardUserDefaults] integerForKey:@"default zoom"]/100];
}

// read module from file
- (BOOL)readFromFile:(NSString *)fileName ofType:(NSString *)docType
{
	if([docType isEqualToString:@"Sword Module"]) {
		NSString		*fullSubDir;
		BOOL			directory;
		NSArray			*subFiles;

		fullSubDir = [fileName stringByAppendingPathComponent:@"mods.d"];
		fullSubDir = [fullSubDir stringByStandardizingPath];
		
		//if its a directory
		if([[NSFileManager defaultManager] fileExistsAtPath:fullSubDir isDirectory:&directory])
		if(directory) {
			if(![SwordManager defaultManager]) {
				[[[NSApplication sharedApplication] delegate] loadManager];
			}
		
			// add the new module in
			[[SwordManager defaultManager] addPath:fileName];
			
			// update module lists
			[[NSNotificationCenter defaultCenter] postNotification:[NSNotification notificationWithName:@"module change" object:NULL]];
			
			subFiles = [[NSFileManager defaultManager] directoryContentsAtPath:fullSubDir];
			
			//find the config file parse it for the module name and open a new document
			int len = [subFiles count];
			for(int i = 0;i < len;i++)
			if([[subFiles objectAtIndex:i] hasSuffix:@".conf"])
			{
				NSString	*mod;
				NSString	*confFile = [NSString stringWithContentsOfFile:[fullSubDir stringByAppendingPathComponent:[subFiles objectAtIndex:i]]];
				NSScanner	*scan = [NSScanner scannerWithString:confFile];
				
				[scan scanString:@"[" intoString:NULL];
				[scan scanUpToString:@"]" intoString:&mod];
				
				[MyDocument openDocumentWithURLString:[NSString stringWithFormat:@"sword://%@/", mod]];
				break;
			}
		}
	}
			
	return NO;
}

- (void)windowWillClose:(NSNotification *)notification {
	MBLOG(MBLOG_DEBUG, @"[windowWillClose:]");
	windowClosing = YES;	// set window will close
    [webView stopLoading:self];
}

- (void)dealloc {
	[self setTag:nil];
	[displayOptions release];
    
	[super dealloc];
}

- (BOOL)isDocumentEdited {
	return NO;
}

- (void)printShowingPrintPanel:(BOOL)showPanels {
    // Obtain a custom view that will be printed
    NSView *printView = [[[webView mainFrame] frameView] documentView];

    // Construct the print operation and setup Print panel
    NSPrintOperation *op = [NSPrintOperation
                printOperationWithView:printView
                printInfo:[self printInfo]];
    [op setShowPanels:showPanels];
    if (showPanels) {
        // Add accessory view, if needed
    }

    // Run operation, which shows the Print panel if showPanels was YES
    [self runModalPrintOperation:op
                delegate:nil
                didRunSelector:NULL
                contextInfo:NULL];
}


- (SwordModule *)module {
    return module;
}

- (ModuleType)type {
	return type;
}

- (NSWindow *)window {
    return myWindow;
}

- (NSURL *)url {
    return url;
}

- (void)setURL:(NSURL *)u {
	[webView stopLoading:self];
    [url autorelease];
    url = [u retain];
}

- (NSDictionary *)displayOptions {
    return [NSDictionary dictionaryWithDictionary:displayOptions];
}

- (void)setDisplayOptions:(NSDictionary *)value {
    [displayOptions release];
    displayOptions = [[NSMutableDictionary dictionaryWithDictionary:value] retain];
    
    // when set from outside, also set in module
    [module setDisplayOptions:displayOptions];
}

- (id)webView {
    return webView;
}

- (void)setTag:(NSString *)t {
	if(tag) {
		[tag autorelease];
	}
	tag = [t retain];
}

- (NSString *)getTag {
	return tag;
}

/**
\brief save all information of this document
 */
- (NSDictionary *)saveInfo {
	return [NSDictionary dictionaryWithObjectsAndKeys:
            [[self window] stringWithSavedFrame], @"Frame", 
            [url absoluteString], @"URL", 
            [NSNumber numberWithFloat:[webView textSizeMultiplier]], @"Zoom", 
            tag, @"Tag",
            displayOptions, @"DisplayOptions",
            nil];
}

// -------------------------------------------------
// controls the contents of the combo boxes
// -------------------------------------------------
- (int)numberOfItemsInComboBox:(NSComboBox *)aComboBox {
	if(aComboBox == dayCombo) {
		if([monthCombo numberOfItems] < 0 || [monthCombo indexOfSelectedItem] < 0) {
			return 31;
		}
		return [[daysPerMonth objectAtIndex:[monthCombo indexOfSelectedItem]] intValue];
	} else if(aComboBox == monthCombo) {
		return 13;
	}
	
	if(type == bible || type == commentary) {
		if(module) {
			if(aComboBox == chapterCombo) {
				return [(SwordBible*)module maxChaptersForBookName:[bookCombo stringValue]];
			} else if(aComboBox == bookCombo) {
				return [[(SwordBible*)module bookNames] count]+1;
			}
		}
		return 0;
	}

	return 0;
}

- (id)comboBox:(NSComboBox *)aComboBox objectValueForItemAtIndex:(int)index {
	if(aComboBox == monthCombo && index >= 0 && index < 13) {
		if(index == 0) {
			return aboutEntrySpecifier;
        } else {
            NSArray *monthArray = [[NSUserDefaults standardUserDefaults] arrayForKey:NSMonthNameArray];
            NSString *monthStr = [monthArray objectAtIndex:index - 1];
            return monthStr;
        }
	} else if(module && aComboBox == bookCombo && (type == bible || type == commentary)) {
		if(index == 0)
			return aboutEntrySpecifier;
		else
			return [[(SwordBible*)module bookNames] objectAtIndex:index-1];
	}
	else
		return [NSString stringWithFormat:@"%d", index+1];
}

- (IBAction)makeTextLarger:(id)sender {
	[webView makeTextLarger:sender];
}

- (IBAction)makeTextSmaller:(id)sender {
	[webView makeTextSmaller:sender];
}

- (IBAction)nextChapter:(id)sender {
	[chapterCombo selectItemAtIndex:[chapterCombo indexOfSelectedItem]+1];
	[self changeChapter:self];
}

- (IBAction)previousChapter:(id)sender {
	[chapterCombo selectItemAtIndex:[chapterCombo indexOfSelectedItem]-1];
	[self changeChapter:self];
}

- (IBAction)nextBook:(id)sender {
	if(type == bible || type == commentary) {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -nextBook:] have moduletype bible or commentary");
		[bookCombo selectItemAtIndex:[bookCombo indexOfSelectedItem]+1];
		[self changeBook:self];
	} else if(type == book) {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -nextBook:] have moduletype general book");
		[contentsHandler nextItem];
	} else if(type == devotional) {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -nextBook:] have moduletype devotional");
		[monthCombo selectItemAtIndex:[monthCombo indexOfSelectedItem]+1];
		[self changeDate:monthCombo];
	} else {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -nextBook:] have moduletype all other");
		[dictCombo selectItemAtIndex:[dictCombo indexOfSelectedItem]+1];
		[self changeEntry:dictCombo];
	}
}

- (IBAction)previousBook:(id)sender {
	if(type == bible || type == commentary) {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -previousBook:] have moduletype bible or commentary");
		[bookCombo selectItemAtIndex:[bookCombo indexOfSelectedItem]-1];
		[self changeBook:self];
    } else if(type == book) {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -previousBook:] have moduletype general book");
		[contentsHandler previousItem];
	} else if(type == devotional) {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -previousBook:] have moduletype devotional");
		[monthCombo selectItemAtIndex:[monthCombo indexOfSelectedItem]-1];
		[self changeDate:monthCombo];
	} else {
		MBLOG(MBLOG_DEBUG, @"[MyDocument -previousBook:] have moduletype all other");
		[dictCombo selectItemAtIndex:[dictCombo indexOfSelectedItem]-1];
		[self changeEntry:dictCombo];
	}
}

- (IBAction)toggleBookmarks:(id)sender {
	if([bookmarkDrawer state] == NSDrawerClosedState || [bookmarkDrawer state] == NSDrawerClosingState) {
		[bookmarkDrawer open];
	} else {
		[bookmarkDrawer close];
	}
}

- (IBAction)toggleContents:(id)sender {
	if([contentsDrawer state] == NSDrawerClosedState || [contentsDrawer state] == NSDrawerClosingState) {
		[contentsDrawer open];
	} else {
		[contentsDrawer close];
	}
}

- (IBAction)addBookmark:(id)sender {
	[bookmarkDrawer open];
	[bookmarkHandler addBookmark:self];
}

- (IBAction)openLocation:(id)sender {
	[myWindow makeFirstResponder:urlBar];
}

- (IBAction)showContext:(id)sender {
    
    NSString *context = [NSString stringWithFormat:@"/%@?highlight=%@#%@",
                         [SwordBible context:[urlBar stringValue]],
                         [urlBar stringValue],
                         [urlBar stringValue]];
    
    
	[self openURLString:context];
}

- (IBAction) changeEntry:(id)sender {
	[self openURLString:[NSString stringWithFormat:@"/%@", [sender stringValue]]];
	[sender scrollItemAtIndexToTop:[sender indexOfSelectedItem]];
}

- (IBAction) changeDate:(id)sender {
	if([monthCombo indexOfSelectedItem] > 0) {
		[self openURLString:[NSString stringWithFormat:@"/%02d.%02d", [monthCombo indexOfSelectedItem], [dayCombo indexOfSelectedItem]+1]];
	} else {
		[self openURLString:@"/(About)"];
	}
}

- (IBAction) changeChapter:(id)sender {
	[self openURLString:[NSString stringWithFormat:@"/%@ %@", [bookCombo stringValue], [chapterCombo stringValue]]];
}

- (IBAction) changeBook:(id)sender {
	int book, chapter;
	
	//one big kludge
	[(SwordBible*)module book:&book andChapter:&chapter fromReference:[bookCombo stringValue]];
	
	if(chapter == 1 && ![[bookCombo stringValue] hasSuffix:@"1"]) {
		if([chapterCombo indexOfSelectedItem] > [(SwordBible *)module maxChaptersForBookName:[bookCombo stringValue]])
			[self openURLString:[NSString stringWithFormat:@"/%@ %d", [bookCombo stringValue], [(SwordBible*)module maxChaptersForBookName:[bookCombo stringValue]]]];
		else
			[self openURLString:[NSString stringWithFormat:@"/%@ %@", [bookCombo stringValue], [chapterCombo stringValue]]];
	}
	else
		[self openURLString:[NSString stringWithFormat:@"/%@", [bookCombo stringValue]]];
}

- (IBAction)changeModule:(id)sender {
	[self changeModuleTo:[[sender selectedItem] representedObject]];
}

- (void)changeModuleTo:(NSString *)aModule {
	NSArray *parts;
	
	parts = [[urlBar stringValue] componentsSeparatedByString:@"/"];
	
	if([parts count] > 1) {
		[self openURLString:[NSString stringWithFormat:@"sword://%@/%@", aModule, [parts objectAtIndex:1]]];
	} else {
		[self openURLString:[NSString stringWithFormat:@"sword://%@/%@", aModule, [urlBar stringValue]]];
	}
}

// Action method invoked when the user enters a new URL
- (IBAction)connectURL:(id)sender {
	NSString *urlString = [urlBar stringValue];
    
    // make sure the leading character is upper case
    urlString = [urlString capitalizedString];
	
	if(![urlString hasPrefix:@"sword://"]) {
		urlString = [NSString stringWithFormat:@"sword://%@/%@", [moduleView hostURL], urlString];
	}
	
	[self openURLString:urlString];
	MBLOGV(MBLOG_DEBUG, @"open %@ = %@", urlString, [urlString addPercentEscapes]);
}

- (void)openURLString:(NSString *)urlString {
	NSURLRequest *request;
	
	urlString = [urlString addPercentEscapes];
	MBLOGV(MBLOG_DEBUG, @"open %@ = %@", urlString, [urlString removePercentEscapes]);	
	
	if([urlString isEqualToString:[url absoluteString]]) {
        [[webView mainFrame] reload];
        return;
	}
	[self setURL:[NSURL URLWithString:urlString relativeToURL:url]];
	
	request = [NSURLRequest requestWithURL:url];
	
	[[webView mainFrame] stopLoading];
	[[webView mainFrame] loadRequest:request];
}

- (BOOL)prepareForDragOperation:(id <NSDraggingInfo>)sender {
    MBLOG(MBLOG_DEBUG, @"start drag?");
	return YES;
}

- (void)updateWindow {
	
	if(windowClosing == NO) {
		if(url != nil) {
			NSString *ref;
			
			if([url path] && [[url path] length] > 0) {
				if([[[url path] pathComponents] count] > 2)
					ref = [[[url path] pathComponents] objectAtIndex:2];
				else if([[[url path] pathComponents] count] > 1)
					ref = [[[url path] pathComponents] objectAtIndex:1];
				else
					ref = @"";
			} else
				ref = @"";
			
			if([url parameterString])
				ref = [ref stringByAppendingFormat:@";%@", [url parameterString]];
			ref = [ref removePercentEscapes];
			[urlBar setStringValue:ref];
			//[modulePopup selectItemWithTitle:[module moduleName]];
			if(moduleView) {
				[moduleView setModules:url];
				[myWindow setTitle: [NSString stringWithFormat:@"%@: %@", [moduleView modulesTitle], ref]];
			}
			else
				[myWindow setTitle: [NSString stringWithFormat:@"%@: %@", [module moduleName], ref]];
		}
		
		[self updateToolbar];
	}
}

- (void)updateToolbar {
	[toolbar validateVisibleItems];
	
	if(type == devotional) {
		[dayCombo reloadData];
	}
	
	[self updateCombos];
}

- (void)updateCombos {
	if(type == bible || type == commentary) {
		[chapterCombo noteNumberOfItemsChanged];
		[bookCombo noteNumberOfItemsChanged];
		
		if([chapterCombo numberOfItems] < 20)
			[chapterCombo setNumberOfVisibleItems:[chapterCombo numberOfItems]];
		else
			[chapterCombo setNumberOfVisibleItems:20];
		
		/*if([verseCombo numberOfItems] < 20)
			[verseCombo setNumberOfVisibleItems:[verseCombo numberOfItems]];
		else
			[verseCombo setNumberOfVisibleItems:20];*/
		if([bookCombo numberOfItems] < 20)
			[bookCombo setNumberOfVisibleItems:[bookCombo numberOfItems]];
		else
			[bookCombo setNumberOfVisibleItems:20];
	}
}

- (BOOL)validateMenuItem:(NSMenuItem *)anItem {

	int	itemtag = [anItem tag];

    switch(itemtag) {
		case 1:
			return [moduleView modulesHaveFeature:@"Footnotes"];
		case 2:
			return [moduleView modulesHaveFeature:@"Headings"];
		case 3:
			return [moduleView modulesHaveFeature:@"Strongs"];
		case 4:
			return [moduleView modulesHaveFeature:@"Morph"];
		case 5:
			return [moduleView modulesHaveFeature:@"Cantillation"];
		case 6:
			return [moduleView modulesHaveFeature:@"HebrewPoints"];
		case 7:
			return [moduleView modulesHaveFeature:@"GreekAccents"];
		case 8:
			return [moduleView modulesHaveFeature:@"Lemma"];
		case 9:
			return [moduleView modulesHaveFeature:@"Scripref"];
		case 10:
		case 11:
		case 12:
			return [moduleView modulesHaveFeature:@"Variants"];
		case 14:
			return [moduleView modulesHaveFeature:@"RedLetterWords"];
		case 20:
			return (type == bible || type == commentary);
		case 21:
			return (type == bible || type == commentary || type == dictionary || type == book);
		case 22:
			return [self canEdit];
		// 15 - Today's Devotional
		default:
			// send message to the app delegate
			return [[[NSApplication sharedApplication] delegate] validateMenuItem:anItem];
	}
	return YES;
}

- (void)setVariants:(id)sender {
	int itemtag = [sender tag];

	[displayOptions removeObjectForKey:@"Primary Reading"];
	[displayOptions removeObjectForKey:@"Secondary Reading"];
	[displayOptions removeObjectForKey:@"All Readings"];
			
    switch(itemtag) {
		case 10:
			[displayOptions setObject:[NSNumber numberWithBool:YES] forKey:@"Primary Reading"];
			break;
		case 11:
			[displayOptions setObject:[NSNumber numberWithBool:YES] forKey:@"Secondary Reading"];
			break;
		case 12:
			[displayOptions setObject:[NSNumber numberWithBool:YES] forKey:@"All Readings"];
			break;
	}
    
    // update options
    [module setDisplayOptions:displayOptions];
    
	[self connectURL:self];
	[self fixVariants];
}

- (void)fixVariants {
    
	int i;
	NSMenu	*optionsMenu = [[[NSApp mainMenu] itemWithTag:3] submenu];
	NSMenu	*varMenu = [[optionsMenu itemWithTag:20] submenu];
	NSArray *vars = [varMenu itemArray];
	
	for(i = 0;i < [vars count];i++)
		[[vars objectAtIndex:i] setState:NSOffState];
				
	if([displayOptions objectForKey:@"Primary Reading"]) {
		[[vars objectAtIndex:0] setState:NSOnState];
	} else if([displayOptions objectForKey:@"Secondary Reading"]) {
		[[vars objectAtIndex:1] setState:NSOnState];
	}
    
	if([displayOptions objectForKey:@"All Readings"]) {
		[[vars objectAtIndex:2] setState:NSOnState];
	}
}

- (void)setState:(id)sender option:(NSString *)option {
    if([displayOptions objectForKey:option])
		[sender setState:NSOnState];
	else
		[sender setState:NSOffState];
}

- (void)setNegativeState:(id)sender option:(NSString*)option {
    if(![displayOptions objectForKey:option])
		[sender setState:NSOnState];
	else
		[sender setState:NSOffState];
}

- (void)getState:(id)sender option:(NSString*)option {
    switch([sender state]) {
		case NSOffState:
			[displayOptions setObject:[NSNumber numberWithBool:YES] forKey:option];
			[sender setState:NSOnState];
			break;
		case NSOnState:
			[displayOptions removeObjectForKey:option];
			[sender setState:NSOffState];
			break;
		default:
			[displayOptions removeObjectForKey:option];
			[sender setState:NSOffState];
			break;
	}
    // update options
    [module setDisplayOptions:displayOptions];

	[self connectURL:self];
}

- (void)getNegativeState:(id)sender option:(NSString*)option {
    switch([sender state]) {
		case NSOnState:
			[displayOptions setObject:[NSNumber numberWithBool:YES] forKey:option];
			[sender setState:NSOnState];
			break;
		case NSOffState:
			[displayOptions removeObjectForKey:option];
			[sender setState:NSOffState];
			break;
		default:
			[displayOptions removeObjectForKey:option];
			[sender setState:NSOffState];
			break;
	}
    // update options
    [module setDisplayOptions:displayOptions];

	[self connectURL:self];
}
 
- (void)setOption:(id)sender {
	switch([sender tag]) {
		case 1:
			[self getState:sender option:@"Footnotes"];
			break;
		case 2:
			[self getState:sender option:@"Headings"];
			break;
		case 3:
			[self getState:sender option:@"Strong's Numbers"];
			break;
		case 4:
			[self getState:sender option:@"Morphological Tags"];
			break;
		case 5:
			[self getState:sender option:@"Hebrew Cantillation"];
			break;
		case 6:
			[self getState:sender option:@"Hebrew Vowel Points"];
			break;
		case 7:
			[self getState:sender option:@"Greek Accents"];
			break;
		case 8:
			[self getState:sender option:@"Lemmas"];
			break;
		case 9:
			[self getState:sender option:@"Cross-references"];
			break;
		case 12:
			[self getNegativeState:sender option:@"No Verse Numbering"];
			break;
		case 13:
			[self getState:sender option:@"Verses On One Line"];
			break;
		case 14:
			[self getState:sender option:@"Words of Christ in Red"];
			break;
		default:
			break;
	}
	// save module options
/*	{
		NSMutableDictionary		*mDict;
		
		mDict = [NSMutableDictionary dictionaryWithDictionary:[[NSUserDefaults standardUserDefaults] dictionaryForKey:@"Module Settings"]];
		
		[mDict setObject:[NSDictionary dictionaryWithDictionary:options] forKey:[module moduleName]];
		
		[[NSUserDefaults standardUserDefaults] setObject:mDict forKey:@"Module Settings"];
	}
	[[NSUserDefaults standardUserDefaults] synchronize];*/
}

- (void)fixOptionsMenu {
	NSMenu	*optionsMenu = [[[NSApp mainMenu] itemWithTag:3] submenu];

	[self setState:[optionsMenu itemWithTag:1] option:@"Footnotes"];
	[self setState:[optionsMenu itemWithTag:2] option:@"Headings"];
	[self setState:[optionsMenu itemWithTag:3] option:@"Strong's Numbers"];
	[self setState:[optionsMenu itemWithTag:4] option:@"Morphological Tags"];
	[self setState:[optionsMenu itemWithTag:5] option:@"Hebrew Cantillation"];
	[self setState:[optionsMenu itemWithTag:6] option:@"Hebrew Vowel Points"];
	[self setState:[optionsMenu itemWithTag:7] option:@"Greek Accents"];
	[self setState:[optionsMenu itemWithTag:8] option:@"Lemmas"];
	[self setState:[optionsMenu itemWithTag:9] option:@"Cross-references"];
	[self setState:[optionsMenu itemWithTag:14] option:@"Words of Christ in Red"];
	
	[self fixVariants];
	
	[self setNegativeState:[optionsMenu itemWithTag:12] option:@"No Verse Numbering"];
	[self setState:[optionsMenu itemWithTag:13] option:@"Verses On One Line"];
}

- (BOOL)becomeFirstResponder {
	[self fixOptionsMenu];
	return YES;
}

// ------------------------------------------
// WebFrameLoadDelegate Methods
// ------------------------------------------
- (void)webView:(WebView *)sender didStartProvisionalLoadForFrame:(WebFrame *)frame {
    // Only report feedback for the main frame.
    if (frame == [sender mainFrame]){
		[url autorelease];
		url = [[[[frame provisionalDataSource] request] URL] retain];
        [self updateWindow];
    }
}

- (void)webView:(WebView *)sender didReceiveTitle:(NSString *)title forFrame:(WebFrame *)frame {
    // Only report feedback for the main frame.
    if (frame == [sender mainFrame]) {
        [self updateWindow];
    }
}

- (void)webView:(WebView *)sender didFinishLoadForFrame:(WebFrame *)frame {
	// only do this if the window is still open
	if(windowClosing == NO) {
		// Only report feedback for the main frame.
		if(frame == [sender mainFrame]) {
			//[self setFrameStatus: nil];
			[backButton setEnabled:[sender canGoBack]];
			[forwardButton setEnabled:[sender canGoForward]];
			
			module = [[SwordManager defaultManager] moduleForURL:url];
			MBLOGV(MBLOG_DEBUG, @"webView:(WebView *)sender didFinishLoadForFrame url=%@",url);
			
			if(isEditing) {
				[editView setURL:[self url]];
			}
			
			[self fixOptionsMenu];
			if(type == bible || type == commentary) {
				int	book, chapter;
				
				NSString *ref;
				
				if([url path] && [[url path] length] > 0) {
					if([[[url path] pathComponents] count] > 2) {
						ref = [[[url path] pathComponents] objectAtIndex:2];
					} else if([[[url path] pathComponents] count] > 1) {
						ref = [[[url path] pathComponents] objectAtIndex:1];
					} else {
						ref = @"";
					}
				} else {
					ref = @"";
				}
							
				if(!ref || ![ref length] || [ref compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame) {
					[bookCombo selectItemAtIndex:0];
					[chapterCombo selectItemAtIndex:0];
				} else {
					[(SwordBible *)module book:&book andChapter:&chapter fromReference:ref];

					if(book+1 >= 1 && book+1 < [bookCombo numberOfItems])
						[bookCombo selectItemAtIndex:book+1];
					if(chapter-1 >= 0 && chapter-1 < [chapterCombo numberOfItems])
						[chapterCombo selectItemAtIndex:chapter-1];
				}
			} else if(type == dictionary) {
				NSString *ref;
				
				if([url path] && [[url path] length] > 0)
					ref = [[url path] substringFromIndex:1];
				else
					ref = @"";
				
				if(!ref || ![ref length] || [ref compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame) {
					[dictCombo selectItemAtIndex:0];
				} else {
					[dictCombo selectItemWithObjectValue:[[(SwordDictionary*)module fullRefName:ref] capitalizedString]];
				}
			} else if(type == devotional) {
				NSScanner		*scan;
				int				day, month;
				
				if([url path] && [[url path] length] > 1) {
					scan = [NSScanner scannerWithString:[url path]];
					
					[scan scanString:@"/" intoString:NULL];
					[scan scanInt:&month];
					[scan scanString:@"." intoString:NULL];
					[scan scanInt:&day];
					
					[monthCombo selectItemAtIndex:month];
					[dayCombo selectItemAtIndex:day-1];
				} else {
					[monthCombo selectItemAtIndex:0];
					[dayCombo selectItemAtIndex:0];
				}
			} else if(type == book) {
				[contentsHandler selectItemForKey:[url path]];
			}
			
			[self updateWindow];
		}
		
		//show no tool-tip
		[ToolTip displayToolTip:NULL forObject:NULL atPoint:NSMakePoint(0,0) forWindow:NULL];
	}
}

- (void)webView:(WebView *)sender didFailProvisionalLoadWithError:(NSError *)error forFrame:(WebFrame *)frame
{
    MBLOGV(MBLOG_DEBUG, @"webView:didFailProvisionalLoadWithError: %@ where sender=%@", error, sender);

    // Only report feedback for the main frame.
    if (frame == [sender mainFrame]) {
		//[self setFrameStatus:[error description]];
		[self updateWindow];
    }
}

// WebUIDelegate Methods

- (WebView *)webView:(WebView *)sender createWebViewWithRequest:(NSURLRequest *)request {
    MBLOGV(MBLOG_DEBUG, @"webView:createWebViewWithRequest: where sender=%@", sender);

    //id myDocument = [[NSDocumentController sharedDocumentController] openUntitledDocumentOfType:@"DocumentType" display:YES];
    //[[[myDocument webView] mainFrame] loadRequest:request];
	id myDocument = [MyDocument openDocumentWithURL:[request URL]];
    
    return [myDocument webView];
}

- (void)webView:(WebView *)sender mouseDidMoveOverElement:(NSDictionary *)elementInformation modifierFlags:(id)modifierFlags {
	NSURL *aUrl;
	
	aUrl = [elementInformation objectForKey:WebElementLinkURLKey];
	
	if(aUrl && ![[aUrl query] isEqualToString:@"notip"]) {
		if([ToolTip willDisplayToolTipForObject:aUrl]) {

			NSMutableString	*urlString;
			NSPoint	position;
            position = [NSEvent mouseLocation];
            urlString = [NSMutableString stringWithString:[SwordServer generateTextForURL:aUrl]];
            
            if([urlString length] > 5) {
                NSRange newlineRange;
                
                //remove extra space before and after tip
                newlineRange = [urlString rangeOfCharacterFromSet:[NSCharacterSet whitespaceAndNewlineCharacterSet] options:0 range:NSMakeRange(0, 5)];
                if(newlineRange.location == 0) {
                    [urlString replaceCharactersInRange:newlineRange withString:@""];
                }
                
                newlineRange = [urlString rangeOfCharacterFromSet:[NSCharacterSet whitespaceAndNewlineCharacterSet] options:NSBackwardsSearch range:NSMakeRange([urlString length]-5, 5)];
                if(newlineRange.location+newlineRange.length == [urlString length]) {
                    [urlString replaceCharactersInRange:newlineRange withString:@""];
                }

                NSString *ref = [aUrl path];
                if([ref length] <= 1) return;
                
                ref = [ref substringFromIndex:1];
                if([urlString length] > 500)
                    urlString = [NSString stringWithFormat:@"%@...", [urlString substringToIndex:500]];
                
                urlString = [NSString stringWithFormat:@"%@\n%@",ref,urlString];
                
                [ToolTip displayToolTip:urlString forObject:aUrl atPoint:position forWindow:myWindow];                
            }
		}
	}
	else
		[ToolTip displayToolTip:NULL forObject:NULL atPoint:[NSEvent mouseLocation] forWindow:NULL];
}

- (void)setupTrackingRect {
	//setup tooltip tracking rect
	trackingTag = [webView addTrackingRect:[webView bounds] owner:self userData:NULL assumeInside:YES];
}

- (void)windowDidResize:(NSNotification *)aNotification {
	[webView removeTrackingRect:trackingTag];
	//setup tooltip tracking rect
	trackingTag = [webView addTrackingRect:[webView bounds] owner:self userData:NULL assumeInside:YES];
}

- (void)mouseEntered:(NSEvent *)theEvent {
}

- (void)mouseExited:(NSEvent *)theEvent {
	if(![webView mouse:[theEvent locationInWindow] inRect:[webView bounds]])
		[ToolTip displayToolTip:NULL forObject:NULL atPoint:[theEvent locationInWindow] forWindow:NULL];
}

- (NSArray *)webView:(WebView *)sender contextMenuItemsForElement:(NSDictionary *)element defaultMenuItems:(NSArray *)defaultMenuItems {
	return [NSArray arrayWithObjects:[defaultMenuItems objectAtIndex:0], [defaultMenuItems objectAtIndex:2], NULL];
}

#pragma mark -

// WebPolicyDelegate

- (void)webView:(WebView *)sender decidePolicyForNavigationAction:(NSDictionary *)actionInformation request:(NSURLRequest *)request frame:(WebFrame *)frame decisionListener:(id<WebPolicyDecisionListener>)listener {
    MBLOG(MBLOG_DEBUG, @"webView:decidePolicyForNavigationAction");
	if([[actionInformation objectForKey:WebActionModifierFlagsKey] intValue] & NSCommandKeyMask) {
		[listener ignore];
		
		[MyDocument openDocumentWithURL:[request URL]];
		
		return;
	}
	
	//if([[actionInformation objectForKey:WebActionNavigationTypeKey] intValue] == WebNavigationTypeLinkClicked)
	{
		ModuleType mType = [[[SwordManager defaultManager] moduleForURL:[request URL]] type];
		
		if(type != mType)
		if(!((type == bible || type == commentary) && (mType == bible || mType == commentary))) {
			[listener ignore];
			
			[MyDocument openDocumentWithURL:[request URL]];
			
			return;
		}
	}
	
	[listener use];
}

- (void)webView:(WebView *)sender decidePolicyForNewWindowAction:(NSDictionary *)actionInformation request:(NSURLRequest *)request newFrameName:(NSString *)frameName decisionListener:(id<WebPolicyDecisionListener>)listener {
    MBLOG(MBLOG_DEBUG, @"webView:decidePolicyForNewWindowAction");
	
	[MyDocument openDocumentWithURL:[request URL]];
		
	return;
}

@end
