/*	BookmarkHandler.m - Handles the bookmark outline view
 
 $Id: BookmarkHandler.m,v 1.5 2006/02/02 00:18:34 willthimbleby Exp $
 
 Copyright 2003 Will Thimbleby (will@thimbleby.net)
 
 This program is free software; you can redistribute it and/or modify it under the terms of the
 GNU General Public License as published by the Free Software Foundation version 2.
 
 This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
 even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
 */

#import "BookmarkHandler.h"
#import "AppDelegate.h"
#import "BookmarkOutlineView.h"
#import "NSOutlineView_Extensions.h"
#import "CocoLogger/CocoLogger.h"
#import "globals.h"

// table colomns
NSString *TableCol_Bookmark = @"bookmark";
NSString *TableCol_Reference = @"reference";
NSString *TableCol_Highlight = @"highlight";

int nodeSort(id node1, id node2, void *context) {
    int v1 = [(TreeNode*)node1 nodeParentIndex];
    int v2 = [(TreeNode*)node2 nodeParentIndex];
    if (v1 < v2)
        return NSOrderedAscending;
    else if (v1 > v2)
        return NSOrderedDescending;
    else
        return NSOrderedSame;
}

@implementation ColourTableColumn

- (void)awakeFromNib {
	blankCell = [[NSCell alloc] initTextCell:@""];
	[blankCell setEditable:NO];
	
	colorCell = [[NSPopUpButtonCell alloc] initTextCell:@"" pullsDown:NO];
	[colorCell addItemWithTitle:NSLocalizedString(@"None", @"Bookmark colour: None")];
	[colorCell addItemWithTitle:NSLocalizedString(@"Red", @"Bookmark colour: Red")];
	[colorCell addItemWithTitle:NSLocalizedString(@"Orange", @"Bookmark colour: Orange")];
	[colorCell addItemWithTitle:NSLocalizedString(@"Yellow", @"Bookmark colour: Yellow")];
	[colorCell addItemWithTitle:NSLocalizedString(@"Green", @"Bookmark colour: Green")];
	[colorCell addItemWithTitle:NSLocalizedString(@"Blue", @"Bookmark colour: Blue")];
	[colorCell addItemWithTitle:NSLocalizedString(@"Magenta", @"Bookmark colour: Magenta")];
	[colorCell addItemWithTitle:NSLocalizedString(@"Underline", @"Bookmark colour: Underline")];
	[colorCell setBordered:NO];
	[colorCell setControlSize:NSSmallControlSize];
	
	for(int i = 0;i < 8;i++) {
		[[colorCell itemAtIndex:i] setImage:[self swatchForColor:i]];
	}
}

- (void)dealloc {
	if(colorCell) [colorCell dealloc];
	[super dealloc];
}

- (NSImage *)swatchForColor:(int)c {
	NSImage *image = [[NSImage alloc] initWithSize:NSMakeSize(10,10)];
	
	[image lockFocus];
	
	NSRect r = NSMakeRect(0,0,10,10);
	switch(c) {
		case 0: [[NSColor whiteColor] drawSwatchInRect:r]; break;
		case 1: [[NSColor redColor] drawSwatchInRect:r]; break;
		case 2: [[NSColor orangeColor] drawSwatchInRect:r]; break;
		case 3: [[NSColor yellowColor] drawSwatchInRect:r]; break;
		case 4: [[NSColor greenColor] drawSwatchInRect:r]; break;
		case 5: [[NSColor blueColor] drawSwatchInRect:r]; break;
		case 6: [[NSColor magentaColor] drawSwatchInRect:r]; break;
		case 7: [[NSColor grayColor] drawSwatchInRect:r]; break;
	}
	[[NSColor darkGrayColor] set];
	NSFrameRect(r);
	[image unlockFocus];
	
	return image;
}

- (id)dataCellForRow:(int)row {
	id item = [(NSOutlineView*)[self tableView] itemAtRow:row];
	
	if(!item || [(NSOutlineView*)[self tableView] isExpandable:item])
		return blankCell;
    
    return colorCell;
}

@end

@implementation BookmarkHandler

+ (NSString *)allBookmarks:(int)colour {
    NSArray *bmarks = [[NSDictionary dictionaryWithContentsOfFile:DEFAULT_BOOKMARK_PATH] objectForKey:@"Bookmarks"];
	
	return [BookmarkHandler _allBookmarks:bmarks :colour];
}

+ (NSString*) _allBookmarks:(NSArray*)array :(int)colour {
	NSEnumerator *entryEnum = [array objectEnumerator];
	id entry;
	NSMutableString *bookmarkString = [NSMutableString string];
	
	[entryEnum nextObject];
	
	while((entry = [entryEnum nextObject])) {
		if ([entry isKindOfClass: [NSArray class]]) {
			[bookmarkString appendString: [BookmarkHandler _allBookmarks: entry :colour] ];
        } else if([entry objectForKey:@"colour"] && [[entry objectForKey:@"colour"] intValue] == colour) {
			[bookmarkString appendString:[entry objectForKey:@"reference"]];
			[bookmarkString appendString:@";"];
		}
	}
	
	return bookmarkString;
}

- (id)init {
    
    NSArray *data = nil;
    
    NSFileManager *fm = [NSFileManager defaultManager];
    if([fm fileExistsAtPath:DEFAULT_BOOKMARK_PATH] == YES) {
        // load from here
        NSDictionary *dict = [NSDictionary dictionaryWithContentsOfFile:DEFAULT_BOOKMARK_PATH];
        data = [dict objectForKey:@"Bookmarks"];
        treeData = [[BookmarkTreeNode treeFromArray:data] retain];
    } else {
        // for version 1.4.0 and above, get the bookmarks out of user defaults and save them seperately
        data = [[NSUserDefaults standardUserDefaults] arrayForKey:@"Bookmarks"];
        treeData = [[BookmarkTreeNode treeFromArray:data] retain];
        // save
        [self save];
    }
    
	[[NSNotificationCenter defaultCenter] addObserver:self selector:@selector(update:) name:@"update bookmarks" object:nil];
	
	return self;
}

- (void)dealloc
{
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	[treeData release];
	//[draggedNodes release];
	
	[super dealloc];
}

- (void)awakeFromNib
{
	[outlineView registerForDraggedTypes:[NSArray arrayWithObjects:NSURLPboardType, DragDropSimplePboardType, DragDropVerseListPboardType, nil]];
	
	[outlineView setTarget:self];
	[outlineView setDoubleAction:@selector(doubleClick:)];
	[outlineView setAutoresizesOutlineColumn:NO];
	
	[actionButton setAlternateImage:[NSImage imageNamed:@"Action_Pressed"]];
	[actionButton setButtonType:NSMomentaryChangeButton];
	[[actionButton cell] setMenu:bookmarkMenu];
	[[actionButton cell] setImagePosition:NSImageOnly];
	[[actionButton cell] setArrowPosition:NSPopUpNoArrow];
	[ [[[actionButton cell] menu] itemAtIndex:0] setImage:[NSImage imageNamed:@"Action"]];
	[actionButton setAutoenablesItems:YES];
}

- (NSArray*)draggedNodes   { return draggedNodes; }
- (NSArray*)selectedNodes  { return [outlineView allSelectedItems]; }

- (id)treeData {
	return treeData;
}

- (void)save {
	[outlineView deselectAll:nil];
	[outlineView reloadData];
	
    NSMutableDictionary *bookm = [NSMutableDictionary dictionary];
	[bookm setObject:[(BookmarkTreeNode *)treeData arrayFromTree] forKey:@"Bookmarks"];
    [bookm writeToFile:DEFAULT_BOOKMARK_PATH atomically:YES];
    
	[[NSNotificationCenter defaultCenter] postNotification:[NSNotification notificationWithName:@"update bookmarks" object:self]];
}

- (void)update:(id)sender {
	if([sender object] != self) {
		NSArray *data = [[NSDictionary dictionaryWithContentsOfFile:DEFAULT_BOOKMARK_PATH] arrayForKey:@"Bookmarks"];
		
		[treeData release];
		treeData = [[BookmarkTreeNode treeFromArray: data] retain];
		
		[outlineView deselectAll:nil];
		[outlineView reloadData];
	}
}

/**
 \brief double click on a bookmark item
 */
- (IBAction)doubleClick:(id)sender
{
	NSArray *refs;
	NSEnumerator *refEnum;
	id ref;
    
	refs = [outlineView allSelectedItems];
	refEnum = [refs objectEnumerator];
	
	// go through all selected tableview items
	while((ref = [refEnum nextObject]))
	{
		// get reference
		BookmarkNodeData *bNodeData = (BookmarkNodeData *)[ref nodeData];
		NSString *ref = [bNodeData reference];
		if(ref != nil)
		{
			NSString *swordRef;
			
			swordRef = ref;
			MBLOGV(MBLOG_DEBUG, @"bookmark double click, swordref: %@", swordRef);
            
			// add protocol prefix
			if(![swordRef hasPrefix:@"sword://"])
			{
				swordRef = [NSString stringWithFormat:@"/%@", swordRef];
			}
			[theDocument openURLString:swordRef];
		}
	}
}

- (IBAction)addGroup:(id)sender
{
	[self _addNewDataToSelection: [BookmarkTreeNode treeNodeWithData: [BookmarkNodeData groupDataWithName:NSLocalizedString(@"New Group", @"Bookmarks default group name")]]];
	[self save];
}

- (IBAction)addBookmark:(id)sender
{
	NSURL *url;
	NSString *bookmarkURL;
	url = [theDocument url];
	if([[url path] length] > 0)
	{
		bookmarkURL = [[url path] substringFromIndex:1];
	}
	else
	{
		bookmarkURL = @"";
	}
	
	[self _addNewDataToSelection:[BookmarkTreeNode treeNodeWithData:
                                  [BookmarkNodeData leafDataWithName:bookmarkURL forReference:bookmarkURL withColour:NULL]]];
	[self save];
}

- (IBAction)deleteBookmark:(id)sender
{
	[self deleteItems:[outlineView allSelectedItems]];
}

- (IBAction)editBookmark:(id)sender
{
	[outlineView editBookmark];
}

- (IBAction)editReference:(id)sender
{
	[outlineView editReference];
}

- (void)addItems:(NSArray*)items
{
	[[theDocument undoManager] registerUndoWithTarget:self
                                             selector:@selector(deleteItems:)
                                               object:items];
    
	[items sortedArrayUsingFunction:nodeSort context:NULL];
	[items makeObjectsPerformSelector: @selector(addToParent)];
	[self save];
}

- (void)deleteItems:(NSArray*)items
{
	[[theDocument undoManager] registerUndoWithTarget:self
                                             selector:@selector(addItems:)
                                               object:items];
    
	[items makeObjectsPerformSelector: @selector(removeFromParent)];
	[self save];
}

- (void) _addNewDataToSelection:(BookmarkTreeNode *)newChild
{
	int					childIndex = 0, newRow = 0;
	NSArray				*selectedNodes = [self selectedNodes];
	BookmarkTreeNode	*selectedNode = ([selectedNodes count] ? [selectedNodes objectAtIndex:0] : treeData);
	BookmarkTreeNode	*parentNode = nil;
	
	if ([(BookmarkNodeData*)[selectedNode nodeData] isGroup])
	{ 
		parentNode = selectedNode; 
		childIndex = 0; 
	}
	else
	{ 
		parentNode = (BookmarkTreeNode*)[selectedNode nodeParent]; 
		childIndex = [parentNode indexOfChildIdenticalTo:selectedNode]+1; 
	}
	
	[parentNode insertChild: newChild atIndex: childIndex];
	[outlineView reloadData];
	
	newRow = [outlineView rowForItem: newChild];
	if (newRow>=0) [outlineView selectRow: newRow byExtendingSelection: NO];
	if (newRow>=0) [outlineView editColumn:0 row:newRow withEvent:nil select:YES];
}

#pragma mark -

- (id)outlineView:(NSOutlineView *)olv child:(int)index ofItem:(id)item
{
	if(!item)
		return [treeData childAtIndex:index];
	
    return [item childAtIndex:index];
}

- (BOOL)outlineView:(NSOutlineView *)olv isItemExpandable:(id)item
{
    return [(BookmarkNodeData*)[item nodeData] isGroup];
}

- (int)outlineView:(NSOutlineView *)olv numberOfChildrenOfItem:(id)item
{
	if(!item)
		return [treeData numberOfChildren];
    
	return [item numberOfChildren];
}

- (id)outlineView:(NSOutlineView *)olv objectValueForTableColumn:(NSTableColumn *)tableColumn byItem:(id)item
{
    id objectValue = nil;
    
    // The return value from this method is used to configure the state of the items cell via setObjectValue:
    if([[tableColumn identifier] isEqualToString: @"bookmark"])
	{
		objectValue = [(BookmarkNodeData*)[item nodeData] name];
    }
	else if([[tableColumn identifier] isEqualToString: @"reference"])
	{
		objectValue = [(BookmarkNodeData*)[item nodeData] reference];
    }
	else if([[tableColumn identifier] isEqualToString: @"highlight"])
	{
		objectValue = [(BookmarkNodeData*)[item nodeData] colour];
    }
	//else if([[tableColumn identifier] isEqualToString: COLUMNID_IS_EXPANDABLE] && [NODE_DATA(item) isGroup])
	//{
	// Here, object value will be used to set the state of a check box.
	/*objectValue = [NSNumber numberWithBool: [NODE_DATA(item) isExpandable]];
     } else if([[tableColumn identifier] isEqualToString: COLUMNID_NODE_KIND]) {
     objectValue = ([NODE_DATA(item) isLeaf] ? @"Leaf" : @"Group");
     }*/
    
    return objectValue;
}

- (void)setName:(NSArray*)data
{
	[[theDocument undoManager] registerUndoWithTarget:self
                                             selector:@selector(setName:)
                                               object:[NSArray arrayWithObjects:[data objectAtIndex:0], [[data objectAtIndex:0] name], NULL]];
	[(BookmarkNodeData*)[data objectAtIndex:0] setName:[data objectAtIndex:1]];
	[self save];
}

/**
 \brief sets the reference the user gave
 @param[in] data an NSArray with the item nodeData and the reference string in it
 */
- (void)setReference:(NSArray*)data
{
	// use undo manager
	[[theDocument undoManager] registerUndoWithTarget:self
                                             selector:@selector(setReference:)
                                               object:[NSArray arrayWithObjects:[data objectAtIndex:0], [[data objectAtIndex:0] reference], NULL]];
	
	BookmarkNodeData *bNodeData = [data objectAtIndex:0];
	[bNodeData setReference:[data objectAtIndex:1]];
	[self save];
}

// Optional method: needed to allow editing.
- (void)outlineView:(NSOutlineView *)olv setObjectValue:(id)object forTableColumn:(NSTableColumn *)tableColumn byItem:(id)item
{
    if([[tableColumn identifier] isEqualToString:TableCol_Bookmark])
	{
		/*[[theDocument undoManager] registerUndoWithTarget:self
         selector:@selector(setName:)
         object:[NSArray arrayWithObjects:[item nodeData], [[item nodeData] name], NULL]];
         [[item nodeData] setName: object];
         */
        
		// object should be string
		NSString *bName = (NSString *)object;
		[self setName:[NSArray arrayWithObjects:[item nodeData], bName, NULL]];
    }
	else if([[tableColumn identifier] isEqualToString:TableCol_Reference])
	{
		/*[[theDocument undoManager] registerUndoWithTarget:self
         selector:@selector(setReference:)
         object:[NSArray arrayWithObjects:[item nodeData], [[item nodeData] reference], NULL]];
         [[item nodeData] setReference: object];*/
		
		NSString *bRef = (NSString *)object;
		[self setReference: [NSArray arrayWithObjects:[item nodeData], bRef, NULL]];
    }
	else if([[tableColumn identifier] isEqualToString:TableCol_Highlight])
	{
		[(BookmarkNodeData*)[item nodeData] setColour:[object intValue]];
		[self save];
    }
	
	/*else if ([[tableColumn identifier] isEqualToString: COLUMNID_IS_EXPANDABLE]) {
     [NODE_DATA(item) setIsExpandable: [object boolValue]];
     if (![NODE_DATA(item) isExpandable] && [outlineView isItemExpanded: item]) [outlineView collapseItem: item];
     } else if([[tableColumn identifier] isEqualToString: COLUMNID_NODE_KIND]) {
     // We don't allow editing of this column, so we should never actually get here.
     }*/
}

// ================================================================
//  NSOutlineView data source methods. (dragging related)
// ================================================================

- (BOOL)outlineView:(NSOutlineView *)olv writeItems:(NSArray*)items toPasteboard:(NSPasteboard*)pboard
{
    draggedNodes = items; // Don't retain since this is just holding temporaral drag information, and it is only used during a drag!  We could put this in the pboard actually.
    
    // Provide data for our custom type, and simple NSStrings.
    [pboard declareTypes:[NSArray arrayWithObjects: NSURLPboardType, @"relative url", DragDropSimplePboardType, nil] owner:self];
    
    // the actual data doesn't matter since DragDropSimplePboardType drags aren't recognized by anyone but us!.
    [pboard setData:[NSData data] forType:DragDropSimplePboardType];
	[[NSURL URLWithString:[(BookmarkNodeData*)[[items objectAtIndex:0] nodeData] reference]] writeToPasteboard:pboard];
	
	
	//relative urls
	NSMutableString		*relativeURL = [NSMutableString string];
	NSEnumerator		*rowEnum;
	id					row;
	
	[relativeURL appendString:@"/"];
	rowEnum = [items objectEnumerator];
	while((row = [rowEnum nextObject]))
	{
		[relativeURL appendString:[(BookmarkNodeData*)[row nodeData] reference]];
		[relativeURL appendString:@";"];
	}
	
	[pboard setString:relativeURL forType:@"relative url"];
    
    return YES;
}

- (unsigned int)outlineView:(NSOutlineView*)olv validateDrop:(id <NSDraggingInfo>)info proposedItem:(id)item proposedChildIndex:(int)childIndex
{
	// This method validates whether or not the proposal is a valid one. Returns NO if the drop should not be allowed.
	BookmarkTreeNode	*targetNode = item;
	BOOL				targetNodeIsValid = YES;
	
	//MBLOG(MBLOG_ERR, @"%@,%@",[[info draggingPasteboard] types],item);
	
	BOOL isOnDropTypeProposal = childIndex==NSOutlineViewDropOnItemIndex;
	
	// Refuse if: dropping "on" the view itself unless we have no data in the view.
	if (targetNode==nil && childIndex==NSOutlineViewDropOnItemIndex && [treeData numberOfChildren]!=0) 
		targetNodeIsValid = NO;
	
	if (targetNode==nil && childIndex==NSOutlineViewDropOnItemIndex)// && [self allowOnDropOnLeaf]==NO)
		targetNodeIsValid = NO;
	
	// Refuse if: we are trying to do something which is not allowed as specified by the UI check boxes.
	if (([(BookmarkNodeData*)[targetNode nodeData] isGroup] && isOnDropTypeProposal==YES) ||
		(![(BookmarkNodeData*)[targetNode nodeData] isGroup] && isOnDropTypeProposal==YES))
        targetNodeIsValid = NO;
    
	// Check to make sure we don't allow a node to be inserted into one of its descendants!
	if (targetNodeIsValid && ([info draggingSource]==outlineView) && [[info draggingPasteboard] availableTypeFromArray:[NSArray arrayWithObject: DragDropSimplePboardType]] != nil)
	{
		NSArray *_draggedNodes = [[[info draggingSource] dataSource] draggedNodes];
		targetNodeIsValid = ![targetNode isDescendantOfNodeInArray: _draggedNodes];
	}
	
	// Set the item and child index in case we computed a retargeted one.
	[outlineView setDropItem:targetNode dropChildIndex:childIndex];
	
	return targetNodeIsValid ? NSDragOperationGeneric : NSDragOperationNone;
}

- (void)_performDropOperation:(id <NSDraggingInfo>)info onNode:(TreeNode*)parentNode atIndex:(int)childIndex
{
    // Helper method to insert dropped data into the model. 
    NSPasteboard		*pboard = [info draggingPasteboard];
    NSMutableArray		*itemsToSelect = nil;
    
    // Do the appropriate thing depending on wether the data is DragDropSimplePboardType or NSStringPboardType.
    if ([pboard availableTypeFromArray:[NSArray arrayWithObjects:DragDropSimplePboardType, nil]] != nil)
	{
        BookmarkHandler		*dragDataSource = [[info draggingSource] dataSource];
        NSArray				*_draggedNodes = [TreeNode minimumNodeCoverFromNodesInArray: [dragDataSource draggedNodes]];
        NSEnumerator		*draggedNodesEnum = [_draggedNodes objectEnumerator];
        BookmarkTreeNode	*_draggedNode = nil, *_draggedNodeParent = nil;
        
		itemsToSelect = [NSMutableArray arrayWithArray:[self selectedNodes]];
        
        while ((_draggedNode = [draggedNodesEnum nextObject]))
		{
            _draggedNodeParent = (BookmarkTreeNode*)[_draggedNode nodeParent];
            if (parentNode==_draggedNodeParent && [parentNode indexOfChild: _draggedNode]<childIndex) childIndex--;
            [_draggedNodeParent removeChild: _draggedNode];
        }
        [parentNode insertChildren: _draggedNodes atIndex: childIndex];
    }
    else if ([pboard availableTypeFromArray:[NSArray arrayWithObject: DragDropVerseListPboardType]])
	{
		NSArray				*stringArray = [[pboard stringForType: DragDropVerseListPboardType] componentsSeparatedByString:@"	"];
        NSString			*string;
		NSEnumerator		*stringEnum;
		
		stringEnum = [stringArray objectEnumerator];
		
		while((string = [stringEnum nextObject]))
		{
			BookmarkTreeNode	*newItem = [BookmarkTreeNode treeNodeWithData: [BookmarkNodeData leafDataWithName:string forReference:string withColour:NULL]];
			
			itemsToSelect = [NSMutableArray arrayWithObject: newItem];
			[parentNode insertChild: newItem atIndex:childIndex++];
		}
    }
    else if ([pboard availableTypeFromArray:[NSArray arrayWithObject: NSURLPboardType]])
	{
		NSString			*url = [[[NSURL URLFromPasteboard:pboard] path] substringFromIndex:1];
		BookmarkTreeNode	*newItem = [BookmarkTreeNode treeNodeWithData: [BookmarkNodeData leafDataWithName:url forReference:url withColour:NULL]];
        
		[parentNode insertChild: newItem atIndex:childIndex++];
    }
    
    [outlineView reloadData];
    [outlineView selectItems: itemsToSelect byExtendingSelection: NO];
	[self save];
}

- (BOOL)outlineView:(NSOutlineView*)olv acceptDrop:(id <NSDraggingInfo>)info item:(id)targetItem childIndex:(int)childIndex
{
	TreeNode * 		parentNode = nil;
	
	// Determine the parent to insert into and the child index to insert at.
	if (targetItem && ![(BookmarkNodeData*)[targetItem nodeData] isGroup])
	{
		parentNode = (BookmarkTreeNode*)[targetItem nodeParent];//(childIndex==NSOutlineViewDropOnItemIndex ? [targetItem nodeParent] : targetItem);
		childIndex = [[targetItem nodeParent] indexOfChild: targetItem]+1;//(childIndex==NSOutlineViewDropOnItemIndex ? [[targetItem nodeParent] indexOfChild: targetItem]+1 : 0);
		
		//if ([NODE_DATA(parentNode) isLeaf])
		//	[[parentNode nodeData] setIsLeaf:NO];
	}
	else
	{            
		if(!targetItem)
			parentNode = treeData;
		else
			parentNode = targetItem;
		
		childIndex = (childIndex==NSOutlineViewDropOnItemIndex?0:childIndex);
	}
	
	[self _performDropOperation:info onNode:parentNode atIndex:childIndex];
    
	return YES;
}

@end
