//
//  SwordSearching.m
//  MacSword
//
//  Created by William Thimbleby on Thu Mar 10 2005.
//  Copyright (c) 2005 __MyCompanyName__. All rights reserved.
//

#import "SwordSearching.h"
#import "PrefHandler.h"
#import "CocoLogger/CocoLogger.h"
#import "IndexingManager.h"
#import "Indexer.h"
#import "SearchResultEntry.h"
#import "utils.h"

NSString *MacSwordIndexVersion = @"2.3";

NSMutableDictionary *scopeFilters = NULL;

@implementation SwordModule(Searching)

- (void)stopSearch
{
	terminateSearch = YES;
}

- (void)stopIndex
{
	terminateIndex = YES;
}

- (NSString *)textForKey:(NSString *)key
{
	sword::SWKey textkey = toUTF8(key);
	module->setKey(textkey);
	
	char *ctxt = (char *)module->StripText();
	int clen = strlen(ctxt);
	if(clen > 3 && ctxt[clen-3] == -96)
	{
		ctxt[clen-3] = 0;
	}

	return fromUTF8(ctxt);
}

/**
 \brief creates index from a VerseKey of Verse number, Chapter number, Book number and Testament number
 */
NSString *indexForVerseKey(sword::VerseKey *vk)
{
	long keyindex = vk->Verse();
	keyindex += vk->Chapter() * 1000;
	keyindex += vk->Book() * 1000 * 1000;
	keyindex += vk->Testament() * 1000 * 1000 * 1000;
	
    NSString *indexStr = [NSString stringWithFormat:@"%010ld", keyindex];
    
	return indexStr;
}

/**
 \brief generates a path index for the given VerseKey
 */
+ (NSString *)indexOfVerseKey:(sword::VerseKey *)vk
{
    int testament = vk->Testament();
    int book = vk->Book();
    
    // testament 2 begins with book 1
    if(testament == 2)
    {
        book = book + 39;   // 39 is the last book in AT
    }
    
    NSString *index = [NSString stringWithFormat:@"%003i/%003i/%003i/%003i", 
        testament,
        book,
        vk->Chapter(),
        vk->Verse()];
    
    return index;
}

- (BOOL)hasIndex
{
	// get IndexingManager
	IndexingManager *im = [IndexingManager sharedManager]; 
    
	NSString *path = [im indexFolderPathForModuleName:[self moduleName]];
    BOOL isDir;
	
	if([[NSFileManager defaultManager] fileExistsAtPath:path isDirectory:&isDir] && isDir)
	{
		NSDictionary *d = [NSDictionary dictionaryWithContentsOfFile:[path stringByAppendingPathComponent:@"version.plist"]];
		
		if(!d)
		{
			return NO;
		}
		
		if([[d objectForKey:@"MacSword Index Version"] isEqualToString: MacSwordIndexVersion])
		{
			if([d objectForKey:@"Sword Module Version"] == NULL) // just in case
			{
				return YES;
			}
			if([[d objectForKey:@"Sword Module Version"] isEqualToString: [self getConfigEntry:@"Version"]])
			{
				return YES;
			}
		}
		
		//index out of date remove it
		[[NSFileManager defaultManager] removeFileAtPath:path handler:nil];
	}
    
	return NO;
}

- (void)indexContentsIntoIndex:(Indexer *)indexer withProgress:(NSProgressIndicator*)indexProgress
{

}

- (void)createIndex:(NSProgressIndicator*)indexProgress
{
	sword::SWKey *savekey = 0;
	sword::SWKey *searchkey = 0;
	sword::SWKey textkey;
	
	[moduleLock lock];
	
	MBLOG(MBLOG_DEBUG, @"start index");
	terminateIndex = NO;

	// turn all filters to default values
	/*sword::StringList filterSettings;
	for (sword::FilterList::iterator filter = module->optionFilters->begin(); filter != module->optionFilters->end(); filter++)
	{
		filterSettings.push_back((*filter)->getOptionValue());
		(*filter)->setOptionValue(*((*filter)->getOptionValues().begin()));
	}*/

	// save key information so as not to disrupt original
	// module position
	if (!module->getKey()->Persist()) {
		savekey = module->CreateKey();
		*savekey = *module->getKey();
	} else
		savekey = module->getKey();

	searchkey = (module->getKey()->Persist())? module->getKey()->clone() : 0;
	if (searchkey) {
		searchkey->Persist(1);
		module->setKey(*searchkey);
	}

	// position module at the beginning
	*module = sword::TOP;

	// iterate thru each entry in module	

	// get Indexer
    Indexer *indexer = [Indexer indexerWithModuleName:[self moduleName] moduleType:[SwordModule moduleTypeForModuleTypeString:[self moduleType]]];
	//Indexer *indexer = [IndexingManager indexerForModule:self];
    if(indexer == nil) {
        MBLOG(MBLOG_ERR, @"Could not create Indexer for this module!");
    } else {
        [self indexContentsIntoIndex:indexer withProgress:indexProgress];
        
        if(terminateIndex) {
            [indexer flushIndex];
            [indexer close];
            
            return;
        }
        
        [indexer flushIndex];
        [indexer close];

        // reposition module back to where it was before we were called
        module->setKey(*savekey);

        if (!savekey->Persist())
            delete savekey;

        if (searchkey)
            delete searchkey;

        
        [moduleLock unlock];

        // reset option filters back to original values
    /*	sword::StringList::iterator origVal = filterSettings.begin();
        for (FilterList::iterator filter = optionFilters->begin(); filter != optionFilters->end(); filter++) {
            (*filter)->setOptionValue(*origVal++);
        }*/
        MBLOG(MBLOG_DEBUG, @"end index");
        
        NSString *path = [(IndexingManager *)[IndexingManager sharedManager] indexFolderPathForModuleName:[self moduleName]];
        
        //save version info
        NSDictionary *d = [NSDictionary dictionaryWithObjectsAndKeys:MacSwordIndexVersion, @"MacSword Index Version", [self getConfigEntry:@"Version"], @"Sword Module Version", nil];
        [d writeToFile:[path stringByAppendingPathComponent:@"version.plist"] atomically:NO];
    }
}

@end

NSString* safeC_NR(const char *bytes) {
	NSString *r;
	if(bytes) {
		 r = [[NSString alloc] initWithCString:bytes];
	}
	
    if(r) {
		return r;
	}
	
	return [[NSString alloc] initWithString:@""];
}

NSString* safeUTF8_NR(const char *bytes) {
	NSString *r;
	if(bytes) {
		r = [[NSString alloc] initWithUTF8String:bytes];
	}
	
    if(r) {
		return r;
	}
	
	return safeC_NR(bytes);
}

@implementation SwordBible(Searching)

- (void)indexContentsIntoIndex:(Indexer *)indexer withProgress:(NSProgressIndicator*)indexProgress {
	char perc = 1;
	sword::VerseKey *vkcheck = 0;
	vkcheck = My_SWDYNAMIC_CAST(VerseKey, module->getKey());
	long highIndex = (vkcheck)? 32300:module->getKey()->Index();
	if(!highIndex) {
		highIndex = 1;		// avoid division by zero errors.
	}
	bool savePEA = module->isProcessEntryAttributes();
	module->processEntryAttributes(true);
	
	while(!module->Error()) {
		NSAutoreleasePool	*pool = [[NSAutoreleasePool alloc] init];
		
		long mindex = 0;
		if (vkcheck) {
			mindex = vkcheck->NewIndex();
		} else {
			mindex = module->getKey()->Index();
		}
		
		// compute percent complete so we can report to our progress callback
		float per = (float)mindex / highIndex;
		// between 5%-98%
		per *= 93; per += 5;
		char newperc = (char)per;
		if(newperc > perc) {
			perc = newperc;
			[indexProgress setDoubleValue:perc];
		}
		
		// get "content" field
		const char *content = module->StripText();
		if (content && *content) {
			// build "strong" field
			sword::SWBuf strong;
			
			sword::AttributeTypeList::iterator words;
			sword::AttributeList::iterator word;
			sword::AttributeValue::iterator strongVal;
			
			words = module->getEntryAttributes().find("Word");
			if (words != module->getEntryAttributes().end()) {
				for (word = words->second.begin();word != words->second.end(); word++) {
					strongVal = word->second.find("Lemma");
					if (strongVal != word->second.end()) {
						// cheeze.  skip empty article tags that weren't assigned to any text
						if (strongVal->second == "G3588") {
							if (word->second.find("Text") == word->second.end())
								continue;	// no text? let's skip
						}
						strong.append(strongVal->second);
						strong.append(' ');
					}
				}
			}
			
			sword::VerseKey *vk = (sword::VerseKey *)module->getKey();
			NSString *keyStr = safeUTF8_NR(vk->getText());
			NSString *contentStr = safeUTF8_NR(content);
			
            NSString *keyIndex = [SwordModule indexOfVerseKey:vk];
            
            NSMutableDictionary *propDict = [NSMutableDictionary dictionaryWithCapacity:2];
            // additionally save content and key string
            [propDict setObject:contentStr forKey:IndexPropSwordKeyContent];
            [propDict setObject:keyStr forKey:IndexPropSwordKeyString];
            
            NSMutableString *strongStr = [NSMutableString string];
			if(strong.length() > 0) {
                [strongStr appendString:safeUTF8_NR(strong.c_str())];
				[strongStr replaceOccurrencesOfString:@"|x-Strongs:" withString:@" " options:0 range:NSMakeRange(0, [strongStr length])];
                
                // also add to dictionary
                [propDict setObject:strongStr forKey:IndexPropSwordStrongString];
			}
			
            // index combined with strongs
            NSString *indexContent = [NSString stringWithFormat:@"%@ - %@", contentStr, strongStr];
                
            // add to index
            [indexer addDocument:keyIndex text:indexContent textType:ContentTextType storeDict:propDict];
			// release stuff
			[contentStr release];
            
            // release key string
			[keyStr release];
		}
		
		(*module)++;
		
		[pool release];
		
		if(terminateIndex) return;
	}
	
	module->processEntryAttributes(savePEA);	
}

@end

@implementation SwordDictionary(Searching)

- (void)indexContentsIntoIndex:(Indexer *)indexer withProgress:(NSProgressIndicator*)indexProgress {
    
	module->setSkipConsecutiveLinks(true);
	*module = sword::TOP;
	module->getRawEntry();
 
	int counter = 0;
    int max = [self entryCount];
    if(max == 0) {
        max = 1;    // avoid division by zero
    }
	
	if(module->isUnicode()) {
		while (!module->Error()) {
			const char *content = module->StripText();
            
            NSString *keyStr = fromUTF8(module->getKey()->getText());
            NSString *contentStr = fromUTF8(content);

            NSMutableDictionary *propDict = [NSMutableDictionary dictionaryWithCapacity:1];
            // additionally save content
            [propDict setObject:contentStr forKey:IndexPropSwordKeyContent];
            [propDict setObject:keyStr forKey:IndexPropSwordKeyString];
            
            // let's add the key also into the searchable content
            NSString *indexContent = [NSString stringWithFormat:@"%@ - %@", keyStr, contentStr];
			// add content
			[indexer addDocument:keyStr text:indexContent textType:ContentTextType storeDict:propDict];

			(*module)++;
			
			[indexProgress setDoubleValue:counter++/max];
			if(terminateIndex) return;
		}
	} else {
		while (!module->Error()) {
			const char *content = module->StripText();
            
            NSString *keyStr = fromLatin1(module->getKey()->getText());
            NSString *contentStr = fromLatin1(content);
            
            NSMutableDictionary *propDict = [NSMutableDictionary dictionaryWithCapacity:1];
            // additionally save content
            [propDict setObject:contentStr forKey:IndexPropSwordKeyContent];
            [propDict setObject:keyStr forKey:IndexPropSwordKeyString];

            // let's add the key also into the searchable content
            NSString *indexContent = [NSString stringWithFormat:@"%@ - %@", keyStr, contentStr];
            // add content
			[indexer addDocument:keyStr text:indexContent textType:ContentTextType storeDict:propDict];
			
			(*module)++;
			
			[indexProgress setDoubleValue:counter++/max];
			if(terminateIndex) return;
		}
	}
}

@end

@implementation SwordBook(Searching)

- (void)indexContentsIntoIndex:(Indexer *)indexer withProgress:(NSProgressIndicator*)indexProgress {
	sword::TreeKeyIdx *treeKey = dynamic_cast<sword::TreeKeyIdx*>((sword::SWKey*)*(module));
	[self indexContents:treeKey intoIndex:indexer];
}

- (id)indexContents:(sword::TreeKeyIdx *)treeKey intoIndex:(Indexer *)indexer {
	// we need to check for any Unicode names here
	char *treeNodeName = (char *)treeKey->getText();

	// use MSStringMgr to check for Unicode
	MSStringMgr *stringMgr = (MSStringMgr *)MSStringMgr::getSystemStringMgr();
	NSString *name = @"";
	bool isUnicode = stringMgr->isUtf8(treeNodeName);
	if(isUnicode) {
		name = fromUTF8(treeNodeName);
	} else {
		name = fromLatin1(treeNodeName);	
	}
	
	// this is for about key
	if([name length] == 0) {
		name = aboutEntrySpecifier;
	}
	
	if(treeKey->hasChildren()) {
		NSMutableArray *c = [NSMutableArray array];
		
		[c addObject:name];
		treeKey->firstChild();
		
		// look over keys
		do {
			*module = (long)treeKey;
			char *keyCString = (char *)module->getKey()->getText();
			const char *content = module->StripText();

            NSString *contentStr = @"";
            NSString *keyStr = @"";
            if(stringMgr->isUtf8(keyCString)) {
                keyStr = fromUTF8(keyCString);
                contentStr = fromUTF8(content);
            } else {
                keyStr = fromLatin1(keyCString);
                contentStr = fromLatin1(content);
			}
            
            NSMutableDictionary *propDict = [NSMutableDictionary dictionaryWithCapacity:2];
            // additionally save content
            [propDict setObject:contentStr forKey:IndexPropSwordKeyContent];
            [propDict setObject:keyStr forKey:IndexPropSwordKeyString];
            
            // let's add the key also into the searchable content
            NSString *indexContent = [NSString stringWithFormat:@"%@ - %@ - %@", keyStr, name, contentStr];

            // add content with key
            [indexer addDocument:keyStr text:indexContent textType:ContentTextType storeDict:propDict];

			[self indexContents:treeKey intoIndex:indexer];
		}
		while(treeKey->nextSibling());
		
		treeKey->parent();
		
		return c;
	}
	
	return name;
}

@end
