/*	SwordModule.mm - Sword API wrapper for Modules.

	$Id: SwordModule.mm,v 1.5 2006/02/28 10:27:22 willthimbleby Exp $

	Copyright 2003 Will Thimbleby (will@thimbleby.net)
	Based on code by Nathan Youngman (http://www.nathany.com)

	This program is free software; you can redistribute it and/or modify it under the terms of the
	GNU General Public License as published by the Free Software Foundation version 2.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
	even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
*/

#import "SwordModule.h"
#import "rtfhtml.h"
#import "utils.h"
#import "SwordManager.h"
#import "SwordURLProtocol.h"

NSString *aboutEntrySpecifier = NULL;

sword::ListKey *lastSearchResult = nil;

void handleSearch(char percent, void *userData)
{
	[(id)userData callback:percent];
}

@interface SwordModule (PrivateAPI)

- (NSString *)typeName;
- (NSString *)htmlForRef:(NSString *)ref;

@end

@implementation SwordModule (PrivateAPI)

// returns the type of the module this object is using
- (NSString *)typeName {
	NSString *stype;
	
	[moduleLock lock];
	
	if(!module->getConfig().has("Category", "Cults / Unorthodox / Questionable Material")) {
		stype = fromC(module->Type());
    } else {
		stype = [NSString stringWithFormat:@"%@ (%@)", fromC(module->Type()), NSLocalizedString(@"Unorthodox", @"Module List: Unorthodox")];
    }
    
	[moduleLock unlock];
	
	return stype;
}

- (NSString *)htmlForRef:(NSString *)ref {
    
    NSString *ret = @"";
    
    [moduleLock lock];
    if(module->isUnicode()) {
        module->SetKey(toUTF8(ref));
    } else {
        module->SetKey(toLatin1(ref));
    }
    
    char *bytes = (char *)module->RenderText();
    [moduleLock unlock];

    ret = [NSString stringWithUTF8String:bytes];
    
    return ret;
}

@end

@implementation SwordModule

+ (NSDictionary *)defaultDisplayOptions {
    NSMutableDictionary *ret = [NSMutableDictionary dictionary];
    
    NSUserDefaults *ud = [NSUserDefaults standardUserDefaults];
    NSString *vool = [ud stringForKey:@"default one-line verses"];
    BOOL voolVal = NO;
    if([vool isEqualToString:@"YES"]) {
        voolVal = YES;
    }
    [ret setObject:[NSNumber numberWithBool:voolVal] forKey:@"Verses On One Line"];
    
    return ret;
}

/**
 \brief maps type string to ModuleType enum
 @param[in] typeStr type String as in -moduleType(SwordModule)
 @return type according to ModuleType enum
 */
+ (ModuleType)moduleTypeForModuleTypeString:(NSString *)typeStr {
     ModuleType ret = bible;
    
    if(typeStr == nil)
    {
        MBLOG(MBLOG_ERR, @"have a nil typeStr!");
        return ret;
    }
    
    if([typeStr isEqualToString:@"Biblical Texts"])
    {
        ret = bible;
    }
    else if([typeStr isEqualToString:@"Commentaries"])
    {
        ret = commentary;
    }
    else if([typeStr isEqualToString:@"Lexicons / Dictionaries"])
    {
        ret = dictionary;
    }
    else if([typeStr isEqualToString:@"Generic Books"])
    {
        ret = book;
    }
    
    return ret;
}

// initalises the module from a manager
- (id)initWithSwordManager: (SwordManager *)theManager moduleName:(NSString*)name {
	if((self = [super init])) {
		[theManager retain];
		myManager = theManager;
		moduleName = [name retain];
		module = [myManager getSWModuleWithName:moduleName];
        [self setDisplayOptions:[SwordModule defaultDisplayOptions]];
		
        [self setTypeString:[self typeName]];

		if([self hasFeature:@"DailyDevotion"]) {
			type = devotional;
		} else {
            type = [SwordModule moduleTypeForModuleTypeString:[self moduleType]];
        }
		
		if(!aboutEntrySpecifier) {
			aboutEntrySpecifier =  NSLocalizedString(@"(About)", @"About entry for popup toolbar buttons");
		}
		
		moduleLock = [[NSRecursiveLock alloc] init];
	}
	
	return self;
}

/** init with given SWModule */
- (id)initWithSWModule:(sword::SWModule *)aModule {
    self = [super init];
    if(self) {
		
        // init lock
        moduleLock = [[NSRecursiveLock alloc] init];
        [self setDisplayOptions:[SwordModule defaultDisplayOptions]];
        
        // copy the module instance
        module = aModule;
        
        // set properties
        moduleName = [[NSString stringWithCString:module->Name() encoding:NSUTF8StringEncoding] retain];
        [self setTypeString:[self typeName]];
        
		if([self hasFeature:@"DailyDevotion"]) {
			type = devotional;
		} else {
            type = [SwordModule moduleTypeForModuleTypeString:typeString];
        }		
    }
    
    return self;
}

// releases strored information
- (void)dealloc {
	[moduleName release];
	[myManager release];
    [typeString release];
	[super dealloc];
}

- (NSLock *)moduleLock {
	return moduleLock;
}

// Sets the unlock key of the modules and writes the key into the pref file.
- (BOOL)unlock:(NSString *)unlockKey
{
	if (![self isEncrypted])
	{
		return NO;
	}

	//remember key in pref file
	{
		NSMutableDictionary		*keys;
	
		keys = [NSMutableDictionary dictionaryWithDictionary:[[NSUserDefaults standardUserDefaults] objectForKey:@"cipher keys"]];
		[keys setObject:unlockKey forKey:moduleName];
		[[NSUserDefaults standardUserDefaults] setObject:keys forKey:@"cipher keys"];
	}
		
	[myManager setCipherKey:unlockKey forModuleNamed:moduleName];
	  
	return YES;
}

// This function returns true if this module is locked, otherwise return false.
- (BOOL)isLocked {
	//still works, but the cipherkey is stored in prefs.
	//Works because it is set in sword on program startup.
	if ([self isEncrypted] && [[self getConfigEntry:@"CipherKey"] length] == 0)
		return YES;
	return NO;
}

// This functions returns true if this module is encrypted (locked or unlocked).
- (BOOL)isEncrypted {
	/**
	* If we have the CipherKey entry the module
	* is encrypted but not necessarily locked
	*/
	return ([self getConfigEntry:@"CipherKey"] != nil);
}

/** returnes the description of the module */
- (NSString *)moduleDescription {
    return [myManager descriptionForModuleName:moduleName];
}

/** returns version number of module */
- (NSString *)versionNumber {
    return [myManager versionForModuleName:moduleName];
}

/** returns about string of module */
- (NSString *)aboutString {
    return [self getConfigEntry:@"About"];
}

/** read config entry for encoding */
- (BOOL)isUnicode {    
    return module->isUnicode();
}

// returns the specified config entry
- (NSString *)getConfigEntry:(NSString *)config {
	[moduleLock lock];

	NSString *result = NULL;
	
	if(module->getConfigEntry([config cString]))
		result = fromUTF8(module->getConfigEntry([config cString]));
	
	[moduleLock unlock];
	
	return result;
}

// returns the type of the module this object is using
- (ModuleType)type {
	return type;
}
// returns the type of the module this object is using
- (NSString *)moduleType {
	NSString	*stype;
	
	[moduleLock lock];
	
	if(!module->getConfig().has("Category", "Cults / Unorthodox / Questionable Material"))
		stype = [NSString stringWithCString:module->Type()];
	else
		stype = [NSString stringWithFormat:@"%@ (%@)", [NSString stringWithCString:module->Type()], NSLocalizedString(@"Unorthodox", @"Module List: Unorthodox")];

	[moduleLock unlock];
	
	return stype;
}

// Returns the name of the module this object is using
- (NSString *)moduleName {
	return moduleName;
}

- (NSString *)name {
	return moduleName;
}

- (int)status {
    return status;
}

- (void)setStatus:(int)value {
    status = value;
}

- (NSDictionary *)displayOptions {
    return [NSDictionary dictionaryWithDictionary:displayOptions];
}

- (void)setDisplayOptions:(NSDictionary *)value {
    [displayOptions release];
    displayOptions = [[NSMutableDictionary dictionaryWithDictionary:value] retain];
}

- (sword::SWModule *)swmodule {
	return module;
}

- (NSString *)typeString {
    return typeString;
}

- (void)setTypeString:(NSString *)value {
    [value retain];
    [typeString release];
    typeString = value;    
}

// general feature access
- (BOOL) hasFeature:(NSString *)feature
{
	BOOL	has = NO;
	
	[moduleLock lock];
	
	if(module->getConfig().has("Feature", [feature cStringUsingEncoding:NSUTF8StringEncoding]))
		has = YES;
	else if (module->getConfig().has("GlobalOptionFilter",[[NSString stringWithFormat:@"GBF%@", feature] cString]))
 		has = YES;
 	else if (module->getConfig().has("GlobalOptionFilter",[[NSString stringWithFormat:@"ThML%@", feature] cString]))
 		has = YES;
 	else if (module->getConfig().has("GlobalOptionFilter",[[NSString stringWithFormat:@"UTF8%@", feature] cString]))
 		has = YES;
 	else if (module->getConfig().has("GlobalOptionFilter",[[NSString stringWithFormat:@"OSIS%@", feature] cString]))
 		has = YES;		
 	else if (module->getConfig().has("GlobalOptionFilter",[feature cStringUsingEncoding:NSUTF8StringEncoding]))
 		has = YES;
	
	[moduleLock unlock];
	
	return has;
}

/** numnber of entries
 abstract method, should be overriden by subclasses
 */
- (long)entryCount {
    return 0;
}

- (id)safeUTF8:(const char *)bytes {
	if(bytes)
		return [NSString stringWithUTF8String:bytes];
	return NULL;
}

- (NSString *)htmlForDescription {
	[moduleLock lock];
	
	NSString			*html, *cryptString;
	sword::RTFHTML		filter;
    
    NSString *aboutStr = @"";
    const char *aboutCStr = module->getConfigEntry("About");
    if(aboutCStr != NULL) {
        if([self isUnicode]) {
            aboutStr = [NSString stringWithUTF8String:aboutCStr];
        } else {
            aboutStr = [NSString stringWithCString:aboutCStr encoding:NSISOLatin1StringEncoding];
        }        
    }
    
	sword::SWBuf buf([aboutStr UTF8String]);
	filter.processText(buf, 0, 0);
	
	if([self isLocked]) {
		NSString *warnPath = [[NSBundle mainBundle] pathForResource:@"warn" ofType:@"tiff"];
		cryptString = [NSString stringWithFormat:@"<div style=\"vertical-align:middle\"><img width=32 height=32 src=\"file://%@\"><form action=\"enterkey\"> (Locked) Unlock module to use. <input type=submit value=\"Enter Key\"></form></div>",warnPath];
	}
	else if([self isEncrypted])
		cryptString = @"<br>(Encrypted)";
	else
		cryptString = @"";
		
	html = [NSString stringWithFormat:
		[NSString stringWithContentsOfFile: [[NSBundle mainBundle] pathForResource:@"about_template" ofType:@"html"]],
		[self moduleName],
		[myManager descriptionForModuleName:moduleName],
		[self moduleType],
		cryptString,
		[myManager languageAbbrevForModuleName:moduleName],
		[myManager versionForModuleName:moduleName],
		[self safeUTF8:buf.c_str()]];
	
	[moduleLock unlock];
	
	return html;
}

- (NSString *)entryForRef:(NSString *)reference {
	return [self htmlForRef:reference];
}

- (NSString *)htmlForRef:(NSString *)reference {
	return [self htmlForRef:reference embelish:YES];
}

- (NSString *)htmlForRef:(NSString *)reference highlightReference:(NSString *)hr {
	return [self htmlForRef:reference];
}

- (NSString *)htmlForRef:(NSString *)reference highlightReference:(NSString *)hr protocol:(NSURLProtocol*)protocol {
	return [self htmlForRef:reference embelish:YES protocol:protocol];
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish {
	if([self isLocked] || !reference || ![reference length] || [reference compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame) {
		return [self htmlForDescription];
	} else {
        return [self htmlForRef:reference];
	}
}

- (void)htmlForRef:(NSString *)reference embelish:(BOOL)embelish protocol:(SwordURLProtocol *)protocol {
	if([self isLocked] || !reference || ![reference length] || [reference compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame) {
		[protocol recieveData:[[self htmlForDescription] dataUsingEncoding: NSUTF8StringEncoding]];
	} else {
        NSString *htmlStr = [self htmlForRef:reference];
		[protocol recieveData:[htmlStr dataUsingEncoding:NSUTF8StringEncoding]];
    }
}

- (void)writeEntry:(NSString *)value forRef:(NSString *)reference {
}

- (NSString *)description {
	return [NSString stringWithFormat:@"SwordModule: %@", moduleName];
}

@end
