/*	SwordManager.mm - Sword API wrapper for Modules.

	$Id: SwordManager.mm,v 1.4 2006/02/02 00:20:07 willthimbleby Exp $
	
	Copyright 2003 Will Thimbleby (will@thimbleby.net)
	Based on code by Nathan Youngman (http://www.nathany.com)

	This program is free software; you can redistribute it and/or modify it under the terms of the
	GNU General Public License as published by the Free Software Foundation version 2.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
	even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
*/

#import "SwordManager.h"
#include <string>
#include <list>

#include "bt_osishtml.h"
#include "bt_thmlhtml.h"
#include "bt_gbfhtml.h"
#include "gbfplain.h"
#include "thmlplain.h"
#include "osisplain.h"
#include "msstringmgr.h"
#import "CocoLogger/CocoLogger.h"
#import "IndexingManager.h"
#import "globals.h"
#import "utils.h"
#import "SwordBook.h"
#import "SwordModule.h"
#import "SwordBible.h"
#import "SwordCommentary.h"
#import "SwordDictionary.h"

using std::string;
using std::list;

static SwordManager *defaultManager = 0L;
 
@interface SwordManager (PrivateAPI)

- (void)setManagerLock:(NSLock *)value;

@end

@implementation SwordManager (PrivateAPI)

- (void)setManagerLock:(NSLock *)value {
    [value retain];
    [managerLock release];
    managerLock = value;
}

@end

@implementation SwordManager

/*" Sword API wrapper for module manager. "*/

/*" Initializes Sword Manager with the path to the folder that contains the mods.d, modules and
(optionally) locals.d folders.
"*/
- (id)initWithPath:(NSString *)path {
	NSEnumerator	*enumerator;
	NSArray			*subDirs;
	NSString		*subDir, *fullSubDir;
	BOOL			directory;
	NSFileManager	*filemanager = [NSFileManager defaultManager];
	
	if((self = [super init])) {
        // this is our main manager
        temporaryManager = NO;
        
        [self setModulePath:path];

		subDirs = [filemanager directoryContentsAtPath:path];
		enumerator = [subDirs objectEnumerator];
	
		// create a manager if there is a module here
		if([subDirs containsObject:@"mods.d"])
			manager = new sword::SWMgr( toUTF8(path) , true, new sword::EncodingFilterMgr(sword::ENC_UTF8));
		else
			manager = NULL;
		
		// for all sub directories add module
		while((subDir = [enumerator nextObject])) {
			// as long as it's not hidden
			if(![subDir hasPrefix:@"."])
			{
				fullSubDir = [path stringByAppendingPathComponent:subDir];
				fullSubDir = [fullSubDir stringByStandardizingPath];
				
				//if its a directory
				if([filemanager fileExistsAtPath:fullSubDir isDirectory:&directory])
				if(directory)
				{
					// if manager has not been created yet then do so - otherwise add path to manager
					if(!manager) {
						manager = new sword::SWMgr(toUTF8(fullSubDir) , true, new sword::EncodingFilterMgr(sword::ENC_UTF8));
                    } else {
						manager->augmentModules(toUTF8(fullSubDir));
                    }
				}
			}
		}
		
		// if manager has not been created yet then do so - otherwise add path to manager
		if(!manager) {
			manager = new sword::SWMgr( toUTF8(path) , true, new sword::EncodingFilterMgr(sword::ENC_UTF8));
        }
        
        // also add the executing path to the path of modules
        NSString *bundlePath = [[NSBundle mainBundle] bundlePath];
        if(bundlePath && manager) {
            // remove last path component
            NSString *appPath = [bundlePath stringByDeletingLastPathComponent];
            manager->augmentModules(toUTF8(appPath));
        }
		
		//string manager
		StringMgr::setSystemStringMgr(new MSStringMgr());
		
		// set locale manager
        NSString *resourcePath = [[NSBundle mainBundle] resourcePath];
        NSString *localePath = [resourcePath stringByAppendingPathComponent:@"locales.d"];
		sword::LocaleMgr *lManager = sword::LocaleMgr::getSystemLocaleMgr();
        lManager->loadConfigDir([localePath UTF8String]);
        
		//get the language
		NSEnumerator *langIter = [[[NSUserDefaults standardUserDefaults] objectForKey:@"AppleLanguages"] objectEnumerator];

		NSString *lang = nil;
		BOOL haveLocale = NO;
        int i = 0;
		// for every language, check if we know the locales
		while((lang = [langIter nextObject])) {
            // if first language is english, then go no further
            if([[lang substringToIndex:2] isEqualToString:@"en"] && (i == 0)) {
                haveLocale = YES;
                break;
            }
            
            // set path for locale
            NSString *filePath = [NSString stringWithFormat:@"%@/locales.d/%@-utf8.conf", resourcePath, lang];
            haveLocale = [filemanager fileExistsAtPath:filePath isDirectory:&directory];
            // do we have this locale?
            if(haveLocale == NO) {
                filePath = [NSString stringWithFormat:@"%@/locales.d/%@.conf", resourcePath, lang];
                haveLocale = [filemanager fileExistsAtPath:filePath isDirectory:&directory];
            } else {
                // if we have the locale, we can break up at once
                break;
            }
            // now check for locale file for only 2 characters
            if(haveLocale == NO) {
                filePath = [NSString stringWithFormat:@"%@/locales.d/%@.conf", resourcePath, [lang substringToIndex:2]];
                haveLocale = [filemanager fileExistsAtPath:filePath isDirectory:&directory];
            } else {
                break;
            }

            i++;
		}

		// if still haveLocale is still NO, we have a problem
		// use english for testing
		if(haveLocale == NO) {
			lang = @"en";
		}
		
		// set the locale
		lManager->setDefaultLocaleName([lang cStringUsingEncoding:NSUTF8StringEncoding]);			
		
		[self setModuleData:[NSDictionary dictionary]];
		[self setModuleTypes:[NSDictionary dictionary]];
		[self setModules:[NSDictionary dictionary]];
		managerLock = [[NSRecursiveLock alloc] init];

		[self refreshModuleData];
		[self addRenderFilters];
		
		modules = [[NSMutableDictionary dictionary] retain];

		// init indexingmanager, set base index path
        IndexingManager *im = [IndexingManager sharedManager];
        NSString *path = [NSString localizedStringWithFormat:@"%@/Library/Application Support/MacSword", NSHomeDirectory()];
        [im setBaseIndexPath:path];        
	}
	
    sword::StringList options = manager->getGlobalOptions();
    sword::StringList::iterator	it;
    
    for(it = options.begin(); it != options.end(); it++)
    {
        [self setGlobalOption:[NSString stringWithCString:it->c_str()] value:@"Off"];
        //MBLOGV(MBLOG_DEBUG, @"Option: %@", [NSString stringWithUTF8String:it->c_str()]);
    }
	
	return self;
}

/** initialize a new SwordManager */
- (id)initWithSWMgr:(sword::SWMgr *)smgr {
    MBLOG(MBLOG_DEBUG, @"[SwordManager -initWithSWMgr:]");
    
    self = [super init];
    if(self) {
        manager = smgr;
        // this is a temporary manager
        temporaryManager = YES;
        [self setModuleData:[NSDictionary dictionary]];
		[self setModules:[NSDictionary dictionary]];
		[self setModuleTypes:[NSDictionary dictionary]];
        managerLock = [[NSRecursiveLock alloc] init];
        
		[self refreshModuleData];
		[self addRenderFilters];
    }
    
    return self;
}

/** reinit the manager */
- (void)reInit {
    MBLOG(MBLOG_DEBUG, @"[SwordManager -reInit]");
    
    if(modulePath && [modulePath length] > 0) {
        manager = (sword::SWMgr *)new sword::SWMgr(toUTF8(modulePath), true, NULL, false, true);    
        
        // clear some data
        [self refreshModuleData];
        [self addRenderFilters];
        [modules removeAllObjects];
        
        // send notification about module change
        SendNotifyModulesChanged(nil);
    }
}

- (NSDictionary *)modules {
    return [NSDictionary dictionaryWithDictionary:modules];
}

- (void)setModules:(NSDictionary *)value {
    [modules release];
    modules = [[NSMutableDictionary dictionaryWithDictionary:value] retain];
}

- (NSDictionary *)moduleData {
    return [NSDictionary dictionaryWithDictionary:moduleData];
}

- (void)setModuleData:(NSDictionary *)value {
    [moduleData release];
    moduleData = [[NSMutableDictionary dictionaryWithDictionary:value] retain];
}

- (NSDictionary *)moduleTypes {
    return [NSDictionary dictionaryWithDictionary:moduleTypes];
}

- (void)setModuleTypes:(NSDictionary *)value {
    [moduleTypes release];
    moduleTypes = [[NSMutableDictionary dictionaryWithDictionary:value] retain];    
}

- (NSString *)modulePath {
    return modulePath;
}

- (void)setModulePath:(NSString *)value {
    [value retain];
    [modulePath release];
    modulePath = value;
}

- (BOOL)temporaryManager {
    return temporaryManager;
}

- (void)setTemporaryManager:(BOOL)value {
    temporaryManager = value;
}

- (void)addPath:(NSString*)path {
	[managerLock lock];

	if([moduleData count] == 0)
		manager = new sword::SWMgr( toUTF8(path) , true, new sword::EncodingFilterMgr(sword::ENC_UTF8));
	else
		manager->augmentModules(toUTF8(path));
	
	[self refreshModuleData];
	[self addRenderFilters];
	
	[managerLock unlock];
}

/*" Unloads Sword Manager.
"*/
- (void)dealloc {
    MBLOG(MBLOG_DEBUG, @"[SwordManager -dealloc]");    
    
    if(!temporaryManager) {
        delete manager;
    }
    
    [self setModuleData:nil];
    [self setModulePath:nil];
    [self setModules:nil];
    [self setModuleTypes:nil];
    [self setManagerLock:nil];
    
	[super dealloc];
}

#pragma mark -

+ (void)initManagersWithPath:(NSString*)path {
	static BOOL		loaded = NO;
	
	if(loaded)
		return;
	
	defaultManager = [[SwordManager alloc] initWithPath:path];
	
	loaded = YES;
}

+ (SwordManager*)defaultManager {
	return defaultManager;
}

#pragma mark -

- (SwordModule *)moduleForURL:(NSURL *)aUrl {
	NSArray	*themodules = NULL;
	NSString *moduleName = @"?";
	SwordModule	*theModule = NULL;
	
	if([aUrl host])
	{
		themodules = [[aUrl host] componentsSeparatedByString:@"."];
	}
	
	if(themodules && [themodules count] > 0)
	{
		moduleName = [themodules objectAtIndex:0];
	}
		
	if([moduleName hasPrefix:@"css-"])
	{
		moduleName = [moduleName substringFromIndex:4];
	}

	
	if([moduleName caseInsensitiveCompare:@"Bible"] == NSOrderedSame || [moduleName caseInsensitiveCompare:@"Commentary"] == NSOrderedSame)
	{
		NSArray	*pathComponents;
		
		//remove first item of path into moduleName
		pathComponents = [[aUrl path] pathComponents];
		if([pathComponents count] > 0)
		{
			moduleName = [pathComponents objectAtIndex:1];
			
			theModule = [self moduleWithName:moduleName];
			
			if(theModule && [theModule type] != bible)
			{
				moduleName = [[NSUserDefaults standardUserDefaults] objectForKey:@"serviceBible"];
				theModule = [self moduleWithName:moduleName];
			}
		}
	}
	else if([moduleName hasPrefix:@"Bible-"] || [moduleName hasPrefix:@"bible-"])
	{		
		moduleName = [moduleName substringFromIndex:6];
		
		theModule = [self moduleWithName:moduleName];
		
		if(theModule && [theModule type] != bible)
		{
			moduleName = [[NSUserDefaults standardUserDefaults] objectForKey:@"serviceBible"];
			theModule = [self moduleWithName:moduleName];
		}
	}
	else if([moduleName caseInsensitiveCompare:@"StrongsGreek"] == NSOrderedSame)
	{
		moduleName = [[NSUserDefaults standardUserDefaults] objectForKey:@"strongs greek dictionary"];
	}
	else if([moduleName caseInsensitiveCompare:@"StrongsHebrew"] == NSOrderedSame)
	{
		moduleName = [[NSUserDefaults standardUserDefaults] objectForKey:@"strongs hebrew dictionary"];
	}
	else if([moduleName caseInsensitiveCompare:@"MorphGreek"] == NSOrderedSame)
	{
		moduleName = [[NSUserDefaults standardUserDefaults] objectForKey:@"morph greek dictionary"];
	}
	else if([moduleName caseInsensitiveCompare:@"MorphHebrew"] == NSOrderedSame)
	{
		moduleName = [[NSUserDefaults standardUserDefaults] objectForKey:@"morph hebrew dictionary"];
	}
	
	if(!theModule)
	{
		theModule = [self moduleWithName:moduleName];
	}
	
	if(theModule)
	{
		return theModule;
	}
	else
	{
		return [self moduleWithName:[[NSUserDefaults standardUserDefaults] objectForKey:@"serviceBible"]];
	}
}

// get module for name from all modules in moduleData
- (SwordModule *)moduleWithName:(NSString *)name {
	SwordModule	*mod = NULL;
	NSString *modtype;
	
	NSArray *keys = [moduleData allKeys];
	int len = [keys count];
	for(int i = 0;i < len;i++) {
		if([[keys objectAtIndex:i] caseInsensitiveCompare:name] == NSOrderedSame) {
			name = [keys objectAtIndex:i];
			break;
		}
	}
	
	mod = [modules objectForKey:name];
	if(mod != NULL) {
		return mod;
	}	
	[managerLock lock];
		
	mod = NULL;
	
	modtype = [self typeForModuleName:name];

	if([modtype isEqualTo:@"Biblical Texts"]) {
		mod = [[SwordBible alloc] initWithSwordManager:self moduleName:name];
	} else if([modtype isEqualTo:@"Lexicons / Dictionaries"]) {
		mod = [[SwordDictionary alloc] initWithSwordManager:self moduleName:name];
	} else if([modtype isEqualTo:@"Commentaries"]) {
		mod = [[SwordCommentary alloc] initWithSwordManager:self moduleName:name];
	} else if([modtype isEqualTo:@"Generic Books"]) {
		mod = [[SwordBook alloc] initWithSwordManager:self moduleName:name];
	} else if([moduleData objectForKey:name]) { // if actually exists {
		mod = [[SwordModule alloc] initWithSwordManager:self moduleName:name];
	}
		
	if(mod) {
		[modules setObject:mod forKey:name];
	}
	
	[managerLock unlock];
				
	return mod;
}

/*" Retrieves C++ SWModule pointer - used internally by SwordBible. "*/
- (sword::SWModule *)getSWModuleWithName: (NSString *)moduleName
{
	[managerLock lock];
	
	sword::SWModule *module;
	module = manager->Modules[[moduleName cStringUsingEncoding:NSUTF8StringEncoding]];
	
	[managerLock unlock];

	if(module == NULL)
	{
		return NULL;
	}
	return module;
}

#pragma mark -

- (void)setCipherKey:(NSString *)key forModuleNamed:(NSString*)name
{
	[managerLock lock];
	
	manager->setCipherKey(toLatin1(name), toLatin1(key));
	
	[managerLock unlock];
}

#pragma mark -

- (void)addRenderFilters
{
	[managerLock lock];
	
	sword::SWModule	*module;
	sword::ModMap::iterator	it;

	for (it = manager->Modules.begin(); it != manager->Modules.end(); it++)
	{
		module = it->second;
		
		switch (module->Markup())
		{
			case sword::FMT_GBF:
				if(!gbfFilter)
					gbfFilter = new BT_GBFHTML(); //new sword::GBFHTMLHREF();
				if(!gbfStripFilter)
					gbfStripFilter = new sword::GBFPlain();
				module->AddRenderFilter(gbfFilter);
				module->AddStripFilter(gbfStripFilter);
				break;
			case sword::FMT_THML:
				if(!thmlFilter)
					thmlFilter = new BT_ThMLHTML();//new sword::ThMLHTMLHREF();
				if(!thmlStripFilter)
					thmlStripFilter = new sword::ThMLPlain();
				module->AddRenderFilter(thmlFilter);
				module->AddStripFilter(thmlStripFilter);
				break;
			case sword::FMT_OSIS:
				if(!osisFilter)
					osisFilter = new BT_OSISHTML();//new sword::OSISHTMLHREF();
				if(!osisStripFilter)
					osisStripFilter = new sword::OSISPlain();
				module->AddRenderFilter(osisFilter);
				module->AddStripFilter(osisStripFilter);
				break;
            case sword::FMT_TEI:
                if(!teiFilter)
                    teiFilter = new sword::TEIHTMLHREF();
                /*
                if(!teiStripFilter)
                    teiStripFilter = new sword::TEIPlain();
                 */
				module->AddRenderFilter(teiFilter);
//				module->AddStripFilter(teiStripFilter);
				break;
            case sword::FMT_PLAIN:
			default:
				if(!plainFilter)
					plainFilter = new sword::PLAINHTML();
				module->AddRenderFilter(plainFilter);
				break;
		}
	}
	
	[managerLock unlock];
}

- (id)safeC:(const char *)bytes
{
	if(bytes)
		return [NSString stringWithCString:bytes];
	return @"";
}

- (id)safeUTF8:(const char *)bytes
{
	NSString *r = nil;
	if(bytes)
		 r =[NSString stringWithUTF8String:bytes];
	if(r) return r;
	
	return [self safeC:bytes];
}

- (void)refreshModuleData {
	[managerLock lock];
	
	if(moduleData) {
        [moduleData removeAllObjects];
	}
    
    if(moduleTypes) {
        [moduleTypes removeAllObjects];
    }
    
    // loop over modules
	sword::ModMap::iterator it;
	for(it = manager->Modules.begin(); it != manager->Modules.end();it++) {
        // the module
		sword::SWModule *module = it->second;
        
		NSMutableDictionary *dictionary = [NSMutableDictionary dictionaryWithCapacity:10];
        NSString *modName = [self safeUTF8:module->Name()];
        NSString *modType = [self safeUTF8:module->Type()];
        NSString *modVersion = [self safeUTF8:module->getConfigEntry("Version")];
        NSString *modDescr = [self safeUTF8:module->Description()];
        NSString *modLang = [self safeUTF8:module->Lang()];
        NSString *modMinVersion = [self safeUTF8:module->getConfigEntry("MinimumVersion")];
        
        [dictionary setObject:modVersion forKey:@"Version"];
        [dictionary setObject:modMinVersion forKey:@"MinVersion"];
		[dictionary setObject:modName forKey:@"Name"];
		[dictionary setObject:modDescr forKey:@"Description"];
		[dictionary setObject:modLang forKey:@"Language"];
		[dictionary setObject:modType forKey:@"Type"];
		[dictionary setObject:[NSNumber numberWithBool:module->getConfig().has("Feature", "HebrewDef")] forKey:@"HebrewDef"];
		[dictionary setObject:[NSNumber numberWithBool:module->getConfig().has("Feature", "GreekDef")] forKey:@"GreekDef"];
		[dictionary setObject:[NSNumber numberWithBool:module->getConfig().has("Feature", "HebrewParse")] forKey:@"HebrewParse"];
		[dictionary setObject:[NSNumber numberWithBool:module->getConfig().has("Feature", "GreekParse")] forKey:@"GreekParse"];
		[dictionary setObject:[NSNumber numberWithBool:module->getConfig().has("Feature", "DailyDevotion")] forKey:@"DailyDevotion"];
		[dictionary setObject:[NSNumber numberWithBool:module->getConfig().has("Category", "Cults / Unorthodox / Questionable Material")] forKey:@"Unorthodox"];
        
        // add dictionary
		[moduleData setObject:dictionary forKey:modName];
        
        // add to types
        [moduleTypes setObject:modType forKey:modType];
	}
	
	[managerLock unlock];
}

/*" Sets global options such as 'Strongs' or 'Footnotes'. "*/
- (void)setGlobalOption:(NSString *)option value:(NSString *)value
{
	[managerLock lock];
	
	if([option isEqualToString:@"All Readings"])
		manager->setGlobalOption("Textual Variants", "All Readings");
	else if([option isEqualToString:@"Secondary Reading"]) {	
        manager->setGlobalOption("Textual Variants", "Secondary Reading");
        MBLOG(MBLOG_DEBUG, @"SECONDARY");
    } else if([option isEqualToString:@"Primary Reading"]) {	
        manager->setGlobalOption("Textual Variants", "Primary Reading");
        MBLOG(MBLOG_DEBUG, @"PRIMARY");
    } else {
		manager->setGlobalOption([option cStringUsingEncoding:NSUTF8StringEncoding], [value cStringUsingEncoding:NSUTF8StringEncoding]);
	}
	
	[managerLock unlock];
}

- (NSString *)descriptionForModuleName:(NSString *)name {
	return [[moduleData objectForKey:name] objectForKey:@"Description"];
}

- (NSString *)typeForModuleName:(NSString *)name {
	return [[moduleData objectForKey:name] objectForKey:@"Type"];
}

- (NSString *)languageAbbrevForModuleName:(NSString *)name {
	return [[moduleData objectForKey:name] objectForKey:@"Language"];
}

- (NSString *)versionForModuleName:(NSString *)name {
	return [[moduleData objectForKey:name] objectForKey:@"Version"];
}

/*" general feature access for modules "*/
- (id)featureForModuleName:(NSString *)name feature:(NSString *)feature {
	return [[moduleData objectForKey:name] objectForKey:feature];
}

/** 
 list all module and return them in a Array 
 */
- (NSArray *)listModules {
    
    NSMutableArray *ret = [NSMutableArray array];
    
    sword::SWModule *mod;
	for (ModMap::iterator it = manager->Modules.begin(); it != manager->Modules.end(); it++) {
		mod = it->second;
        // create SwordModule out of it
        SwordModule *sm = [[[SwordModule alloc] initWithSWModule:mod] autorelease];
        [ret addObject:sm];
	}    
    
    return [NSArray arrayWithArray:ret];
}

- (NSEnumerator *)moduleNameEnumerator {
	return [[moduleData keysSortedByValueUsingSelector:@selector(sortModulesByTypeLanguageDescription:)] objectEnumerator];
}


/*" Retrieve list of installed modules as an array, where the module has a specific feature
"*/
- (NSMutableArray *)swordModulesForFeature:(NSString *)feature
{
	[managerLock lock];
	
	sword::SWModule			*module;
	sword::ModMap::iterator	it;
	NSMutableArray			*moduleList = [[NSMutableArray alloc] init];

	for (it = manager->Modules.begin(); it != manager->Modules.end(); it++)
	{
		module = it->second;
		if( module->getConfig().has("Feature", [feature cStringUsingEncoding:NSUTF8StringEncoding]) )
		{
			[moduleList addObject:[NSString stringWithCString:module->Name()]];
		}
	}

	[moduleList autorelease];	// release memory later
	
	[managerLock unlock];
	
	return moduleList;
}

/*" Retrieve list of installed modules as an array, where type is: @"Biblical Texts", @"Commentaries", ..., @"ALL"
"*/
- (NSMutableArray *)swordModulesForType:(NSString *)type {
	[managerLock lock];
	
	sword::SWModule			*module;
	sword::ModMap::iterator	it;
	NSMutableArray			*moduleList = [[NSMutableArray alloc] init];

	for (it = manager->Modules.begin(); it != manager->Modules.end(); it++) {
		module = it->second;
		if( [type caseInsensitiveCompare: @"ALL"] == 0 || [type caseInsensitiveCompare:[NSString stringWithCString:module->Type()]] == 0 ) {
			[moduleList addObject:[NSString stringWithCString:module->Name()]];
		}
	}

	[moduleList autorelease];	// release memory later	
	[managerLock unlock];
	
	return moduleList;
}

/** return the sword manager of this class */
- (sword::SWMgr *)swMgr {
    return manager;
}

- (NSArray *)allModuleNames {
	return [moduleData keysSortedByValueUsingSelector:@selector(sortModulesByTypeLanguageDescription:)];
}

@end

@implementation NSDictionary (sortModules)

- (NSComparisonResult)sortModulesByTypeLanguageDescription:(NSDictionary *)aDictionary
{
	NSComparisonResult result;
	NSString *s1, *s2;
	
	//sort by name
	s1 = [self objectForKey:@"Name"];
	s2 = [aDictionary objectForKey:@"Name"];
	result = [s1 caseInsensitiveCompare:s2];
	return result;
}

@end
