/*	SwordDict.mm - Sword API wrapper for lexicons and Dictionaries.

	$Id: SwordDictionary.mm,v 1.4 2006/02/02 00:20:07 willthimbleby Exp $

	Copyright 2003 Will Thimbleby (will@thimbleby.net)
	Based on code by Nathan Youngman (http://www.nathany.com)

	This program is free software; you can redistribute it and/or modify it under the terms of the
	GNU General Public License as published by the Free Software Foundation version 2.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
	even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
*/

#import "SwordDictionary.h"
#import "PrefHandler.h"
#import "utils.h"

@implementation SwordDictionary

- (id)initWithSwordManager: (SwordManager *)theManager moduleName:(NSString*)name {
	self = [super initWithSwordManager:theManager moduleName:name];
    numberOfEntries = -1;
	[self getEntries];
	
	return self;
}

- (void)dealloc {
    [entries release];
	[super dealloc];
}

// does dict have key?
- (BOOL)hasReference:(NSString*)ref
{
	[moduleLock lock];
	
	sword::SWKey *key = module->CreateKey();
	
	if(module->isUnicode())
		(*key) = toUTF8([ref uppercaseString]);
	else
		(*key) = toLatin1([ref uppercaseString]);
	
	BOOL result = !key->Error();
	
	[moduleLock unlock];
	
	return result;
}


-(NSString*) fullRefName:(NSString*)ref
{
	[moduleLock lock];
	
	sword::SWKey		*key = module->CreateKey();
	
	if(module->isUnicode())
		(*key) = toUTF8([ref uppercaseString]);
	else
		(*key) = toLatin1([ref uppercaseString]);
	
	module->SetKey(key);
	module->getRawEntry();
	
	NSString *result;
	if(module->isUnicode())
		result = fromUTF8(module->KeyText());
	else
		result =  fromLatin1(module->KeyText());
	
	[moduleLock unlock];
	
	return result;
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish
{
	return [super htmlForRef:[reference uppercaseString] embelish:embelish];
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish protocol:(NSURLProtocol*)protocol
{
	return [super htmlForRef:[reference uppercaseString] embelish:embelish protocol:protocol];
}

- (void)openCache
{
	NSString	*cachePath;
	
	//open cached file
	cachePath = [NSString localizedStringWithFormat:@"%@/Library/Application Support/MacSword/cache-%@", NSHomeDirectory(), [self moduleName]];
	entries = [[NSArray arrayWithContentsOfFile:cachePath] retain];
}

- (void)saveCache
{
	NSString	*appSupportPath = [PrefHandler getCacheFolder];
	
	// save cached file
	[entries writeToFile:[NSString stringWithFormat:@"%@/cache-%@", appSupportPath, [self moduleName]] atomically:NO];
}

- (long)entryCount {
    long ret = 0;
    
    if(numberOfEntries == -1) {
        ret = [[self getEntries] count];
    } else {
        ret = numberOfEntries;
    }
    
    return ret;
}

//	Returns an array containing all the entries in this module.
- (NSMutableArray *) getEntries {
	if(entries)
		return entries;
	[self openCache];
	if(entries)
		return entries;
		
	[moduleLock lock];
	
	entries = [[NSMutableArray alloc] init];
	
	module->setSkipConsecutiveLinks(true);
	*module = sword::TOP;
	module->getRawEntry();
	
	if(module->isUnicode()) {
		while (!module->Error()) {
			[entries addObject: [fromUTF8(module->KeyText()) capitalizedString]];
			(*module)++;
		}
	} else {
		while (!module->Error()) {
			[entries addObject: [fromLatin1(module->KeyText()) capitalizedString]];
			(*module)++;
		}
	}
	[moduleLock unlock];
	
	[self saveCache];
	
    // also cache the number of entries
    numberOfEntries = [entries count];
    
	return entries;
}

@end
