/*	SwordBook.mm - Sword API wrapper for Commentaries.

	$Id: SwordBook.mm,v 1.1 2006/02/02 00:20:07 willthimbleby Exp $

	Copyright 2003 Will Thimbleby (will@thimbleby.net)

	This program is free software; you can redistribute it and/or modify it under the terms of the
	GNU General Public License as published by the Free Software Foundation version 2.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
	even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
*/

#import "SwordBook.h"
#import "SwordURLProtocol.h"
#import "utils.h"

@interface SwordBook (PrivateAPI)

- (void)setContents:(NSMutableArray *)value;
- (NSString *)htmlForRef:(NSString *)ref;

@end

@implementation SwordBook (PrivateAPI)

- (void)setContents:(NSMutableArray *)value {
    [value retain];
    [contents release];
    contents = value;
}

- (NSString *)htmlForRef:(NSString *)ref {
    NSString *ret = @"";
    
    // lock
    [moduleLock lock];

    const char *refStr = NULL;
    // encoding depends on module encoding
    if([self isUnicode]) {
        refStr = [ref cStringUsingEncoding:NSUTF8StringEncoding];
    } else {
        refStr = [ref cStringUsingEncoding:NSISOLatin1StringEncoding];
    }
    
    //sword::TreeKeyIdx *key = new sword::TreeKeyIdx(refStr);
    sword::SWKey *key = new sword::SWKey(refStr);
    module->setKey(key);
    if(key) {
        delete key;
    }
    
    char *bytes = (char *)module->RenderText();
    [moduleLock unlock];
    
    ret = [NSString stringWithUTF8String:bytes];
    
    return ret;
}

@end


@implementation SwordBook

- (id)initWithSwordManager:(SwordManager *)theManager moduleName:(NSString *)name {
	self = [super initWithSwordManager:theManager moduleName:name];

    [self setContents:nil];

	sword::TreeKeyIdx *treeKey = dynamic_cast<sword::TreeKeyIdx*>((sword::SWKey*)*(module));
	[self createContents:treeKey];
                         
	return self;
}

- (NSString *)fullRefName:(NSString *)ref {
	return ref;
}

- (long)entryCount {
    return [contents count];
}

- (NSArray *)contents {
	return [NSArray arrayWithArray:contents];
}

- (void)createContents:(sword::TreeKeyIdx*)treeKey {

	[moduleLock lock];
	
    if(contents == nil) {
		[self setContents:(NSMutableArray *)[self createContentsB:treeKey]];
	}
	
	[moduleLock unlock];
}

// fill tree content with keys of book
- (id)createContentsB:(sword::TreeKeyIdx*)treeKey {
    
	// we need to check for any Unicode names here
	char *treeNodeName = (char *)treeKey->getText();
	// use MSStringMgr to check for Unicode
	//MSStringMgr *stringMgr = (MSStringMgr *)MSStringMgr::getSystemStringMgr();
	NSString *name = @"";
    // key encoding depends on module encoding
	//bool isUnicode = stringMgr->isUtf8(treeNodeName);
	if([self isUnicode]) {
		name = fromUTF8(treeNodeName);
	} else {
		name = fromLatin1(treeNodeName);
	}
	
	// if empty this is about Specifier
	if([name length] == 0) {
		name = aboutEntrySpecifier;
	}
    
    // if this node has children, walk them
	if(treeKey->hasChildren()) {
		NSMutableArray *c = [NSMutableArray array];
		
		[c addObject:name];
        // get first child
		treeKey->firstChild();
		
        // walk the subtree
		do {
			[c addObject:[self createContentsB:treeKey]];
		}
		while(treeKey->nextSibling());
		
        // back to parent
		treeKey->parent();
	
        // returns the subtree array
		return c;
	}
	
    // returns the node name
	return name;
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish {
	if([self isLocked] || !reference || ![reference length] || [reference compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame) {
		return [self htmlForDescription];
	} else {
        return [self htmlForRef:reference];
	}
}

- (void)htmlForRef:(NSString *)reference embelish:(BOOL)embelish protocol:(SwordURLProtocol *)protocol {
	if([self isLocked] || !reference || ![reference length] || [reference compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame) {
		[protocol recieveData:[[self htmlForDescription] dataUsingEncoding: NSUTF8StringEncoding]];
	} else {
        NSString *htmlStr = [self htmlForRef:reference];
		[protocol recieveData:[htmlStr dataUsingEncoding:NSUTF8StringEncoding]];
    }
}

@end
