/*	SwordBible.mm - Sword API wrapper for Biblical Texts.

	$Id: SwordBible.mm,v 1.4 2006/02/02 00:20:07 willthimbleby Exp $

	Copyright 2003 Will Thimbleby (will@thimbleby.net)
	Based on code by Nathan Youngman (http://www.nathany.com)

	This program is free software; you can redistribute it and/or modify it under the terms of the
	GNU General Public License as published by the Free Software Foundation version 2.

	This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without
	even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
	General Public License for more details. (http://www.gnu.org/licenses/gpl.html)
*/

#import "SwordBible.h"
#import "BookmarkHandler.h"
#import "WebKit/WebKit.h"
#import "utils.h"

@implementation SwordBible

- (id)initWithSwordManager: (SwordManager *)theManager moduleName:(NSString*)name
{
	self = [super initWithSwordManager:theManager moduleName:name];
	[self getBookNames];
	
	return self;
}

- (void)dealloc
{
	[books release];
	[bookNumbers release];
	[engBookNames release];
	[super dealloc];
}

// just checks whether module contains book atm
- (BOOL)hasReference:(NSString*)ref
{
	[moduleLock lock];
	
	sword::VerseKey	*key = (sword::VerseKey *)(module->CreateKey());
	(*key) = toUTF8(ref);
	BOOL result = !key->Error() && [bookNumbers containsObject:[NSNumber numberWithInt:key->Book()+key->Testament()*100]];
	
	[moduleLock unlock];
	
	return result;
}

NSLock  *bibleLock = 0;

// changes an abbreviated reference into a full
// eg. Dan4:5-7 => Daniel4:5
+(void) decodeRef:(NSString*)ref intoBook:(NSString**)book chapter:(int*)chapter verse:(int*)verse
{
	if(!bibleLock) bibleLock = [[NSLock alloc] init];
	[bibleLock lock];
	
	sword::VerseKey		vk(toUTF8(ref));
	
	*book = fromUTF8(vk.getBookName());
	*chapter = vk.Chapter();
	*verse = vk.Verse();
		
	[bibleLock unlock];
}

// changes an abbreviated reference into a full
// eg. Dan4:5-7 => Daniel4:5-7
+ (NSString *)fullRefName_old:(NSString*)abbr {
	if(!bibleLock) bibleLock = [[NSLock alloc] init];
	[bibleLock lock];
	
	sword::VerseKey		vk(toUTF8(abbr));
	NSScanner			*scan;
	int					toVerse;
	NSString			*result;
	
	scan = [NSScanner scannerWithString:abbr];
	[scan scanUpToString:@"-" intoString:NULL];
	[scan scanString:@"-" intoString:NULL];

	if([scan scanInt:&toVerse])
		result = [NSString stringWithFormat:@"%@-%d", fromUTF8(vk), toVerse];
	else
		result = fromUTF8(vk);
		
	[bibleLock unlock];
	
	return result;
}

- (NSString *)fullRefName:(NSString *)ref { 	
	[moduleLock lock];

	sword::ListKey		listkey;
	sword::VerseKey		vk;
	int					lastIndex;
	int					chapter=-1, book=-1, verse=-1, startChapter, startVerse;
	NSMutableString		*reference = [NSMutableString string];
	
	listkey = vk.ParseVerseList(toUTF8(ref), "Gen1", true);
	
	int len = listkey.Count();
	for(int i = 0; i < len; i++) {
		sword::VerseKey *element = My_SWDYNAMIC_CAST(VerseKey, listkey.GetElement(i));
		
		// is it a chapter or book - not atomic
		if(element) {
			// start at lower bound
			module->Key(element->LowerBound());
			// find the upper bound
			vk = element->UpperBound();
			vk.Headings();
		} else {
			// set it up
			module->Key(*listkey.GetElement(i));
		}
		
		unichar mdashchars[1] = {0x2013};
		NSString *mdash = [NSString stringWithCharacters:mdashchars length:1];
			
		// while not past the upper bound
		do {
			int newBook = ((sword::VerseKey)(module->Key())).Book();
			int newChapter = ((sword::VerseKey)(module->Key())).Chapter();
			int newVerse = ((sword::VerseKey)(module->Key())).Verse();
			
			if(book != newBook) {
				if(book != -1) {
					if(startChapter != chapter) [reference appendString:[NSString stringWithFormat:@"%@%d:%d; ", mdash, chapter, verse]];
					else if(startVerse != verse) [reference appendString:[NSString stringWithFormat:@"%@%d; ", mdash, verse]];
					else [reference appendString:@"; "];
				}
				[reference appendString:[NSString stringWithFormat:@"%@",fromUTF8(module->Key().getText())]];
				startChapter = newChapter; startVerse = newVerse;
			} else if(chapter != newChapter && lastIndex != (module->Key()).Index()-2) {
				if(book != -1) {
					if(startChapter != chapter) [reference appendString:[NSString stringWithFormat:@"%@%d:%d; ", mdash, chapter, verse]];
					else if(startVerse != verse) [reference appendString:[NSString stringWithFormat:@"%@%d; ", mdash, verse]];
					else [reference appendString:@"; "];
				}
				[reference appendString:[NSString stringWithFormat:@"%d:%d", newChapter, newVerse]];
				startChapter = newChapter; startVerse = newVerse;
			} else if(verse != newVerse-1 && chapter == newChapter) {
				[reference appendString:[NSString stringWithFormat:@",%d",newVerse]];
				startVerse = newVerse;
			}
			
			MBLOGV(MBLOG_DEBUG, @"reference:%@", reference);
			
			book = newBook;
			chapter = newChapter;
			verse = newVerse;
							
			lastIndex = (module->Key()).Index();
			(*module)++;
			if(lastIndex == (module->Key()).Index())
				break;
		}while (element && module->Key() <= vk);
		
		if(startChapter != chapter)
			[reference appendString:[NSString stringWithFormat:@"%@%d:%d", mdash, chapter, verse]];
		else if(startVerse != verse)
			[reference appendString:[NSString stringWithFormat:@"%@%d", mdash, verse]];
	}
	
	[moduleLock unlock];
	
	return reference;
}

+ (NSString *)firstRefName:(NSString*)abbr {
	if(!bibleLock) bibleLock = [[NSLock alloc] init];
	[bibleLock lock];
	
	sword::VerseKey		vk(toUTF8(abbr));
	NSString *result = fromUTF8(vk);
		
	[bibleLock unlock];
	
	return result;
}

+ (NSString *)context:(NSString*)abbr {
	//get parsed simple ref
	NSString	*first = [SwordBible firstRefName:abbr];
	NSArray		*firstbits = [first componentsSeparatedByString:@":"];
	
	//if abbr contains : or . then we are a verse so return a chapter
	if([abbr rangeOfString:@":"].location != NSNotFound || [abbr rangeOfString:@"."].location != NSNotFound)
		return [firstbits objectAtIndex:0];
	
	//otherwise return a book
	firstbits = [first componentsSeparatedByString:@" "];
	
	if([firstbits count] > 0)
		return [firstbits objectAtIndex:0];
	
	return abbr;
}

-(void) book:(int*)book andChapter:(int*)chapter fromReference:(NSString*)ref {
	[moduleLock lock];
	
	sword::VerseKey		vk(toUTF8(ref));
	
	*book = [books indexOfObject:fromUTF8(vk.getBookName())];
	*chapter = vk.Chapter();
	
	[moduleLock unlock];
}

// returns an array containing all the book names found in this module.
- (NSMutableArray *)bookNames {	
	return books;
}

- (NSMutableArray *)engBookNames {	
	return engBookNames;
}

// calculates an array containing all the book names found in this module.
- (void)getBookNames {	
	[moduleLock lock];
	
	bool skipsLinks;

	skipsLinks = module->getSkipConsecutiveLinks();
	module->setSkipConsecutiveLinks(true);
	
	sword::VerseKey	currentKey, bottom;
	int				lastBook;
	
	books = [[[NSMutableArray array] retain] retain];
	engBookNames = [[NSMutableArray array] retain];
	bookNumbers = [[NSMutableArray array] retain];

	bottom.setPosition(sword::BOTTOM);
	*module = sword::BOTTOM;
	bottom = module->Key();
	
	*module = sword::TOP;
	
	currentKey = (sword::VerseKey)module->Key();
	
	while (!module->Key().Error() && currentKey < bottom) {
		currentKey = (sword::VerseKey)module->Key();

		//ctxt = (char *)module->RenderText();
		//if(strlen(ctxt) > 1)
		{
            NSString *bookName = fromUTF8(currentKey.getBookName());
			[books addObject:bookName];
			[bookNumbers addObject: [NSNumber numberWithInt:currentKey.Book()+currentKey.Testament()*100]];
            
            // get english bookname
            sword::VerseKey engKey = currentKey;
            engKey.setLocale("en");
            NSString *engBookName = fromUTF8(engKey.getBookName());
            [engBookNames addObject:engBookName];
		}
		lastBook = currentKey.Book();
		currentKey.Book( currentKey.Book() + 1 );
		module->SetKey(currentKey);
	}
	
	module->setSkipConsecutiveLinks(skipsLinks);
	
	[moduleLock unlock];
}

/*" Returns the number of chapters for the specified bookName.

	%{Doesn't validate the book name specified.}
"*/
- (int)maxChaptersForBookName:(NSString *)bookName {
	[moduleLock lock];
	
	int maxChapters;
	sword::VerseKey *key = (sword::VerseKey *)module->CreateKey();
	(*key) = toUTF8(bookName);
	(*key) = sword::MAXCHAPTER;
	maxChapters = key->Chapter();
	delete key;
	
	[moduleLock unlock];
	
	return maxChapters;
}


/*" Returns the number of verses for the specified chapter and bookName, or 0 if the chapter specified is out of range.

	%{Doesn't validate the book name specified.}
"*/
- (int)maxVersesForChapter:(int)chapter bookName:(NSString *)bookName {
	[moduleLock lock];
	
	int maxVerses;
	sword::VerseKey *key;

	if( chapter < 1 || chapter > [self maxChaptersForBookName:bookName] ) return 0;

	key = (sword::VerseKey *)module->CreateKey();
	(*key) = toUTF8(bookName);
	key->Chapter(chapter);
	(*key) = sword::MAXVERSE;
	maxVerses = key->Verse();
	delete key;
	
	[moduleLock unlock];
	
	return maxVerses;
}

/**
 \brief calculate all verses for this bible book
 */
- (int)maxVersesForBible
{
    int ret = 0;
    
    int len = [books count];
    for(int i = 0;i < len;i++) {
        // get book name
        NSString *bookName = [books objectAtIndex:i];
        
        // get chapters for book
        int chapters = [self maxChaptersForBookName:bookName];
        // calculate all verses for this book
        int verses = 0;
        for(int j = 1;j <= chapters;j++) {
            verses += [self maxVersesForChapter:j bookName:bookName];
        }
        ret += verses;
    }
    
    return ret;
}

/** numnber of entries
 abstract method, should be overriden by subclasses
 */
- (long)entryCount {
    
    *module = sword::TOP;
    unsigned long verseLowIndex = module->Index();
    *module = sword::BOTTOM;
    unsigned long verseHighIndex = module->Index();
    
    return verseHighIndex - verseLowIndex;
}

- (NSString *)entryForRef:(NSString *)reference
{
	return [super htmlForRef:reference];
}

- (NSString *)textForRef:(NSString *)reference
{
	return [self htmlForRef:reference embelish:NO];
}

- (NSString *)htmlForRef:(NSString *)reference
{
	return [self htmlForRef:reference embelish:YES];
}

- (NSString *)htmlForRef:(NSString *)reference highlightReference:(NSString *)hr
{
	return [self htmlForRef:reference embelish:YES highlight:YES highlightReference:hr];
}

- (NSString *)htmlForRef:(NSString *)reference highlightReference:(NSString *)hr protocol:(SwordURLProtocol *)protocol
{
	return [self htmlForRef:reference embelish:YES highlight:YES highlightReference:hr protocol:protocol];
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish
{
	return [self htmlForRef:reference embelish:embelish highlight:embelish highlightReference:NULL];
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish protocol:(SwordURLProtocol *)protocol
{
	return [self htmlForRef:reference embelish:embelish highlight:embelish highlightReference:NULL protocol:protocol];
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish highlight:(BOOL)highlight highlightReference:(NSString*)highlightRef protocol:(SwordURLProtocol *)protocol
{	
	NSMutableString *ret = [NSMutableString string];
	
	if(!reference || ![reference length] || [reference compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame)
	{
		return [self htmlForDescription];
	}
	else
	{
		[moduleLock lock];

		sword::ListKey		listkey, bookmarkslk[9];
		sword::VerseKey		vk;
		int					i, j, lastIndex;
		char				*ctxt;
		int					clen;
		NSString			*verseNumbering;
		NSMutableString		*html = [[NSMutableString alloc] init];
		int					chapter=-1, book=-1, verse=-1;
		
		BOOL				warnIncomplete = NO;
		
		if(highlightRef)
			bookmarkslk[0] = vk.ParseVerseList(toUTF8(highlightRef), "Gen1", true);
		
		if(highlight)
		{
			for(i=1;i<8;i++)
			{
				bookmarkslk[i] = vk.ParseVerseList(toUTF8([BookmarkHandler allBookmarks:i]), "Gen1", true);
			}
		}
		
		((sword::VerseKey*)(module->getKey()))->Headings(1);
		
		listkey = vk.ParseVerseList(toUTF8(reference), "Gen1", true);
		
		for (i = 0; i < listkey.Count(); i++)
		{
			sword::VerseKey *element = My_SWDYNAMIC_CAST(VerseKey, listkey.GetElement(i));
			
			// is it a chapter or book - not atomic
			if(element)
			{
				// start at lower bound
				element->Headings(1);
				module->Key(element->LowerBound());
				
				// find the upper bound
				vk = element->UpperBound();
				vk.Headings(true);
			}
			else
			{
				// set it up
				module->Key(*listkey.GetElement(i));
			}
				
			// while not past the upper bound
			do
			{
				//add verse index to dictionary
				ctxt = (char *)module->RenderText();
                clen = strlen(ctxt);
                NSString *cString = @"";
				if(clen > 3) {
					if(ctxt[clen-3] == -96) ctxt[clen-3] = 0;
					cString = [NSString stringWithUTF8String:ctxt];
                }
				                
				//if(embelish)
				{
					int newBook, newChapter, newVerse;
					
					newBook = ((sword::VerseKey)(module->Key())).Book();
					newChapter = ((sword::VerseKey)(module->Key())).Chapter();
					newVerse = ((sword::VerseKey)(module->Key())).Verse();
					
					if(book != newBook)
					{
						book = newBook;
						chapter = -1;
						if(embelish)
							[html appendString:[NSString stringWithFormat:@"<div class=\"booktitle\"><a name=\"%@\">%@</a></div>\n",
								fromUTF8(module->Key().getText()), fromUTF8(((sword::VerseKey)(module->Key())).getBookName())]];
						else
							if(verse != -1) [html appendString:@"<br><br>"];
					}
					if(chapter != newChapter)
					{
						chapter = newChapter;
						verse = newVerse;
						
						if(embelish)
							[html appendString:[NSString stringWithFormat:@"<br><a name=\"%@\" class=\"chapternumber\"  href=\"%@?notip\">%d</a> ",
								fromUTF8(module->Key().getText()), [SwordBible context:fromUTF8(module->Key().getText())], ((sword::VerseKey)(module->Key())).Chapter()]];
						else
							if(verse != -1) [html appendString:@"<br>"];
					}
					
					if(verse < newVerse-1)
						[html appendString:@"<br><br><div>...</div><br>"];
					verse = newVerse;
					
					if(embelish)
					{
						NSMutableString	*hclass = [NSMutableString string];
		
						if(highlightRef)
						{
							bookmarkslk[0] = module->Key().getText();
							
							if(!bookmarkslk[0].Error())
								[hclass appendString:@" highlight"];
						}
						
						if(highlight)
						{
							for(j=1;j<8;j++)
							{
								bookmarkslk[j] = module->Key().getText();
								
								if(!bookmarkslk[j].Error())
									[hclass appendString:[NSString stringWithFormat:@" highlighted%c", 'A'+j-1]];
							}
						}
						
						[html appendString:[NSString stringWithFormat:@"\n<span id=\"%@\" class=\"verse%@\">", fromUTF8(module->Key().getText()), hclass]];
						
						if(clen > 0)
						{
							NSString *verse = fromUTF8(module->Key().getText());
							
							verseNumbering = [NSString stringWithFormat:@"<a name=\"%@\" class=\"versenumber\" href=\"%@?notip\">%d</a> ", 
								verse, verse, 
								((sword::VerseKey)(module->Key())).Verse()];
							//verseNumbering = [NSString stringWithFormat:@"%d ", ((sword::VerseKey)(module->Key())).Verse()];
							[html appendString:verseNumbering];
						}
						
					}
				}
                
                if(cString)
                    [html appendString:cString];
				
				if(clen == 0 && !warnIncomplete)
				{
					[html appendString:NSLocalizedString(@"This module maybe incomplete. Try a different reference.", @"Bible incomplete warning - try a different reference")];
					warnIncomplete = YES;
				}
				else warnIncomplete = YES;
				
				if(clen > 0)
				{
					[html appendString:@"\n"];
				}
				
				//if(embelish)
					[html appendString:@"</span>"];
				
				if(![protocol recieveData:[html dataUsingEncoding:NSUTF8StringEncoding]])
					break;
                
                // add
                [ret appendString:html];
                // delete for next round
				[html deleteCharactersInRange:NSMakeRange(0, [html length])];
				
				lastIndex = (module->Key()).Index();
				(*module)++;
				if(lastIndex == (module->Key()).Index())
					break;
			}while (element && module->Key() <= vk);
		}
		
		[html autorelease];
		[moduleLock unlock];
	}

#ifdef DEBUG    
    [ret writeToFile:@"/Users/mbergmann/Desktop/mod.html" atomically:NO encoding:NSUTF8StringEncoding error:nil];
#endif
    
	return ret;
}

- (NSString *)htmlForRef:(NSString *)reference embelish:(BOOL)embelish highlight:(BOOL)highlight highlightReference:(NSString*)highlightRef
{	
	if(!reference || ![reference length] || [reference compare:aboutEntrySpecifier options:NSCaseInsensitiveSearch range:NSMakeRange(0,[aboutEntrySpecifier length])] == NSOrderedSame)
	{
		return [self htmlForDescription];
	}
	else
	{
	[moduleLock lock];

	sword::ListKey		listkey, bookmarkslk[9];
	sword::VerseKey		vk;
	int					i, j, lastIndex;
	char				*ctxt;
	int					clen;
	NSString			*verseNumbering;
	NSMutableString		*html = [[NSMutableString alloc] init];
	int					chapter=-1, book=-1, verse=-1;
	
	BOOL				warnIncomplete = NO;
	
	if(highlightRef)
		bookmarkslk[0] = vk.ParseVerseList(toUTF8(highlightRef), "Gen1", true);
	
	if(highlight)
	{
		for(i=1;i<8;i++)
		{
			bookmarkslk[i] = vk.ParseVerseList(toUTF8([BookmarkHandler allBookmarks:i]), "Gen1", true);
		}
	}
	
	((sword::VerseKey*)(module->getKey()))->Headings(1);
	
	listkey = vk.ParseVerseList(toUTF8(reference), "Gen1", true);
	
	for (i = 0; i < listkey.Count(); i++)
	{
		sword::VerseKey *element = My_SWDYNAMIC_CAST(VerseKey, listkey.GetElement(i));
		
		// is it a chapter or book - not atomic
		if(element)
		{
			module->Key(element->LowerBound());

			// find the upper bound
			vk = element->UpperBound();
			vk.Headings(true);
		}
		else
		{
			// set it up
			module->Key(*listkey.GetElement(i));
		}
			
		// while not past the upper bound
		do
		{
			
			//add verse index to dictionary
			ctxt = (char *)module->RenderText();
			clen = strlen(ctxt);
			
			//if(embelish)
			{
				int newBook, newChapter, newVerse;
				
				newBook = ((sword::VerseKey)(module->Key())).Book();
				newChapter = ((sword::VerseKey)(module->Key())).Chapter();
				newVerse = ((sword::VerseKey)(module->Key())).Verse();
				
				if(book != newBook)
				{
					book = newBook;
					chapter = -1;
					if(embelish)
						[html appendString:[NSString stringWithFormat:@"<div class=\"booktitle\"><a name=\"%@\">%@</a></div>\n",
							fromUTF8(module->Key().getText()), fromUTF8(((sword::VerseKey)(module->Key())).getBookName())]];
					else
						if(verse != -1) [html appendString:@"<br><br>"];
				}
				if(chapter != newChapter)
				{
					chapter = newChapter;
					verse = newVerse;
					
					if(embelish)
						[html appendString:[NSString stringWithFormat:@"<br><a name=\"%@\" class=\"chapternumber\"  href=\"%@?notip\">%d</a> ",
							fromUTF8(module->Key().getText()), [SwordBible context:fromUTF8(module->Key().getText())], ((sword::VerseKey)(module->Key())).Chapter()]];
					else
						if(verse != -1) [html appendString:@"<br>"];
				}
				
				if(verse < newVerse-1)
					[html appendString:@"<br><br><div>...</div><br>"];
				verse = newVerse;
				
				if(embelish)
				{
					NSMutableString	*hclass = [NSMutableString string];
	
					if(highlightRef)
					{
						bookmarkslk[0] = module->Key().getText();
						
						if(!bookmarkslk[0].Error())
							[hclass appendString:@" highlight"];
					}
					
					if(highlight)
					{
						for(j=1;j<8;j++)
						{
							bookmarkslk[j] = module->Key().getText();
							
							if(!bookmarkslk[j].Error())
								[hclass appendString:[NSString stringWithFormat:@" highlighted%c", 'A'+j-1]];
						}
					}
					
					[html appendString:[NSString stringWithFormat:@"\n<span id=\"%@\" class=\"verse%@\">", fromUTF8(module->Key().getText()), hclass]];
					
					if(clen > 0)
					{
						NSString *verse = fromUTF8(module->Key().getText());
						
						verseNumbering = [NSString stringWithFormat:@"<a name=\"%@\" class=\"versenumber\" href=\"%@?notip\">%d</a> ", 
							verse, verse, 
							((sword::VerseKey)(module->Key())).Verse()];
						//verseNumbering = [NSString stringWithFormat:@"%d ", ((sword::VerseKey)(module->Key())).Verse()];
						[html appendString:verseNumbering];
					}
					
				}
			}
			if(clen > 3)
			{
				if(ctxt[clen-3] == -96) ctxt[clen-3] = 0;
				
				NSString	*string = [NSString stringWithUTF8String:ctxt];
				
				if(string)
					[html appendString:string];
			}
			
			if(clen == 0 && !warnIncomplete)
			{
				[html appendString:NSLocalizedString(@"This module maybe incomplete. Try a different reference.", @"Bible incomplete warning - try a different reference")];
				warnIncomplete = YES;
			}
			else warnIncomplete = YES;
			
			if(clen > 0)
			{
				[html appendString:@"\n"];
			}
			
			//if(embelish)
				[html appendString:@"</span>"];
			
			lastIndex = (module->Key()).Index();
			(*module)++;
			if(lastIndex == (module->Key()).Index())
				break;
		}while (element && module->Key() <= vk);
	}
	
	[html autorelease];
	[moduleLock unlock];
	
	return html;
	}
}

+ (void)parallel:(NSArray*)swordmodules htmlForRef:(NSString *)reference embelish:(BOOL)embelish highlight:(BOOL)highlight highlightReference:(NSString*)highlightRef protocol:(SwordURLProtocol *)protocol
{	
	[[swordmodules collectWithSelector:@selector(moduleLock)] makeObjectsPerformSelector:@selector(lock)]; 

	sword::ListKey		listkey;
	sword::VerseKey		vk;
	int					i, j, lastIndex;
	NSString			*verseNumbering;
	NSMutableString		*html = [[NSMutableString alloc] init];
	int					chapter=-1, book=-1, verse=-1;
	
	//BOOL				warnIncomplete = NO;
	
	int					moduleCount = [swordmodules count];
	sword::SWModule		*modules[moduleCount];
	bool				skipLinks[moduleCount];
	bool				unique[moduleCount];
	char				*ctxt[moduleCount];
	int					clen[moduleCount];
	int					rtol[moduleCount];
	
	for(j=0;j<moduleCount;j++)
	{
		modules[j] = [[swordmodules objectAtIndex:j] swmodule];
	
		skipLinks[j] = modules[j]->getSkipConsecutiveLinks();
		modules[j]->setSkipConsecutiveLinks(false);
		
		if([[swordmodules objectAtIndex:j] getConfigEntry:@"Direction"] && [[[swordmodules objectAtIndex:j] getConfigEntry:@"Direction"] isEqualToString:@"RtoL"])
			rtol[j] = YES;
		else rtol[j] = NO;		
		
				NSLog(@"RTL %d %@", rtol[j], [swordmodules objectAtIndex:j]);

		
		unique[j] = YES;
		int k;
		for(k=j-1; k>=0; k--)
		{
			if(modules[k] == modules[j])
			{
				unique[j] = NO;
			}
		}
	}
	//parse references
/*	if(highlightRef)
		bookmarkslk[0] = vk.ParseVerseList(toUTF8(highlightRef), "Gen1", true);
	
	if(highlight)
	{
		for(i=1;i<8;i++)
		{
			bookmarkslk[i] = vk.ParseVerseList(toUTF8([BookmarkHandler allBookmarks:i]), "Gen1", true);
		}
	}*/
					
	listkey = vk.ParseVerseList(toUTF8(reference), "Gen1", true);
	
	for (i = 0; i < listkey.Count(); i++)
	{
		sword::VerseKey *element = My_SWDYNAMIC_CAST(VerseKey, listkey.GetElement(i));
		
		// is it a chapter or book - not atomic
		if(element)
		{
			// start at lower bound
			//module->Key(element->LowerBound());
			for(j=0;j<moduleCount;j++)
				modules[j]->Key(element->LowerBound());
			// find the upper bound
			vk = element->UpperBound();
			vk.Headings();
		}
		else
		{
			// set it up
			//module->Key(*listkey.GetElement(i));
			for(j=0;j<moduleCount;j++)
				modules[j]->Key(*listkey.GetElement(i));
		}
			
		// while not past the upper bound
		do
		{
			//add verse index to dictionary
			//ctxt = (char *)module->RenderText();
			//clen = strlen(ctxt);
			for(j=0;j<moduleCount;j++)
			{
				ctxt[j] = (char *)modules[j]->RenderText();
				clen[j] = strlen(ctxt[j]);
			}
			
			//[html appendString:@"<tr>"];
			
			//if(embelish)
			{
				int newBook, newChapter, newVerse;
				
				newBook = ((sword::VerseKey)(modules[0]->Key())).Book();
				newChapter = ((sword::VerseKey)(modules[0]->Key())).Chapter();
				newVerse = ((sword::VerseKey)(modules[0]->Key())).Verse();
				
				if(book != newBook)
				{
					book = newBook;
					chapter = -1;
					if(embelish)
						[html appendString:[NSString stringWithFormat:@"<div class=\"booktitle\"><a name=\"%@\">%@</a></div>\n",
							fromUTF8(modules[0]->Key().getText()), fromUTF8(((sword::VerseKey)(modules[0]->Key())).getBookName())]];
					//else
					//	[html appendString:@"<br><br>"];
				}
				if(chapter != newChapter)
				{
					chapter = newChapter;
					verse = newVerse;
					
					if(embelish)
						[html appendString:[NSString stringWithFormat:@"<br><a name=\"%@\" class=\"chapternumber\"  href=\"%@?notip\">%d</a> ",
							fromUTF8(modules[0]->Key().getText()), [SwordBible context:fromUTF8(modules[0]->Key().getText())], ((sword::VerseKey)(modules[0]->Key())).Chapter()]];
				}
				
				if(verse < newVerse-1)
					[html appendString:@"<br><br><div>...</div><br>"];
				verse = newVerse;
				
				/*if(embelish)
				{
					NSMutableString	*hclass = [NSMutableString string];
	
					if(highlightRef)
					{
						bookmarkslk[0] = modules[0]->Key().getText();
						
						if(!bookmarkslk[0].Error())
							[hclass appendString:@" highlight"];
					}
					
					if(highlight)
					{
						for(j=1;j<8;j++)
						{
							bookmarkslk[j] = modules[0]->Key().getText();
							
							if(!bookmarkslk[j].Error())
								[hclass appendString:[NSString stringWithFormat:@" highlighted%c", 'A'+j-1]];
						}
					}
					
					[html appendString:[NSString stringWithFormat:@"\n<span id=\"%@\" class=\"verse%@\">", fromUTF8(modules[0]->Key().getText()), hclass]];
					
					if(clen > 0)
					{
						NSString *verse = fromUTF8(modules[0]->Key().getText());
						
						verseNumbering = [NSString stringWithFormat:@"<a name=\"%@\" class=\"versenumber\" href=\"%@?notip\">%d</a> ", 
							verse, verse, 
							((sword::VerseKey)(modules[0]->Key())).Verse()];
						//verseNumbering = [NSString stringWithFormat:@"%d ", ((sword::VerseKey)(module->Key())).Verse()];
						[html appendString:verseNumbering];
					}
					
				}*/
			}
			
			static int odd=0;
			
			if((odd++)%2)
				[html appendString:@"<table cellspacing=0 cellpadding=5 width=100% style=\"border-bottom: 1px solid #333;\"><tr class=\"odd\">"];
			else
				[html appendString:@"<table cellspacing=0 cellpadding=5 width=100% style=\"border-bottom: 1px solid #333;\"><tr>"];
			
			float width = 100.0/moduleCount;
			
			for(j=0;j<moduleCount;j++)
			{
				NSString	*string = [NSString stringWithUTF8String:ctxt[j]];
				
				[html appendString:[NSString stringWithFormat:@"<td valign=top style=\"width: %.2f%%; %@\">",
						width, j==0? @"" : @"border-left: 1px solid #333;"]];
					
				[html appendString:[NSString stringWithFormat:@"<span class=verse %s>", 
										rtol[j] ? "style=\"display:block; direction:rtl; text-align:right;\"" : ""]];
										
					
					{
						NSString *verse = fromUTF8(modules[0]->Key().getText());
						
						verseNumbering = [NSString stringWithFormat:@"<a name=\"%@\" class=\"versenumber\" href=\"%@?notip\">%d</a> ", 
										verse, verse, 
										((sword::VerseKey)(modules[0]->Key())).Verse()];
						//verseNumbering = [NSString stringWithFormat:@"%d ", ((sword::VerseKey)(module->Key())).Verse()];
						[html appendString:verseNumbering];
					}
					
				if(string)
					[html appendString:string];
					
				[html appendString:@"</span></td>"];
			}
			[html appendString:@"</tr></table>"];
			
			/*if(clen == 0 && !warnIncomplete)
			{
				[html appendString:NSLocalizedString(@"This module maybe incomplete. Try a different reference.", @"Bible incomplete warning - try a different reference")];
				warnIncomplete = YES;
			}
			else warnIncomplete = YES;*/
			
			//if(clen > 0)
			{
				[html appendString:@"\n"];
			}
			
			//if(embelish)
			//	[html appendString:@"</span>"];
			
			if(![protocol recieveData:[html dataUsingEncoding: NSUTF8StringEncoding]])
				break;
			[html deleteCharactersInRange:NSMakeRange(0, [html length])];
			
			for(j=0;j<moduleCount;j++)
			{
				if(unique[j])
				{
					lastIndex = (modules[j]->Key()).Index();
					(*modules[j])++;
					if(lastIndex == (modules[j]->Key()).Index())
						break;
				}
			}
			/*lastIndex = (module->Key()).Index();
			(*module)++;
			if(lastIndex == (module->Key()).Index())
				break;*/
			for(j=0;j<moduleCount;j++)
			{
				if(modules[j]->Key() > vk)
					break;
			}
			if(j<moduleCount) break;
		}while (element);//element && module->Key() <= vk);
	}
	
	[html autorelease];
	
	[[swordmodules collectWithSelector:@selector(moduleLock)] makeObjectsPerformSelector:@selector(unlock)]; 
	
	
	for(j=0;j<moduleCount;j++)
	{
		if(skipLinks[j]);
			modules[j]->setSkipConsecutiveLinks(true);
	}
	
	
	//return html;
	//}
}

+ (NSString *)parallel:(NSArray*)swordmodules htmlForRef:(NSString *)reference embelish:(BOOL)embelish highlight:(BOOL)highlight highlightReference:(NSString*)highlightRef
{	
	[[swordmodules collectWithSelector:@selector(moduleLock)] makeObjectsPerformSelector:@selector(lock)]; 

	sword::ListKey		listkey;
	sword::VerseKey		vk;
	int					i, j, lastIndex;
	NSString			*verseNumbering;
	NSMutableString		*html = [[NSMutableString alloc] init];
	int					chapter=-1, book=-1, verse=-1;
	
	//BOOL				warnIncomplete = NO;
	
	int					moduleCount = [swordmodules count];
	sword::SWModule		*modules[moduleCount];
	bool				skipLinks[moduleCount];
	bool				unique[moduleCount];
	char				*ctxt[moduleCount];
	int					clen[moduleCount];
	
	for(j=0;j<moduleCount;j++)
	{
		modules[j] = [[swordmodules objectAtIndex:j] swmodule];
	
		skipLinks[j] = modules[j]->getSkipConsecutiveLinks();
		modules[j]->setSkipConsecutiveLinks(false);
		
		unique[j] = YES;
		int k;
		for(k=j-1; k>=0; k--)
		{
			if(modules[k] == modules[j])
			{
				unique[j] = NO;
			}
		}
	}
	//parse references
/*	if(highlightRef)
		bookmarkslk[0] = vk.ParseVerseList(toUTF8(highlightRef), "Gen1", true);
	
	if(highlight)
	{
		for(i=1;i<8;i++)
		{
			bookmarkslk[i] = vk.ParseVerseList(toUTF8([BookmarkHandler allBookmarks:i]), "Gen1", true);
		}
	}*/
					
	listkey = vk.ParseVerseList(toUTF8(reference), "Gen1", true);
	
	for (i = 0; i < listkey.Count(); i++)
	{
		sword::VerseKey *element = My_SWDYNAMIC_CAST(VerseKey, listkey.GetElement(i));
		
		// is it a chapter or book - not atomic
		if(element)
		{
			// start at lower bound
			//module->Key(element->LowerBound());
			for(j=0;j<moduleCount;j++)
				modules[j]->Key(element->LowerBound());
			// find the upper bound
			vk = element->UpperBound();
			vk.Headings();
		}
		else
		{
			// set it up
			//module->Key(*listkey.GetElement(i));
			for(j=0;j<moduleCount;j++)
				modules[j]->Key(*listkey.GetElement(i));
		}
			
		// while not past the upper bound
		do
		{
			//add verse index to dictionary
			//ctxt = (char *)module->RenderText();
			//clen = strlen(ctxt);
			for(j=0;j<moduleCount;j++)
			{
				ctxt[j] = (char *)modules[j]->RenderText();
				clen[j] = strlen(ctxt[j]);
			}
			
			//[html appendString:@"<tr>"];
			
			//if(embelish)
			{
				int newBook, newChapter, newVerse;
				
				newBook = ((sword::VerseKey)(modules[0]->Key())).Book();
				newChapter = ((sword::VerseKey)(modules[0]->Key())).Chapter();
				newVerse = ((sword::VerseKey)(modules[0]->Key())).Verse();
				
				if(book != newBook)
				{
					book = newBook;
					chapter = -1;
					if(embelish)
						[html appendString:[NSString stringWithFormat:@"<div class=\"booktitle\"><a name=\"%@\">%@</a></div>\n",
							fromUTF8(modules[0]->Key().getText()), fromUTF8(((sword::VerseKey)(modules[0]->Key())).getBookName())]];
					//else
					//	[html appendString:@"<br><br>"];
				}
				if(chapter != newChapter)
				{
					chapter = newChapter;
					verse = newVerse;
					
					if(embelish)
						[html appendString:[NSString stringWithFormat:@"<br><a name=\"%@\" class=\"chapternumber\"  href=\"%@?notip\">%d</a> ",
							fromUTF8(modules[0]->Key().getText()), [SwordBible context:fromUTF8(modules[0]->Key().getText())], ((sword::VerseKey)(modules[0]->Key())).Chapter()]];
				}
				
				if(verse < newVerse-1)
					[html appendString:@"<br><br><div>...</div><br>"];
				verse = newVerse;
				
				/*if(embelish)
				{
					NSMutableString	*hclass = [NSMutableString string];
	
					if(highlightRef)
					{
						bookmarkslk[0] = modules[0]->Key().getText();
						
						if(!bookmarkslk[0].Error())
							[hclass appendString:@" highlight"];
					}
					
					if(highlight)
					{
						for(j=1;j<8;j++)
						{
							bookmarkslk[j] = modules[0]->Key().getText();
							
							if(!bookmarkslk[j].Error())
								[hclass appendString:[NSString stringWithFormat:@" highlighted%c", 'A'+j-1]];
						}
					}
					
					[html appendString:[NSString stringWithFormat:@"\n<span id=\"%@\" class=\"verse%@\">", fromUTF8(modules[0]->Key().getText()), hclass]];
					
					if(clen > 0)
					{
						NSString *verse = fromUTF8(modules[0]->Key().getText());
						
						verseNumbering = [NSString stringWithFormat:@"<a name=\"%@\" class=\"versenumber\" href=\"%@?notip\">%d</a> ", 
							verse, verse, 
							((sword::VerseKey)(modules[0]->Key())).Verse()];
						//verseNumbering = [NSString stringWithFormat:@"%d ", ((sword::VerseKey)(module->Key())).Verse()];
						[html appendString:verseNumbering];
					}
					
				}*/
			}
			
			static int odd=0;
			
			if((odd++)%2)
				[html appendString:@"<table cellspacing=0 cellpadding=5 width=100% style=\"border-bottom: 1px solid #333;\"><tr class=\"odd\">"];
			else
				[html appendString:@"<table cellspacing=0 cellpadding=5 width=100% style=\"border-bottom: 1px solid #333;\"><tr>"];
			
			float width = 100.0/moduleCount;
			
			for(j = 0;j < moduleCount;j++)
			{
				NSString	*string = [NSString stringWithUTF8String:ctxt[j]];
				
				[html appendString:[NSString stringWithFormat:@"<td valign=top style=\"width: %.2f%%; %@\">",
						width, j==0? @"" : @"border-left: 1px solid #333;"]];
					
					{
						NSString *verse = fromUTF8(modules[0]->Key().getText());
						
						verseNumbering = [NSString stringWithFormat:@"<a name=\"%@\" class=\"versenumber\" href=\"%@?notip\">%d</a> ", 
										verse, verse, 
										((sword::VerseKey)(modules[0]->Key())).Verse()];
						//verseNumbering = [NSString stringWithFormat:@"%d ", ((sword::VerseKey)(module->Key())).Verse()];
						[html appendString:verseNumbering];
					}
					
				if(string)
					[html appendString:string];
				[html appendString:@"</td>"];
			}
			[html appendString:@"</tr></table>"];
			
			/*if(clen == 0 && !warnIncomplete)
			{
				[html appendString:NSLocalizedString(@"This module maybe incomplete. Try a different reference.", @"Bible incomplete warning - try a different reference")];
				warnIncomplete = YES;
			}
			else warnIncomplete = YES;*/
			
			//if(clen > 0)
			{
				[html appendString:@"\n"];
			}
			
			//if(embelish)
			//	[html appendString:@"</span>"];
			
			for(j = 0;j < moduleCount;j++)
			{
				if(unique[j])
				{
					lastIndex = (modules[j]->Key()).Index();
					(*modules[j])++;
					if(lastIndex == (modules[j]->Key()).Index())
						break;
				}
			}
			/*lastIndex = (module->Key()).Index();
			(*module)++;
			if(lastIndex == (module->Key()).Index())
				break;*/
			for(j=0;j<moduleCount;j++)
			{
				if(modules[j]->Key() > vk)
					break;
			}
			if(j<moduleCount) break;
		}while (element);//element && module->Key() <= vk);
	}
	
	[html autorelease];
	
	[[swordmodules collectWithSelector:@selector(moduleLock)] makeObjectsPerformSelector:@selector(unlock)]; 
	
	
	for(j=0;j<moduleCount;j++)
	{
		if(skipLinks[j]);
			modules[j]->setSkipConsecutiveLinks(true);
	}
	
	
	return html;
	//}
}

/*- (void)highlightRef:(NSString *)reference forWebView:(WebView*)web
{
	if(!reference || ![reference length])
	{
		return;
	}
	else
	{

	sword::ListKey		listkey;
	sword::VerseKey		vk;
	int					i, lastIndex;
	
	listkey = vk.ParseVerseList(toLatin1(reference), "Gen1", true);
	for (i = 0; i < listkey.Count(); i++)
	{
		sword::VerseKey *element = My_SWDYNAMIC_CAST(VerseKey, listkey.GetElement(i));
		
		// is it a chapter or book - not atomic
		if(element)
		{
			// start at lower bound
			module->Key(element->LowerBound());
			// find the upper bound
			vk = element->UpperBound();
			vk.Headings();
		}
		else
		{
			// set it up
			module->Key(*listkey.GetElement(i));
		}
			
		// while not past the upper bound
		do
		{
			[web stringByEvaluatingJavaScriptFromString:[NSString stringWithFormat:@"highlight('%s')", (char *)(module->Key())]];
			
			lastIndex = (module->Key()).Index();
			(*module)++;
			if(lastIndex == (module->Key()).Index())
				break;
		}while (element && module->Key() <= vk);
	}
	}
}*/

#pragma mark -

- (void)writeEntry:(NSString*)value forRef:(NSString*)reference
{
	[moduleLock lock];
	
	sword::ListKey		listkey;
	int					i, lastIndex;
	bool				havefirst = false;
	sword::VerseKey		firstverse;
	sword::VerseKey		vk;
	
	listkey = vk.ParseVerseList(toUTF8(reference), "Gen1:1", true);
	
	for (i = 0; i < listkey.Count(); i++)
	{
		sword::VerseKey *element = My_SWDYNAMIC_CAST(VerseKey, listkey.GetElement(i));
		
		// is it a chapter or book - not atomic
		if(element)
		{
			// start at lower bound
			module->Key(element->LowerBound());
			// find the upper bound
			vk = element->UpperBound();
			vk.Headings();
		}
		else
		{
			// set it up
			module->Key(*listkey.GetElement(i));
		}
			
		// while not past the upper bound
		do
		{
			if (!havefirst)
			{
				havefirst = true;
				firstverse = module->Key();
				
				const char	*data;
				int		dLen;
				
				data = toUTF8(value);
				dLen = strlen(data);

				module->setEntry(data, dLen);	// save text to module at current position
			}
			else
			{
				*(sword::SWModule*)module << &firstverse;
			}
			
			lastIndex = (module->Key()).Index();
			(*module)++;
			if(lastIndex == (module->Key()).Index())
				break;
		}while (element && module->Key() <= vk);
	}
	
	[moduleLock unlock];
}

@end
